/*
    Copyright (C) 2003  Anthony R. Jansen.

    This file is part of the CIDER Toolkit.

    The CIDER Toolkit is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    Foobar is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Foobar; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

// ----------------------------------------
// CIDER Toolkit: Compiler Main Method
// 
// Author:  Anthony R. Jansen
// Begun:   June 2003
// Class:   CiderCompiler
//
// This class processes command line
// parameters for the CIDER compiler.
// ----------------------------------------

import au.edu.monash.csse.tonyj.cider.compiler.*;

public class CiderCompiler {

	// Prints out command line errors
	private static boolean commandLineError(String error)
	{
		System.err.println("Command Line Error: " + error);
		return true;
	}

	// Prints out the standard preamble
	private static void printPreamble()
	{
		System.out.println("");
		System.out.println("CIDER Compiler (Version " + au.edu.monash.csse.tonyj.cider.compiler.Compiler.VERSION + ")");
		System.out.println("Copyright (C) 2003-2004  Anthony R. Jansen");
		System.out.println("CIDER Compiler comes with ABSOLUTELY NO WARRANTY.");
		System.out.println("This is free software, and you are welcome to redistribute it");
		System.out.println("under the terms and conditions of the GNU General Public License.");
		System.out.println("This product includes software developed by the");
		System.out.println("Apache Software Foundation (http://www.apache.org/).");
		System.out.println("");
	}

	// Prints out the command line options
	private static void printHelp()
	{
		System.out.println("Command line options:");
		System.out.println("     -h                 this help");
		System.out.println("     -g [name]          grammar file          (CIDER format)");
		System.out.println("     -t [name]          transformations file  (CIDER format)");
		System.out.println("     -gx [name]         grammar file          (XML format)");
		System.out.println("     -tx [name]         transformations file  (XML format)");
		System.out.println("     -translate_only    translate to XML format but do not compile");
		System.out.println("     -w                 suppress compiler warnings");
		System.out.println("");
		System.out.println("Note: All files in CIDER format will be translated into XML format.");
		System.out.println("");
	}

	// Main method
	public static void main(String argv[])
	{
		boolean errorFlag = false;
		boolean translateOnly = false;
		String g = null;
		String t = null;
		String gx = null;
		String tx = null;

		// Always print preamble
		printPreamble();

		// See if help is called for (or no arguments are given) 
		if (argv.length == 0) {
			printHelp();
			System.exit(1);
		}
		for (int i = 0; i < argv.length; i++) {
			if (argv[i].equals("-h") || argv[i].equals("-help")) {

				// If help requested, print help list
				printHelp();
				System.exit(1);
			}
		}

		// Read in all arguments, checking for duplications
		try {
			for (int i = 0; i < argv.length; i++) {

				if (argv[i].equals("-translate_only")) {
					if (translateOnly)
						errorFlag = commandLineError("Option \"-translate_only\" specified more than once");
					translateOnly = true;
				}
				else if (argv[i].equals("-g")) {
					if (g != null)
						errorFlag = commandLineError("Option \"-g\" specified more than once");
					g = argv[++i];
				}
				else if (argv[i].equals("-t")) {
					if (t != null)
						errorFlag = commandLineError("Option \"-t\" specified more than once");
					t = argv[++i];
				}
				else if (argv[i].equals("-gx")) {
					if (gx != null)
						errorFlag = commandLineError("Option \"-gx\" specified more than once");
					gx = argv[++i];
				}
				else if (argv[i].equals("-tx")) {
					if (tx != null)
						errorFlag = commandLineError("Option \"-tx\" specified more than once");
					tx = argv[++i];
				}
				else if (argv[i].equals("-w")) 
					au.edu.monash.csse.tonyj.cider.compiler.Error.warningsOn = false;
				else
					errorFlag = commandLineError("Unrecognized option \"" + argv[i] + "\"");
			}
		}
		catch (ArrayIndexOutOfBoundsException e) {
			errorFlag = commandLineError("Value missing for last option \"" + argv[argv.length-1] + "\"");
		}

		// Check for illegal pairings
		if (g != null && gx != null)
			errorFlag = commandLineError("Options \"-g\" and \"-gx\" cannot both be specified");
		if (t != null && tx != null)
			errorFlag = commandLineError("Options \"-t\" and \"-tx\" cannot both be specified");
		if (translateOnly && gx != null)
			errorFlag = commandLineError("Option \"-gx\" cannot be used with option \"-translate_only\"");
		if (translateOnly && tx != null)
			errorFlag = commandLineError("Option \"-tx\" cannot be used with option \"-translate_only\"");
		if (translateOnly && g == null && t == null)
			errorFlag = commandLineError("Option \"-translate_only\" must be used with option \"-g\" and/or option \"-t\"");
		if (!translateOnly && g == null && gx == null)
			errorFlag = commandLineError("Option \"-g\" or option \"-gx\" must be used when compiling");

		// Proceed no further if error flag is active
		if (errorFlag) {
			System.err.println("");
			System.exit(1);
		}

		// Carry out translations if needed
		if (g != null) {
			gx = g + ".xml";
			GrammarConverter gConverter = new GrammarConverter(g, gx);
			gConverter.convert();
			System.out.println("Grammar file \"" + g + "\" has been converted to \"" + gx + "\"");
		}
		if (t != null) {
			tx = t + ".xml";
			TransformationConverter tConverter = new TransformationConverter(t, tx);
			tConverter.convert();
			System.out.println("Transformation file \"" + t + "\" has been converted to \"" + tx + "\"");
		}

		// Compile if appropriate
		if (!translateOnly) {
			au.edu.monash.csse.tonyj.cider.compiler.Compiler compiler = 
				new au.edu.monash.csse.tonyj.cider.compiler.Compiler(gx, tx);
			System.out.println("");
			System.out.println("CIDER Compiler successfully finished.");
			System.out.println("");
		}
	}
}

