/*
    Copyright (C) 2003  Anthony R. Jansen.

    This file is part of the CIDER Toolkit.

    The CIDER Toolkit is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    The CIDER Toolkit is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with the CIDER Toolkit; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

// ----------------------------------------
// CIDER Toolkit: Compiler Package
//
// Author:  Anthony R. Jansen
// Begun:   January 2003
// Class:   AttDependencyGraph
//
// This class implements a dependency
// graph for the attributes in the language 
// given to the compiler. It allows for the
// grammar validity to be checked.
// ----------------------------------------

package au.edu.monash.csse.tonyj.cider.compiler;

import java.util.Set;
import java.util.HashSet;
import java.util.List;
import java.util.LinkedList;
import java.util.Iterator;
import java.io.PrintStream;

public class AttDependencyGraph {

	// Private variables
	private List	nodes;
	private List	symbolDefinitionList;

	// Constructor, which creates a node for every non-abstract symbol type
	public AttDependencyGraph(List symbolDefs)
	{
		symbolDefinitionList = symbolDefs;
		nodes = new LinkedList();
		Iterator iter = symbolDefs.iterator();
		while (iter.hasNext()) {
			SymbolDefinition sDef = (SymbolDefinition) iter.next();
			if (!sDef.isAbstract()) {

				// Get all attributes
				Iterator iter2 = sDef.getAllAttributes().iterator();
				while (iter2.hasNext())
					nodes.add(new AttDependencyGraphNode(sDef, (Attribute) iter2.next()));
			}
		}
	}

	// Returns the node corresponding to a given name
	private AttDependencyGraphNode getNode(String name)
	{
		Iterator iter = nodes.iterator();
		while (iter.hasNext()) {
			AttDependencyGraphNode node = (AttDependencyGraphNode) iter.next();
			if (node.getName().equals(name)) 
				return node;
		}
		return null;
	}

	// Returns the set of nodes corresponding to a given symbol definition
	// This is the node for either the attribute, or if it is from an abstract symbol, 
	// the attributes from the non-abstract symbols that extend it
	private Set getNodes(String name)
	{
		Set toBeReplaced = new HashSet();

		// Parse name
		int dotIndex = name.indexOf('.');
		if (dotIndex == -1)
			Error.die(this, "Invalid attribute reference format: dot missing in: " + name);
		String symbolName = name.substring(0, dotIndex);
		String attName = name.substring(dotIndex + 1);

		// Get symbol definition
		SymbolDefinition type = null;
		Iterator iter3 = symbolDefinitionList.iterator();
		while (iter3.hasNext()) {
			SymbolDefinition sDef = (SymbolDefinition) iter3.next();
			if (sDef.getName().equals(symbolName)) {
				type = sDef;
				break;
			}
		}	
		if (type == null)
			Error.die(this, "Could not find symbol definition: " + symbolName);

		// Start with a set containing original symbol type
		Set symbolTypes = new HashSet();
		symbolTypes.add(type);

		do {
			// Look for abstract types that need to be replaced
			toBeReplaced.clear();
			Iterator iter = symbolTypes.iterator();
			while (iter.hasNext()) {
				SymbolDefinition sDef = (SymbolDefinition) iter.next();
				if (sDef.isAbstract())
					toBeReplaced.add(sDef);
			}

			// Add any symbols that extend the symbols to be replaced
			iter = toBeReplaced.iterator();
			while (iter.hasNext()) {
				SymbolDefinition s1 = (SymbolDefinition) iter.next();

				Iterator iter2 = symbolDefinitionList.iterator();
				while (iter2.hasNext()) {
					SymbolDefinition s2 = (SymbolDefinition) iter2.next();

					if (s2.getExtendsName().equals(s1.getName()))
						symbolTypes.add(s2);
				}
			}

			// Remove the symbols to be replaced
			symbolTypes.removeAll(toBeReplaced);

		} while (!toBeReplaced.isEmpty());

		// Return the nodes corresponding to the symbol types
		Set nodeSet = new HashSet();
		iter3 = symbolTypes.iterator();
		while (iter3.hasNext()) {
			SymbolDefinition sDef = (SymbolDefinition) iter3.next();
			nodeSet.add(getNode(sDef.getName() + "." + attName));
		}
		return nodeSet;
	}

	// Add a dependecy
	public void addDependency(String att, String setBy, int from)
	{
		Iterator iter = getNodes(att).iterator();
		while (iter.hasNext()) {
			AttDependencyGraphNode attNode = (AttDependencyGraphNode) iter.next();
			Iterator iter2 = getNodes(setBy).iterator();
			while (iter2.hasNext()) {
				AttDependencyGraphNode setByNode = (AttDependencyGraphNode) iter2.next();
				attNode.isSetBy(setByNode, from);
			}
		}
	}

	// This method checks to make sure that the dependencies do not cause any dependency loops that 
	// might lead to infinite loops when setting the attributes
	public void checkDependencies()
	{
		Iterator iter = nodes.iterator();
		while (iter.hasNext()) {
			AttDependencyGraphNode node1 = (AttDependencyGraphNode) iter.next();

			// Check each dependency
			Iterator iter2 = node1.getSetBy().iterator();
			while (iter2.hasNext()) {
				AttDependencyGraphNode node2 = (AttDependencyGraphNode) iter2.next();

				if (isDependent(node2, node1))
					Error.warn(this, "Potentially dangerous dependency loop between attributes \"" + node1.getName() +
						"\" and \"" + node2.getName() + "\"");
			}
		}
	}

	// This method returns true if node 1 is dependent on node 2
	private boolean isDependent(AttDependencyGraphNode node1, AttDependencyGraphNode node2)
	{
		int oldSize;
		Set dependentNodes = new HashSet();
		Set tempSet = new HashSet();

		// Keep looping as long as there are more dependencies to add
		dependentNodes.add(node1);
		do {
			oldSize = dependentNodes.size();
			tempSet.clear();

			// Get all nodes that the current nodes immediately depend on
			Iterator iter = dependentNodes.iterator();
			while (iter.hasNext()) {
				AttDependencyGraphNode node = (AttDependencyGraphNode) iter.next();
				tempSet.addAll(node.getSetBy());
			}
			dependentNodes.addAll(tempSet);

			// See if node 2 is one of the dependent nodes
			if (dependentNodes.contains(node2))
				return true;

		} while (oldSize < dependentNodes.size());

		return false;
	}

	// Debug method
	public void printGraph(PrintStream stream)
	{
		stream.println("*** Attribute Dependency Graph ***");
		stream.println("");

		// Go through all nodes
		Iterator iter1 = nodes.iterator();
		while (iter1.hasNext()) {
			AttDependencyGraphNode node1 = (AttDependencyGraphNode) iter1.next();
			stream.println(node1.getName() + ":");
			stream.print("\tDependencies:");
			Iterator iter2 = node1.getSetBy().iterator();
			while (iter2.hasNext()) {
				AttDependencyGraphNode node2 = (AttDependencyGraphNode) iter2.next();
				stream.print("  " + node2.getSymbolDefinition().getName());
			}
			stream.print("\n\n");
		}
	}

	// Returns true iff specified attribute is set from nowhere 
	// If attribute of an abstract symbol, all extensions of it must be set from nowhere
	public boolean isTerminal(String name)
	{
		Iterator iter = getNodes(name).iterator();
		while (iter.hasNext()) {
			AttDependencyGraphNode node = (AttDependencyGraphNode) iter.next();;
			if (node.getSetFrom() != AttDependencyGraphNode.NOWHERE)
				return false;
		}
		return true;
	}
}

