/*
    Copyright (C) 2003  Anthony R. Jansen.

    This file is part of the CIDER Toolkit.

    The CIDER Toolkit is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    The CIDER Toolkit is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with the CIDER Toolkit; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

// ----------------------------------------
// CIDER Toolkit: Compiler Package 
//
// Author:  Anthony R. Jansen
// Begun:   June 2003
// Class:   BaseConverter
//
// This class is the abstract base class
// used for converting a cider input file 
// into an XML format.
// ----------------------------------------

package au.edu.monash.csse.tonyj.cider.compiler;

import java.io.IOException;
import java.io.PrintWriter;
import java.io.FileWriter;

abstract class BaseConverter {

	// Protected variables
	protected TextInputFile		input;
	protected PrintWriter		output;

	// Constructor
	public BaseConverter(String inputFilename, String outputFilename)
	{
		input = new TextInputFile(inputFilename);
		try {
			output = new PrintWriterNL(new FileWriter(outputFilename), true);
		}
		catch(IOException e) {
			Error.die(this, "Could not create the file: " + outputFilename);
		}
	}

	// Method to perform the conversion 
	public abstract void convert();

	// Calls the Error.die method, giving the location in the input file of the problem
	protected void InputError(String string)
	{
		Error.die(this, string + ": line " + input.getLineNumber() + " of file " + input.getFilename());
	}

	// Adds a comment to the XML file indicating the current input file line number
	protected void addLineComment()
	{
		output.println("\t <!-- Corresponding position in input file \"" + input.getFilename() + "\": line " + 
			input.getLineNumber() + " -->");
	}

	// Reads in a string of form: name = "value";
	protected String readEqualsString(String name)
	{
		if (!input.readNextToken(name))
			InputError("Expecting \"" + name + "\" to be defined");
		if (!input.readNextToken("="))
			InputError("Expecting = to follow \"" + name + "\"");
		String value= input.readQuoteString();
		if (!input.readNextToken(";"))
			InputError("Expecting \";\" to follow \"" + name + "\" value");
		return value;
	}
	
	// Processes a symbol or symbolset declaration: name:type
	protected void processSymbol(String tabs)
	{
		String name = input.readStandardToken();
		if (!input.readNextToken(":"))
				InputError("Expecting : to follow a symbol name declaration");
		String type = input.readStandardToken();
		
		if (input.readNextToken("*"))
			output.println(tabs + "<SymbolSet type=\"" + type + "\" name=\"" + name + "\" />");
		else
			output.println(tabs + "<Symbol type=\"" + type + "\" name=\"" + name + "\" />");
	}

	// Processes a constraint set (either the main one, or one for an existential symbol)
	protected void processConstraintSet(String tabs)
	{
		if (!input.readNextToken("where")) 
			InputError("Expecting a \"where\" declaration");
			
		if (input.readNextToken("true")) {	
			if (!input.readNextToken(";"))
				InputError("Expecting ; to follow a \"where true\" declaration");
			return;
		}

		if (!input.readNextToken("{"))
			InputError("Expecting { or \"true\" to follow \"where\"");
		output.println(tabs + "<ConstraintSet>");

		// Read in individual constraints
		while (!input.readNextToken("}")) {
			if (input.isNextToken(TextInputFile.EOF))
				InputError("EOF reached in the \"where\" declaration");
			processConstraint(tabs + "\t");
		}
		output.println(tabs + "</ConstraintSet>");
		addLineComment();
	}

	// Processes a constraint (solver or normal)
	protected void processConstraint(String tabs)
	{
		// Deal with a solver constraint
		if (input.readNextToken("@sc")) {
			String line = tabs + "<SolverConstraint ";

			// Read the weight if present
			if (input.readNextToken("[")) {
				line += "weight=\"" + input.readStandardToken() + "\" ";
				if (!input.readNextToken("]"))
					InputError("Expecting ] to close the solver constraint weight");
			}

			// Read in the tolerance
			if (!input.readNextToken("("))
				InputError("Expecting ( to follow a \"@sc\" declaration");
			line += "tolerance=\"" + input.readStandardToken() + "\" >";
			if (!input.readNextToken(","))
				InputError("Expecting , to follow the solver constraint tolerance value");
			
			output.print(line);
			output.print(input.readQuoteString());
			output.println("</SolverConstraint>");

			if (!input.readNextToken(")"))
				InputError("Expecting ) to close a \"@sc\" declaration");
			if (!input.readNextToken(";"))
				InputError("Expecting ; at the end of a constraint declaration");
		}
		else {
			output.println(tabs + "<Constraint>");

			// Check for exists and not exists constraints
			if (input.readNextToken("exists")) {
				output.println(tabs + "\t<Exists>");
				processSymbol(tabs + "\t\t");
				if (input.isNextToken(","))
					InputError("Only one symbol can be specified per \"exists\" declaration");
				processConstraintSet(tabs + "\t\t");
				output.println(tabs + "\t</Exists>");
			}
			else if (input.readNextToken("not_exists")) {
				output.println(tabs + "\t<NotExists>");
				processSymbol(tabs + "\t\t");
				if (input.isNextToken(","))
					InputError("Only one symbol can be specified per \"not_exists\" declaration");
				processConstraintSet(tabs + "\t\t");
				output.println(tabs + "\t</NotExists>");
			}

			// All other constraints should of the form: Argument Relationship Argument
			else {
				String arg1 = getArgument(tabs + "\t\t");
				String rel = "";

				// Process relationship
				if (input.readNextToken("==")) 
					rel = "EQ";
				else if (input.readNextToken("!=")) 
					rel = "NEQ";
				else if (input.readNextToken(">=")) 
					rel = "GEQ";
				else if (input.readNextToken("<=")) 
					rel = "LEQ";
				else if (input.readNextToken(">")) 
					rel = "GT";
				else if (input.readNextToken("<")) 
					rel = "LT";
				else if (input.readNextToken("eq")) 
					rel = "ObjectEQ";
				else if (input.readNextToken("ne")) 
					rel = "ObjectNEQ";

				output.println(tabs + "\t<" + rel + ">");
				output.println(arg1);
				output.println(getArgument(tabs + "\t\t"));
				output.println(tabs + "\t</" + rel + ">");

				if (!input.readNextToken(";"))
					InputError("Expecting ; at the end of a constraint declaration");
			}
			output.println(tabs + "</Constraint>");
		}
		addLineComment();
	}

	// Process an argument which can be either an attribute value, a function call or a constant
	protected String getArgument(String tabs)
	{
		String string = "";

		// See if a character constant is present
		if (input.isNextToken("\'"))		
			return tabs + "<ConstantValue value=\"\'" + input.readQuoteString() + "\'\" />";

		// See if a string constant is present
		if (input.isNextToken("\""))		
			return tabs + "<ConstantValue value=\'\"" + input.readQuoteString() + "\"\' />";

		String token = input.readStandardToken();

		// Check if it is a boolean constant or null value
		if (token.equals("true") || token.equals("false") || token.equals("null"))
			return tabs + "<ConstantValue value=\"" + token + "\" />";

		// Check if it is a valid constant number
		try {
			Double d = Double.valueOf(token);
			return tabs + "<ConstantValue value=\"" + token + "\" />";
		}
		catch (NumberFormatException e) { ; }

		// Check if a literal java string has been defined
		if (token.equals("@java")) {
			if (!input.readNextToken("("))
				InputError("Expecting ( to follow a \"@java\" declaration");

			if (input.isNextToken("\'"))
				string = tabs + "<ConstantValue value=\"" + input.readQuoteString() + "\" />";
			else if (input.isNextToken("\""))
				string = tabs + "<ConstantValue value=\'" + input.readQuoteString() + "\' />";
			else
				InputError("Expecting a string inside quotation marks inside a \"@java\" declaration");

			if (!input.readNextToken(")"))
				InputError("Expecting ) at the end of a \"@java\" declaration");
			return string;
		}

		// Assume now the token is either an attribute value or a function name
		// Check if it is a function
		if (input.readNextToken("(")) {
			string = tabs + "<FunctionValue name=\"" + token + "\" >\n";

			// Read in arguments
			if (!input.isNextToken(")"))
				string += getArgument(tabs + "\t") + "\n";
			while (!input.readNextToken(")")) {
				if (!input.readNextToken(","))
					InputError("Expecting a , between function arguments");
				string += getArgument(tabs + "\t") + "\n";
			}
			string += tabs + "</FunctionValue>";
			return string;
		}

		// If it doesn't contain a . it must be a symbol value
		if (token.indexOf('.') == -1)
			return tabs + "<SymbolValue name=\"" + token + "\" />";

		// If we end up here, must be an attribute value
		return tabs + "<AttributeValue name=\"" + token + "\" />";
	}

	// Process a constraint expression
	protected void processConstraintExpression(String tabs)
	{
		if (!input.readNextToken("@ce"))
			InputError("Expecting \"@ce\" to define a constraint expression");
		if (!input.readNextToken("("))
			InputError("Expecting ( to follow a \"@ce\" declaration");

		output.print(tabs + "<ConstraintExpression>");
		output.print(input.readQuoteString());
		output.println("</ConstraintExpression>");

		if (!input.readNextToken(")"))
			InputError("Expecting ) to conclude a \"@ce\" declaration");
	}

	// Close file before class is destroyed
	public void finalize()
	{
		if (output != null)
			output.close();
	}
}

