/*
    Copyright (C) 2003  Anthony R. Jansen.

    This file is part of the CIDER Toolkit.

    The CIDER Toolkit is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    The CIDER Toolkit is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with the CIDER Toolkit; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

// ----------------------------------------
// CIDER Toolkit: Compiler Package 
// 
// Author:  Anthony R. Jansen
// Begun:   September 2002
// Class:   Compiler
//
// This class is the primary compiler class
// that is used to read in an XML document
// containing the specification for a
// diagrammatic language, and then create 
// the classes that will be used by an IDE
// interpreter.
// ----------------------------------------

package au.edu.monash.csse.tonyj.cider.compiler;

import org.apache.xerces.parsers.DOMParser;
import org.w3c.dom.DocumentType;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.w3c.dom.Element;
import org.w3c.dom.Document;
import org.w3c.dom.DOMException;
import org.xml.sax.SAXException;
import org.xml.sax.SAXParseException;
import org.xml.sax.ErrorHandler;
import java.util.List;
import java.util.LinkedList;
import java.util.Iterator;
import java.util.Set;
import java.util.HashSet;
import java.io.IOException;
import java.io.File;
import java.io.FileWriter;
import java.io.PrintWriter;
import au.edu.monash.csse.tonyj.cider.constraints.ConstraintCompiler;

public class Compiler implements ErrorHandler {

	// Grammar definitions
	private String		grammarName;
	private List 		symbolDefinitionList;
	private List 		productionList;
	private List 		transformationList;
	private List 		disjunctionList;
	private List 		regExpList;

	// Constraint classes (using QOCA)
	private static final String	CONSTRAINT_SOLVER_CLASS = "au.edu.monash.csse.tonyj.cider.qoca.QocaConstraintSolver";
	private static final String	CONSTRAINT_CLASS = "au.edu.monash.csse.tonyj.cider.qoca.QocaConstraint";
	private static final String	CONSTRAINT_EXPRESSION_CLASS = "au.edu.monash.csse.tonyj.cider.qoca.QocaConstraintExpression";
	private static final String	CONSTRAINT_VARIABLE_CLASS = "au.edu.monash.csse.tonyj.cider.qoca.QocaConstraintVariable";
	private static final String	CONSTRAINT_COMPILER_CLASS = "au.edu.monash.csse.tonyj.cider.qoca.QocaConstraintCompiler";

	// Constraint compiler
	private ConstraintCompiler	constraintCompiler;

	// XML elements
	private Element		symbolDefinitionsElement;
	private Element		productionRulesElement;
	private Element		transformationRulesElement;

	// Dependency Graphs
	private DependencyGraph		dependencyGraph;
	private AttDependencyGraph	attDependencyGraph;

	// Error flag
	private boolean 	parserError;

	// Constants
	public static final String	PACKAGE_DIRECTORY = "au" + File.separator + "edu" + File.separator + "monash" + File.separator + "csse" + 
						File.separator + "tonyj" + File.separator + "cider" + File.separator + "interpreter" + File.separator;
	public static final String	PACKAGE_LINE = "package au.edu.monash.csse.tonyj.cider.interpreter;";
	public static final String	ORIGIN_LINE = "// This file was automatically generated by the CIDER Compiler";
	public static final String	VERSION = "0.3.0";

	// Default Constructor
	public Compiler() { ; }

	// Constructor
	public Compiler(String xmlGrammarFile, String xmlTransformationsFile)
	{
		// Make directory for output files (if needed)
		File outputDir = new File(PACKAGE_DIRECTORY);
		if (!outputDir.exists()) {
			if (!outputDir.mkdirs())
				Error.die(this, "Could not create output directory: " + PACKAGE_DIRECTORY);
		}

		// Create constraint compiler 
		try {
			constraintCompiler = (ConstraintCompiler) Class.forName(CONSTRAINT_COMPILER_CLASS).newInstance();
		}
		catch (Exception e) {
			Error.die(this, "Could not create instance of constraint compiler: " + CONSTRAINT_COMPILER_CLASS);
		}

		//  Create an Xerces DOM Parser
		parserError = false;
		symbolDefinitionList = new LinkedList();
		productionList = new LinkedList();
		transformationList = new LinkedList();
		disjunctionList = new LinkedList();
		regExpList = new LinkedList();

		try {
			Node node = parseXMLFile(xmlGrammarFile, "Grammar");

			// Get grammar name
			grammarName = ((Element) node).getAttribute("name");
			if (!isValidNameString(grammarName))
				Error.die(this, "Invalid name given in <Grammar> tag");

			// Get children of the root node
			symbolDefinitionsElement = null;
			productionRulesElement = null;
			NodeList children = node.getChildNodes();
			for (int i = 0; i < children.getLength(); i++) {
				node = children.item(i);
				if (node.getNodeName().equals("SymbolDefinitions"))
					symbolDefinitionsElement = (Element) node;
				else if (node.getNodeName().equals("ProductionRules"))
					productionRulesElement = (Element) node;
			}

			// Read in the transformations xml file if it is specified
			if (xmlTransformationsFile != null) 
				transformationRulesElement = (Element) parseXMLFile(xmlTransformationsFile, "Transformations");

			// Extract and check the symbol definitions
			createSymbolDefinitionsList();
			checkSymbolDefinitionDependencies();
			updateSymbolInheritedAttributes();

			// Initialize the dependency graphs
			dependencyGraph = new DependencyGraph(symbolDefinitionList);
			attDependencyGraph = new AttDependencyGraph(symbolDefinitionList);

			// Extract and check the production definitions
			checkSymbolNames(productionRulesElement, "Symbol", "name");
			createProductionList();
			dependencyGraph.checkDependencies();
			attDependencyGraph.checkDependencies();
			//attDependencyGraph.printGraph(System.out);

			// Extract and check the transformation definitions (if needed)
			if (transformationRulesElement != null) {
				checkSymbolNames(transformationRulesElement, "Symbol", "name");
				checkSymbolNames(transformationRulesElement, "NewTerminalSymbol", "name");
				checkSymbolNames(transformationRulesElement, "OutputSymbol", "label");
				createTransformationLists();
			}

			// Create the symbol classes
			for (int i = 0; i < symbolDefinitionList.size(); i++) 
				((SymbolDefinition) symbolDefinitionList.get(i)).createSymbolClass(CONSTRAINT_VARIABLE_CLASS, attDependencyGraph);
			System.out.println("Symbol classes created");

			// Create the symbol sets class
			createSymbolNodeSetsClass();
			System.out.println("SymbolNodeSets class created");

			// Create production related classes
			createProductionClasses();
			System.out.println("Production related classes created");

			// Create transfromation related classes (if needed)
			if (transformationRulesElement != null) {
				createTransformationClasses();
				System.out.println("Transformation related classes created");
			}
			else
				System.out.println("No transformation classes to create");

			// Create interpreter class
			createInterpreterClass();
			System.out.println("Interpreter class created");
		} 
		catch (SAXException e) {
			Error.die(this, e.toString());
		} 
		catch (IOException e) {
			Error.die(this, e.toString());
		}
	}

	// This method parses an xml file, confirming the input type,
	// eliminating non-element nodes, and returning the first child of the root element node
	private Node parseXMLFile(String filename, String inputType) throws IOException, SAXException
	{
		DOMParserLN parser = new DOMParserLN();

		// Set up parser validation
		parser.setFeature("http://xml.org/sax/features/validation", true);
		parser.setIncludeIgnorableWhitespace(false);
		parser.setErrorHandler(this);

		//  Parse the Document
		File file = new File(filename);
		parser.parse(file.toURI().toString());
		if (parserError)
			Error.die(this, "Too many errors and warnings when parsing the XML grammar document");

		Document document = parser.getDocument();
		Node node = document;

		// Check that correct DTD is used
		boolean flag = false;
		NodeList children = node.getChildNodes();
		for (int i = 0; i < children.getLength(); i++) {
			if (children.item(i).getNodeType() == Node.DOCUMENT_TYPE_NODE) {
				node = children.item(i);
				flag = true;
				break;
			}
		}
		if (!flag || !((DocumentType) node).getSystemId().equals("Cider-" + VERSION + ".dtd"))
			Error.die(this, "XML document \"" + filename + "\" must utilize the Cider-" + VERSION + " DTD");

		// Get root node
		for (int i = 0; i < children.getLength(); i++) {
			if (children.item(i).getNodeType() == Node.ELEMENT_NODE) {
				node = children.item(i);
				break;
			}
		}

		// Eliminate all unneeded non-element nodes
		eliminateNonElementNodes((Element) node);

		// Check that the first child of this node corresponds with the input type
		node = node.getFirstChild();
		if (!(node.getNodeType() == Node.ELEMENT_NODE && node.getNodeName().equals(inputType)))
			Error.die(this, "Expecting \"" + filename + "\" to contain specifications for the " + inputType);

		// Add layout symbols if a grammar file
		if (inputType.equals("Grammar")) {
			try {
				NodeList nodeList = ((Element) node).getElementsByTagName("SymbolDefinitions");
				Element symbolDefsElement = (Element) nodeList.item(0);
				addLayoutSymbols(document, symbolDefsElement);
			}
			catch (DOMException dome) {
				Error.die(this, "Could not add pre-defined GraphNode and GraphEdge symbol types: " + dome.toString());
			}
		}

		// Add <Variable> nodes for all solver constraint variables used
		processConstraintVariables(document, (Element) node);

		return node;
	}

	// Adds the GraphNode and GraphEdge abstract symbols to the DOM tree
	private void addLayoutSymbols(Document document, Element symbolDefsElement) throws DOMException
	{
		// Check that GraphNode and GraphEdge do not already exist
		NodeList nodeList = symbolDefsElement.getElementsByTagName("SymbolDef");
		for (int i = 0; i < nodeList.getLength(); i++) {
			Element element = (Element) nodeList.item(i);
			if (element.getAttribute("name").equals("GraphNode"))
				Error.die(this, "GraphNode is a reserved symbol type: it cannot be defined by the user");
			if (element.getAttribute("name").equals("GraphEdge"))
				Error.die(this, "GraphEdge is a reserved symbol type: it cannot be defined by the user");
		}

		// Create GraphNode element
		Element graphNodeElement = document.createElement("SymbolDef");
		graphNodeElement.setAttribute("name", "GraphNode");
		graphNodeElement.setAttribute("interface", "au.edu.monash.csse.tonyj.cider.layout.GraphNodeInt");
		graphNodeElement.setAttribute("type", "abstract");
		graphNodeElement.appendChild(createLayoutSymbolAttribute(document, "mid_x", "ConstraintVariable"));
		graphNodeElement.appendChild(createLayoutSymbolAttribute(document, "mid_y", "ConstraintVariable"));
		graphNodeElement.appendChild(createLayoutSymbolAttribute(document, "width", "double"));
		graphNodeElement.appendChild(createLayoutSymbolAttribute(document, "height", "double"));

		// Create GraphEdge element
		Element graphEdgeElement = document.createElement("SymbolDef");
		graphEdgeElement.setAttribute("name", "GraphEdge");
		graphEdgeElement.setAttribute("interface", "au.edu.monash.csse.tonyj.cider.layout.GraphEdgeInt");
		graphEdgeElement.setAttribute("type", "abstract");
		graphEdgeElement.appendChild(createLayoutSymbolAttribute(document, "mid_x", "ConstraintVariable"));
		graphEdgeElement.appendChild(createLayoutSymbolAttribute(document, "mid_y", "ConstraintVariable"));
		graphEdgeElement.appendChild(createLayoutSymbolAttribute(document, "node1", "au.edu.monash.csse.tonyj.cider.layout.GraphNodeInt"));
		graphEdgeElement.appendChild(createLayoutSymbolAttribute(document, "node2", "au.edu.monash.csse.tonyj.cider.layout.GraphNodeInt"));

		// Add new elements to the parent
		symbolDefsElement.appendChild(graphNodeElement);
		symbolDefsElement.appendChild(graphEdgeElement);
	}

	// Add an attribute to a layout symbol (with default weights)
	private Element createLayoutSymbolAttribute(Document document, String name, String datatype) throws DOMException
	{
		Element attElement = document.createElement("Attribute");
		attElement.setAttribute("name", name);
		attElement.setAttribute("datatype", datatype);
		attElement.setAttribute("stayweight", "10.0");
		attElement.setAttribute("editweight", "1000.0");
		return attElement;
	}

	// Returns the TEXT and CDATA contents of an element
	public static String getTextContents(Element element)
	{
		// Get contents
		String contents = "";
		NodeList children = element.getChildNodes();
		for (int i = 0; i < children.getLength(); i++) {
			Node child = children.item(i);
			if (child.getNodeType() == Node.TEXT_NODE || child.getNodeType() == Node.CDATA_SECTION_NODE)
				contents += child.getNodeValue();
		}
		return contents;
	}

	// This method processes a string and returns an array containing contents
	// of all substrings that are inside square brackets [] (in order they occurred)
	private String[] processSolverString(String string, Element element)
	{
		int start = 0;
		int end = 0;
		int total = 0;
		int count = 0;

		for (int i = 0; i < string.length(); i++)
			if (string.charAt(i) == '[')
				total++;

		String vars[] = new String[total];
		while (count < total) {
			while (string.charAt(start) != '[')
				start++;
			end = start + 1;
			while (string.charAt(end) != ']') {
				if (string.charAt(end) == '[' || end >= string.length())
					Error.die(this, element.getAttribute("_line_number_"), "Could not extract solver constraint variables " +
						"from string: " + string);
				end++;
			}
			vars[count] = string.substring(start + 1, end);
			count++;
			start = end + 1;
		}
		return vars;
	}

	// This method extracts all constraint variables from string in <SolverConstraint> and
	// <ConstraintExpression> strings, and adds them as <Variable> child elements with an attribute called "name"
	private void processConstraintVariables(Document document, Element root)
	{
		try {
			NodeList nodeList = root.getElementsByTagName("SolverConstraint");
			for (int i = 0; i < nodeList.getLength(); i++) {
				Element element = (Element) nodeList.item(i);
				String vars[] = processSolverString(getTextContents(element), element);
				for (int j = 0; j < vars.length; j++) {
					Element child = document.createElement("Variable");
					child.setAttribute("name", vars[j]);
					child.setAttribute("_line_number_", element.getAttribute("_line_number_"));
					element.appendChild(child);
				}
			}
			nodeList = root.getElementsByTagName("ConstraintExpression");
			for (int i = 0; i < nodeList.getLength(); i++) {
				Element element = (Element) nodeList.item(i);
				String vars[] = processSolverString(getTextContents(element), element);
				for (int j = 0; j < vars.length; j++) {
					Element child = document.createElement("Variable");
					child.setAttribute("name", vars[j]);
					child.setAttribute("_line_number_", element.getAttribute("_line_number_"));
					element.appendChild(child);
				}
			}
		}
		catch (DOMException e) {
			Error.die(this, "Could not process solver constraint variables: " + e.toString());
		}
	}

	// This method eliminates all child nodes of a node that are not element nodes
	// unless the current node is a <SolverConstraint> or <ConstraintExpression> element
	private void eliminateNonElementNodes(Element element)
	{
		List removeNodes = new LinkedList();

		// Do not alter the following elements
		if (element.getTagName().equals("SolverConstraint") || element.getTagName().equals("ConstraintExpression"))
			return;

		// Check each node and recursively call this method for all element nodes
		NodeList children = element.getChildNodes();
		for (int i = 0; i < children.getLength(); i++) {
			Node node = children.item(i);
			if (node.getNodeType() == Node.ELEMENT_NODE)
				eliminateNonElementNodes((Element) node);
			else
				removeNodes.add(node);
		}

		// Remove all non-element nodes
		Iterator iter = removeNodes.iterator();
		while (iter.hasNext()) {
			Node node = (Node) iter.next();
			try {
				element.removeChild(node);
			}
			catch (DOMException e) { 
				Error.die(this, "Error occurred while removing non-essential non-element nodes from DOM tree");
			}
		}
	}

	// This method extracts the symbol definitions from the DOM tree
	public void createSymbolDefinitionsList()
	{
		NodeList defs = symbolDefinitionsElement.getChildNodes();
		for (int i = 0; i < defs.getLength(); i++) {
			symbolDefinitionList.add(new SymbolDefinition((Element) defs.item(i)));
		}
	}

	// Check to make sure that there is no symbol name duplication and make 
	// sure all symbols needed for inheritence relationships are present
	public void checkSymbolDefinitionDependencies()
	{
		SymbolDefinition s1, s2;
		boolean present;

		// Check for name duplication
		for (int i = 0; i < symbolDefinitionList.size(); i++) {
			s1 = (SymbolDefinition) symbolDefinitionList.get(i);

			for (int j = i+1; j < symbolDefinitionList.size(); j++) {
				s2 = (SymbolDefinition) symbolDefinitionList.get(j);
				if (s1.getName().equals(s2.getName()))
					Error.die(this, "Symbol definition name duplication: " + s1.getName());	
			}
		}

		// Check that all extended symbols are present and abstract
		for (int i = 0; i < symbolDefinitionList.size(); i++) {
			s1 = (SymbolDefinition) symbolDefinitionList.get(i);
			if (!s1.getExtendsName().equals("")) {
				present = false;
				for (int j = 0; j < symbolDefinitionList.size(); j++) {
					s2 = (SymbolDefinition) symbolDefinitionList.get(j);
					if (s2.getName().equals(s1.getExtendsName())) {
						present = true;

						// Make sure symbol does not extend itself
						if (j == i)
							Error.die(this, "Symbol cannot extend itself: " + s2.getName());

						// Make sure symbol being extended is abstract
						if (!s2.isAbstract())
							Error.die(this, "Cannot extend symbol types that are not abstract: " + s2.getName());
						break;
					}
				}
				if (!present)
        	                        Error.die(this, "An abstract symbol definition is missing: " + s1.getExtendsName());
			}
		}
	}

	// This method updates the inherited attributes of each symbol based
	// on whether it extends any other symbols
	public void updateSymbolInheritedAttributes()
	{
		SymbolDefinition s1, s2;
		boolean moreToGo;

		do {
			moreToGo = false;

			// Check if each symbol needs more attributes
			for (int i = 0; i < symbolDefinitionList.size(); i++) {
				s1 = (SymbolDefinition) symbolDefinitionList.get(i);
				if (s1.doesNeedMoreAttributes()) {

					// Find symbol it extends 
					for (int j = 0; j < symbolDefinitionList.size(); j++) {
						s2 = (SymbolDefinition) symbolDefinitionList.get(j);
						if (s2.getName().equals(s1.getExtendsName())) {

							// Can copy attributes if symbol to be extended is complete
							if (s2.doesNeedMoreAttributes()) 
								moreToGo = true;
							else {
								s1.addInheritedAttributes(s2.getAllAttributes());
								s1.noMoreAttributes();
							}
							break;
						}
					}
				}
			}
		} while (moreToGo);
	}

	// This method opens a file for writing text output
	public static PrintWriter openTextFile(String filename)
	{
		PrintWriter writer = null;
		try {
			writer = new PrintWriterNL(new FileWriter(filename));
			writer.println(ORIGIN_LINE + " (Version " + VERSION + ")");
		}
		catch(IOException e) {
			Error.die(new Compiler(), "Could not create the text file: " + filename);
		}
		return writer;
	}

	// This method checks to see if a name string starts with a letter
	// and only contains letters, digits and the underscore character
	public static boolean isValidNameString(String str)
	{
		if (str == null || str.length() == 0)
			return false;
		if (!Character.isLetter(str.charAt(0)))
			return false;
		for (int i = 1; i < str.length(); i++) {
			if (!(str.charAt(i) == '_' || Character.isLetterOrDigit(str.charAt(i))))
				return false;
		}
		return true;
	}

	// Checks that the specified attribute of all specified child elements are valid
	public void checkSymbolNames(Element parent, String elementName, String attributeName)
	{
		NodeList nodes = parent.getElementsByTagName(elementName);
		for (int i = 0; i < nodes.getLength(); i++) {
			Element child = (Element) nodes.item(i);
			if (!isValidNameString(child.getAttribute(attributeName)))
				Error.die(this, child.getAttribute("_line_number_"), "Invalid string in \"" + attributeName + "\" attribute: " +
					child.getAttribute(attributeName));
		}
	}

	// Create the Symbol Node Sets class for the grammar
	public void createSymbolNodeSetsClass()
	{
		SymbolDefinition s;
		String line;

		// Open file for the class
		PrintWriter output = Compiler.openTextFile(PACKAGE_DIRECTORY + grammarName + "SymbolNodeSets.java");

		// Print class header details (package scope)
		output.println("");
		output.println(PACKAGE_LINE);
		output.println("");
		output.println("import au.edu.monash.csse.tonyj.cider.constraints.ConstraintSolver;");
		output.println("import java.io.PrintStream;");
		output.println("import java.util.Iterator;");
		output.println("import java.util.Set;");
		output.println("import java.util.HashSet;");
		output.println("");
		output.println("class " + grammarName + "SymbolNodeSets implements SymbolNodeSets {");
		output.println("");

		// Print private variables (sets for each symbol type)
		for (int i = 0; i < symbolDefinitionList.size(); i++) {
			s = (SymbolDefinition) symbolDefinitionList.get(i);
			output.println("\tprivate Set " + s.getName() + "_set;");
		}
		output.println("");

		// Print constructor
		output.println("\tpublic " + grammarName + "SymbolNodeSets()");
		output.println("\t{");
		for (int i = 0; i < symbolDefinitionList.size(); i++) {
			s = (SymbolDefinition) symbolDefinitionList.get(i);
			output.println("\t\t" + s.getName() + "_set = new HashSet();");
		}
		output.println("\t}");
		output.println("");
		
		// Print the print method
		output.println("\tpublic void print(PrintStream stream)");
		output.println("\t{");
		output.println("\t\tIterator iter;");
		output.println("");
		for (int i = 0; i < symbolDefinitionList.size(); i++) {
			s = (SymbolDefinition) symbolDefinitionList.get(i);
			if (!s.isAbstract()) {
				output.println("\t\titer = " + s.getName() + "_set.iterator();");
				output.println("\t\twhile (iter.hasNext()) {");
				output.println("\t\t\tGrammarSymbol symbol = ((SymbolNode) iter.next()).getSymbol();");
				output.println("\t\t\tif (symbol.isRegistered())");
				output.println("\t\t\t\tstream.println(symbol.toString());");
				output.println("\t\t}");
			}
		}
		output.println("\t}");
		output.println("");

		// Print the set interpreter method
		output.println("\tpublic void setInterpreter(Interpreter interpreter)");
		output.println("\t{");
		output.println("\t\tIterator iter;");
		output.println("");
		for (int i = 0; i < symbolDefinitionList.size(); i++) {
			s = (SymbolDefinition) symbolDefinitionList.get(i);
			if (!s.isAbstract()) {
				output.println("\t\titer = " + s.getName() + "_set.iterator();");
				output.println("\t\twhile (iter.hasNext())");
				output.println("\t\t\t((SymbolNode) iter.next()).setInterpreter(interpreter);");
			}
		}
		output.println("\t}");
		output.println("");

		// Print the get symbols method
		output.println("\tpublic Set getSymbols()");
		output.println("\t{");
		output.println("\t\tIterator iter;");
		output.println("\t\tSet symbols = new HashSet();");
		output.println("");
		for (int i = 0; i < symbolDefinitionList.size(); i++) {
			s = (SymbolDefinition) symbolDefinitionList.get(i);
			if (!s.isAbstract()) {
				output.println("\t\titer = " + s.getName() + "_set.iterator();");
				output.println("\t\twhile (iter.hasNext())");
				output.println("\t\t\tsymbols.add(((SymbolNode) iter.next()).getSymbol());");
			}
		}
		output.println("");
		output.println("\t\treturn symbols;");
		output.println("\t}");
		output.println("");

		// Print get methods
		for (int i = 0; i < symbolDefinitionList.size(); i++) {
			s = (SymbolDefinition) symbolDefinitionList.get(i);
			output.println("\tpublic Set get_" + s.getName() + "_set()");
			output.println("\t{");
			output.println("\t\treturn " + s.getName() + "_set;");
			output.println("\t}");
			output.println("");
		}

		// Print clear method
		output.println("\tpublic void clear()");
		output.println("\t{");
		for (int i = 0; i < symbolDefinitionList.size(); i++) {
			s = (SymbolDefinition) symbolDefinitionList.get(i);
			output.println("\t\t" + s.getName() + "_set.clear();");
		}
		output.println("\t}");
		output.println("");
		
		// Print add method
		output.println("\tpublic void add(SymbolNode node)");
		output.println("\t{");
		for (int i = 0; i < symbolDefinitionList.size(); i++) {
			s = (SymbolDefinition) symbolDefinitionList.get(i);
			output.println("\t\tif (node.getSymbol() instanceof " + s.getName() + ")");
			output.println("\t\t\t" + s.getName() + "_set.add(node);");
		}
		output.println("\t}");
		output.println("");

		// Print the add all method
		output.println("\tpublic void addAll(SymbolNodeSets sns)");
		output.println("\t{");
		output.println("\t\t" + grammarName + "SymbolNodeSets sets = (" + grammarName + "SymbolNodeSets) sns;");
		for (int i = 0; i < symbolDefinitionList.size(); i++) {
			s = (SymbolDefinition) symbolDefinitionList.get(i);
			output.println("\t\t" + s.getName() + "_set.addAll(sets.get_" + s.getName() + "_set());");
		}
		output.println("\t}");
		output.println("");

		// Print remove method
		output.println("\tpublic void remove(SymbolNode node)");
		output.println("\t{");
		for (int i = 0; i < symbolDefinitionList.size(); i++) {
			s = (SymbolDefinition) symbolDefinitionList.get(i);
			output.println("\t\tif (node.getSymbol() instanceof " + s.getName() + ")");
			output.println("\t\t\t" + s.getName() + "_set.remove(node);");
		}
		output.println("\t}");
		output.println("");

		// Print contains method
		output.println("\tpublic boolean contains(SymbolNode node)");
		output.println("\t{");
		for (int i = 0; i < symbolDefinitionList.size(); i++) {
			s = (SymbolDefinition) symbolDefinitionList.get(i);
			output.println("\t\tif (" + s.getName() + "_set.contains(node))");
			output.println("\t\t\treturn true;");
		}
		output.println("\t\treturn false;");
		output.println("\t}");
		output.println("");

		// End class and close file
		output.println("}");
		output.println("");
		output.close();
	}

	// This method extracts the productions from the DOM tree
	public void createProductionList()
	{
		NodeList prods = productionRulesElement.getChildNodes();
		for (int i = 0; i < prods.getLength(); i++) 
			productionList.add(new ProductionDefinition(i+1, grammarName, (Element) prods.item(i), symbolDefinitionList, 
				dependencyGraph, attDependencyGraph, CONSTRAINT_CLASS, CONSTRAINT_EXPRESSION_CLASS, constraintCompiler));

		// Check for name duplication
		Set names = new HashSet();
		for (int i = 0; i < productionList.size(); i++) {
			ProductionDefinition pDef = (ProductionDefinition) productionList.get(i);
			if (!pDef.getName().equals("")) {
				if (!names.add(pDef.getName()))
					Error.die(this, "Production name duplication: " + pDef.getName());
			}
		}
	}

	// This method creates the production record and production rule classes for each production in the list
	public void createProductionClasses()
	{
		for (int i = 0; i < productionList.size(); i++) {
			ProductionDefinition prod = (ProductionDefinition) productionList.get(i);
			prod.createProductionRecordClass();
			prod.createProductionRuleClass(dependencyGraph);
		}
	}

	// This method extracts the transformations from their DOM tree
	public void createTransformationLists()
	{
		// Create transformation definitions
		NodeList trans = transformationRulesElement.getChildNodes();
		for (int i = 0; i < trans.getLength(); i++) {
			Element transElement = (Element) trans.item(i);
			if (transElement.getTagName().equals("Transformation"))
				transformationList.add(new TransformationDefinition(i+1, grammarName, transElement, symbolDefinitionList, 
					productionList, CONSTRAINT_CLASS, CONSTRAINT_EXPRESSION_CLASS, constraintCompiler));
		}

		// Check for name duplication
		Set names = new HashSet();

		// Reserve these names
		names.add("ForceDirectedLayout");
		names.add("CheckForest");

		for (int i = 0; i < transformationList.size(); i++) {
			TransformationDefinition tDef = (TransformationDefinition) transformationList.get(i);
			if (!names.add(tDef.getName()))
				Error.die(this, "Transformation name duplication: " + tDef.getName());
		}

		// Create disjunction definitions
		for (int i = 0; i < trans.getLength(); i++) {
			Element transElement = (Element) trans.item(i);
			if (transElement.getTagName().equals("Disjunction"))
				disjunctionList.add(new DisjunctionDefinition(i+1, grammarName, transElement, transformationList));
		}

		// Check for name duplication
		for (int i = 0; i < disjunctionList.size(); i++) {
			DisjunctionDefinition dDef = (DisjunctionDefinition) disjunctionList.get(i);
			if (!names.add(dDef.getName()))
				Error.die(this, "Transformation/Disjunction name duplication: " + dDef.getName());
		}

		// Create regular expression definitions
		for (int i = 0; i < trans.getLength(); i++) {
			Element transElement = (Element) trans.item(i);
			if (transElement.getTagName().equals("CompoundTransformation"))
				regExpList.add(new RegExpDefinition(i+1, grammarName, transElement, transformationList, disjunctionList));
		}

		// Check for name duplication
		for (int i = 0; i < regExpList.size(); i++) {
			RegExpDefinition reDef = (RegExpDefinition) regExpList.get(i);
			if (!names.add(reDef.getName()))
				Error.die(this, "Transformation/Disjunction/CompoundTransformation name duplication: " + reDef.getName());
		}
	}

	// This method creates the transformation classes for each transformation, disjunction and regular expression
	public void createTransformationClasses()
	{
		for (int i = 0; i < transformationList.size(); i++) {
			TransformationDefinition tDef = (TransformationDefinition) transformationList.get(i);
			tDef.createTransformationRecordClass();
			tDef.createTransformationRuleClass(dependencyGraph);
		}
		for (int i = 0; i < disjunctionList.size(); i++) {
			DisjunctionDefinition dDef = (DisjunctionDefinition) disjunctionList.get(i);
			dDef.createDisjunctionRuleClass();
		}
		for (int i = 0; i < regExpList.size(); i++) {
			RegExpDefinition reDef = (RegExpDefinition) regExpList.get(i);
			reDef.createRegularExpressionClass();
		}
	}

	// Create the Interpreter class
	public void createInterpreterClass()
	{
		SymbolDefinition s;
		String line;

		// Open file for the class
		PrintWriter output = Compiler.openTextFile(PACKAGE_DIRECTORY + grammarName + "Interpreter.java");

		// Print class header details (public scope)
		output.println("");
		output.println(PACKAGE_LINE);
		output.println("");
		output.println("import java.util.Iterator;");
		output.println("import java.util.Map;");
		output.println("import java.util.List;");
		output.println("import java.util.LinkedList;");
		output.println("");
		output.println("public class " + grammarName + "Interpreter extends Interpreter {");
		output.println("");
		output.println("\tprivate boolean rulesBeingEvaluated;");
		output.println("");

		// Print out constants for transformations
		for (int i = 0; i < transformationList.size(); i++) {
			TransformationDefinition tDef = (TransformationDefinition) transformationList.get(i);
			output.println("\tpublic static final int TRANSFORMATION_" + tDef.getName() + " = " + tDef.getNumber() + ";");
		}
		for (int i = 0; i < disjunctionList.size(); i++) {
			DisjunctionDefinition dDef = (DisjunctionDefinition) disjunctionList.get(i);
			output.println("\tpublic static final int TRANSFORMATION_" + dDef.getName() + " = " + dDef.getNumber() + ";");
		}
		for (int i = 0; i < regExpList.size(); i++) {
			RegExpDefinition reDef = (RegExpDefinition) regExpList.get(i);
			output.println("\tpublic static final int TRANSFORMATION_" + reDef.getName() + " = " + reDef.getNumber() + ";");
		}
		output.println("");

		// Print the constructor
		output.println("\tpublic " + grammarName + "Interpreter()");
		output.println("\t{");
		output.println("\t\trulesBeingEvaluated = false;");
		output.println("\t\tsolver = new " + CONSTRAINT_SOLVER_CLASS + "();");
		output.println("\t\tsolver.addConstraintVariableListener(this);");
		output.println("\t\tsolverClass = solver.getClass();");
		output.println("\t\ttry {");
		output.println("\t\t\tforest = new ParseForest(this, Class.forName(\"au.edu.monash.csse.tonyj.cider.interpreter." + 
			grammarName + "SymbolNodeSets\"));");
		output.println("\t\t}");
		output.println("\t\tcatch (Exception e) {");
		output.println("\t\t\tLog.getLogger().severe(\"Could not create parse forest: SymbolNodeSets class not found\");");
		output.println("\t\t\tError.die(\"Could not create parse forest: SymbolNodeSets class not found\");");
		output.println("\t\t}");
		output.println("\t\tLog.getLogger().info(\"Interpreter created for " + grammarName + " grammar\");");
		output.println("\t}");
		output.println("");

		// Print evaluate production rules method
		output.println("\tpublic synchronized void evaluateProductionRules()");
		output.println("\t{");
		output.println("\t\tboolean checkRules;");
		output.println("");
		output.println("\t\tif (forestBeingChecked || rulesBeingEvaluated)");
		output.println("\t\t\treturn;");
		output.println("");
		output.println("\t\trulesBeingEvaluated = true;");
		output.println("\t\tdo {");
		output.println("\t\t\tcheckRules = false;");
		for (int i = 0; i < productionList.size(); i++) {
			ProductionDefinition prodDef = (ProductionDefinition) productionList.get(i);
			output.println("\t\t\tif (ProductionRule" + prodDef.getNumber() + ".evaluateEntireForest(forest, solver) != null) {");
			output.println("\t\t\t\tcheckRules = true;");
			output.println("\t\t\t\tcheckForest();");
			output.println("\t\t\t}");
		}
		output.println("\t\t} while (checkRules);");
		output.println("\t\trulesBeingEvaluated = false;");
		output.println("\t}");
		output.println("");

		// Print apply transformation not in parallel method
		output.println("\tpublic synchronized boolean applyTransformation(int transformation) throws IllegalArgumentException");
		output.println("\t{");
		output.println("\t\treturn processTransformation(transformation, false);");
		output.println("\t}");
		output.println("");

		// Print apply transformation in parallel method
		output.println("\tpublic synchronized boolean applyTransformationInParallel(int transformation) throws IllegalArgumentException");
		output.println("\t{");
		output.println("\t\treturn processTransformation(transformation, true);");
		output.println("\t}");
		output.println("");

		// Print apply transformation method
		output.println("\tprivate boolean processTransformation(int transformation, boolean parallel) throws IllegalArgumentException");
		output.println("\t{");
		output.println("\t\tswitch (transformation) {");
		for (int i = 0; i < transformationList.size(); i++) {
			TransformationDefinition tDef = (TransformationDefinition) transformationList.get(i);
			output.println("\t\t\tcase TRANSFORMATION_" + tDef.getName() + ": return applyTransformation_" + tDef.getName() + 
				"(parallel, null);");
		}
		for (int i = 0; i < disjunctionList.size(); i++) {
			DisjunctionDefinition dDef = (DisjunctionDefinition) disjunctionList.get(i);
			output.println("\t\t\tcase TRANSFORMATION_" + dDef.getName() + ": return applyDisjunction_" + dDef.getName() + "(parallel);");
		}
		for (int i = 0; i < regExpList.size(); i++) {
			RegExpDefinition reDef = (RegExpDefinition) regExpList.get(i);
			output.println("\t\t\tcase TRANSFORMATION_" + reDef.getName() + ":");
			output.println("\t\t\t\tif (parallel)");
			output.println("\t\t\t\t\tthrow new IllegalArgumentException(\"Compound transformations cannot be applied in parallel\");"); 
			output.println("\t\t\t\telse");
			output.println("\t\t\t\t\treturn applyCompoundTransformation_" + reDef.getName() + "();");
		}
		output.println("\t\t\tdefault: throw new IllegalArgumentException(\"Invalid transformation value\");");
		output.println("\t\t}");
		output.println("\t}");
		output.println("");

		// Print apply transformation with mapping method
		output.println("\tpublic synchronized boolean applyTransformationWithMapping(int transformation, boolean parallel, " +
			"Map candidatesMap) throws IllegalArgumentException");
		output.println("\t{");
		output.println("\t\tswitch (transformation) {");
		for (int i = 0; i < transformationList.size(); i++) {
			TransformationDefinition tDef = (TransformationDefinition) transformationList.get(i);
			output.println("\t\t\tcase TRANSFORMATION_" + tDef.getName() + ": return applyTransformation_" + tDef.getName() + 
				"(parallel, candidatesMap);");
		}
		output.println("\t\t\tdefault: throw new IllegalArgumentException(\"Invalid transformation value - only basic transformations " +
			"can have candidate mappings\");");
		output.println("\t\t}");
		output.println("\t}");
		output.println("");

		// Create transformation methods (if required)
		for (int i = 0; i < transformationList.size(); i++) {
			TransformationDefinition tDef = (TransformationDefinition) transformationList.get(i);
			output.println("\tboolean applyTransformation_" + tDef.getName() + "(boolean parallel, Map candidatesMap)");
			output.println("\t\tthrows IllegalArgumentException");
			output.println("\t{");
			output.println("\t\tboolean success;");
			output.println("\t\tboolean tempAFC = autoForestChecking;");
			output.println("\t\tautoForestChecking = false;");
			output.println("\t\tTransformationRule" + tDef.getNumber() + " rule = (candidatesMap == null ? new TransformationRule" +  
				tDef.getNumber() + "(forest, solver) : new TransformationRule" + tDef.getNumber() + 
				"(forest, solver, candidatesMap));");
			output.println("\t\tif (parallel)");
			output.println("\t\t\tsuccess = rule.parallelApply();");
			output.println("\t\telse");
			output.println("\t\t\tsuccess = rule.normalApply();");
			output.println("\t\tautoForestChecking = tempAFC;");
			output.println("\t\tif (success && autoForestChecking) {");
			output.println("\t\t\tcheckForest();");
			output.println("\t\t\tevaluateProductionRules();");
			output.println("\t\t}");
			output.println("\t\treturn success;");
			output.println("\t}");
			output.println("");
		}

		// Create disjunction methods (if required)
		for (int i = 0; i < disjunctionList.size(); i++) {
			DisjunctionDefinition dDef = (DisjunctionDefinition) disjunctionList.get(i);
			output.println("\tboolean applyDisjunction_" + dDef.getName() + "(boolean parallel)");
			output.println("\t{");
			output.println("\t\tboolean success;");
			output.println("\t\tboolean tempAFC = autoForestChecking;");
			output.println("\t\tautoForestChecking = false;");
			output.println("\t\tDisjunctionRule" + dDef.getNumber() + " rule = new DisjunctionRule" +  dDef.getNumber() +
				"(forest, solver);");
			output.println("\t\tif (parallel)");
			output.println("\t\t\tsuccess = rule.parallelApply();");
			output.println("\t\telse");
			output.println("\t\t\tsuccess = rule.normalApply();");
			output.println("\t\tautoForestChecking = tempAFC;");
			output.println("\t\tif (success && autoForestChecking) {");
			output.println("\t\t\tcheckForest();");
			output.println("\t\t\tevaluateProductionRules();");
			output.println("\t\t}");
			output.println("\t\treturn success;");
			output.println("\t}");
			output.println("");
		}

		// Create regular expression methods (if required)
		for (int i = 0; i < regExpList.size(); i++) {
			RegExpDefinition reDef = (RegExpDefinition) regExpList.get(i);
			output.println("\tboolean applyCompoundTransformation_" + reDef.getName() + "()");
			output.println("\t{");
			output.println("\t\tboolean tempAFC = autoForestChecking;");
			output.println("\t\tautoForestChecking = false;");
			output.println("\t\tRegularExpression" + reDef.getNumber() + " expression = new RegularExpression" +  reDef.getNumber() +
				"(forest, solver);");
			output.println("\t\tboolean success = expression.normalApply();");
			output.println("\t\tautoForestChecking = tempAFC;");
			output.println("\t\tif (success && autoForestChecking) {");
			output.println("\t\t\tcheckForest();");
			output.println("\t\t\tevaluateProductionRules();");
			output.println("\t\t}");
			output.println("\t\treturn success;");
			output.println("\t}");
			output.println("");
		}

		// End class and close file
		output.println("}");
		output.println("");
		output.close();
	}

	// Warning Event Handler
	public void warning(SAXParseException e) 
		throws SAXException 
	{
		System.err.println ("Parser Warning: " + e);
		parserError = true;
	}

	// Error Event Handler
	public void error(SAXParseException e)
		throws SAXException 
	{
		System.err.println ("Parser Error: " + e);
		parserError = true;
	}

	// Fatal Error Event Handler
	public void fatalError(SAXParseException e)
		throws SAXException 
	{
		System.err.println ("Parser Fatal Error: " + e);
		parserError = true;
	}
}

