/*
    Copyright (C) 2003  Anthony R. Jansen.

    This file is part of the CIDER Toolkit.

    The CIDER Toolkit is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    The CIDER Toolkit is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with the CIDER Toolkit; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

// ----------------------------------------
// CIDER Toolkit: Compiler Package
//
// Author:  Anthony R. Jansen
// Begun:   November 2002
// Class:   DependencyGraph
//
// This class implements a dependency
// graph for the language given to the
// compiler. It allows for aspects of the
// languages validity to be checked.
// ----------------------------------------

package au.edu.monash.csse.tonyj.cider.compiler;

import java.util.Set;
import java.util.HashSet;
import java.util.List;
import java.util.LinkedList;
import java.util.Iterator;
import java.io.PrintStream;

public class DependencyGraph {

	// Private variables
	private List	symbolDefinitionList;
	private List	nodes;

	// Constructor, which creates a node for every non-abstract symbol type
	public DependencyGraph(List symbolDefs)
	{
		symbolDefinitionList = symbolDefs;
		nodes = new LinkedList();
		Iterator iter = symbolDefs.iterator();
		while (iter.hasNext()) {
			SymbolDefinition sDef = (SymbolDefinition) iter.next();
			if (!sDef.isAbstract())
				nodes.add(new DependencyGraphNode(sDef));
		}
	}

	// Returns the node corresponding to a given symbol definition
	private DependencyGraphNode getNode(SymbolDefinition type)
	{
		Iterator iter = nodes.iterator();
		while (iter.hasNext()) {
			DependencyGraphNode node = (DependencyGraphNode) iter.next();
			if (node.getSymbolDefinition().equals(type)) 
				return node;
		}
		return null;
	}

	// Returns the set of nodes corresponding to a given symbol definition
	// This is the node for either the symbol, or if it is abstract, the non-abstract symbols that extend it
	private Set getNodes(SymbolDefinition type)
	{
		Set toBeReplaced = new HashSet();

		// Start with a set containing original symbol type
		Set symbolTypes = new HashSet();
		symbolTypes.add(type);

		do {
			// Look for abstract types that need to be replaced
			toBeReplaced.clear();
			Iterator iter = symbolTypes.iterator();
			while (iter.hasNext()) {
				SymbolDefinition sDef = (SymbolDefinition) iter.next();
				if (sDef.isAbstract())
					toBeReplaced.add(sDef);
			}

			// Add any symbols that extend the symbols to be replaced
			iter = toBeReplaced.iterator();
			while (iter.hasNext()) {
				SymbolDefinition s1 = (SymbolDefinition) iter.next();

				Iterator iter2 = symbolDefinitionList.iterator();
				while (iter2.hasNext()) {
					SymbolDefinition s2 = (SymbolDefinition) iter2.next();

					if (s2.getExtendsName().equals(s1.getName()))
						symbolTypes.add(s2);
				}
			}

			// Remove the symbols to be replaced
			symbolTypes.removeAll(toBeReplaced);

		} while (!toBeReplaced.isEmpty());

		// Return the nodes corresponding to the symbol types
		Set nodeSet = new HashSet();
		Iterator iter = nodes.iterator();
		while (iter.hasNext()) {
			DependencyGraphNode node = (DependencyGraphNode) iter.next();
			if (symbolTypes.contains(node.getSymbolDefinition())) 
				nodeSet.add(node);
		}
		return nodeSet;
	}

	// Add a reduction dependecy
	public void addReductionDependency(SymbolDefinition from, SymbolDefinition to)
	{
		Iterator iter = getNodes(from).iterator();
		while (iter.hasNext()) 
			((DependencyGraphNode) iter.next()).addReductionDependency(getNode(to));
	}

	// Add a positive existential dependecy
	public void addPositiveExistentialDependency(SymbolDefinition from, SymbolDefinition to)
	{
		Iterator iter = getNodes(from).iterator();
		while (iter.hasNext()) 
			((DependencyGraphNode) iter.next()).addPositiveExistentialDependency(getNode(to));
	}

	// Add a negative existential dependecy
	public void addNegativeExistentialDependency(SymbolDefinition from, SymbolDefinition to)
	{
		Iterator iter = getNodes(from).iterator();
		while (iter.hasNext()) 
			((DependencyGraphNode) iter.next()).addNegativeExistentialDependency(getNode(to));
	}

	// This method checks to make sure that the negative existential dependencies on the symbol types
	// do not cause any dependency loops that might lead to infinite loops when evaluating the productions
	public void checkDependencies()
	{
		Iterator iter = nodes.iterator();
		while (iter.hasNext()) {
			DependencyGraphNode node1 = (DependencyGraphNode) iter.next();

			// Check each negative dependency
			Iterator iter2 = node1.getNegativeExistentialDependencies().iterator();
			while (iter2.hasNext()) {
				DependencyGraphNode node2 = (DependencyGraphNode) iter2.next();

				if (isPositivelyDependent(node2, node1))
					Error.die(this, "Illegal negative dependency between symbol types \"" + node1.getSymbolType() +
						"\" and \"" + node2.getSymbolType() + "\"");
			}
		}
	}

	// This method returns true if node 1 is positively dependent on node 2
	private boolean isPositivelyDependent(DependencyGraphNode node1, DependencyGraphNode node2)
	{
		int oldSize;
		Set dependentNodes = new HashSet();
		Set tempSet = new HashSet();

		// Keep looping as long as there are more dependencies to add
		dependentNodes.add(node1);
		do {
			oldSize = dependentNodes.size();
			tempSet.clear();

			// Get all nodes that the current nodes immediately depend on
			Iterator iter = dependentNodes.iterator();
			while (iter.hasNext()) {
				DependencyGraphNode node = (DependencyGraphNode) iter.next();
				tempSet.addAll(node.getReductionDependencies());
				tempSet.addAll(node.getPositiveExistentialDependencies());
			}
			dependentNodes.addAll(tempSet);

			// See if node 2 is one of the dependent nodes
			if (dependentNodes.contains(node2))
				return true;

		} while (oldSize < dependentNodes.size());

		return false;
	}

	// This method returns true iff the product symbol type can be created from the component symbol type (using reduction links)
	public boolean hasCreationRelationship(SymbolDefinition product, SymbolDefinition component)
	{
		Set productSet = getNodes(product);
		Set componentSet = getNodes(component);

		Iterator iter1 = productSet.iterator();
		while (iter1.hasNext()) {
			DependencyGraphNode productNode = (DependencyGraphNode) iter1.next();

			Iterator iter2 = componentSet.iterator();
			while (iter2.hasNext()) {
				DependencyGraphNode componentNode = (DependencyGraphNode) iter2.next();
				if (hasCreationRelationship(productNode, componentNode))
					return true;
			}
		}
		return false;
	}

	// This method returns true iff the product symbol type can be created from the component symbol type (using reduction links)
	// This need not be a direct relationship, but can happen via intermediary symbol types (e.g. a -> b -> c)
	// Note that if the product and component types are the same, a reduction link must still exist in order to return true
	private boolean hasCreationRelationship(DependencyGraphNode productNode, DependencyGraphNode componentNode)
	{
		int oldSize;
		Set potentialProducts = new HashSet();
		Set tempSet = new HashSet();

		// Keep looping as long as there are more dependencies to add
		potentialProducts.addAll(componentNode.getReductionDependencies());
		do {
			oldSize = potentialProducts.size();
			tempSet.clear();

			// Get all nodes that the current nodes immediately depend on with a reduction link
			Iterator iter = potentialProducts.iterator();
			while (iter.hasNext()) {
				DependencyGraphNode node = (DependencyGraphNode) iter.next();
				tempSet.addAll(node.getReductionDependencies());
			}
			potentialProducts.addAll(tempSet);

			// See if product node is one of the dependent nodes
			if (potentialProducts.contains(productNode))
				return true;

		} while (oldSize < potentialProducts.size());

		return false;
	}

	// Debug method
	public void printGraph(PrintStream stream)
	{
		stream.println("*** Dependency Graph ***");
		stream.println("");

		// Go through all nodes
		Iterator iter1 = nodes.iterator();
		while (iter1.hasNext()) {
			DependencyGraphNode node1 = (DependencyGraphNode) iter1.next();
			stream.println(node1.getSymbolDefinition().getName() + ":");
			stream.print("\tReduction Dependencies:");
			Iterator iter2 = node1.getReductionDependencies().iterator();
			while (iter2.hasNext()) {
				DependencyGraphNode node2 = (DependencyGraphNode) iter2.next();
				stream.print("  " + node2.getSymbolDefinition().getName());
			}
			stream.print("\n\tPositive Dependencies:");
			iter2 = node1.getPositiveExistentialDependencies().iterator();
			while (iter2.hasNext()) {
				DependencyGraphNode node2 = (DependencyGraphNode) iter2.next();
				stream.print("  " + node2.getSymbolDefinition().getName());
			}
			stream.print("\n\tNegative Dependencies:");
			iter2 = node1.getNegativeExistentialDependencies().iterator();
			while (iter2.hasNext()) {
				DependencyGraphNode node2 = (DependencyGraphNode) iter2.next();
				stream.print("  " + node2.getSymbolDefinition().getName());
			}
			stream.print("\n\n");
		}
	}
}

