/*
    Copyright (C) 2003  Anthony R. Jansen.

    This file is part of the CIDER Toolkit.

    The CIDER Toolkit is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    The CIDER Toolkit is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with the CIDER Toolkit; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

// ----------------------------------------
// CIDER Toolkit: Compiler Package
//
// Author:  Anthony R. Jansen
// Begun:   February 2003
// Class:   DisjunctionDefinition
//
// This class is used to represent the
// information contained in a disjunction
// definition, part of a diagrammatic
// language specification. It defines the
// disjunction of simple transformations.
// The information is taken from an XML DOM 
// tree.
// ----------------------------------------

package au.edu.monash.csse.tonyj.cider.compiler;

import org.w3c.dom.Node;
import org.w3c.dom.Element;
import org.w3c.dom.NodeList;
import java.io.PrintWriter;
import java.util.List;
import java.util.LinkedList;

public class DisjunctionDefinition {

	// Private variables
	private int			number;
	private String			disjunctionName;
	private String			grammarName;
	private List			transformationsList;

	// Constructor, which takes a number, name, a DOM Node for the Transformation and a list of transformation definitions 
	public DisjunctionDefinition(int num, String gName, Element disjunctionElement, List transList)
	{
		number = num;
		grammarName = gName;
		disjunctionName = disjunctionElement.getAttribute("name");
		if (disjunctionName.equals(""))
			Error.die(this, disjunctionElement.getAttribute("_line_number_"), "Disjunction element does not have a valid name");

		// Read in the transformations
		transformationsList = new LinkedList();
		readInTransformations(disjunctionElement, transList);
		if (transformationsList.size() < 2)
			Error.die(this, disjunctionElement.getAttribute("_line_number_"), "Disjunction element must have at least two components");
	}
	
	// Reads all the transformations in
	private void readInTransformations(Element disjunctionElement, List transList)
	{
		NodeList nodes = disjunctionElement.getChildNodes();
		for (int i = 0; i < nodes.getLength(); i++) {
			Element componentElement = (Element) nodes.item(i);
			String name = componentElement.getAttribute("name");

			boolean success = false;
			for (int j = 0; j < transList.size(); j++) {
				TransformationDefinition tDef = (TransformationDefinition) transList.get(j);
				if (tDef.getName().equals(name)) {
					if (transformationsList.contains(tDef))
						Error.die(this, disjunctionElement.getAttribute("_line_number_"), "Duplication error: " +
							"Transformation specified more than once: " + name);
					transformationsList.add(tDef);
					success = true;
					break;
				}
			}
			if (!success)
				Error.die(this, disjunctionElement.getAttribute("_line_number_"), "Disjunction refers to a transformation " +
					"that has not been specified: " + name);
		}
	}

	// Produces a disjunction rule class for this disjunction
	public void createDisjunctionRuleClass()
	{
		// Open file for the class
		PrintWriter output = Compiler.openTextFile(Compiler.PACKAGE_DIRECTORY + "DisjunctionRule" + number + ".java");

		// Print class header details (package scope)
		output.println("");
		output.println(Compiler.PACKAGE_LINE);
		output.println("");
		output.println("import java.util.List;");
		output.println("import java.util.Vector;");
		output.println("import java.util.Map;");
		output.println("import java.util.HashMap;");
		output.println("import java.util.Iterator;");
		output.println("import au.edu.monash.csse.tonyj.cider.constraints.ConstraintSolver;");
		output.println("");
		output.println("class DisjunctionRule" + number + " implements TransformationRule {");
		output.println("");

		// Print out private variables 
		output.println("\tprivate int listIndex;");
		output.println("\tprivate List transformationsList;");
		output.println("");

		// Print constructor 
		output.println("\tpublic DisjunctionRule" + number + "(ParseForest f, ConstraintSolver cs)");
		output.println("\t{");
		output.println("\t\tlistIndex = 0;");
		output.println("\t\ttransformationsList = new Vector(" + (transformationsList.size() + 1) + ");");
		for (int i = 0; i < transformationsList.size(); i++) {
			TransformationDefinition tDef = (TransformationDefinition) transformationsList.get(i);
			output.println("\t\ttransformationsList.add(new TransformationRule" + tDef.getNumber() + "(f, cs));"); 
		}
		output.println("\t}");

		// Print out the method to apply the transformation normally
		output.println("\tpublic boolean normalApply()");
		output.println("\t{");
		output.println("\t\twhile (listIndex < transformationsList.size()) {");
		output.println("\t\t\tBasicTransformationRule rule = (BasicTransformationRule) transformationsList.get(listIndex);");
		output.println("\t\t\tif (rule.normalApply()) {");
		output.println("\t\t\t\tLog.getLogger().info(\"Disjunction " + disjunctionName + " successfully asynchronously applied\");");
		output.println("\t\t\t\treturn true;");
		output.println("\t\t\t}");
		output.println("\t\t\tlistIndex++;");
		output.println("\t\t}");
		output.println("\t\treturn false;");
		output.println("\t}");
		output.println("");

		// Print out the method to apply the transformation in parallel
		output.println("\tpublic boolean parallelApply()");
		output.println("\t{");
		output.println("\t\tboolean success = false;");
		output.println("\t\tMap recordMap = new HashMap();");
		output.println("\t\tTransformationRecord record;");
		output.println("\t\tBasicTransformationRule rule;");
		output.println("\t\twhile (listIndex < transformationsList.size()) {");
		output.println("\t\t\trule = (BasicTransformationRule) transformationsList.get(listIndex);");
		output.println("\t\t\trecord = rule.evaluate();");
		output.println("\t\t\twhile (record != null) {");
		output.println("\t\t\t\trecordMap.put(record, rule);");
		output.println("\t\t\t\trecord = rule.evaluate();");
		output.println("\t\t\t}");
		output.println("\t\t\tlistIndex++;");
		output.println("\t\t}");
		output.println("\t\tIterator iter = recordMap.keySet().iterator();");
		output.println("\t\twhile (iter.hasNext()) {");
		output.println("\t\t\trecord = (TransformationRecord) iter.next();");
		output.println("\t\t\trule = (BasicTransformationRule) recordMap.get(record);");
		output.println("\t\t\tif (rule.apply(record))");
		output.println("\t\t\t\tsuccess = true;");
		output.println("\t\t}");
		output.println("\t\tif (success)");
		output.println("\t\t\tLog.getLogger().info(\"Disjunction " + disjunctionName + " successfully synchronously applied\");");
		output.println("\t\treturn success;");
		output.println("\t}");

		// End class and close file
		output.println("}");
		output.println("");
		output.close();
	}

	// Returns the name of this disjunction
	public String getName()
	{
		return disjunctionName;
	}

	// Returns the number of this disjunction
	public int getNumber()
	{
		return number;
	}
}


