/*
    Copyright (C) 2003  Anthony R. Jansen.

    This file is part of the CIDER Toolkit.

    The CIDER Toolkit is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    The CIDER Toolkit is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with the CIDER Toolkit; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

// ----------------------------------------
// CIDER Toolkit: Compiler Package 
//
// Author:  Anthony R. Jansen
// Begun:   May 2003
// Class:   GrammarConverter
//
// This class is used for converting a 
// cider grammar file into an XML format.
// ----------------------------------------

package au.edu.monash.csse.tonyj.cider.compiler;

public class GrammarConverter extends BaseConverter {

	// Constructor
	public GrammarConverter(String inputFilename, String outputFilename)
	{
		super(inputFilename, outputFilename);
	}

	// Performs the conversion (should be called only once)
	public void convert()
	{
		// Output XML and DOCTYPE tags
		output.println("<?xml version=\"1.0\" encoding=\"UTF-8\" ?>");
		output.println("<!DOCTYPE Cider SYSTEM \"Cider-" + Compiler.VERSION + ".dtd\" >");
		output.println("");

		// Output cider and grammar start tags
		output.println("<Cider>");
		output.println("<Grammar name=\"" + readGrammarName() + "\" >");
		addLineComment();
		output.println("");

		processSymbolDefinitions();
		processProductionRules();

		// Output cider and grammar end tags
		output.println("</Grammar>");
		output.println("</Cider>");
		output.println("");
	}

	// Reads in the name of the grammar
	private String readGrammarName()
	{
		return readEqualsString("grammar");
	}

	// Processes the symbol definitions
	private void processSymbolDefinitions()
	{
		if (!input.readNextToken("symbol_definitions"))
			InputError("Expecting \"symbol_definitions\" to be defined");
		if (!input.readNextToken("{"))
			InputError("Expecting { to follow \"symbol_definitions\"");

		output.println("\t<SymbolDefinitions>");
		output.println("");

		while (!input.readNextToken("}")) {
			if (input.isNextToken(TextInputFile.EOF))
				InputError("EOF reached in the \"symbol_definitions\" declarations");
			processSymbolDefinition();
		}

		output.println("\t</SymbolDefinitions>");
		output.println("");
	}

	// Processes the declaration for a specific symbol definition
	private void processSymbolDefinition()
	{
		String name = input.readStandardToken();		
		String type = input.readBracketString();
		String drawable = "";
		String extending = "";

		// Read the drawable class if present
		if (input.readNextToken("[")) {
			drawable = input.readStandardToken();
			if (!input.readNextToken("]"))
				InputError("Expecting ] to close the drawable class string");
		}

		// Read the extends symbol type if present
		if (input.readNextToken(":")) 
			extending = input.readStandardToken();

		String line = "\t\t<SymbolDef name=\"" + name + "\" type=\"" + type + "\" ";
		if (!drawable.equals(""))
			line += "drawable=\"" + drawable + "\" ";
		if (!extending.equals(""))
			line += "extends=\"" + extending + "\" ";
		output.println(line + ">");

		// Read in the attributes
		while (!input.readNextToken(";")) {
			if (input.isNextToken(TextInputFile.EOF))
				InputError("EOF reached in the symbol declaration");
			processAttribute();
			if (!(input.isNextToken(",") || input.isNextToken(";")))
				InputError("Symbol attribute must be followed by either a , or ; character");
			input.readNextToken(",");
		}
		output.println("\t\t</SymbolDef>");
		addLineComment();
		output.println("");
	}

	// Processes attribute declarations
	private void processAttribute()
	{
		String name = input.readStandardToken();
		if (!input.readNextToken(":"))
				InputError("Expecting : to follow an attribute name declaration");
		String datatype = input.readStandardToken();

		String line = "\t\t\t<Attribute name=\"" + name + "\" datatype=\"";
		if (datatype.equals("@cv")) {
			line += "ConstraintVariable\" ";

			// Read the weights if present
			if (input.readNextToken("[")) {
				line += "stayweight=\"" + input.readStandardToken() + "\" ";
				if (!input.readNextToken(","))
					InputError("Expecting , between constraint variable weights");
				line += "editweight=\"" + input.readStandardToken() + "\" ";
				if (!input.readNextToken("]"))
					InputError("Expecting ] to close the constraint variable weights");
			}
			line += "/>";
		}
		else
			line += datatype + "\" />";
		output.println(line);
	}

	// Processes all production rules up to the end-of-file
	private void processProductionRules()
	{
		output.println("\t<ProductionRules>");
		output.println("");

		while (!input.isEndOfFile())
			processProduction();

		output.println("\t</ProductionRules>");
		output.println("");
	}

	// Processes a single production
	private void processProduction()
	{
		String productionName = "";

		// Read in declaration and name (if there is one)
		if (!input.isNextToken(":"))
			productionName = input.readStandardToken();
		if (!input.readNextToken(":"))
			InputError("Expecting : when declaring a production");
		if (!input.readNextToken("production"))
			InputError("Expecting \"production\" when declaring a production");
		if (!input.readNextToken("{"))
			InputError("Expecting { to follow \"production\"");

		if (productionName.equals(""))
			output.println("\t\t<Production>");
		else
			output.println("\t\t<Production name=\"" + productionName + "\" >");

		// Process output symbols
		output.println("\t\t\t<OutputSymbols>");
		do {
			processSymbol("\t\t\t\t");
		} while (input.readNextToken(","));
		output.println("\t\t\t</OutputSymbols>");
		addLineComment();

		if (!input.readNextToken("::="))
			InputError("Expecting ::= after the output symbols have been defined");

		// Process input symbols
		output.println("\t\t\t<InputSymbols>");
		do {
			processSymbol("\t\t\t\t");
		} while (input.readNextToken(","));
		output.println("\t\t\t</InputSymbols>");
		addLineComment();

		processConstraintSet("\t\t\t");

		// Process attribute settings and inits
		output.println("\t\t\t<AttributeAssignment>");
		processInitSection("\t\t\t\t");
		processSetSection();
		output.println("\t\t\t</AttributeAssignment>");

		output.println("\t\t</Production>");
		if (!input.readNextToken("}"))
			InputError("Expecting } to conclude a \"production\" declaration");
	}	

	// Process statements in the init section 
	// Returns true iff there is one
	private boolean processInitSection(String tabs)
	{
		// If no init section is declared, return false
		if (!input.readNextToken("init"))
			return false;
		if (!input.readNextToken("{"))
			InputError("Expecting { to follow \"init\"");

		// Read in individual init statements
		while (!input.readNextToken("}")) {
			if (input.isNextToken(TextInputFile.EOF))
				InputError("EOF reached in the \"init\" declaration");

			String name = input.readStandardToken();
			if (!input.readNextToken(":="))
				InputError("Expecting := to follow the name of the attribute to be initialized");
			output.println(tabs + "<InitAttribute name=\"" + name + "\" >");
			output.println(getArgument(tabs + "\t"));
			output.println(tabs + "</InitAttribute>");
			addLineComment();
			if (!input.readNextToken(";"))
				InputError("Expecting ; to follow an attribute initialization");
		}
		return true;
	}

	// Process statements in the set section (if there is one)
	private void processSetSection()
	{
		// If no set section is declared, return
		if (!input.readNextToken("set"))
			return;
		if (!input.readNextToken("{"))
			InputError("Expecting { to follow \"set\"");

		// Read in individual init statements
		while (!input.readNextToken("}")) {
			if (input.isNextToken(TextInputFile.EOF))
				InputError("EOF reached in the \"set\" declaration");

			String name = input.readStandardToken();
			if (!input.readNextToken(":="))
				InputError("Expecting := to follow the name of the attribute to be initialized");
			output.println("\t\t\t\t<SetAttribute name=\"" + name + "\" >");

			// Deal with constraint expressions if appropriate
			if (input.isNextToken("@ce"))
				processConstraintExpression("\t\t\t\t\t");
			else
				output.println(getArgument("\t\t\t\t\t"));
			output.println("\t\t\t\t</SetAttribute>");
			addLineComment();
			if (!input.readNextToken(";"))
				InputError("Expecting ; to follow an attribute setting");
		}
	}
}

