/*
    Copyright (C) 2003  Anthony R. Jansen.

    This file is part of the CIDER Toolkit.

    The CIDER Toolkit is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    The CIDER Toolkit is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with the CIDER Toolkit; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

// ----------------------------------------
// CIDER Toolkit: Compiler Package 
// 
// Author:  Anthony R. Jansen
// Begun:   June 2003
// Class:   PrintWriterNL
//
// This class modifies the PrintWriter 
// class by ensuring that every newline
// character sequence passed to a print 
// or println method is converted to the
// appropriate line separator for the
// current platform. 
// ----------------------------------------

package au.edu.monash.csse.tonyj.cider.compiler;

import java.io.PrintWriter;
import java.io.Writer;
import java.io.OutputStream;

class PrintWriterNL extends PrintWriter {

	// Constructor 1
	public PrintWriterNL(Writer out)
	{
		super(out);
	}

	// Constructor 2
	public PrintWriterNL(Writer out, boolean autoFlush)
	{
		super(out, autoFlush);
	}

	// Ensure all newlines combos (\n, \r\n, and \r) are represented by \n
	private	String processNewlines(String string)
	{
		String output = "";

		try {
			for (int i = 0; i < string.length(); i++) {
				if (string.startsWith("\r\n", i)) {
					output += "\n";
					i++;
				}
				else if (string.startsWith("\r", i))
					output += "\n";
				else
					output += "" + string.charAt(i);
			}
		}
		catch(IndexOutOfBoundsException e) { ; }

		return output;
	}

	// Override the println method in PrintWriter
	// Ensure that each newline character is handled consistently by the underlying PrintWriter
	public void println(String string)
	{
		string = processNewlines(string);
		int index = string.indexOf("\n");
		while (index != -1) {
			super.println(string.substring(0, index));
			string = string.substring(index + 1);
			index = string.indexOf("\n");
		}
		super.println(string);
		if (checkError())
			Error.die(this, "Error occurred when printing output");
	}

	// Override the print method in PrintWriter
	// Ensure that each newline character is handled consistently by the underlying PrintWriter
	public void print(String string)
	{
		string = processNewlines(string);
		int index = string.indexOf("\n");
		while (index != -1) {
			super.println(string.substring(0, index));
			string = string.substring(index + 1);
			index = string.indexOf("\n");
		}
		super.print(string);
		if (checkError())
			Error.die(this, "Error occurred when printing output");
	}
}

