/*
    Copyright (C) 2003  Anthony R. Jansen.

    This file is part of the CIDER Toolkit.

    The CIDER Toolkit is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    The CIDER Toolkit is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with the CIDER Toolkit; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

// ----------------------------------------
// CIDER Toolkit: Compiler Package
//
// Author:  Anthony R. Jansen
// Begun:   September 2002
// Class:   ProductionDefinition
//
// This class is used to represent the
// information contained in a production
// definition, part of a diagrammatic
// language specification. The information
// is taken from an XML DOM tree.
// ----------------------------------------

package au.edu.monash.csse.tonyj.cider.compiler;

import org.w3c.dom.Node;
import org.w3c.dom.Element;
import org.w3c.dom.NodeList;
import org.w3c.dom.DOMException;
import java.io.PrintWriter;
import java.util.Iterator;
import java.util.Set;
import java.util.HashSet;
import java.util.List;
import java.util.LinkedList;
import java.util.ArrayList;
import java.util.Map;
import java.util.HashMap;
import java.util.NoSuchElementException;
import au.edu.monash.csse.tonyj.cider.constraints.ConstraintCompiler;

public class ProductionDefinition extends RuleDefinition {

	// Symbols involved - maps name to a symbol definition
	private Map		inputSymbolSets;

	// XML element
	private Element		attAssignmentElement;

	// Other required mappings
	private Set		setAtts;			// Contains the attributes that are set
	private Set		initAtts;			// Contains the attributes that need to be initialized

	// Constructor, which takes an identification number, the grammar name, 
	// a DOM Node for the Production, a list of symbol definitions and a dependency graph
	public ProductionDefinition(int num, String gName, Element production, List sdList, DependencyGraph dependencyGraph,
		AttDependencyGraph attDependencyGraph, String conClass, String conExpClass, ConstraintCompiler cc)
	{
		super(true, num, gName, production, sdList, conClass, conExpClass, cc);
		inputSymbolSets = new HashMap();
		setAtts = new HashSet();
		initAtts = new HashSet();

		// Check node name
		if (!production.getTagName().equals("Production")) 
			Error.die(this, production.getAttribute("_line_number_"), "Expecting a <Production> element");

		// Get XML elements
		mainConstraintSetElement = getChildElement(production, "ConstraintSet");
		attAssignmentElement = getChildElement(production, "AttributeAssignment");

		// Read in the output, input and existential symbols (in that order)
		// and determine the dependencies that result from each production
		readOutputSymbols();
		readInputSymbols(dependencyGraph);
		if (mainConstraintSetElement != null)
			readExistentialSymbols(mainConstraintSetElement, true, true, dependencyGraph);

		// Parse all attribute strings
		parseAttributeStrings(ruleElement);

		// Check all constraint attribute dependencies
		if (mainConstraintSetElement != null)
			checkVariables(mainConstraintSetElement, new HashMap());

		// Check all attribute assignments
		if (attAssignmentElement != null) 
			checkVariables(attAssignmentElement, referableExistentialSymbols);
		checkAssignments(attDependencyGraph);

		// Move all referable symbol constraints to main constraint set
		relocateReferableSymbolConstraints();

		// Label all constraints
		if (mainConstraintSetElement != null) {
			labelConstraints(mainConstraintSetElement, "");
			labelConstraintExistsDependencies();
		}

		// Generate code for all the function references
		generateCodeForAllFunctions();
	}

	// Reads in the input symbols
	private void readInputSymbols(DependencyGraph dependencyGraph)
	{
		// Get input symbols
		Element inputs = getChildElement(ruleElement, "InputSymbols");
		if (inputs == null)
			Error.die(this, inputs.getAttribute("_line_number_"), 
				"Expecting <Production> element to contain an <InputSymbols> element");

		// Make sure at least one input symbol exists
                NodeList children = inputs.getChildNodes();
		if (children.getLength() == 0)
			Error.die(this, inputs.getAttribute("_line_number_"), 
				"Expecting <InputSymbols> element to contain at least one input symbol");

		// Read in input symbols and symbol sets
                for (int i = 0; i < children.getLength(); i++) {
                        Element element = (Element) children.item(i);

			// Check node name
			if (element.getTagName().equals("Symbol")) {

				// Check for symbol name duplication
				if (outputSymbols.containsKey(element.getAttribute("name")) || 
					inputSymbols.containsKey(element.getAttribute("name")) || 
					inputSymbolSets.containsKey(element.getAttribute("name")))
					Error.die(this, element.getAttribute("_line_number_"), 
						"Production Rule contains a symbol name duplication: " + element.getAttribute("name"));
				
				// Check for valid symbol type
				SymbolDefinition sDef = getSymbolDefinition(element.getAttribute("type"));
				if (sDef == null)
					Error.die(this, element.getAttribute("_line_number_"),
						"Production rule refers to undefined symbol type: " + element.getAttribute("type"));

				inputSymbols.put(element.getAttribute("name"), sDef);

				// For each output symbol, add a reduction dependency
				Set outputSymbolTypes = new HashSet(outputSymbols.values());
				Iterator iter = outputSymbolTypes.iterator();
				while (iter.hasNext()) 
					dependencyGraph.addReductionDependency(sDef, (SymbolDefinition) iter.next());
			}
			else if (element.getTagName().equals("SymbolSet")) {

				// Check for symbol name duplication
				if (outputSymbols.containsKey(element.getAttribute("name")) || 
					inputSymbols.containsKey(element.getAttribute("name")) || 
					inputSymbolSets.containsKey(element.getAttribute("name")))
					Error.die(this, element.getAttribute("_line_number_"),
						"Production Rule contains a symbol name duplication: " + element.getAttribute("name"));
				
				// Check for valid symbol type
				SymbolDefinition sDef = getSymbolDefinition(element.getAttribute("type"));
				if (sDef == null)
					Error.die(this, element.getAttribute("_line_number_"),
						"Production rule refers to undefined symbol type: " + element.getAttribute("type"));

				inputSymbolSets.put(element.getAttribute("name"), sDef);

				// For each output symbol, add existential dependencies
				Set outputSymbolTypes = new HashSet(outputSymbols.values());
				Iterator iter = outputSymbolTypes.iterator();
				while (iter.hasNext()) {
					SymbolDefinition s2 = (SymbolDefinition) iter.next();
					dependencyGraph.addPositiveExistentialDependency(sDef, s2);
					dependencyGraph.addNegativeExistentialDependency(sDef, s2);
				}
			}
			else 
				Error.die(this, element.getAttribute("_line_number_"),
					"Expecting <InputSymbols> to contain only <Symbol> or <SymbolSets> elements");
                }
	}

	// Reads in the output symbols
	private void readOutputSymbols()
	{
		// Get output symbols
		Element outputs = getChildElement(ruleElement, "OutputSymbols");
		if (outputs == null)
			Error.die(this, outputs.getAttribute("_line_number_"),
				"Expecting <Production> element to contain an <OutputSymbols> element");

		// Read in output symbols 
                NodeList children = outputs.getChildNodes();
                for (int i = 0; i < children.getLength(); i++) {
                        Element element = (Element) children.item(i);

			// Check node name
			if (element.getTagName().equals("Symbol")) {

				// Check for symbol name duplication
				if (outputSymbols.containsKey(element.getAttribute("name")))
					Error.die(this, element.getAttribute("_line_number_"),
						"Production rule contains a symbol name duplication: " + element.getAttribute("name"));
				
				// Check for valid symbol type
				SymbolDefinition sDef = getSymbolDefinition(element.getAttribute("type"));
				if (sDef == null)
					Error.die(this, element.getAttribute("_line_number_"),
						"Production rule refers to undefined symbol type: " + element.getAttribute("type"));

				// Check that output symbol is not abstract
				if (sDef.isAbstract())
					Error.die(this, element.getAttribute("_line_number_"),
						"Output symbol of production rule cannot be abstract: " + element.getAttribute("type"));

				outputSymbols.put(element.getAttribute("name"), sDef);
			}
			else 
				Error.die(this, element.getAttribute("_line_number_"),
					"Expecting <OutputSymbols> to contain only <Symbol> elements");
                }
	}

	// Recursively looks through a DOM subtree for nodes with a Symbol tag
	private void readExistentialSymbols(Node node, boolean referable, boolean positiveDependency, DependencyGraph dependencyGraph)
	{
		NodeList children = node.getChildNodes();
		for (int i = 0; i < children.getLength(); i++) {
			Node child = children.item(i);
			if (child.getNodeName().equals("Symbol")) {
				Element element = (Element) child;

				// Check for symbol name duplication
				if (inputSymbols.containsKey(element.getAttribute("name")) || 
					inputSymbolSets.containsKey(element.getAttribute("name")) ||
					outputSymbols.containsKey(element.getAttribute("name")) ||
					existentialSymbols.containsKey(element.getAttribute("name")))
					Error.die(this, element.getAttribute("_line_number_"),
						"Production Rule contains a symbol name duplication: " + element.getAttribute("name"));

				// Check for valid symbol type
				SymbolDefinition sDef = getSymbolDefinition(element.getAttribute("type"));
				if (sDef == null)
					Error.die(this, element.getAttribute("_line_number_"),
						"Production rule refers to undefined symbol type: " + element.getAttribute("type"));

				existentialSymbols.put(element.getAttribute("name"), sDef);
				if (referable)
					referableExistentialSymbols.put(element.getAttribute("name"), sDef);
				else if (positiveDependency)
					positiveExistentialSymbols.put(element.getAttribute("name"), sDef);
				else
					negativeExistentialSymbols.put(element.getAttribute("name"), sDef);

				// For each output symbol, add appropriate existential dependency
				Set outputSymbolTypes = new HashSet(outputSymbols.values());
				Iterator iter = outputSymbolTypes.iterator();
				while (iter.hasNext()) {
					SymbolDefinition s2 = (SymbolDefinition) iter.next();
					if (positiveDependency)
						dependencyGraph.addPositiveExistentialDependency(sDef, s2);
					else 
						dependencyGraph.addNegativeExistentialDependency(sDef, s2);
				}
			}

			// Recursive call, adjusting the dependency type if needed
			if (child.getNodeName().equals("NotExists"))
				readExistentialSymbols(child, false, !positiveDependency, dependencyGraph);
			else
				readExistentialSymbols(child, referable, positiveDependency, dependencyGraph);
		}
	}

	// Parses all string of the form [symbol_name].[attribute_name]
	private void parseAttributeStrings(Element element)
	{
		NodeList children = element.getChildNodes();
		for (int i = 0; i < children.getLength(); i++) {

			// Ignore non-element nodes
			if (children.item(i).getNodeType() != Node.ELEMENT_NODE)
				continue;
			Element child = (Element) children.item(i);

			// See if it is a SymbolValue element
			if (child.getTagName().equals("SymbolValue")) {

				// Check symbol of that symbol name exists
				String symbolName = child.getAttribute("name");
				SymbolDefinition sDef = (SymbolDefinition) inputSymbols.get(symbolName);
				if (sDef == null)
					sDef = (SymbolDefinition) outputSymbols.get(symbolName);
				if (sDef == null)
					sDef = (SymbolDefinition) existentialSymbols.get(symbolName);
				if (sDef == null)
					Error.die(this, child.getAttribute("_line_number_"),
						"Symbol name not defined for this production: " + symbolName);
			}

			// See if it is an InitAttribute, SetAttribute, AttributeValue or Variable element
			if (child.getTagName().equals("InitAttribute") || child.getTagName().equals("SetAttribute") ||
				child.getTagName().equals("AttributeValue") || child.getTagName().equals("Variable")) {

				// Split name string from [symbol_name].[attribute_name] format
				String name = child.getAttribute("name");
				int dotIndex = name.indexOf('.');
				if (dotIndex == -1)
					Error.die(this, child.getAttribute("_line_number_"),
						"Invalid attribute reference format: dot missing in: " + name);
				String symbolName = name.substring(0, dotIndex);
				String attName = name.substring(dotIndex + 1);

				// Add name components to node
				child.setAttribute("symbol_name", symbolName);
				child.setAttribute("attribute_name", attName);

				// Check symbol of that symbol name exists
				SymbolDefinition sDef = (SymbolDefinition) inputSymbols.get(symbolName);
				if (sDef == null)
					sDef = (SymbolDefinition) outputSymbols.get(symbolName);
				if (sDef == null)
					sDef = (SymbolDefinition) existentialSymbols.get(symbolName);
				if (sDef == null)
					Error.die(this, child.getAttribute("_line_number_"),
						"Symbol name not defined for this production: " + symbolName);

				// Check that symbol has specified attribute
				Attribute attr = sDef.getAttribute(attName);
				if (attr == null)
					Error.die(this, child.getAttribute("_line_number_"),
						"Attribute \'" + attName + "\' not defined for symbol \'" + symbolName + "\'");

				// Check that Variable nodes only contain constraint variable attributes
				if (child.getTagName().equals("Variable")) {
					if (!attr.getDataType().equals("ConstraintVariable"))
						Error.die(this, child.getAttribute("_line_number_"),
							"<Variable> element refers to an attribute which is not a constraint variable");
				}

				// Add symbol type and attribute datatype to node 
				child.setAttribute("symbol_type", sDef.getName());
				child.setAttribute("attribute_datatype", attr.getDataType());
			}
			parseAttributeStrings(child);
		}
	}

	// Recursively validates all variables in a DOM subtree
	private void checkVariables(Element element, Map validExistentialSymbols)
	{
		NodeList children = element.getChildNodes();
		for (int i = 0; i < children.getLength(); i++) {

			// Ignore non-element nodes
			if (children.item(i).getNodeType() != Node.ELEMENT_NODE)
				continue;
			Element child = (Element) children.item(i);

			// If it is an exists node, add to valid existential symbols (no need to recheck it)
			if (child.getTagName().equals("Exists") || child.getTagName().equals("NotExists")) {
				Element symbol = getChildElement(child, "Symbol");
				SymbolDefinition sDef = getSymbolDefinition(symbol.getAttribute("type"));
				validExistentialSymbols.put(symbol.getAttribute("name"), sDef);
			}

			// If it is a set attribute node for an input symbol attribute, allow output symbols
			if (child.getTagName().equals("SetAttribute")) {
				if (inputSymbols.containsKey(child.getAttribute("symbol_name")))
					validExistentialSymbols.putAll(outputSymbols);
			}

			// See if it is an AttributeValue or Variable element
			if (child.getTagName().equals("AttributeValue") || child.getTagName().equals("Variable")) {

				// Check symbol of a valid type
				String symbolName = child.getAttribute("symbol_name");
				if (!(inputSymbols.containsKey(symbolName) || validExistentialSymbols.containsKey(symbolName)))
					Error.die(this, child.getAttribute("_line_number_"),
						"Invalid symbol name for this part of the production: " + symbolName);
			}
			checkVariables(child, validExistentialSymbols);

			// Remove existential symbol if exiting exists node subtree
			if (child.getTagName().equals("Exists") || child.getTagName().equals("NotExists")) {
				Element symbol = getChildElement(child, "Symbol");
				validExistentialSymbols.remove(symbol.getAttribute("name"));
			}

			// Remove output symbols if exiting a set attribute subtree
			if (child.getTagName().equals("SetAttribute")) {
				if (inputSymbols.containsKey(child.getAttribute("symbol_name"))) {
					Iterator iter = outputSymbols.keySet().iterator();
					while (iter.hasNext())
						validExistentialSymbols.remove(iter.next());
				}
			}
		}
	}

	// This method checks for valid attribute assignments
	private void checkAssignments(AttDependencyGraph attDependencyGraph)
	{
		Set outputAtts = new HashSet();

		// Examine all the init attribute elements
		NodeList nodes = ruleElement.getElementsByTagName("InitAttribute");
		for (int i = 0; i < nodes.getLength(); i++) {
			Element initAttElement = (Element) nodes.item(i);

			// Check for duplication
			String name = initAttElement.getAttribute("name");
			if (!initAtts.add(name))
				Error.die(this, initAttElement.getAttribute("_line_number_"), 
					"Name duplication when initializing attribute: " + name);

			// Check symbol of a valid type
			String symbolName = initAttElement.getAttribute("symbol_name");
			if (!outputSymbols.containsKey(symbolName))
				Error.die(this, initAttElement.getAttribute("_line_number_"),
					"No such output symbol name defined for this production: " + symbolName);
		}

		// Examine all the set attribute elements
		nodes = ruleElement.getElementsByTagName("SetAttribute");
		for (int i = 0; i < nodes.getLength(); i++) {
			Element setAttElement = (Element) nodes.item(i);

			// Check for duplication
			String name = setAttElement.getAttribute("name");
			if (!setAtts.add(name))
				Error.die(this, setAttElement.getAttribute("_line_number_"), "Cannot set attribute more than once: " + name);
			if (initAtts.contains(name))
				Error.die(this, setAttElement.getAttribute("_line_number_"), "Attribute cannot be both initialized and set: " + name);

			// Check symbol of a valid type
			String symbolName = setAttElement.getAttribute("symbol_name");
			if (!(inputSymbols.containsKey(symbolName) || outputSymbols.containsKey(symbolName)))
				Error.die(this, setAttElement.getAttribute("_line_number_"),
					"No such input or output symbol name defined for this production: " + symbolName);

			// Update the attribute dependency graph
			String setteeName = setAttElement.getAttribute("symbol_type") + "." + setAttElement.getAttribute("attribute_name");
			NodeList dependencies = setAttElement.getElementsByTagName("AttributeValue");
			for (int j = 0; j < dependencies.getLength(); j++) {
				Element setByElement = (Element) dependencies.item(j);
				String setByName = setByElement.getAttribute("symbol_type") + "." + setByElement.getAttribute("attribute_name");
				if (outputSymbols.containsKey(symbolName))
					attDependencyGraph.addDependency(setteeName, setByName, AttDependencyGraphNode.BELOW);
				else
					attDependencyGraph.addDependency(setteeName, setByName, AttDependencyGraphNode.ABOVE);
			}
			dependencies = setAttElement.getElementsByTagName("Variable");
			for (int j = 0; j < dependencies.getLength(); j++) {
				Element setByElement = (Element) dependencies.item(j);
				String setByName = setByElement.getAttribute("symbol_type") + "." + setByElement.getAttribute("attribute_name");
				if (outputSymbols.containsKey(symbolName))
					attDependencyGraph.addDependency(setteeName, setByName, AttDependencyGraphNode.BELOW);
				else
					attDependencyGraph.addDependency(setteeName, setByName, AttDependencyGraphNode.ABOVE);
			}
		}

		// Create a list of all the output attributes
		Iterator iter = outputSymbols.keySet().iterator();
		while (iter.hasNext()) {
			String name = (String) iter.next();
			SymbolDefinition sDef = (SymbolDefinition) outputSymbols.get(name);

			Iterator iter2 = sDef.getAllAttributes().iterator();
			while (iter2.hasNext()) {
				Attribute attr = (Attribute) iter2.next();
				outputAtts.add(name + "." + attr.getName());
			}
		}

		// Make sure all output attributes were set or initialized
		Set setAndInitAtts = new HashSet(initAtts);
		setAndInitAtts.addAll(setAtts);
		if (!setAndInitAtts.containsAll(outputAtts))
			Error.die(this, "Production rule " + number + " contains output symbol attributes which were not set or initialized");

		// Make sure none of the constraints use an attribute that is 
		// set by the rule, as this may result in an infinite loop
		if (mainConstraintSetElement != null) {
			nodes = mainConstraintSetElement.getElementsByTagName("AttributeValue");
			for (int i = 0; i < nodes.getLength(); i++) {
				Element element = (Element) nodes.item(i);
				if (setAtts.contains(element.getAttribute("name"))) 
					Error.die(this, "Production rule " + number + 
						" constraints cannot use an attribute that is set by that rule: " + element.getAttribute("name"));
			}
			nodes = mainConstraintSetElement.getElementsByTagName("Variable");
			for (int i = 0; i < nodes.getLength(); i++) {
				Element element = (Element) nodes.item(i);
				if (setAtts.contains(element.getAttribute("name"))) 
					Error.die(this, "Production rule " + number + 
						" constraints cannot use an attribute that is set by that rule: " + element.getAttribute("name"));
			}
		}

		// Verify that all constraint expressions apply to a constraint variable
		nodes = ruleElement.getElementsByTagName("ConstraintExpression");
		for (int i = 0; i < nodes.getLength(); i++) {
			Element element = (Element) nodes.item(i);
			Element parent = (Element) element.getParentNode();
			if (!parent.getAttribute("attribute_datatype").equals("ConstraintVariable"))
				Error.die(this, parent.getAttribute("_line_number_"), 
					"Only constraint variables can be set with a constraint expression");
		}
	}

	// Produces a production record class for this production
	public void createProductionRecordClass()
	{
		int num, num2;
		Iterator iter;
		NodeList nodes;
		String constructor = "";
		String upToDate = "";
		String recheck = "";
		String print = "";
		String line;

		// Open file for the class
		PrintWriter output = Compiler.openTextFile(Compiler.PACKAGE_DIRECTORY + "ProductionRecord" + number + ".java");

		// Print class header details (package scope)
		output.println("");
		output.println(Compiler.PACKAGE_LINE);
		output.println("");
		output.println("import au.edu.monash.csse.tonyj.cider.constraints.ConstraintSolver;");
		output.println("import au.edu.monash.csse.tonyj.cider.constraints.Constraint;");
		output.println("import au.edu.monash.csse.tonyj.cider.constraints.ConstraintExpression;");
		output.println("import java.io.PrintStream;");
		output.println("import java.util.Iterator;");
		output.println("import java.util.List;");
		output.println("import java.util.LinkedList;");
		output.println("import java.util.Set;");
		output.println("import java.util.HashSet;");
		output.println("");
		output.println("class ProductionRecord" + number + " extends ProductionRecord {");
		output.println("");

		// Print output symbol variables
		print += "\t\tstream.println(\"Output Symbols:\");\n";
		iter = outputSymbols.keySet().iterator();
		while (iter.hasNext()) {
			String name = (String) iter.next();
			SymbolDefinition sDef = (SymbolDefinition) outputSymbols.get(name);
			output.println("\tprivate " + sDef.getName() + " _" + name + ";");
			constructor += "\t\t_" + name + " = null;\n";
			print += "\t\tstream.println(\"    " + name + ": \" + _" + name + ".toString());\n";
		}

		// Print input symbol variables
		print += "\t\tstream.println(\"Input Symbols:\");\n";
		iter = inputSymbols.keySet().iterator();
		while (iter.hasNext()) {
			String name = (String) iter.next();
			SymbolDefinition sDef = (SymbolDefinition) inputSymbols.get(name);
			output.println("\tprivate " + sDef.getName() + " _" + name + ";");
			constructor += "\t\t_" + name + " = null;\n";
			print += "\t\tstream.println(\"    " + name + ": \" + _" + name + ".toString());\n";
		}

		// Print input symbol set variables
		iter = inputSymbolSets.keySet().iterator();
		while (iter.hasNext()) {
			String name = (String) iter.next();
			output.println("\tprivate Set _" + name + ";");
			constructor += "\t\t_" + name + " = new HashSet();\n";
			print += "\t\titer = _" + name + ".iterator();\n";
			print += "\t\twhile (iter.hasNext())\n";
			print += "\t\t\tstream.println(\"    Member of " + name + ": \" + ((GrammarSymbol) iter.next()).toString());\n";
		}

		// Print existential symbol variables
		iter = existentialSymbols.keySet().iterator();
		while (iter.hasNext()) {
			String name = (String) iter.next();
			SymbolDefinition sDef = (SymbolDefinition) existentialSymbols.get(name);
			output.println("\tprivate " + sDef.getName() + " _" + name + ";");
			constructor += "\t\t_" + name + " = null;\n";
		}

		// Only print existential symbols that can be referred to
		if (!referableExistentialSymbols.keySet().isEmpty())
			print += "\t\tstream.println(\"Referable Existential Symbols:\");\n";
		iter = referableExistentialSymbols.keySet().iterator();
		while (iter.hasNext()) {
			String name = (String) iter.next();
			SymbolDefinition sDef = (SymbolDefinition) referableExistentialSymbols.get(name);
			print += "\t\tstream.println(\"    " + name + ": \" + (_" + name + " == null ? \"<No Symbol>\" : _" + 
				name + ".toString()));\n";
		}

		output.println("");
		output.println("\tprivate boolean alive;");
		constructor += "\n\t\talive = true;\n";

		// Print check constraint variables
		if (mainConstraintSetElement != null) {
			nodes = mainConstraintSetElement.getChildNodes();
			for (int i = 0; i < nodes.getLength(); i++) {
				Element element = (Element) nodes.item(i);
				output.println("\tprivate boolean checkConstraint" + element.getAttribute("label") + ";");
				constructor += "\t\tcheckConstraint" + element.getAttribute("label") + " = false;\n";
				recheck += "\t\tcheckConstraint" + element.getAttribute("label") + " = true;\n";
			}
		}

		// Print update symbol attributes variables
		iter = setAtts.iterator();
		while (iter.hasNext()) {
			String name = (String) iter.next();
			name = name.replace('.', '_');
			output.println("\tprivate boolean update_" + name + ";");		
			constructor += "\t\tupdate_" + name + " = false;\n";
			recheck += "\t\tupdate_" + name + " = true;\n";
		}

		// Print solver constraint variables for constraint set
		if (mainConstraintSetElement != null) {
			nodes = mainConstraintSetElement.getElementsByTagName("SolverConstraint");
			for (int i = 0; i < nodes.getLength(); i++) {
				Element element = (Element) nodes.item(i);
				if (element.getAttribute("dependsOn").equals("")) {
					output.println("\tprivate boolean constraint" + element.getAttribute("label") + "InSolver;");
					constructor += "\t\tconstraint" + element.getAttribute("label") + "InSolver = false;\n";
				}
			}
			for (int i = 0; i < nodes.getLength(); i++) {
				Element element = (Element) nodes.item(i);
				output.println("\tprivate Constraint constraint" + element.getAttribute("label") + ";");
			}
		}

		// Print solver constraint variables for set attributes
		if (attAssignmentElement != null) {
			nodes = attAssignmentElement.getElementsByTagName("ConstraintExpression");
			for (int i = 0; i < nodes.getLength(); i++) {
				Element element = (Element) nodes.item(i);
				Element parent = (Element) element.getParentNode();
				String name = parent.getAttribute("symbol_name") + "_" + parent.getAttribute("attribute_name");

				output.println("\tprivate boolean att_" + name + "_ConstraintInSolver;");
				output.println("\tprivate Constraint att_" + name + "_Constraint;");
				output.println("\tprivate ConstraintExpression att_" + name + "_ConExp;");
				constructor += "\t\tatt_" + name + "_ConstraintInSolver = false;\n";
			}
		}

		// Print solver constraints to make attributes constant
		Set constantAttributes = new HashSet();
		if (attAssignmentElement != null) {
			nodes = attAssignmentElement.getElementsByTagName("SetAttribute");
			for (int i = 0; i < nodes.getLength(); i++) {
				Element element = (Element) nodes.item(i);
				Element ce = getChildElement(element, "ConstraintExpression");

				// Only apply to constraint variables that are not set by a constraint expression
				if (element.getAttribute("attribute_datatype").equals("ConstraintVariable") && ce == null) {
					String fullName =  element.getAttribute("symbol_name") + "_" + element.getAttribute("attribute_name");
					output.println("\tprivate boolean constant_" + fullName + "_ConstraintInSolver;");
					output.println("\tprivate Constraint constant_" + fullName + "_Constraint;");
					constructor += "\t\tconstant_" + fullName + "_ConstraintInSolver = false;\n";
					element.setAttribute("full_name", fullName);
					constantAttributes.add(element);
				}
			}
		}

		// Print constructor
		output.println("");
		output.println("\tpublic ProductionRecord" + number + "()");
		output.println("\t{");
		output.print(constructor);
		output.println("\t}");

		// Print full recheck method
		output.println("");
		output.println("\tpublic void fullRecheck()");
		output.println("\t{");
		output.print(recheck);
		output.println("\t}");

		// Print print method
		output.println("");
		output.println("\tpublic void print(PrintStream stream)");
		output.println("\t{");
		output.println("\t\tIterator iter;");
		output.println("");
		output.println("\t\tstream.println(\"Production Rule " + number + "\");");
		output.print(print);
		output.println("\t}");

		// Reinitialize input symbol attributes that need it (when production no longer valid)
		output.println("");
		output.println("\tpublic void reinitializeInputSymbols(Interpreter interpreter)");
		output.println("\t{");

		// See if any constraint variables need modifying
		boolean cvFlag = false;
		iter = setAtts.iterator();
		while (iter.hasNext()) {
			String name = (String) iter.next();
			int dotIndex = name.indexOf('.');
			String symbolName = name.substring(0, dotIndex);
			String attName = name.substring(dotIndex + 1);

			if (inputSymbols.containsKey(symbolName)) { 
				SymbolDefinition sDef = (SymbolDefinition) inputSymbols.get(symbolName);
				Attribute attr = sDef.getAttribute(attName);
				if (attr.isConstraintVariable())
					cvFlag = true;
			}
		}

		if (cvFlag) {
			output.println("\t\tboolean tempACS = interpreter.isAutoConstraintSolvingOn();");
			output.println("\t\tinterpreter.setAutoConstraintSolving(false);");
		}
		iter = setAtts.iterator();
		while (iter.hasNext()) {
			String name = (String) iter.next();
			int dotIndex = name.indexOf('.');
			String symbolName = name.substring(0, dotIndex);
			String attName = name.substring(dotIndex + 1);

			if (inputSymbols.containsKey(symbolName)) 
				output.println("\t\t_" + symbolName + ".reinitialize_" + attName + "();");
		}
		if (cvFlag) {
			output.println("\t\tinterpreter.setAutoConstraintSolving(tempACS);");
			output.println("\t\tinterpreter.resolve();");
		}
		output.println("\t}");

		// Print output symbol get and set methods
		iter = outputSymbols.keySet().iterator();
		while (iter.hasNext()) {
			String name = (String) iter.next();
			SymbolDefinition sDef = (SymbolDefinition) outputSymbols.get(name);
			output.println("");
			output.println("\tpublic " + sDef.getName() + " get_" + name + "()");
			output.println("\t{");
			output.println("\t\treturn _" + name + ";");
			output.println("\t}");
			output.println("");
			output.println("\tpublic void set_" + name + "(" + sDef.getName() + " value)");
			output.println("\t{");
			output.println("\t\t_" + name + " = value;");
			output.println("\t}");
		}

		// Print input symbolset get methods
		iter = inputSymbolSets.keySet().iterator();
		while (iter.hasNext()) {
			String name = (String) iter.next();
			output.println("");
			output.println("\tpublic Set get_" + name + "()");
			output.println("\t{");
			output.println("\t\treturn _" + name + ";");
			output.println("\t}");
		}

		// Print input symbol get and set methods
		iter = inputSymbols.keySet().iterator();
		while (iter.hasNext()) {
			String name = (String) iter.next();
			SymbolDefinition sDef = (SymbolDefinition) inputSymbols.get(name);
			output.println("");
			output.println("\tpublic " + sDef.getName() + " get_" + name + "()");
			output.println("\t{");
			output.println("\t\treturn _" + name + ";");
			output.println("\t}");
			output.println("");
			output.println("\tpublic void set_" + name + "(" + sDef.getName() + " value)");
			output.println("\t{");
			output.println("\t\t_" + name + " = value;");
			output.println("\t}");
		}

		// Print existential symbol get and set methods
		iter = existentialSymbols.keySet().iterator();
		while (iter.hasNext()) {
			String name = (String) iter.next();
			SymbolDefinition sDef = (SymbolDefinition) existentialSymbols.get(name);
			output.println("");
			output.println("\tpublic " + sDef.getName() + " get_" + name + "()");
			output.println("\t{");
			output.println("\t\treturn _" + name + ";");
			output.println("\t}");
			output.println("");
			output.println("\tpublic void set_" + name + "(" + sDef.getName() + " value)");
			output.println("\t{");
			output.println("\t\t_" + name + " = value;");
			output.println("\t}");
		}

		// Print solver constraint get and set methods
		if (mainConstraintSetElement != null) {
			nodes = mainConstraintSetElement.getElementsByTagName("SolverConstraint");
			for (int i = 0; i < nodes.getLength(); i++) {
				Element element = (Element) nodes.item(i);
				output.println("");
				output.println("\tpublic Constraint getConstraint" + element.getAttribute("label") + "()");
				output.println("\t{");
				output.println("\t\treturn constraint" + element.getAttribute("label") + ";");
				output.println("\t}");
				output.println("");
				output.println("\tpublic void setConstraint" + element.getAttribute("label") + "(Constraint c)");
				output.println("\t{");
				output.println("\t\tconstraint" + element.getAttribute("label") + " = c;");
				output.println("\t}");
			}
		}

		// Print attribute constraint get and set methods
		if (attAssignmentElement != null) {
			nodes = attAssignmentElement.getElementsByTagName("ConstraintExpression");
			for (int i = 0; i < nodes.getLength(); i++) {
				Element element = (Element) nodes.item(i);
				Element parent = (Element) element.getParentNode();
				String name = parent.getAttribute("symbol_name") + "_" + parent.getAttribute("attribute_name");
				output.println("");
				output.println("\tpublic Constraint getAtt_" + name + "_Constraint()");
				output.println("\t{");
				output.println("\t\treturn att_" + name + "_Constraint;");
				output.println("\t}");
				output.println("");
				output.println("\tpublic void setAtt_" + name + "_Constraint(Constraint c)");
				output.println("\t{");
				output.println("\t\tatt_" + name + "_Constraint = c;");
				output.println("\t}");
				output.println("");
				output.println("\tpublic ConstraintExpression getAtt_" + name + "_ConExp()");
				output.println("\t{");
				output.println("\t\treturn att_" + name + "_ConExp;");
				output.println("\t}");
				output.println("");
				output.println("\tpublic void setAtt_" + name + "_ConExp(ConstraintExpression ce)");
				output.println("\t{");
				output.println("\t\tatt_" + name + "_ConExp = ce;");
				output.println("\t}");
			}
		}

		// Print method to return output symbols
		output.println("");
		output.println("\tpublic Set getOutputSymbols()");
		output.println("\t{");
		output.println("\t\tSet symbols = new HashSet();");
		iter = outputSymbols.keySet().iterator();
		while (iter.hasNext()) {
			String name = (String) iter.next();
			output.println("\t\tsymbols.add(_" + name + ");");
		}
		output.println("\t\treturn symbols;");
		output.println("\t}");
		
		// Print method to return output symbol nodes
		output.println("");
		output.println("\tpublic Set getOutputSymbolNodes()");
		output.println("\t{");
		output.println("\t\tSet nodes = new HashSet();");
		iter = outputSymbols.keySet().iterator();
		while (iter.hasNext()) {
			String name = (String) iter.next();
			output.println("\t\tnodes.add(_" + name + ".getSymbolNode());");
		}
		output.println("\t\treturn nodes;");
		output.println("\t}");

		// Print method to return input symbol nodes
		output.println("");
		output.println("\tpublic Set getInputSymbolNodes()");
		output.println("\t{");
		output.println("\t\tIterator iter;");
		output.println("\t\tGrammarSymbol symbol;");
		output.println("\t\tSet nodes = new HashSet();");
		output.println("");
		iter = inputSymbols.keySet().iterator();
		while (iter.hasNext()) {
			String name = (String) iter.next();
			output.println("\t\tnodes.add(_" + name + ".getSymbolNode());");
		}
		iter = inputSymbolSets.keySet().iterator();
		while (iter.hasNext()) {
			String name = (String) iter.next();
			output.println("\t\titer = _" + name + ".iterator();");
			output.println("\t\twhile (iter.hasNext()) {");
			output.println("\t\t\tsymbol = (GrammarSymbol) iter.next();");
			output.println("\t\t\tnodes.add(symbol.getSymbolNode());");
			output.println("\t\t}");
		}
		output.println("\t\treturn nodes;");
		output.println("\t}");

		// Print method to return referable existential symbol nodes
		output.println("");
		output.println("\tpublic Set getReferableExistentialSymbolNodes()");
		output.println("\t{");
		output.println("\t\tSet nodes = new HashSet();");
		iter = referableExistentialSymbols.keySet().iterator();
		while (iter.hasNext()) {
			String name = (String) iter.next();
			output.println("\t\tnodes.add(_" + name + ".getSymbolNode());");
		}
		output.println("\t\treturn nodes;");
		output.println("\t}");

		// Print symbol added method, which only affects negative existential symbols and input symbol sets
		output.println("");
		output.println("\tpublic void symbolAdded(GrammarSymbol symbol)");
		output.println("\t{");

		// Check negative existential symbols
		iter = negativeExistentialSymbols.keySet().iterator();
		while (iter.hasNext()) {
			String name = (String) iter.next();
			SymbolDefinition sDef = (SymbolDefinition) negativeExistentialSymbols.get(name);

			// Get label for the base constraint
			String label = (String) existentialConstraintLabels.get(name);
			int index = label.indexOf('_');
			if (index != -1)
				label = label.substring(0, index);
			output.println("\t\tif (symbol instanceof " + sDef.getName() + ")");
			output.println("\t\t\tcheckConstraint" + label + " = true;");
		}

		// Check input symbol sets
		num = 0;
		iter = inputSymbolSets.keySet().iterator();
		while (iter.hasNext()) {
			String name = (String) iter.next();
			SymbolDefinition sDef = (SymbolDefinition) inputSymbolSets.get(name);
			output.println("\t\t" + (num > 0 ? "else if" : "if") + " (symbol instanceof " + sDef.getName() + ") {");
			output.println("\t\t\talive = false;");
			output.println("\t\t\treturn;");
			output.println("\t\t}");
			num++;
		}
		output.println("\t}");

		// Print symbol removed method, which involves checking the input, referable and 
		// positive existential symbols, as well as input symbol sets
		output.println("");
		output.println("\tpublic void symbolRemoved(GrammarSymbol symbol)");
		output.println("\t{");

		// Check input symbols
		num = 0;
		iter = inputSymbols.keySet().iterator();
		while (iter.hasNext()) {
			String name = (String) iter.next();
			output.println("\t\t" + (num > 0 ? "else if" : "if") + " (symbol == _" + name + ") {");
			output.println("\t\t\talive = false;");
			output.println("\t\t\treturn;");
			output.println("\t\t}");
			num++;
		}

		// Check referable existential symbols
		iter = referableExistentialSymbols.keySet().iterator();
		while (iter.hasNext()) {
			String name = (String) iter.next();
			output.println("\t\t" + (num > 0 ? "else if" : "if") + " (symbol == _" + name + ") {");
			output.println("\t\t\talive = false;");
			output.println("\t\t\treturn;");
			output.println("\t\t}");
			num++;
		}

		// Check positive existential symbols
		iter = positiveExistentialSymbols.keySet().iterator();
		while (iter.hasNext()) {
			String name = (String) iter.next();
			SymbolDefinition sDef = (SymbolDefinition) positiveExistentialSymbols.get(name);

			// Get label for the base constraint
			String label = (String) existentialConstraintLabels.get(name);
			int index = label.indexOf('_');
			if (index != -1)
				label = label.substring(0, index);
			output.println("\t\tif (symbol instanceof " + sDef.getName() + ")");
			output.println("\t\t\tcheckConstraint" + label + " = true;");
		}

		// Check input symbol sets
		num = 0;
		iter = inputSymbolSets.keySet().iterator();
		while (iter.hasNext()) {
			String name = (String) iter.next();
			SymbolDefinition sDef = (SymbolDefinition) inputSymbolSets.get(name);
			output.println("\t\t" + (num > 0 ? "else if" : "if") + " (symbol instanceof " + sDef.getName() + ") {");
			output.println("\t\t\talive = false;");
			output.println("\t\t\treturn;");
			output.println("\t\t}");
			num++;
		}
		output.println("\t}");

		// Print symbol modified method, which affects input and existential symbols
		output.println("");
		output.println("\tpublic void symbolModified(GrammarSymbol symbol, int attribute)");
		output.println("\t{");

		// Print required constraint checking 
		num2 = 0;
		Map allSymbols = new HashMap(inputSymbols);
		allSymbols.putAll(existentialSymbols);
		allSymbols.putAll(outputSymbols);
		iter = allSymbols.keySet().iterator();
		while (iter.hasNext()) {
			String name = (String) iter.next();
			SymbolDefinition sDef = (SymbolDefinition) allSymbols.get(name);

			// If a non-referable existential, account for if it doesn't exist as well
			if (positiveExistentialSymbols.containsKey(name) || negativeExistentialSymbols.containsKey(name))
				output.println("\t\tif (symbol instanceof " + sDef.getName() + ") {");
			else {
				output.println("\t\t" + (num2 > 0 ? "else if" : "if") + " (symbol == _" + name + ") {");
				num2++;
			}

			// Get symbol attributes
			num = 0;
			Iterator iter2 = sDef.getAllAttributes().iterator();
			while (iter2.hasNext()) {
				boolean required = false;
				Attribute attr = (Attribute) iter2.next();
				String string = name + "." + attr.getName();
				line = "\t\t\t" + (num > 0 ? "else if" : "if") + " (attribute == " + sDef.getName() + "._att_" + 
					attr.getName() + "_) {\n";
				
				// See which output symbol attribute sets are affected by this attribute
				nodes = ruleElement.getElementsByTagName("SetAttribute");
				for (int i = 0; i < nodes.getLength(); i++) {
					Element setAttElement = (Element) nodes.item(i);

					// Check attributes needed 
					NodeList attNodes = setAttElement.getElementsByTagName("AttributeValue");
					for (int j = 0; j < attNodes.getLength(); j++) {
						Element attElement = (Element) attNodes.item(j);
						if (attElement.getAttribute("name").equals(string)) {
							line += "\t\t\t\tupdate_" + setAttElement.getAttribute("symbol_name") + "_" +
								setAttElement.getAttribute("attribute_name") + " = true;\n";
							required = true;
							break;
						}
					}

					// Also check constraint expressions
					attNodes = setAttElement.getElementsByTagName("Variable");
					for (int j = 0; j < attNodes.getLength(); j++) {
						Element attElement = (Element) attNodes.item(j);
						if (attElement.getAttribute("name").equals(string)) {
							line += "\t\t\t\tif (!att_" + setAttElement.getAttribute("symbol_name") + "_" +
								setAttElement.getAttribute("attribute_name") + "_ConstraintInSolver)\n\t";
							line += "\t\t\t\tupdate_" + setAttElement.getAttribute("symbol_name") + "_" +
								setAttElement.getAttribute("attribute_name") + " = true;\n";
							required = true;
							break;
						}
					}
				}

				// See which constraints are affected by this attribute
				if (mainConstraintSetElement != null) {
					nodes = mainConstraintSetElement.getChildNodes();
					for (int i = 0; i < nodes.getLength(); i++) {
						Element element = (Element) nodes.item(i);

						// Check attributes needed 
						NodeList attNodes = element.getElementsByTagName("AttributeValue");
						for (int j = 0; j < attNodes.getLength(); j++) {
							Element attElement = (Element) attNodes.item(j);
							if (attElement.getAttribute("name").equals(string)) {
								line += "\t\t\t\tcheckConstraint" + element.getAttribute("label") + " = true;\n";
								required = true;
								break;
							}
						}

						// Also check solver constraints if necessary
						attNodes = element.getElementsByTagName("Variable");
						for (int j = 0; j < attNodes.getLength(); j++) {
							Element attElement = (Element) attNodes.item(j);
							Element scElement = (Element) attElement.getParentNode();
							if (attElement.getAttribute("name").equals(string)) {
								if (scElement.getAttribute("dependsOn").equals("")) 
									line += "\t\t\t\tif (!constraint" + element.getAttribute("label") + 
										"InSolver)\n\t";
								line += "\t\t\t\tcheckConstraint" + element.getAttribute("label") + " = true;\n";
								required = true;
								break;
							}
						}
					}
				}
				line += "\t\t\t}\n";
				if (required) {
					output.print(line);
					num++;
				}
			}
			output.println("\t\t}");
		}
		output.println("\t}");

		// Print the update symbol attributes method
		num = 0;
		output.println("");
		output.println("\tpublic boolean updateSymbolAttributes(Interpreter interpreter)");
		output.println("\t{");
		output.println("\t\tboolean updateOccurred = false;");

		if (!constantAttributes.isEmpty())
			output.println("\t\tConstraintSolver csolver = interpreter.getConstraintSolver();");

		output.println("");
		iter = setAtts.iterator();
		while (iter.hasNext()) {
			String name = (String) iter.next();
			int dotIndex = name.indexOf('.');
			String symbolName = name.substring(0, dotIndex);
			String attName = name.substring(dotIndex + 1);
			name = name.replace('.', '_');

			// See if value being updated is a constant
			Element constant = null;
			Iterator iter2 = constantAttributes.iterator();
			while (iter2.hasNext()) {
				Element element = (Element) iter2.next();
				if (element.getAttribute("full_name").equals(name)) {
					constant = element;
					break;
				}
			}

			output.println("\t\tif (update_" + name + ") {");		
			output.println("\t\t\tupdate_" + name + " = false;");
			output.println("\t\t\tupdateOccurred = true;");

			// If a constant constraint is present, change constraint to a new constant constraint
			// with the updated value. If this value cannot be accepted by the solver, invalidate the production
			if (constant != null) {
				output.println("\t\t\tif (constant_" + name + "_ConstraintInSolver) {");
				output.println("\t\t\t\tcsolver.removeConstraint(constant_" + name + "_Constraint);");
				output.println("\t\t\t\tconstant_" + name + "_ConstraintInSolver = false;");
				output.println("\t\t\t}");

				output.println("\t\t\tconstant_" + name + "_Constraint = new " + constraintClass + "();");
				output.println("\t\t\tconstant_" + name + "_Constraint.initializeConstant(_" + constant.getAttribute("symbol_name") 
					+ ".getConstraintVariable_" + constant.getAttribute("attribute_name") + "(), ProductionRule" 
					+ number + ".valueOf_" + name + "(this));");
				output.println("\t\t\tif (!csolver.addConstraint(constant_" + name + "_Constraint)) {");
				output.println("\t\t\t\t\tLog.getLogger().info(\"Production invalidated because new constant solver constraint " +
					"for attribute " + constant.getAttribute("name") + " could not be added in production " + number + "\");");
				output.println("\t\t\t\talive = false;");
				output.println("\t\t\t}");
				output.println("\t\t\telse {");
				output.println("\t\t\t\tconstant_" + name + "_ConstraintInSolver = true;");
				output.println("\t\t\t\tinterpreter.solve();");
				output.println("\t\t\t}");
			}
			else
				output.println("\t\t\t_" + symbolName + ".set_" + attName + "(ProductionRule" + number + ".valueOf_" + 
					name + "(this));");
			output.println("\t\t}");
			upToDate += (num > 0 ? " || " : "") + "update_" + name;
			num++;
		}
		output.println("\t\treturn updateOccurred;");
		output.println("\t}");

		// Print the reevaluate production method
		output.println("");
		output.println("\tpublic boolean reevaluateProduction(ParseForest forest)");
		output.println("\t{");
		output.println("\t\tif (!alive)");
		output.println("\t\t\treturn false;");
		if (mainConstraintSetElement != null) {
			nodes = mainConstraintSetElement.getChildNodes();
			for (int i = 0; i < nodes.getLength(); i++) {
				Element element = (Element) nodes.item(i);
				output.println("\t\tif (checkConstraint" + element.getAttribute("label") + ") {");
				output.println("\t\t\tcheckConstraint" + element.getAttribute("label") + " = false;");
				output.println("\t\t\talive = ProductionRule" + number + ".evaluateConstraint" + element.getAttribute("label") + 
					"(this, forest);");
				output.println("\t\t\tif (!alive)");
				output.println("\t\t\t\treturn false;");
				output.println("\t\t}");
				upToDate += (num > 0 ? " || " : "") + "checkConstraint" + element.getAttribute("label");
				num++;
			}
		}
		output.println("\t\treturn true;");
		output.println("\t}");

		// Print is up to date method
		output.println("");
		output.println("\tpublic boolean isUpToDate()");
		output.println("\t{");
		if (num == 0)
			output.println("\t\treturn true;");
		else
			output.println("\t\treturn !(" + upToDate + ");");
		output.println("\t}");

		// Print add constraints method
		List validNodes = new LinkedList();
		output.println("");
		output.println("\tpublic boolean addSolverConstraints(Interpreter interpreter)");
		output.println("\t{");
		if (mainConstraintSetElement != null) {
			nodes = mainConstraintSetElement.getElementsByTagName("SolverConstraint");
			for (int i = 0; i < nodes.getLength(); i++) {
				Element constraint = (Element) nodes.item(i);
				if (constraint.getAttribute("dependsOn").equals("")) 
					validNodes.add(constraint);
			}
		}
		if (attAssignmentElement != null) {
			nodes = attAssignmentElement.getElementsByTagName("ConstraintExpression");
			for (int j = 0; j < nodes.getLength(); j++) {
				Element constraintExp = (Element) nodes.item(j);
				validNodes.add(constraintExp.getParentNode());
			}
		}

		// See if any constraints to add
		if (!(validNodes.isEmpty() && constantAttributes.isEmpty())) {
			output.println("\t\tConstraintSolver csolver = interpreter.getConstraintSolver();");
			output.println("");
			iter = constantAttributes.iterator();
			while (iter.hasNext()) {
				Element element = (Element) iter.next();

				output.println("\t\tif (!constant_" + element.getAttribute("full_name") + "_ConstraintInSolver) {");
				output.println("\t\t\tif (constant_" + element.getAttribute("full_name") + "_Constraint == null) {");
				output.println("\t\t\t\tconstant_" + element.getAttribute("full_name") + "_Constraint = new " +
					constraintClass + "();");
				output.println("\t\t\t\tconstant_" + element.getAttribute("full_name") + "_Constraint.initializeConstant(_" 
					+ element.getAttribute("symbol_name") + ".getConstraintVariable_" + element.getAttribute("attribute_name") 
					+ "(), _" + element.getAttribute("symbol_name") + ".get_" + element.getAttribute("attribute_name") + "());");
				output.println("\t\t\t}");
				output.println("\t\t\tif (!csolver.addConstraint(constant_" + element.getAttribute("full_name") + "_Constraint))");
				output.println("\t\t\t\treturn false;");
				output.println("\t\t\tconstant_" + element.getAttribute("full_name") + "_ConstraintInSolver = true;");
				output.println("\t\t}");
			}
			iter = validNodes.iterator();
			while (iter.hasNext()) {
				Element constraint = (Element) iter.next();

				// See what sort of constraint it is
				if (constraint.getTagName().equals("SolverConstraint")) {
					output.println("\t\tif (!constraint" + constraint.getAttribute("label") + "InSolver) {");
					output.println("\t\t\tif (!csolver.addConstraint(constraint" + 
						constraint.getAttribute("label") + "))");
					output.println("\t\t\t\treturn false;");
					output.println("\t\t\tconstraint" + constraint.getAttribute("label") + "InSolver = true;");
					output.println("\t\t\tcheckConstraint" + constraint.getAttribute("label") + " = false;");
					output.println("\t\t}");
				}
				else {
					String name = constraint.getAttribute("symbol_name") + "_" + 
						constraint.getAttribute("attribute_name");
					output.println("\t\tif (!att_" + name + "_ConstraintInSolver) {");
					output.println("\t\t\tif (!csolver.addConstraint(att_" + name + "_Constraint))");
					output.println("\t\t\t\treturn false;");
					output.println("\t\t\tatt_" + name + "_ConstraintInSolver = true;");
					output.println("\t\t\tupdate_" + name + " = false;");
					output.println("\t\t}");
				}
			}
			output.println("\t\tinterpreter.solve();");
		}
		output.println("\t\treturn true;");
		output.println("\t}");

		// Print remove constraints method
		validNodes.clear();
		output.println("");
		output.println("\tpublic void removeSolverConstraints(Interpreter interpreter)");
		output.println("\t{");
		if (mainConstraintSetElement != null) {
			nodes = mainConstraintSetElement.getElementsByTagName("SolverConstraint");
			validNodes = new LinkedList();
			for (int i = 0; i < nodes.getLength(); i++) {
				Element constraint = (Element) nodes.item(i);
				if (constraint.getAttribute("dependsOn").equals("")) 
					validNodes.add(constraint);
			}
		}
		if (attAssignmentElement != null) {
			nodes = attAssignmentElement.getElementsByTagName("ConstraintExpression");
			for (int j = 0; j < nodes.getLength(); j++) {
				Element constraintExp = (Element) nodes.item(j);
				validNodes.add(constraintExp.getParentNode());
			}
		}

		// See if any constraints to remove
		if (!(validNodes.isEmpty() && constantAttributes.isEmpty())) {
			output.println("\t\tConstraintSolver csolver = interpreter.getConstraintSolver();");
			output.println("");
			iter = validNodes.iterator();
			while (iter.hasNext()) {
				Element constraint = (Element) iter.next();

				// See what sort of constraint it is
				if (constraint.getTagName().equals("SolverConstraint")) {
					String constraintName = "constraint" + constraint.getAttribute("label");
					output.println("\t\tif (" + constraintName + "InSolver) {");
					output.println("\t\t\tcsolver.removeConstraint(" + constraintName + ");");
					output.println("\t\t\t" + constraintName + "InSolver = false;");
					output.println("\t\t}");
				}
				else {
					String name = constraint.getAttribute("symbol_name") + "_" + 
						constraint.getAttribute("attribute_name");
					output.println("\t\tif (att_" + name + "_ConstraintInSolver) {");
					output.println("\t\t\tcsolver.removeConstraint(att_" + name + "_Constraint);");
					output.println("\t\t\tatt_" + name + "_ConstraintInSolver = false;");
					output.println("\t\t}");

				}
			}
			iter = constantAttributes.iterator();
			while (iter.hasNext()) {
				Element element = (Element) iter.next();

				output.println("\t\tif (constant_" + element.getAttribute("full_name") + "_ConstraintInSolver) {");
				output.println("\t\t\tcsolver.removeConstraint(constant_" + element.getAttribute("full_name") + "_Constraint);");
				output.println("\t\t\tconstant_" + element.getAttribute("full_name") + "_ConstraintInSolver = false;");
				output.println("\t\t}");
			}
			output.println("\t\tinterpreter.solve();");
		}
		output.println("\t}");

		// End class and close file
		output.println("}");
		output.println("");
		output.close();
	}

	// Produces a production rule class for this production
	public void createProductionRuleClass(DependencyGraph dependencyGraph)
	{
		String line;
		Iterator iter;
		NodeList nodes;

		// Open file for the class
		PrintWriter output = Compiler.openTextFile(Compiler.PACKAGE_DIRECTORY + "ProductionRule" + number + ".java");

		// Print class header details (package scope)
		output.println("");
		output.println(Compiler.PACKAGE_LINE);
		output.println("");
		output.println("import au.edu.monash.csse.tonyj.cider.constraints.ConstraintSolver;");
		output.println("import au.edu.monash.csse.tonyj.cider.constraints.ConstraintExpression;");
		output.println("import au.edu.monash.csse.tonyj.cider.constraints.Constraint;");
		output.println("import au.edu.monash.csse.tonyj.cider.constraints.ConstraintVariable;");
		output.println("import java.util.Iterator;");
		output.println("import java.util.Set;");
		output.println("import java.util.HashSet;");
		output.println("");
		output.println("class ProductionRule" + number + " {");
		output.println("");

		// Print simple evaluate method (with no candidate sets for input or referable existential symbols)
		output.println("\tpublic static ProductionRecord" + number + " evaluateEntireForest(ParseForest forest, ConstraintSolver csolver)");
		output.println("\t{");
		line = "\t\treturn evaluate(forest, csolver";
		for (int i = 0; i < inputSymbols.size(); i++)
			line += ", null";
		for (int i = 0; i < referableExistentialSymbols.size(); i++)
			line += ", null";
		line += ");";
		output.println(line);
		output.println("\t}");
		output.println("");

		// Print evaluate method, which returns the record if successful, or null if not
		line = "\tpublic static ProductionRecord" + number + " evaluate(ParseForest forest, ConstraintSolver csolver";
		iter = inputSymbols.keySet().iterator();
		while (iter.hasNext()) {
			String name = (String) iter.next();
			line += ", Set " + name + "_candidateNodes";
		}
		iter = referableExistentialSymbols.keySet().iterator();
		while (iter.hasNext()) {
			String name = (String) iter.next();
			line += ", Set " + name + "_candidateNodes";
		}
		line += ")";
		output.println(line);
		output.println("\t{");
		if (!inputSymbolSets.isEmpty())
			output.println("\t\tIterator iter;");
		output.println("\t\tSet usedSymbols = new HashSet();");
		output.println("");

		// Check for input symbol set conflicts
		Set inputTypes = new HashSet(inputSymbols.values());
		Set referableTypes = new HashSet(referableExistentialSymbols.values());
		Set existentialTypes = new HashSet(existentialSymbols.values());
		iter = inputSymbolSets.keySet().iterator();
		while (iter.hasNext()) {
			SymbolDefinition sDef = (SymbolDefinition) inputSymbolSets.get(iter.next());
			if (!inputTypes.add(sDef))
				Error.die(this, "Production rule contains an input symbol of same type as one of its input symbol sets");
			if (!existentialTypes.add(sDef))
				Error.die(this, "Production rule contains an existential symbol of same type as one of its input symbol sets");
		}

		// Print node sets
		iter = inputTypes.iterator();
		while (iter.hasNext()) {
			SymbolDefinition sDef = (SymbolDefinition) iter.next();
			output.println("\t\tSet " + sDef.getName() + "_root_nodes = ((" + grammarName + 
				"SymbolNodeSets) forest.getAllRootSymbolNodes()).get_" + sDef.getName() + "_set();");
			output.println("\t\tif (" + sDef.getName() + "_root_nodes.isEmpty())");
			output.println("\t\t\treturn null;");
		}
		iter = referableTypes.iterator();
		while (iter.hasNext()) {
			SymbolDefinition sDef = (SymbolDefinition) iter.next();
			output.println("\t\tSet " + sDef.getName() + "_all_nodes = ((" + grammarName + 
				"SymbolNodeSets) forest.getAllSymbolNodes()).get_" + sDef.getName() + "_set();");
			output.println("\t\tif (" + sDef.getName() + "_all_nodes.isEmpty())");
			output.println("\t\t\treturn null;");
		}
		output.println("");
		output.println("\t\tProductionRecord" + number + " record = new ProductionRecord" + number + "();");
		output.println("");

		// Fill input symbol sets
		iter = inputSymbolSets.keySet().iterator();
		while (iter.hasNext()) {
			String name = (String) iter.next();
			SymbolDefinition sDef = (SymbolDefinition) inputSymbolSets.get(name);
			output.println("\t\titer = " + sDef.getName() + "_root_nodes.iterator();");
			output.println("\t\twhile (iter.hasNext())");
			output.println("\t\t\trecord.get_" + name + "().add(((SymbolNode) iter.next()).getSymbol());");
		}

		// Fill input symbols
		String tabs = "\t";
		String close = "";
		int count = 1;
		List usedSymbolTypes = new LinkedList();
		iter = inputSymbols.keySet().iterator();
		while (iter.hasNext()) {
			String name = (String) iter.next();
			SymbolDefinition sDef = (SymbolDefinition) inputSymbols.get(name);
			tabs = "\t";
			for (int i = 0; i < count; i++)
				tabs += "\t";
			output.println(tabs + "Iterator i" + count + ";");
			output.println(tabs + "if (" + name + "_candidateNodes != null)");
			output.println(tabs + "\ti" + count + " = " + name + "_candidateNodes.iterator();");
			output.println(tabs + "else");
			output.println(tabs + "\ti" + count + " = " + sDef.getName() + "_root_nodes.iterator();");
			output.println(tabs + "while (i" + count + ".hasNext()) {");
			output.println(tabs + "\t" + sDef.getName() + " s" + count + " = (" + sDef.getName() + ") ((SymbolNode) i" + count + 
				".next()).getSymbol();");
			output.println(tabs + "\tif (!usedSymbols.add(s" + count + "))");
			output.println(tabs + "\t\tcontinue;");
			output.println(tabs + "\trecord.set_" + name + "(s" + count + ");");
			output.println("");
			
			close = tabs + "\tusedSymbols.remove(s" + count + ");\n" + tabs + "\trecord.set_" + name + "(null);\n" + tabs + "}\n" + close;
			count++;
			usedSymbolTypes.add(sDef);
		}

		// Fill referable existential symbols
		iter = referableExistentialSymbols.keySet().iterator();
		while (iter.hasNext()) {
			String name = (String) iter.next();
			SymbolDefinition sDef = (SymbolDefinition) referableExistentialSymbols.get(name);
			tabs = "\t";
			for (int i = 0; i < count; i++)
				tabs += "\t";
			output.println(tabs + "Iterator i" + count + ";");
			output.println(tabs + "if (" + name + "_candidateNodes != null)");
			output.println(tabs + "\ti" + count + " = " + name + "_candidateNodes.iterator();");
			output.println(tabs + "else");
			output.println(tabs + "\ti" + count + " = " + sDef.getName() + "_all_nodes.iterator();");
			output.println(tabs + "while (i" + count + ".hasNext()) {");
			output.println(tabs + "\t" + sDef.getName() + " s" + count + " = (" + sDef.getName() + ") ((SymbolNode) i" + count + 
				".next()).getSymbol();");

			// Check that there are no subtree issues with the referable existential symbols and already used symbols
			for (int i = 0; i < usedSymbolTypes.size(); i++) {
				SymbolDefinition usedType = (SymbolDefinition) usedSymbolTypes.get(i);
				if (dependencyGraph.hasCreationRelationship(usedType, sDef)) {
					Error.warn(this, "Production " + number + ": existential symbol \"" + name + 
						"\" could be in the subtree of another input or existential symbol");
					output.println(tabs + "\tif (forest.inSubTree(s" + count + ", s" + (i+1) + "))");
					output.println(tabs + "\t\tcontinue;");
				}

				// Check reverse direction with other referable existentials
				if (i >= inputSymbols.size()) {
					if (dependencyGraph.hasCreationRelationship(sDef, usedType)) {
						Error.warn(this, "Production " + number + ": existential symbol \"" + name + 
							"\" could contain another existential symbol in its subtree");
						output.println(tabs + "\tif (forest.inSubTree(s" + (i+1) + ", s" + count + "))");
						output.println(tabs + "\t\tcontinue;");
					}
				}
			}
			output.println(tabs + "\tif (!usedSymbols.add(s" + count + "))");
			output.println(tabs + "\t\tcontinue;");
			output.println(tabs + "\trecord.set_" + name + "(s" + count + ");");
			output.println("");
			
			close = tabs + "\tusedSymbols.remove(s" + count + ");\n" + tabs + "\trecord.set_" + name + "(null);\n" + tabs + "}\n" + close;
			count++;
			usedSymbolTypes.add(sDef);
		}

		// Add solver constraints that do not depend on any non-referable symbols
		if (mainConstraintSetElement != null) {
			nodes = mainConstraintSetElement.getElementsByTagName("SolverConstraint");
			for (int i = 0; i < nodes.getLength(); i++) {
				Element constraint = (Element) nodes.item(i);
				if (constraint.getAttribute("dependsOn").equals("")) {

					// Create the constraint
					printConstraintVariableArray(constraint, tabs + "\t", output);
					String contents = Compiler.getTextContents(constraint);

					// See if constraint compiler can be used
					if (constraintCompiler == null) {
						output.println(tabs + "\tConstraint c" +  constraint.getAttribute("label") + " = new " + 
							constraintClass + "();");
						output.println(tabs + "\tc" + constraint.getAttribute("label") + ".initialize(\"" + contents + 
							"\", cvArray" + constraint.getAttribute("label") + ");");
					}
					else {
						String className = "Constraint_P" + number + "__" + constraint.getAttribute("label");
						constraintClasses.put(className, contents);
						output.println(tabs + "\tConstraint c" +  constraint.getAttribute("label") + " = (Constraint) new " + 
							className + "(cvArray" + constraint.getAttribute("label") + ");");
					}
					output.println(tabs + "\trecord.setConstraint" + constraint.getAttribute("label") + "(c" + 
						constraint.getAttribute("label") + ");");
				}
			}
		}

		// Check symbols against constraints
		output.println("");
		if (mainConstraintSetElement != null) {
			nodes = mainConstraintSetElement.getChildNodes();
			if (nodes.getLength() == 0) 
				output.println(tabs + "\tif (true) {");
			else {
				output.print(tabs + "\tif (");
				for (int i = 0; i < nodes.getLength(); i++) {
					Element child = (Element) nodes.item(i);
					if (i > 0) 
						output.print(" &&\n" + tabs + "\t\t");
					output.print("evaluateConstraint" + child.getAttribute("label") + "(record, forest)");
				}
				output.println(") {");
			}
		}
		else 
			output.println(tabs + "\tif (true) {");

		// Create output symbols
		output.println("");
		output.println(tabs + "\t\tLog.getLogger().info(\"Production Rule " + number + 
			" successfully evaluated\");");
		iter = outputSymbols.keySet().iterator();
		while (iter.hasNext()) {
			String name = (String) iter.next();
			SymbolDefinition sDef = (SymbolDefinition) outputSymbols.get(name);
			if (sDef.getDrawableClass().equals(""))
				line = tabs + "\t\t" + sDef.getName() + " _" + name + " = new " + sDef.getName() + "(";
			else
				line = tabs + "\t\t" + sDef.getName() + " _" + name + " = new " + sDef.getDrawableClass() + "(";

			// Go through attributes
			List atts = sDef.getAllAttributes();
			int numArgs = 0;
			Iterator iter2 = atts.iterator();
			while (iter2.hasNext()) {
				Attribute attr = (Attribute) iter2.next();
				String fullName = name + "." + attr.getName();
				if (numArgs > 0)
					line += ",\n" + tabs + "\t\t\t";
				if (setAtts.contains(fullName))
					line += "valueOf_" + name + "_" + attr.getName() + "(record)";
				else
					line += "initial_" + name + "_" + attr.getName() + "(record)";
				numArgs++;
			}
			line += ");";
			output.println(line);
			output.println(tabs + "\t\trecord.set_" + name + "(_" + name + ");");
		}

		// See if any constraint variables need modifying
		boolean cvFlag = false;
		iter = setAtts.iterator();
		while (iter.hasNext()) {
			String name = (String) iter.next();
			int dotIndex = name.indexOf('.');
			String symbolName = name.substring(0, dotIndex);
			String attName = name.substring(dotIndex + 1);

			if (inputSymbols.containsKey(symbolName)) { 
				SymbolDefinition sDef = (SymbolDefinition) inputSymbols.get(symbolName);
				Attribute attr = sDef.getAttribute(attName);
				if (attr.isConstraintVariable())
					cvFlag = true;
			}
		}

		// Set input symbol attributes that need setting
		if (cvFlag) {
			output.println(tabs + "\t\tboolean tempACS = forest.getInterpreter().isAutoConstraintSolvingOn();");
			output.println(tabs + "\t\tforest.getInterpreter().setAutoConstraintSolving(false);");
		}
		iter = setAtts.iterator();
		while (iter.hasNext()) {
			String name = (String) iter.next();
			int dotIndex = name.indexOf('.');
			String symbolName = name.substring(0, dotIndex);
			String attName = name.substring(dotIndex + 1);

			if (inputSymbols.containsKey(symbolName)) 
				output.println(tabs + "\t\trecord.get_" + symbolName + "().set_" + attName + "(valueOf_" + symbolName +
					"_" + attName + "(record));");
		}
		if (cvFlag) {
			output.println(tabs + "\t\tforest.getInterpreter().setAutoConstraintSolving(tempACS);");
			output.println(tabs + "\t\tforest.getInterpreter().resolve();");
		}

		// Create set attribute constraints
		output.println("");
		if (attAssignmentElement != null) {
			nodes = attAssignmentElement.getElementsByTagName("ConstraintExpression");
			for (int i = 0; i < nodes.getLength(); i++) {
				Element element = (Element) nodes.item(i);
				Element parent = (Element) element.getParentNode();
				String name = parent.getAttribute("symbol_name") + "_" + parent.getAttribute("attribute_name");

				output.println(tabs + "\t\tConstraint c_" + name + " = new " + constraintClass + "();");
				output.println(tabs + "\t\tc_" + name + ".initializeWithEqualsRelationship(record.get_" + 
					parent.getAttribute("symbol_name") + "().getConstraintVariable_" +
					parent.getAttribute("attribute_name") + "(), record.getAtt_" + name + "_ConExp());");
				output.println(tabs + "\t\trecord.setAtt_" + name + "_Constraint(c_" + name + ");");
			}
		}

		// Try adding production to the forest
		output.println("");
		output.println(tabs + "\t\tif (forest.addProduction(record))");
		output.println(tabs + "\t\t\treturn record;");
		output.println("");

		// If production could not be added to the forest, log this and reinitialize input symbols
		output.println(tabs + "\t\trecord.reinitializeInputSymbols(forest.getInterpreter());");
		output.println(tabs + "\t\tLog.getLogger().info(\"Production Rule " + number + 
			" rejected by the Parse Forest\");");
		output.println(tabs + "\t}");
		output.println("");

		// Remove solver constraints that do not depend on any non-referable symbols
		if (mainConstraintSetElement != null) {
			nodes = mainConstraintSetElement.getElementsByTagName("SolverConstraint");
			for (int i = 0; i < nodes.getLength(); i++) {
				Element constraint = (Element) nodes.item(i);
				if (constraint.getAttribute("dependsOn").equals("")) 
					output.println(tabs + "\trecord.setConstraint" + constraint.getAttribute("label") + "(null);");
			}
		}
		output.print(close);
		output.println("\t\treturn null;");
		output.println("\t}");

		// Print out the constraints
		printConstraintSet(mainConstraintSetElement, output);
		while (!constraintSets.isEmpty()) {
			try {
				Element cSet = (Element) constraintSets.removeFirst();
				printConstraintSet(cSet, output);
			}
			catch (NoSuchElementException e) { ; }
		}

		// Print out the attribute value calculations
		if (attAssignmentElement != null) {

			NodeList children = attAssignmentElement.getElementsByTagName("InitAttribute");
			for (int i = 0; i < children.getLength(); i++) {
				Element initAttElement = (Element) children.item(i);
				SymbolDefinition sDef = (SymbolDefinition) outputSymbols.get(initAttElement.getAttribute("symbol_name"));
				Attribute attr = sDef.getAttribute(initAttElement.getAttribute("attribute_name"));

				// Get <InitAttribute> child element
				NodeList children2 = initAttElement.getChildNodes();
				Element child = (Element) children2.item(0);

				// Print method
				output.println("");
				output.println("\tpublic static " + (attr.isConstraintVariable() ? "double" : attr.getDataType()) + " initial_" + 
					initAttElement.getAttribute("symbol_name") + "_" + initAttElement.getAttribute("attribute_name") + 
					"(ProductionRecord" + number + " record)");
				output.println("\t{");
				output.println("\t\treturn " + printArgument(child) + ";");
				output.println("\t}");
			}

			children = attAssignmentElement.getElementsByTagName("SetAttribute");
			for (int i = 0; i < children.getLength(); i++) {
				Element setAttElement = (Element) children.item(i);
				SymbolDefinition sDef = null;
				sDef = (SymbolDefinition) outputSymbols.get(setAttElement.getAttribute("symbol_name"));
				if (sDef == null)
					sDef = (SymbolDefinition) inputSymbols.get(setAttElement.getAttribute("symbol_name"));
				Attribute attr = sDef.getAttribute(setAttElement.getAttribute("attribute_name"));

				// Get <SetAttribute> child element
				NodeList children2 = setAttElement.getChildNodes();
				Element child = (Element) children2.item(0);

				// Print method
				String name = setAttElement.getAttribute("symbol_name") + "_" + setAttElement.getAttribute("attribute_name");
				output.println("");
				output.println("\tpublic static " + (attr.isConstraintVariable() ? "double" : attr.getDataType()) + " valueOf_" + 
					name + "(ProductionRecord" + number + " record)");
				output.println("\t{");
				if (child.getTagName().equals("ConstraintExpression")) {
					output.println("\t\tif (record.getAtt_" + name + "_ConExp() == null) {");

					printConstraintVariableArray(child, "\t\t\t", output);
					String contents = Compiler.getTextContents(child);

					// See if constraint compiler can be used
					if (constraintCompiler == null) {
						output.println("\t\t\tConstraintExpression ce = new " + constraintExpressionClass + "();");
						output.println("\t\t\tce.initialize(\"" + contents + "\", cvArray" + 
							child.getAttribute("label") + ");");
					}
					else {
						String className = "ConstraintExpression" + number + "__" + name;
						constraintExpressionClasses.put(className, contents);
						output.println("\t\t\tConstraintExpression ce = (ConstraintExpression) new " + 
							className + "(cvArray" + child.getAttribute("label") + ");");
					}
					output.println("\t\t\trecord.setAtt_" + name + "_ConExp(ce);");
					output.println("\t\t}");
					output.println("\t\treturn record.getAtt_" + name + "_ConExp().evaluate();");
				}
				else
					output.println("\t\treturn " + printArgument(child) + ";");
				output.println("\t}");
			}
		}

		// End class and close file
		output.println("}");
		output.println("");
		output.close();

		createConstraintClasses();
	}

	// Returns the output symbol mapping
	public Map getOutputSymbols()
	{
		return outputSymbols;
	}

	// Returns the input symbol mapping
	public Map getInputSymbols()
	{
		return inputSymbols;
	}

	// Returns the referable existential symbol mapping
	public Map getReferableExistentialSymbols()
	{
		return referableExistentialSymbols;
	}
}

