/*
    Copyright (C) 2003  Anthony R. Jansen.

    This file is part of the CIDER Toolkit.

    The CIDER Toolkit is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    The CIDER Toolkit is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with the CIDER Toolkit; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

// ----------------------------------------
// CIDER Toolkit: Compiler Package
//
// Author:  Anthony R. Jansen
// Begun:   February 2003
// Class:   RegExpDefinition
//
// This class is used to represent the
// information contained in a regular
// expression definition which is made up
// of transformations, and is part of a 
// diagrammatic language specification. 
// The information is taken from an XML DOM 
// tree.
// ----------------------------------------

package au.edu.monash.csse.tonyj.cider.compiler;

import org.w3c.dom.Node;
import org.w3c.dom.Element;
import org.w3c.dom.NodeList;
import java.io.PrintWriter;
import java.util.List;

public class RegExpDefinition {

	// Private variables
	private int			number;
	private String			regExpName;
	private String			grammarName;
	private Element			regExpElement;

	// Constructor, which takes a number, name, a DOM Node for the Transformation and lists of transformation and disjunction definitions 
	public RegExpDefinition(int num, String gName, Element reElement, List transList, List disList)
	{
		number = num;
		grammarName = gName;
		regExpElement = reElement;
		regExpName = regExpElement.getAttribute("name");
		if (regExpName.equals(""))
			Error.die(this, regExpElement.getAttribute("_line_number_"), "Regular Expression element does not have a valid name");

		// Calculate necessary class names
		labelTermsAndComponents();
		getComponentClassNames(transList, disList);
	}
	
	// Label all the terms and components with class names
	private void labelTermsAndComponents()
	{
		NodeList nodes = regExpElement.getElementsByTagName("Component");
		for (int i = 0; i < nodes.getLength(); i++) {
			Element element = (Element) nodes.item(i);
			element.setAttribute("label", "RegExp" + number + "_Component" + (i+1));
		}
		nodes = regExpElement.getElementsByTagName("ConcatenationTerm");
		for (int i = 0; i < nodes.getLength(); i++) {
			Element element = (Element) nodes.item(i);
			element.setAttribute("label", "RegExp" + number + "_ConcatentationTerm" + (i+1));
		}
		nodes = regExpElement.getElementsByTagName("StarTerm");
		for (int i = 0; i < nodes.getLength(); i++) {
			Element element = (Element) nodes.item(i);
			element.setAttribute("label", "RegExp" + number + "_StarTerm" + (i+1));
		}
		nodes = regExpElement.getElementsByTagName("OrTerm");
		for (int i = 0; i < nodes.getLength(); i++) {
			Element element = (Element) nodes.item(i);
			element.setAttribute("label", "RegExp" + number + "_OrTerm" + (i+1));
		}
	}
	
	// Determines the name of the class for each component
	private void getComponentClassNames(List transList, List disList)
	{
		NodeList nodes = regExpElement.getElementsByTagName("Component");
		for (int i = 0; i < nodes.getLength(); i++) {
			Element element = (Element) nodes.item(i);
			String name = element.getAttribute("name");

			boolean success = false;

			// Check for pre-defined transformations first
			if (name.equals("ForceDirectedLayout")) {
				element.setAttribute("ruleClass", "TransformationRuleFDLayout");
				success = true;
			}
			else if (name.equals("CheckForest")) {
				element.setAttribute("ruleClass", "TransformationRuleCheckForest");
				success = true;
			}
			if (!success) {
				for (int j = 0; j < transList.size(); j++) {
					TransformationDefinition tDef = (TransformationDefinition) transList.get(j);
					if (tDef.getName().equals(name)) {
						element.setAttribute("ruleClass", "TransformationRule" + tDef.getNumber());
						success = true;
						break;
					}
				}
			}
			if (!success) {
				for (int j = 0; j < disList.size(); j++) {
					DisjunctionDefinition dDef = (DisjunctionDefinition) disList.get(j);
					if (dDef.getName().equals(name)) {
						element.setAttribute("ruleClass", "DisjunctionRule" + dDef.getNumber());
						success = true;
						break;
					}
				}
			}
			if (!success)
				Error.die(this, regExpElement.getAttribute("_line_number_"), "Regular expression contains a component " +
					"Transformation/Disjunction which is not defined: " + name);
		}
	}

	// Writes out the inner classes for the terms and components
	private void writeInnerClasses(PrintWriter output)
	{
		NodeList nodes = regExpElement.getElementsByTagName("Component");
		for (int i = 0; i < nodes.getLength(); i++) 
			writeComponentInnerClass((Element) nodes.item(i), output);
		nodes = regExpElement.getElementsByTagName("ConcatenationTerm");
		for (int i = 0; i < nodes.getLength(); i++) 
			writeConcatenationTermInnerClass((Element) nodes.item(i), output);
		nodes = regExpElement.getElementsByTagName("StarTerm");
		for (int i = 0; i < nodes.getLength(); i++) 
			writeStarTermInnerClass((Element) nodes.item(i), output);
		nodes = regExpElement.getElementsByTagName("OrTerm");
		for (int i = 0; i < nodes.getLength(); i++) 
			writeOrTermInnerClass((Element) nodes.item(i), output);
	}

	// Writes out the inner class for a component
	private void writeComponentInnerClass(Element element, PrintWriter output)
	{
		output.println("\tprivate class " + element.getAttribute("label") + " implements RegExpTerm {");
		output.println("");
		output.println("\t\tprivate TransformationRule rule;");
		output.println("");
		output.println("\t\tpublic " +  element.getAttribute("label") + "(ParseForest f, ConstraintSolver cs)");
		output.println("\t\t{");
		output.println("\t\t\trule = new " + element.getAttribute("ruleClass") + "(f, cs);");
		output.println("\t\t}");
		output.println("");
		output.println("\t\tpublic boolean apply()");
		output.println("\t\t{");
		if (element.getAttribute("parallel").equalsIgnoreCase("true")) 
			output.println("\t\t\treturn rule.parallelApply();");
		else
			output.println("\t\t\treturn rule.normalApply();");
		output.println("\t\t}");
		output.println("\t}");
		output.println("");
	}

	// Writes out the inner class for a star term
	private void writeStarTermInnerClass(Element element, PrintWriter output)
	{
		// This node should only have one child
		Element child = (Element) element.getFirstChild();

		output.println("\tprivate class " + element.getAttribute("label") + " implements RegExpTerm {");
		output.println("");
		output.println("\t\tprivate boolean alreadyApplied;");
		output.println("\t\tprivate ParseForest forest;");
		output.println("\t\tprivate ConstraintSolver csolver;");
		output.println("");
		output.println("\t\tpublic " +  element.getAttribute("label") + "(ParseForest f, ConstraintSolver cs)");
		output.println("\t\t{");
		output.println("\t\t\talreadyApplied = false;");
		output.println("\t\t\tforest = f;");
		output.println("\t\t\tcsolver = cs;");
		output.println("\t\t}");
		output.println("");
		output.println("\t\tpublic boolean apply()");
		output.println("\t\t{");
		output.println("\t\t\tif (alreadyApplied)");
		output.println("\t\t\t\treturn false;");
		output.println("");
		output.println("\t\t\tRegExpTerm term = null;");
		output.println("\t\t\tdo {");
		output.println("\t\t\t\tterm = new " + child.getAttribute("label") + "(forest, csolver);");
		output.println("\t\t\t} while (term.apply());");
		output.println("\t\t\talreadyApplied = true;");
		output.println("\t\t\treturn true;");
		output.println("\t\t}");
		output.println("\t}");
		output.println("");
	}

	// Writes out the inner class for an or term
	private void writeOrTermInnerClass(Element element, PrintWriter output)
	{
		NodeList children = element.getChildNodes();

		output.println("\tprivate class " + element.getAttribute("label") + " implements RegExpTerm {");
		output.println("");
		output.println("\t\tprivate int level;");
		output.println("\t\tprivate RegExpTerm terms[];");
		output.println("");
		output.println("\t\tpublic " +  element.getAttribute("label") + "(ParseForest f, ConstraintSolver cs)");
		output.println("\t\t{");
		output.println("\t\t\tlevel = 0;");
		output.println("\t\t\tterms = new RegExpTerm[" + children.getLength() + "];");
		for (int i = 0; i < children.getLength(); i++) {
			Element child = (Element) children.item(i);
			output.println("\t\t\tterms[" + i + "] = new " + child.getAttribute("label") + "(f, cs);");
		}
		output.println("\t\t}");
		output.println("");
		output.println("\t\tpublic boolean apply()");
		output.println("\t\t{");
		output.println("\t\t\twhile (level < " + children.getLength() + ") {");
		output.println("\t\t\t\tif (terms[level].apply())");
		output.println("\t\t\t\t\treturn true;");
		output.println("\t\t\t\tlevel++;");
		output.println("\t\t\t}");
		output.println("\t\t\treturn false;");
		output.println("\t\t}");
		output.println("\t}");
		output.println("");
	}

	// Writes out the inner class for a concatenation term
	private void writeConcatenationTermInnerClass(Element element, PrintWriter output)
	{
		Element child;
		NodeList children = element.getChildNodes();

		output.println("\tprivate class " + element.getAttribute("label") + " implements RegExpTerm {");
		output.println("");
		output.println("\t\tprivate boolean existingSolutionFlag;");
		output.println("\t\tprivate ParseForest forest;");
		output.println("\t\tprivate ConstraintSolver csolver;");
		output.println("\t\tprivate int undoPoint[];");
		output.println("\t\tprivate RegExpTerm terms[];");
		output.println("");
		output.println("\t\tpublic " +  element.getAttribute("label") + "(ParseForest f, ConstraintSolver cs)");
		output.println("\t\t{");
		output.println("\t\t\texistingSolutionFlag = false;");
		output.println("\t\t\tforest = f;");
		output.println("\t\t\tcsolver = cs;");
		output.println("\t\t\tundoPoint = new int[" + children.getLength() + "];");
		output.println("\t\t\tterms = new RegExpTerm[" + children.getLength() + "];");
		child = (Element) children.item(0);
		output.println("\t\t\tterms[0] = new " + child.getAttribute("label") + "(forest, csolver);");
		output.println("\t\t}");
		output.println("");
		output.println("\t\tpublic boolean apply()");
		output.println("\t\t{");
		output.println("\t\t\tif (existingSolutionFlag) {");
		output.println("\t\t\t\tfor (int level = " + (children.getLength()-1) + "; level > 0; level--) {");
		output.println("\t\t\t\t\tforest.getInterpreter().undoActions(undoPoint[level]);");
		output.println("\t\t\t\t\tif (processLevel(level))");
		output.println("\t\t\t\t\t\treturn true;");
		output.println("\t\t\t\t}");
		output.println("\t\t\t\tforest.getInterpreter().undoActions(undoPoint[0]);");
		output.println("\t\t\t}");
		output.println("\t\t\texistingSolutionFlag = processLevel(0);");
		output.println("\t\t\treturn existingSolutionFlag;");
		output.println("\t\t}");
		output.println("");
		output.println("\t\tpublic boolean processLevel(int level)");
		output.println("\t\t{");
		output.println("\t\t\tundoPoint[level] = forest.getInterpreter().getUndoIndex();");
		output.println("\t\t\twhile (terms[level].apply()) {");
		output.println("\t\t\t\tswitch (level) {");
		for (int i = 1; i < children.getLength(); i++) {
			child = (Element) children.item(i);
			output.println("\t\t\t\t\tcase " + (i-1) + ":");
			output.println("\t\t\t\t\t\tterms[" + i + "] = new " + child.getAttribute("label") + "(forest, csolver);");
			output.println("\t\t\t\t\t\tbreak;");
		}
		output.println("\t\t\t\t\tcase " + (children.getLength()-1) + ":");
		output.println("\t\t\t\t\t\treturn true;");
		output.println("\t\t\t\t\tdefault:");
		output.println("\t\t\t\t\t\tLog.getLogger().severe(\"Invalid level found when processing Concatenation Term\");");
		output.println("\t\t\t\t\t\tError.die(\"Invalid level found when processing Concatenation Term\");");
		output.println("\t\t\t\t\t\tbreak;");
		output.println("\t\t\t\t}");
		output.println("\t\t\t\tif (processLevel(level+1))");
		output.println("\t\t\t\t\treturn true;");
		output.println("\t\t\t\tforest.getInterpreter().undoActions(undoPoint[level]);");
		output.println("\t\t\t}");
		output.println("\t\t\treturn false;");
		output.println("\t\t}");
		output.println("\t}");
		output.println("");
	}

	// Produces a regular expression class for this expression
	public void createRegularExpressionClass()
	{
		// This expression node should only have one child
		Element baseElement = (Element) regExpElement.getFirstChild();

		// Open file for the class
		PrintWriter output = Compiler.openTextFile(Compiler.PACKAGE_DIRECTORY + "RegularExpression" + number + ".java");

		// Print class header details (package scope)
		output.println("");
		output.println(Compiler.PACKAGE_LINE);
		output.println("");
		output.println("import au.edu.monash.csse.tonyj.cider.constraints.ConstraintSolver;");
		output.println("");
		output.println("class RegularExpression" + number + " implements TransformationRule {");
		output.println("");

		// Print out private variables 
		output.println("\tprivate boolean expAlreadyApplied;");
		output.println("\tprivate ParseForest pForest;");
		output.println("\tprivate RegExpTerm baseTerm;");
		output.println("");

		// Print constructor 
		output.println("\tpublic RegularExpression" + number + "(ParseForest forest, ConstraintSolver csolver)");
		output.println("\t{");
		output.println("\t\texpAlreadyApplied = false;");
		output.println("\t\tpForest = forest;");
		output.println("\t\tbaseTerm = new " + baseElement.getAttribute("label") + "(forest, csolver);");
		output.println("\t}");

		// Print out the method to apply the transformation normally (always false after the first time used)
		output.println("\tpublic boolean normalApply()");
		output.println("\t{");
		output.println("\t\tif (expAlreadyApplied)");
		output.println("\t\t\treturn false;");
		output.println("");
		output.println("\t\tboolean tempUE = pForest.getInterpreter().isUndoEnabled();");
		output.println("\t\tpForest.getInterpreter().enableUndo();");
		output.println("\t\tboolean success = baseTerm.apply();");
		output.println("\t\tif (!tempUE)");
		output.println("\t\t\tpForest.getInterpreter().disableUndo();");
		output.println("\t\texpAlreadyApplied = true;");
		output.println("\t\tif (success)");
		output.println("\t\t\tLog.getLogger().info(\"Regular Expression " + regExpName + " successfully (asynchronously) applied\");");
		output.println("\t\treturn success;");
		output.println("\t}");
		output.println("");

		// Print out the method to apply the transformation in parallel (must return false)
		output.println("\tpublic boolean parallelApply()");
		output.println("\t{");
		output.println("\t\treturn false;");
		output.println("\t}");
		output.println("");

		// Add inner classes to this class
		writeInnerClasses(output);

		// End class and close file
		output.println("}");
		output.println("");
		output.close();
	}

	// Returns the name of this disjunction
	public String getName()
	{
		return regExpName;
	}

	// Returns the number of this disjunction
	public int getNumber()
	{
		return number;
	}
}


