/*
    Copyright (C) 2003  Anthony R. Jansen.

    This file is part of the CIDER Toolkit.

    The CIDER Toolkit is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    The CIDER Toolkit is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with the CIDER Toolkit; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

// ----------------------------------------
// CIDER Toolkit: Compiler Package
//
// Author:  Anthony R. Jansen
// Begun:   June 2003
// Class:   RegExpParser
//
// This class is used for processing a
// regular expression string where all
// tokens are separated by spaces.
// ----------------------------------------

package au.edu.monash.csse.tonyj.cider.compiler;

import java.util.List;
import java.util.LinkedList;

class RegExpParser {

	// Constants
	public static final int UNDEF = 0;
	public static final int STAR = 1;
	public static final int CONCATENATION = 2;
	public static final int OR = 3;
	public static final int LITERAL = 4;

	// Variables
	protected Node		root;
	protected String	exp;

	// Constructor, which expects a string containing spaces around
	// eacg token, tokens being: * ( ) | and name
	public RegExpParser(String expression)
	{
		exp = expression.trim();
		root = new Node();
		checkBrackets();
		String tokens[] = exp.split(" +");
		for (int i = 0; i < tokens.length; i++)
			root.contents.add(tokens[i]);
		processLiterals(root);
		processBrackets(root);
		processStar(root);
		processOr(root);
		processConcatenation(root);
		removeUndefNodes(root);
	}

	// Makes sure the brackets in the expression are correct
	private void checkBrackets()
	{
		int level = 0;

		for (int i = 0; i < exp.length(); i++) {
			if (exp.charAt(i) == '(')
				level++;
			else if (exp.charAt(i) == ')')
				level--;
			if (level < 0)
				Error.die(this, "Invalid brackets in regular expression: " + exp);
		}
		if (level != 0)
			Error.die(this, "Invalid brackets in regular expression: " + exp);
	}

	// Puts all literals in an appropriate node
	private void processLiterals(Node node)
	{
		for (int i = 0; i < node.contents.size(); i++) {
			Object item = node.contents.get(i);

			if (item instanceof Node)
				processLiterals((Node) item);
			else if (item instanceof String) {
				String str = (String) item;

				if (!(str.equals("(") || str.equals(")") || str.equals("*") || str.equals("|"))) {
					Node child = new Node();
					child.type = LITERAL;
					child.contents.add(item);
					node.contents.set(i, child);
				}
			}
		}
	}

	// Puts all bracket terms in child nodes
	private void processBrackets(Node node)
	{
		int level = 0;

		for (int i = 0; i < node.contents.size(); i++) {
			Object item = node.contents.get(i);

			// Create new node for bracket term
			if (item instanceof String && ((String) item).equals("(")) {
				level++;
				Node child = new Node();
				node.contents.remove(i);

				// Get contents of new node
				while (level > 0) {
					item = node.contents.get(i);
					if (item instanceof String && ((String) item).equals("(")) 
						level++;
					else if (item instanceof String && ((String) item).equals(")")) 
						level--;
					node.contents.remove(i);
					if (level > 0)
						child.contents.add(item);
				}
				node.contents.add(i, child);
				processBrackets(child);
			}
		}
	}

	// Process all star terms
	private void processStar(Node node)
	{
		for (int i = 0; i < node.contents.size(); i++) {
			Object item = node.contents.get(i);
			if (item instanceof Node)
				processStar((Node) item);

			// If star is found, create new node
			else if (item instanceof String && ((String) item).equals("*")) {
				if (i == 0)
					Error.die(this, "Invalid * in regular expression: " + exp);
				Node child = new Node();
				child.type = STAR;
				node.contents.remove(i);

				// Add child node to list
				i--;
				child.contents.add(node.contents.get(i));
				node.contents.set(i, child);
			}
		}
	}

	// Process all or terms
	// Assumes all star terms have already been processed
	private void processOr(Node node)
	{
		boolean orPresent = false;

		// See if a | token is present
		for (int i = 0; i < node.contents.size(); i++) {
			Object item = node.contents.get(i);
			if (item instanceof Node)
				processOr((Node) item);
			else if (item instanceof String && ((String) item).equals("|")) 
				orPresent = true;
		}
		if (!orPresent) 
			return;

		// If or term is present, create new nodes
		Object item = null;
		List newList = new LinkedList();
		int i = 0;
		while (i < node.contents.size()) {
			Node child = new Node();

			item = node.contents.get(i);
			while (i < node.contents.size() && !(item instanceof String && ((String) item).equals("|"))) {
				child.contents.add(item);
				i++;
				if (i < node.contents.size())
					item = node.contents.get(i);	
			}
			i++;
			if (child.contents.size() == 0)
				Error.die(this, "Invalid | in regular expression: " + exp);
			newList.add(child);
		}
		if (item instanceof String && ((String) item).equals("|"))
			Error.die(this, "Invalid | in regular expression: " + exp);

		node.contents = newList;
		node.type = OR;
	}

	// Process all concatenation terms
	// Assumes all literals, star and or terms have already been processed
	private void processConcatenation(Node node)
	{
		if (node.type == UNDEF && node.contents.size() > 1)
			node.type = CONCATENATION;

		for (int i = 0; i < node.contents.size(); i++) {
			Object item = node.contents.get(i);
			if (item instanceof Node)
				processConcatenation((Node) item);
		}
	}

	// Removes all undef nodes from the tree
	// Assumes concatenation terms have been processed
	private void removeUndefNodes(Node node)
	{
		for (int i = 0; i < node.contents.size(); i++) {
			Object item = node.contents.get(i);
			if (item instanceof Node)
				removeUndefNodes((Node) item);
		}
		if (node.type == UNDEF) {
			if (node.contents.size() == 0)
				Error.die(this, "Empty term in regular expression: " + exp);
			Node child = (Node) node.contents.get(0);
			node.contents = child.contents;
			node.type = child.type;
		}
	}

	// Node for tree of regular expression
	protected class Node {

		int	type;
		List	contents;

		// Constructor
		Node()
		{
			type = UNDEF;
			contents = new LinkedList();
		}
	}
}

