/*
    Copyright (C) 2003  Anthony R. Jansen.

    This file is part of the CIDER Toolkit.

    The CIDER Toolkit is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    The CIDER Toolkit is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with the CIDER Toolkit; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

// ----------------------------------------
// CIDER Toolkit: Compiler Package
//
// Author:  Anthony R. Jansen
// Begun:   June 2003
// Class:   RegExpXMLWriter
//
// This class output the XML format of a
// regular expression as required by the
// transformation specifications.
// ----------------------------------------

package au.edu.monash.csse.tonyj.cider.compiler;

import java.io.PrintWriter;

class RegExpXMLWriter extends RegExpParser {

	// Constructor
	public RegExpXMLWriter(String expression)
	{
		super(expression);
	}

	// Writes out the XML for this regular expression
	public void write(PrintWriter output)
	{
		writeTerm(output, root, "\t\t");
	}

	// Writes a term
	private void writeTerm(PrintWriter output, Node node, String tabs)
	{
		// Deal with literals
		if (node.type == LITERAL) {
			String name = (String) node.contents.get(0);
			if (name.endsWith("!"))
				output.println(tabs + "<Component name=\"" + name.substring(0, name.length() - 1) + "\" parallel=\"true\" />");
			else
				output.println(tabs + "<Component name=\"" + name + "\" parallel=\"false\" />");
			return;
		}

		// Deal with other node types
		if (node.type == STAR)
			output.println(tabs + "<StarTerm>");
		else if (node.type == OR)
			output.println(tabs + "<OrTerm>");
		else if (node.type == CONCATENATION)
			output.println(tabs + "<ConcatenationTerm>");
		else
			Error.die(this, "Could not process regular expression: " + exp);

		// Go through child nodes
		for (int i = 0; i < node.contents.size(); i++) {
			Node child = (Node) node.contents.get(i);
			writeTerm(output, child, tabs + "\t");
		}

		if (node.type == STAR)
			output.println(tabs + "</StarTerm>");
		else if (node.type == OR)
			output.println(tabs + "</OrTerm>");
		else if (node.type == CONCATENATION)
			output.println(tabs + "</ConcatenationTerm>");
	}
}

