/*
    Copyright (C) 2003  Anthony R. Jansen.

    This file is part of the CIDER Toolkit.

    The CIDER Toolkit is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    The CIDER Toolkit is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with the CIDER Toolkit; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

// ----------------------------------------
// CIDER Toolkit: Compiler Package
//
// Author:  Anthony R. Jansen
// Begun:   July 2003
// Class:   RuleDefinition
//
// This abstract class is used to represent 
// the information contained common to both
// production and transformation rule
// definitions, as part of a diagrammatic
// language specification. The information
// is taken from an XML DOM tree.
// ----------------------------------------

package au.edu.monash.csse.tonyj.cider.compiler;

import au.edu.monash.csse.tonyj.cider.constraints.ConstraintCompiler;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.w3c.dom.DOMException;
import java.util.List;
import java.util.LinkedList;
import java.util.Map;
import java.util.HashMap;
import java.util.Iterator;
import java.io.PrintWriter;

public abstract class RuleDefinition {

	private boolean			isProduction;

	// Protected variables
	protected int			number;
	protected String		ruleName;
	protected String		grammarName;
	protected List			symbolDefinitionList;
	protected String		constraintClass;
	protected String		constraintExpressionClass;
	protected ConstraintCompiler	constraintCompiler;

	// XML elements
	protected Element		ruleElement;
	protected Element		mainConstraintSetElement;

	// Symbols involved - maps name to a symbol definition
	protected Map			inputSymbols;
	protected Map			outputSymbols;
	protected Map			existentialSymbols;
	protected Map			referableExistentialSymbols;
	protected Map			positiveExistentialSymbols;
	protected Map			negativeExistentialSymbols;

	// Other required mappings
	protected Map		existentialConstraintLabels;    // Contains the constraint label associated with each existential symbol
	protected LinkedList	constraintSets;                 // Used to store constraint sets that need to be printed for the rule
        protected Map		constraintClasses;              // Contains the name and strings of all constraints with their own class
        protected Map		constraintExpressionClasses;    // Contains the name and strings of all constraint expressions with their own class

	// Constructor, which takes an identification number, the grammar name, 
	// a DOM Node for the rule, a list of symbol definitions and constraint relevent stuff 
	public RuleDefinition(boolean isP, int num, String gName, Element rule, List sdList, String conClass, String conExpClass, 
		ConstraintCompiler cc)
	{
		isProduction = isP;
		number = num;
		ruleName = rule.getAttribute("name");
		grammarName = gName;
		symbolDefinitionList = sdList;
		ruleElement = rule;
		constraintClass = conClass;
		constraintExpressionClass = conExpClass;
		constraintCompiler = cc;

		// Create require data structures
		inputSymbols = new HashMap();
		outputSymbols = new HashMap();
		existentialSymbols = new HashMap();
		referableExistentialSymbols = new HashMap();
		positiveExistentialSymbols = new HashMap();
		negativeExistentialSymbols = new HashMap();

		existentialConstraintLabels = new HashMap();
		constraintSets = new LinkedList();
		constraintClasses = new HashMap();
		constraintExpressionClasses = new HashMap();
	}

	// Returns the child element of the specified name, or null
	protected Element getChildElement(Element element, String name)
	{
		NodeList children = element.getChildNodes();
		for (int i = 0; i < children.getLength(); i++) {
			if (children.item(i).getNodeName().equals(name))
				return (Element) children.item(i);
		}
		return null;
	}

	// Returns the symbol definition for a specified type, or null if it does not exist
	protected SymbolDefinition getSymbolDefinition(String type)
	{
		for (int i = 0; i < symbolDefinitionList.size(); i++) {
			SymbolDefinition s = (SymbolDefinition) symbolDefinitionList.get(i);
			if (s.getName().equals(type))
				return s;
		}
		return null;
	}

	// Each constraint in a constraint set is given the label "[prefix]_[index]"
	// where [prefix] is predetermined, and [index] is the constraint number in this set
	// If the [prefix] is empty, the label just becomes the [index]
	protected void labelConstraints(Element element, String prefix)
	{
		String label = "";
		int index = 1;

		// Check all child nodes of the element
		NodeList children = element.getChildNodes();
		for (int i = 0; i < children.getLength(); i++) {

			// Ignore non-element nodes
			if (children.item(i).getNodeType() != Node.ELEMENT_NODE)
				continue;
			Element child = (Element) children.item(i);

			// If we find an exists symbol, record its associated constraint
			if (child.getTagName().equals("Symbol"))
				existentialConstraintLabels.put(child.getAttribute("name"), prefix);

			// If parent was a constraint set, child must be a constraint
			if (element.getTagName().equals("ConstraintSet")) {
				label = (prefix.equals("") ? "" + index : prefix + "_" + index);
				child.setAttribute("label", label);
				index++;
				labelConstraints(child, label);
			}
			else
				labelConstraints(child, prefix);
		}
	}

	// This method labels each constraint with the existential symbol
	// it immediately depends on, or the empty string otherwise
	protected void labelConstraintExistsDependencies()
	{
		NodeList existsNodes = mainConstraintSetElement.getElementsByTagName("Exists");
		for (int i = 0; i < existsNodes.getLength(); i++) {
			Element exists = (Element) existsNodes.item(i);
			Element symbol = getChildElement(exists, "Symbol");
			Element cSet = getChildElement(exists, "ConstraintSet");

			if (cSet != null) {
				NodeList children = cSet.getChildNodes();
				for (int j = 0; j < children.getLength(); j++) {
					Element child = (Element) children.item(j);
					child.setAttribute("dependsOn", symbol.getAttribute("name"));
				}
			}
		}
		existsNodes = mainConstraintSetElement.getElementsByTagName("NotExists");
		for (int i = 0; i < existsNodes.getLength(); i++) {
			Element exists = (Element) existsNodes.item(i);
			Element symbol = getChildElement(exists, "Symbol");
			Element cSet = getChildElement(exists, "ConstraintSet");

			if (cSet != null) {
				NodeList children = cSet.getChildNodes();
				for (int j = 0; j < children.getLength(); j++) {
					Element child = (Element) children.item(j);
					child.setAttribute("dependsOn", symbol.getAttribute("name"));
				}
			}
		}
	}

	// Adds the constraints from referable existential symbols to the main constraint set
	protected void relocateReferableSymbolConstraints()
	{
		if (mainConstraintSetElement == null)
			return;

		List validConstraints = new LinkedList();
		List existsNodes = new LinkedList();

		// Find all constraints for referable symbols
		NodeList nodes = mainConstraintSetElement.getElementsByTagName("Exists");
		for (int i = 0; i < nodes.getLength(); i++) {
			Element existsElement = (Element) nodes.item(i);
			existsNodes.add(existsElement);
			Element symbolElement = getChildElement(existsElement, "Symbol");
			Element csElement = getChildElement(existsElement, "ConstraintSet");

			// See if symbol is in the referable symbol set
			if (referableExistentialSymbols.containsKey(symbolElement.getAttribute("name")) && csElement != null) {
				NodeList constraintNodes = csElement.getChildNodes();
				for (int j = 0; j < constraintNodes.getLength(); j++) {
					Element constraintElement = (Element) constraintNodes.item(j);
					validConstraints.add(constraintElement);
				}
			}
		}

		// Remove all exists nodes for referable symbols 
		try {
			for (int i = 0; i < existsNodes.size(); i++) {
				Element existsElement = (Element) existsNodes.get(i);
				Element symbolElement = getChildElement(existsElement, "Symbol");

				// See if symbol is in the referable symbol set
				if (referableExistentialSymbols.containsKey(symbolElement.getAttribute("name"))) {
					Node constraintNode = existsElement.getParentNode();
					Node constraintSetNode = constraintNode.getParentNode();
					constraintSetNode.removeChild(constraintNode);

					// Make sure constraint is not included among the valid constraints
					validConstraints.remove(constraintNode);
				}
			}
		}
		catch (DOMException e) {
			Error.die(this, "Could not remove referable exists symbols from the DOM tree");
		}

		// Add valid constraints to main constraint set
		try {
			Iterator iter = validConstraints.iterator();
			while (iter.hasNext()) {
				Element constraintElement = (Element) iter.next();
				mainConstraintSetElement.appendChild(constraintElement);
			}
		}
		catch (DOMException e) {
			Error.die(this, "Could not move constraints to new location in the DOM tree");
		}
	}

	// This method generated the function code for all functions in the rule
	protected void generateCodeForAllFunctions()
	{
		NodeList nodes = ruleElement.getElementsByTagName("FunctionValue");

		// Go through each function
		for (int i = 0; i < nodes.getLength(); i++) {
			Element functionElement = (Element) nodes.item(i);
			generateFunctionCode(functionElement);
		}
	}

	// This method generates the code for a function iff it hasn't already been created
	protected void generateFunctionCode(Element functionElement)
	{
		String code = functionElement.getAttribute("code");

		// Return if code already there
		if (!code.equals(""))
			return;

		code = functionElement.getAttribute("name") + "(";

		// Get arguments
		int numArgs = 0;
		NodeList children = functionElement.getChildNodes();
		for (int j = 0; j < children.getLength(); j++) {
			Element child = (Element) children.item(j);
			if (numArgs > 0)
				code += ", ";
			if (child.getTagName().equals("AttributeValue")) 
				code += "record.get_" + child.getAttribute("symbol_name") + "().get_" + child.getAttribute("attribute_name") + "()";
			else if (child.getTagName().equals("SymbolValue"))
				code += "record.get_" + child.getAttribute("name") + "()";
			else if (child.getTagName().equals("ConstantValue"))
				code += child.getAttribute("value");
			else if (child.getTagName().equals("FunctionValue")) {
				generateFunctionCode(child);
				code += child.getAttribute("code");
			}
			numArgs++;
		}
		code += ")";
		functionElement.setAttribute("code", code);
	}

	// Prints out the constraints
	protected void printConstraintSet(Element cSet, PrintWriter output)
	{
		// Check if any constraint set exists
		if (cSet == null)
			return;

		// Get constraints from the constraint set
		NodeList constraints = cSet.getChildNodes();
		for (int i = 0; i < constraints.getLength(); i++) {
			Element constraint = (Element) constraints.item(i);

			if (constraint.getTagName().equals("Constraint"))
				printConstraint(constraint, output);
			else if (constraint.getTagName().equals("SolverConstraint"))
				printSolverConstraint(constraint, output);
		}
	}

	// Print out a specific solver constraint evaluation function
	protected void printSolverConstraint(Element constraint, PrintWriter output)
	{
		// Print method
		output.println("");
		if (isProduction)
			output.println("\tpublic static boolean evaluateConstraint" + constraint.getAttribute("label") + 
				"(ProductionRecord" + number + " record, ParseForest forest)");
		else 
			output.println("\tpublic boolean evaluateConstraint" + constraint.getAttribute("label") + 
				"(TransformationRecord" + number + " record)");
		output.println("\t{");

		// If infinite tolerance is specified, constraint is always satisfied
		if (constraint.getAttribute("tolerance").equalsIgnoreCase("infinite")) {
			output.println("\t\treturn true;");
		}
		else {
			// Get tolerance 
			double tolerance = 0.0;
			try {
				tolerance = Double.parseDouble(constraint.getAttribute("tolerance"));
			}
			catch (NumberFormatException e) {
				Error.die(this, constraint.getAttribute("_line_number_"), 
					"Tolerance of solver constraint was not a valid double");
			}
			output.println("\t\tConstraint constraint = record.getConstraint" + constraint.getAttribute("label") + "();");
			output.println("\t\tif (constraint == null)");
			output.println("\t\t\treturn false;");
			output.println("\t\telse");
			output.println("\t\t\treturn constraint.isSatisfied(" + tolerance + ");");
		}
		output.println("\t}");
	}

	// Print out a specific constraint evaluation function
	protected void printConstraint(Element constraint, PrintWriter output)
	{
		output.println("");
		if (isProduction)
			output.println("\tpublic static boolean evaluateConstraint" + constraint.getAttribute("label") + 
				"(ProductionRecord" + number + " record, ParseForest forest)");
		else 
			output.println("\tpublic boolean evaluateConstraint" + constraint.getAttribute("label") + 
				"(TransformationRecord" + number + " record)");
		output.println("\t{");

		// Get operation for this constraint 
		NodeList ops = constraint.getChildNodes();
		Element operation = (Element) ops.item(0);

		output.println("\t\treturn " + printOperation(operation, output) + ";");
		output.println("\t}");
	}

	// Print out a operation
	protected String printOperation(Element operation, PrintWriter output)
	{
		String string = "";

		// Deal with operator as is appropriate
		if (operation.getTagName().equals("ObjectEQ")) 
			string = printTwoArguments(").equals(", operation);
		else if (operation.getTagName().equals("ObjectNEQ")) 
			string = "(!" + printTwoArguments(").equals(", operation) + ")";
		else if (operation.getTagName().equals("EQ")) 
			string = printTwoArguments(") == (", operation);
		else if (operation.getTagName().equals("NEQ")) 
			string = printTwoArguments(") != (", operation);
		else if (operation.getTagName().equals("LEQ")) 
			string = printTwoArguments(") <= (", operation);
		else if (operation.getTagName().equals("GEQ")) 
			string = printTwoArguments(") >= (", operation);
		else if (operation.getTagName().equals("LT")) 
			string = printTwoArguments(") < (", operation);
		else if (operation.getTagName().equals("GT")) 
			string = printTwoArguments(") > (", operation);
		else if (operation.getTagName().equals("Exists")) {
			printExists(operation, output);
			string = "symbolExists";
		}
		else if (operation.getTagName().equals("NotExists")) {
			printExists(operation, output);
			string = "!symbolExists";
		}
		return string;
	}

	// Prints out code for an existential symbol
	protected void printExists(Element operation, PrintWriter output)
	{
		Element argument;
		int count = 0;
		NodeList nodes;

		// Find out type allowed for existential symbol
		String existsType = operation.getAttribute("type");
		Element cSet = getChildElement(operation, "ConstraintSet");
		Element symbol = getChildElement(operation, "Symbol");
		String name = symbol.getAttribute("name");
		String type = symbol.getAttribute("type");

		output.println("\t\tSet " + type + "_all_nodes = ((" + grammarName + "SymbolNodeSets) forest.getAllSymbolNodes()).get_" + 
			type + "_set();");
		output.println("\t\tboolean symbolExists = false;");
		output.println("\t\trecord.set_" + name + "(null);");
		output.println("");

		// Set solver constraints that depend on this existential symbol to null
		String tabs = "\t\t";
		if (mainConstraintSetElement != null) {
			nodes = mainConstraintSetElement.getElementsByTagName("SolverConstraint");
			for (int i = 0; i < nodes.getLength(); i++) {
				Element constraint = (Element) nodes.item(i);
				if (constraint.getAttribute("dependsOn").equals(name)) 
					output.println(tabs + "record.setConstraint" + constraint.getAttribute("label") + "(null);");
			}
			output.println("");
		}

		// Go through all candidate symbols
		output.println("\t\tIterator iter = " + type + "_all_nodes.iterator();");
		output.println("\t\twhile (iter.hasNext() && !symbolExists) {");
		output.println("\t\t\t" + type + " _" + name + " = (" + type + ") ((SymbolNode) iter.next()).getSymbol();");
		output.println("");

		// Make sure symbol does not conflict with any already defined symbols in the record
		int num = 0;
		Iterator iter = inputSymbols.keySet().iterator();
		while (iter.hasNext()) {
			String sName = (String) iter.next();
			SymbolDefinition sDef = (SymbolDefinition) inputSymbols.get(sName);
			if (type.equals(sDef.getName())) {
				output.println("\t\t\t" + (num > 0 ? "else if" : "if") + " (_" + name + " == record.get_" + sName + "())");
				output.println("\t\t\t\tcontinue;");
				num++;
			}
		}
		iter = existentialSymbols.keySet().iterator();
		while (iter.hasNext()) {
			String sName = (String) iter.next();
			if (name.equals(sName))
				continue;
			SymbolDefinition sDef = (SymbolDefinition) existentialSymbols.get(sName);
			if (type.equals(sDef.getName())) {
				output.println("\t\t\t" + (num > 0 ? "else if" : "if") + " (_" + name + " == record.get_" + sName + "())");
				output.println("\t\t\t\tcontinue;");
				num++;
			}
		}
		output.println("\t\t\trecord.set_" + name + "(_" + name + ");");
		output.println("\t\t\tsymbolExists = true;");
		output.println("");

		// Add solver constraints that depend on this existential symbol
		tabs = "\t\t\t";
		if (mainConstraintSetElement != null) {
			nodes = mainConstraintSetElement.getElementsByTagName("SolverConstraint");
			for (int i = 0; i < nodes.getLength(); i++) {
				Element constraint = (Element) nodes.item(i);
				if (constraint.getAttribute("dependsOn").equals(name)) {

					// Create the constraint
					printConstraintVariableArray(constraint, tabs, output);
					String contents = Compiler.getTextContents(constraint);

					// See if constraint compiler can be used
					if (constraintCompiler == null) {
						output.println(tabs + "Constraint c" +  constraint.getAttribute("label") + " = new " + 
							constraintClass + "();");
						output.println(tabs + "c" + constraint.getAttribute("label") + ".initialize(\"" + contents + 
							"\", cvArray" + constraint.getAttribute("label") + ");");
					}
					else {
						String className = "Constraint_" + (isProduction ? "P" : "T") + number + "__" + 
							constraint.getAttribute("label");
						constraintClasses.put(className, contents);
						output.println(tabs + "Constraint c" +  constraint.getAttribute("label") + " = (Constraint) new " + 
							className + "(cvArray" + constraint.getAttribute("label") + ");");
					}
					output.println(tabs + "record.setConstraint" + constraint.getAttribute("label") + "(c" + 
						constraint.getAttribute("label") + ");");
				}
			}
		}

		// Test symbol against constraints
		num = 0;
		if (cSet != null) {
			NodeList constraints = cSet.getChildNodes();
			if (constraints.getLength() != 0) {
				output.print("\t\t\tif (!(");
				for (int i = 0; i < constraints.getLength(); i++) {
					Element constraint = (Element) constraints.item(i);
					if (i > 0)
						output.print(" &&\n\t\t\t\t");
					output.print("evaluateConstraint" + constraint.getAttribute("label") + "(record" + 
						(isProduction ? ", forest)" : ")"));
				}
				output.println(")) {");
				output.println("\t\t\t\tsymbolExists = false;");
				output.println("\t\t\t\trecord.set_" + name + "(null);");

				// Remove solver constraints that depend on this existential symbol, if symbol not valid
				tabs = "\t\t\t\t";
				if (mainConstraintSetElement != null) {
					nodes = mainConstraintSetElement.getElementsByTagName("SolverConstraint");
					for (int j = 0; j < nodes.getLength(); j++) {
						Element child = (Element) nodes.item(j);
						if (child.getAttribute("dependsOn").equals(name)) 
							output.println(tabs + "record.setConstraint" + child.getAttribute("label") + "(null);");
					}
				}
				output.println("\t\t\t}");
				constraintSets.add(cSet);
			}
		}
		output.println("\t\t}");
	}

	// Prints two arguments
	protected String printTwoArguments(String middle, Element operation)
	{
		String params[] = new String[2];

		// Print child nodes
		NodeList args = operation.getChildNodes();
		for (int i = 0; i < args.getLength(); i++) {
			Element argument = (Element) args.item(i);
			params[i] = printArgument(argument);
		}
		return "(" + params[0] + middle + params[1] + ")";
	}

	// Prints out an argument
	protected String printArgument(Element argument)
	{
		String string = "";

		// See what type of argument it is
		if (argument.getTagName().equals("ConstantValue"))
			string = argument.getAttribute("value");
		else if (argument.getTagName().equals("SymbolValue"))
			string = "record.get_" + argument.getAttribute("name") + "()";
		else if (argument.getTagName().equals("AttributeValue"))
			string = "record.get_" + argument.getAttribute("symbol_name") + "().get_" +
				argument.getAttribute("attribute_name") + "()";
		else if (argument.getTagName().equals("FunctionValue"))
			string = argument.getAttribute("code");

		if (string.equals(""))
			Error.die(this, argument.getAttribute("_line_number_"), "Could not generate argument string");
		return string;
	}

	// Prints the array for a constraint variable, named "cvArray[label]"
	protected void printConstraintVariableArray(Element constraint, String tabs, PrintWriter output)
	{
		// Create the variable array
		NodeList variables = constraint.getElementsByTagName("Variable");
		if (variables.getLength() == 0) {
			output.println(tabs + "ConstraintVariable cvArray" + constraint.getAttribute("label") + "[] = null;");
			return;
		}
		output.println(tabs + "ConstraintVariable cvArray" + constraint.getAttribute("label") + "[] = new ConstraintVariable[" +
			variables.getLength() + "];");

		// Add variables to the array
		for (int i = 0; i < variables.getLength(); i++) {
			Element variable = (Element) variables.item(i);
			output.println(tabs + "cvArray" +  constraint.getAttribute("label") + "[" + i + "] = record.get_" +
				variable.getAttribute("symbol_name") + "().getConstraintVariable_" + variable.getAttribute("attribute_name") + "();");
		}
	}

	// This method creates all the classes need for precompiled constraints and constraint expressions
	protected void createConstraintClasses()
	{
		Iterator iter;

		iter = constraintClasses.keySet().iterator();
		while (iter.hasNext()) {
			String className = (String) iter.next();
			String contents = (String) constraintClasses.get(className);

			PrintWriter output = Compiler.openTextFile(Compiler.PACKAGE_DIRECTORY + className + ".java");
			constraintCompiler.createConstraintClass(className, "au.edu.monash.csse.tonyj.cider.interpreter", contents, output);
			output.close();
		}
		iter = constraintExpressionClasses.keySet().iterator();
		while (iter.hasNext()) {
			String className = (String) iter.next();
			String contents = (String) constraintExpressionClasses.get(className);

			PrintWriter output = Compiler.openTextFile(Compiler.PACKAGE_DIRECTORY + className + ".java");
			constraintCompiler.createConstraintExpressionClass(className, "au.edu.monash.csse.tonyj.cider.interpreter", contents, output);
			output.close();
		}
	}

	// Returns the number of this rule
	public int getNumber()
	{
		return number;
	}

	// Returns the name of this rule
	public String getName()
	{
		return ruleName;
	}
}

