/*
    Copyright (C) 2003  Anthony R. Jansen.

    This file is part of the CIDER Toolkit.

    The CIDER Toolkit is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    The CIDER Toolkit is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with the CIDER Toolkit; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

// ----------------------------------------
// CIDER Toolkit: Compiler Package
//
// Author:  Anthony R. Jansen
// Begun:   September 2002
// Class:   SymbolDefinition
//
// This class is used to represent the
// information contained in a symbol 
// definition, part of a diagrammatic
// language specification. The information
// is taken from an XML DOM tree.
// ----------------------------------------

package au.edu.monash.csse.tonyj.cider.compiler;

import org.w3c.dom.Node;
import org.w3c.dom.Element;
import org.w3c.dom.NodeList;
import java.util.List;
import java.util.LinkedList;
import java.util.Iterator;
import java.io.PrintWriter;

public class SymbolDefinition {

	// Private variables
	private boolean		term;
	private boolean		abs;
	private String		name;
	private String		drawableName;
	private String		extendsName;
	private String		interfaceName;
	private List		localAttributes;
	private List		inheritedAttributes;
	private boolean 	needsMoreAttributes;

	private static int	attValue = 1;		// Used to give each attribute a unique identifier

	private static String	illegalFullNames[] = { "Log", "Error", "RegExpTerm", "ParseForest" };
	private static String	illegalNameSubstrings[] = { "Interpreter", "Production", "Transformation", "Undo", "RegularExpression", 
								"Constraint", "Symbol", "Cider" };

	// Constructor, which takes a DOM Node for the SymbolDef
	public SymbolDefinition(Element symbol)
	{
		localAttributes = new LinkedList();
		inheritedAttributes = new LinkedList();

		// Check node name
		if (!symbol.getTagName().equals("SymbolDef")) 
			Error.die(this, symbol.getAttribute("_line_number_"), "Expecting a <SymbolDef> element");

		// Set symbol variables
		name = symbol.getAttribute("name");
		if (!Compiler.isValidNameString(name))
			Error.die(this, symbol.getAttribute("_line_number_"), "Invalid name string in <SymbolDef> tag");
                extendsName = symbol.getAttribute("extends");
                interfaceName = symbol.getAttribute("interface");
                drawableName = symbol.getAttribute("drawable");
		needsMoreAttributes = (extendsName.equals("") ? false : true);


		// Check name is not illegal
		for (int i = 0; i < illegalFullNames.length; i++)
			if (name.equals(illegalFullNames[i]))
				Error.die(this, symbol.getAttribute("_line_number_"), "Illegal symbol name in <SymbolDef> tag: " + name);
		for (int i = 0; i < illegalNameSubstrings.length; i++)
			if (name.indexOf(illegalNameSubstrings[i]) != -1)
				Error.die(this, symbol.getAttribute("_line_number_"), "Illegal symbol name in <SymbolDef> tag " +
					"(name cannot contain substring \"" + illegalNameSubstrings[i] + "\"): " + name);

		String type = symbol.getAttribute("type");
		abs = (type.equals("abstract") ? true : false); 
		term = (type.equals("terminal") ? true : false); 

		// Warn if a terminal symbol does not have a drawable class name
		if (term && drawableName.equals(""))
			Error.warn(this, "Terminal symbol does not have a \"Drawable Class\" attribute: " + name);

                // Get details of symbol attributes
                NodeList atts = symbol.getChildNodes();
                for (int i = 0; i < atts.getLength(); i++) {
                        Element attr = (Element) atts.item(i);

			// Check node name
			if (!attr.getTagName().equals("Attribute")) 
				Error.die(this, attr.getAttribute("_line_number_"), 
					"Expecting the <SymbolDef> element to contain only <Attribute> elements");

			// Check for valud attribute name
			if (!Compiler.isValidNameString(attr.getAttribute("name")))
				Error.die(this, attr.getAttribute("_line_number_"), "Invalid name string in <Attribute> tag");


			// Check for attribute name duplication
			if (doesLocalAttributeExist(attr.getAttribute("name")))
				Error.die(this, attr.getAttribute("_line_number_"), 
					"There is an attribute name duplication for symbol " + name + ": " + attr.getAttribute("name")); 

			// Add attribute to the list
			localAttributes.add(new Attribute(attr.getAttribute("name"), attr.getAttribute("datatype"), attr.getAttribute("editweight"), 
				attr.getAttribute("stayweight"), attr.getAttribute("_line_number_")));
                }
	}

	// This method returns true iff an attribute with the specified name 
	// is already in the local attribute list
	public boolean doesLocalAttributeExist(String name)
	{
		Iterator iter = localAttributes.iterator();
		while (iter.hasNext()) {
			Attribute attr = (Attribute) iter.next();
			if (attr.getName().equals(name))
				return true;
		}
		return false;
	}

	// This method returns true iff an attribute with the specified name 
	// is already in the inherited attribute list
	public boolean doesInheritedAttributeExist(String name)
	{
		Iterator iter = inheritedAttributes.iterator();
		while (iter.hasNext()) {
			Attribute attr = (Attribute) iter.next();
			if (attr.getName().equals(name))
				return true;
		}
		return false;
	}

	// Returns the name of the symbol
	public String getName()
	{
		return name;
	}

	// Returns name of symbol that this symbol extends, or empty 
	// string if there is none
	public String getExtendsName()
	{
		return extendsName;
	}

	// Returns the drawable class name if this is a terminal symbol
	// or empty string (or whatever else) otherwise
	public String getDrawableClass()
	{
		return drawableName;
	}

	// Returns true if symbol is a terminal symbol, false otherwise
	public boolean isTerminal()
	{
		return term;
	}

	// Returns true if symbol is an abstract symbol, false otherwise
	public boolean isAbstract()
	{
		return abs;
	}

	// Returns true iff symbol needs more inherited attributes
	public boolean doesNeedMoreAttributes()
	{
		return needsMoreAttributes;
	}

	// One called, this symbol no longer needs any inherited attributes
	public void noMoreAttributes()
	{
		needsMoreAttributes = false;
	}

	// Returns the specified attribute (either local or inherited), 
	// or null if it does not exist
	public Attribute getAttribute(String name)
	{
		Iterator iter = localAttributes.iterator();
		while (iter.hasNext()) {
			Attribute attr = (Attribute) iter.next();
			if (attr.getName().equals(name))
				return attr;
		}
		iter = inheritedAttributes.iterator();
		while (iter.hasNext()) {
			Attribute attr = (Attribute) iter.next();
			if (attr.getName().equals(name))
				return attr;
		}
		return null;
	}

	// Returns the list of local attributes this symbol has
	public List getLocalAttributes()
	{
		return localAttributes;
	}

	// Returns the list of inherited attributes this symbol has
	public List getInheritedAttributes()
	{
		return inheritedAttributes;
	}

	// Returns the list of all attributes this symbol has
	public List getAllAttributes()
	{
		List allList = new LinkedList(localAttributes);
		allList.addAll(inheritedAttributes);
		return allList;
	}

	// Adds to the list of inherited attributes this symbol has
	// if the symbol is still needs more attributes
	public void addInheritedAttributes(List atts)
	{
		if (needsMoreAttributes) {

			Iterator iter = atts.iterator();
			while (iter.hasNext()) {
				Attribute attr = (Attribute) iter.next();

				// Check for attribute name duplication
				if (doesLocalAttributeExist(attr.getName()) || doesInheritedAttributeExist(attr.getName()))
					Error.die(this, "There is an inherited attribute name duplication for symbol " + name + ": " 
						+ attr.getName()); 
				
				inheritedAttributes.add(attr);
			}
		}
	}

	// This method creates the Java class for this symbol definition
	public void createSymbolClass(String constraintVariableClass, AttDependencyGraph attDependencyGraph)
	{
		int numArgs;
		String line;
		Iterator iter;

		// Open file for the class
		PrintWriter output = Compiler.openTextFile(Compiler.PACKAGE_DIRECTORY + name + ".java");

		// Print class header details (public scope)
		output.println("");
		output.println(Compiler.PACKAGE_LINE);
		output.println("");
		output.println("import au.edu.monash.csse.tonyj.cider.constraints.ConstraintVariable;");
		output.println("import au.edu.monash.csse.tonyj.cider.constraints.ConstraintSolver;");
		output.println("import java.util.Set;");
		output.println("import java.util.HashSet;");
		output.println("");
		line = "public ";
		if (isAbstract())
			line = line + "abstract ";
		line = line + "class " + name;
		if (getExtendsName().equals(""))
			line = line + " extends GrammarSymbol";
		else
			line = line + " extends " + getExtendsName();
		if (interfaceName.equals(""))
			line += " {";
		else
			line += " implements " + interfaceName + " {";
		output.println(line);
		output.println("");
		
		// Print attribute variables (private scope)
		iter = localAttributes.iterator();
		while (iter.hasNext()) {
			Attribute attr = (Attribute) iter.next();
			output.println("\tprivate " + attr.getDataType() + " _" + attr.getName() + ";");
		}

		// Print initial attribute value variables (private scope)
		iter = localAttributes.iterator();
		while (iter.hasNext()) {
			Attribute attr = (Attribute) iter.next();
			output.println("\tprivate " + (attr.isConstraintVariable() ? "double" : attr.getDataType()) + 
				" _initial_" + attr.getName() + ";");
		}
		if (!localAttributes.isEmpty())
			output.println("");

		// Print static codes for each attribute (package scope)
		iter = localAttributes.iterator();
		while (iter.hasNext()) {
			Attribute attr = (Attribute) iter.next();
			output.println("\tstatic final int _att_" + attr.getName() + "_ = " + attValue + ";");
			attValue++;
		}
		if (!localAttributes.isEmpty())
			output.println("");

		// Print constructor parameter list (public scope)
		line = "\tpublic " + name + "(";
		numArgs = 0;
		if (isAbstract()) {
			line += "boolean terminal";
			numArgs++;
		}
		iter = getAllAttributes().iterator();
		while (iter.hasNext()) {
			Attribute attr = (Attribute) iter.next();
			if (numArgs > 0)
				line += ",\n\t\t";
			line += (attr.isConstraintVariable() ? "double" : attr.getDataType()) + " _init_" + attr.getName();
			numArgs++;
		}
		line = line + ")\n\t{";
		output.println(line);

		// Print call to superclass constructor, starting with the terminal symbol value
		if (isAbstract())
			line = "\t\tsuper(terminal";
		else if (isTerminal())
			line = "\t\tsuper(true";
		else
			line = "\t\tsuper(false";

		// Pass on all inherited attributes
		iter = inheritedAttributes.iterator();
		while (iter.hasNext()) {
			Attribute attr = (Attribute) iter.next();
			line = line + ",  _init_" + attr.getName();
		}
		line = line + ");";
		output.println(line);

		// Initialize all attributes of this class
		boolean cvPresent = false;
		iter = localAttributes.iterator();
		while (iter.hasNext()) {
			Attribute attr = (Attribute) iter.next();
			if (attr.isConstraintVariable()) {
				output.println("\t\t_" + attr.getName() + " = new " + constraintVariableClass + "();");
				output.println("\t\t_" + attr.getName() + ".setDesiredValue(_init_" + attr.getName() + ");");
				output.println("\t\t_" + attr.getName() + ".setEditWeight(" + attr.getEditWeight() + ");");
				output.println("\t\t_" + attr.getName() + ".setStayWeight(" + attr.getStayWeight() + ");");
				cvPresent = true;
			}
			else
				output.println("\t\t_" + attr.getName() + " = _init_" + attr.getName() + ";");
			output.println("\t\t_initial_" + attr.getName() + " = _init_" + attr.getName() + ";");
		}

		// End constructor
		output.println("\t}");
		output.println("");

		// Register symbol and add all constraint variables to the solver
		output.println("\tboolean registerSymbol(SymbolNode node)");
		output.println("\t{");
		output.println("\t\tboolean success = super.registerSymbol(node);");
		if (cvPresent) {
			output.println("\t\tif (success) {");
			iter = localAttributes.iterator();
			while (iter.hasNext()) {
				Attribute attr = (Attribute) iter.next();
				if (attr.isConstraintVariable()) 
					output.println("\t\t\tgetSymbolNode().getInterpreter().getConstraintSolver().addVariable(_" + 
						attr.getName() + ");");
			}
			output.println("\t\t\tgetSymbolNode().getInterpreter().getConstraintSolver().resolve();");
			output.println("\t\t}");
		}
		output.println("\t\treturn success;");
		output.println("\t}");
		output.println("");

		// Deregister symbol and remove all constraint variables from the solver
		output.println("\tboolean deregisterSymbol()");
		output.println("\t{");
		if (cvPresent) {
			output.println("\t\tif (isRegistered()) {");
			iter = localAttributes.iterator();
			while (iter.hasNext()) {
				Attribute attr = (Attribute) iter.next();
				if (attr.isConstraintVariable()) 
					output.println("\t\t\tgetSymbolNode().getInterpreter().getConstraintSolver().removeVariable(_" + 
						attr.getName() + ");");
			}
			output.println("\t\t\tgetSymbolNode().getInterpreter().getConstraintSolver().resolve();");
			output.println("\t\t}");
		}
		output.println("\t\treturn super.deregisterSymbol();");
		output.println("\t}");
		output.println("");

		// Print reinitialize attribute methods (package scope)
		iter = localAttributes.iterator();
		while (iter.hasNext()) {
			Attribute attr = (Attribute) iter.next();
			output.println("\tvoid reinitialize_" + attr.getName() + "()");
			output.println("\t{");
			output.println("\t\tset_" + attr.getName() + "(_initial_" + attr.getName() + ");");
			output.println("\t}");
			output.println("");
		}

		// Print get methods (public scope)
		iter = localAttributes.iterator();
		while (iter.hasNext()) {
			Attribute attr = (Attribute) iter.next();
			output.println("\tpublic " + (attr.isConstraintVariable() ? "double" : attr.getDataType()) + " get_" + attr.getName() + "()");
			output.println("\t{");
			if (attr.isConstraintVariable())
				output.println("\t\treturn (isRegistered() ? _" + attr.getName() + ".getSolvedValue() : _" + attr.getName() +
					".getDesiredValue());");
			else
				output.println("\t\treturn _" + attr.getName() + ";");
			output.println("\t}");
			output.println("");

			// For constraint variables print get methods for ConstraintVariable class (public scope)
			if (attr.isConstraintVariable()) {
				output.println("\tpublic ConstraintVariable getConstraintVariable_" + attr.getName() + "()");
				output.println("\t{");
				output.println("\t\treturn _" + attr.getName() + ";");
				output.println("\t}");
				output.println("");
			}
		}

		// Print set methods for all symbols (public scope for terminal attributes, package scope for others)
		iter = localAttributes.iterator();
		while (iter.hasNext()) {
			Attribute attr = (Attribute) iter.next();

			// Setting of constraint variables can always be public; for non constraint variables it depends on att dependency graph
			if (attr.isConstraintVariable())
				output.println("\tpublic void set_" + attr.getName() + "(double value)");
			else 
				output.println((attDependencyGraph.isTerminal(name + "." + attr.getName()) ? "\tpublic " : "\t") + 
					"void set_" + attr.getName() + "(" + attr.getDataType() + " value)");
			output.println("\t{");

			if (attr.isConstraintVariable()) {
				output.println("\t\tif (isRegistered()) {");
				output.println("\t\t\tgetSymbolNode().getInterpreter().getConstraintSolver().addEditVariable(_" + 
					attr.getName() + ");");
				output.println("\t\t\t_" + attr.getName() + ".setDesiredValue(value);");
				output.println("\t\t\tLog.getLogger().info(\"Symbol modified (CV attribute: " + attr.getName() + 
					"): \" + toString());");
				output.println("\t\t\tgetSymbolNode().getInterpreter().resolve();");
				output.println("\t\t}");
				output.println("\t\telse");
				output.println("\t\t\t_" + attr.getName() + ".setDesiredValue(value);");
			}
			else {
				// Get the old values
				if (attr.getDataType().equals("double"))
					output.println("\t\tObject oldValue = new Double(_" + attr.getName() + ");");
				else if (attr.getDataType().equals("float"))
					output.println("\t\tObject oldValue = new Float(_" + attr.getName() + ");");
				else if (attr.getDataType().equals("int"))
					output.println("\t\tObject oldValue = new Integer(_" + attr.getName() + ");");
				else if (attr.getDataType().equals("short"))
					output.println("\t\tObject oldValue = new Short(_" + attr.getName() + ");");
				else if (attr.getDataType().equals("long"))
					output.println("\t\tObject oldValue = new Long(_" + attr.getName() + ");");
				else if (attr.getDataType().equals("boolean"))
					output.println("\t\tObject oldValue = new Boolean(_" + attr.getName() + ");");
				else if (attr.getDataType().equals("byte"))
					output.println("\t\tObject oldValue = new Byte(_" + attr.getName() + ");");
				else if (attr.getDataType().equals("char"))
					output.println("\t\tObject oldValue = new Character(_" + attr.getName() + ");");
				else 
					output.println("\t\tObject oldValue = _" + attr.getName() + ";");

				output.println("\t\t_" + attr.getName() + " = value;");
				output.println("\t\tif (isRegistered()) {");
				output.println("\t\t\tLog.getLogger().info(\"Symbol modified (attribute: " + attr.getName() + "): \" + toString());");
				output.println("\t\t\tgetSymbolNode().getInterpreter().symbolModified(this, " + name + "._att_" + attr.getName() + 
					"_, oldValue);");
				output.println("\t\t}");
			}
			output.println("\t}");
			output.println("");
		}

		// Print restore attribute value methods (package scope)
		output.println("\tvoid restoreAttributeValue(int attribute, Object value)");
		output.println("\t{");
		output.println("\t\tif (!isRegistered()) {");
		output.println("\t\t\tLog.getLogger().warning(\"Attempt to restore an attribute value on an unregistered " + name + " symbol\");");
		output.println("\t\t\treturn;");
		output.println("\t\t}");
		output.println("");

		numArgs = 0;
		iter = localAttributes.iterator();
		while (iter.hasNext()) {
			Attribute attr = (Attribute) iter.next();
			output.println("\t\t" + (numArgs == 0 ? "if" : "else if") + " (attribute == " + name + "._att_" + attr.getName() + "_) {");
			if (attr.isConstraintVariable()) {
				output.println("\t\t\tgetSymbolNode().getInterpreter().getConstraintSolver().addEditVariable(_" + 
					attr.getName() + ");");
				output.println("\t\t\t_" + attr.getName() + ".setDesiredValue(((Double) value).doubleValue());");
			}
			else {
				if (attr.getDataType().equals("double"))
					output.println("\t\t\t_" + attr.getName() + " = ((Double) value).doubleValue();");
				else if (attr.getDataType().equals("float"))
					output.println("\t\t\t_" + attr.getName() + " = ((Float) value).floatValue();");
				else if (attr.getDataType().equals("int"))
					output.println("\t\t\t_" + attr.getName() + " = ((Integer) value).intValue();");
				else if (attr.getDataType().equals("short"))
					output.println("\t\t\t_" + attr.getName() + " = ((Short) value).shortValue();");
				else if (attr.getDataType().equals("long"))
					output.println("\t\t\t_" + attr.getName() + " = ((Long) value).longValue();");
				else if (attr.getDataType().equals("boolean"))
					output.println("\t\t\t_" + attr.getName() + " = ((Boolean) value).booleanValue();");
				else if (attr.getDataType().equals("byte"))
					output.println("\t\t\t_" + attr.getName() + " = ((Byte) value).byteValue();");
				else if (attr.getDataType().equals("char"))
					output.println("\t\t\t_" + attr.getName() + " = ((Character) value).charValue();");
				else
					output.println("\t\t\t_" + attr.getName() + " = (" + attr.getDataType() + ") value;");
			}
			output.println("\t\t\tLog.getLogger().info(\"Symbol modified (restoring attribute: " + attr.getName() + 
				"): \" + toString());");
			output.println("\t\t}");
			numArgs++;
		}
		if (!extendsName.equals("")) {
			if (numArgs == 0)
				output.println("\t\tsuper.restoreAttributeValue(attribute, value);");
			else {
				output.println("\t\telse");
				output.println("\t\t\tsuper.restoreAttributeValue(attribute, value);");
			}
		}
		output.println("\t}");
		output.println("");

		// Print getName method
		if (isAbstract()) 
			output.println("\tpublic abstract String getName();");
		else {
			output.println("\tpublic String getName()");
			output.println("\t{");
			output.println("\t\treturn \"" + name + "\";");
			output.println("\t}");
		}
		output.println("");

		// Print debugging string for non-abstract classes (public scope)
		if (isAbstract()) 
			output.println("\tpublic abstract String toString();");
		else {
			output.println("\tpublic String toString()");
			output.println("\t{");
			line = "\t\treturn \"Symbol \" + getIDNumber() + \": " + name + " [";

			// Print attribute values
			numArgs = 0;
			iter = localAttributes.iterator();
			while (iter.hasNext()) {
				Attribute attr = (Attribute) iter.next();
				if (numArgs > 0)
					line = line + ", \" + \n\t\t\t\"";
				line = line + attr.getName() + " = \\\"\" + get_" + attr.getName() + "() + \"\\\"";
				numArgs++;
			}
			iter = inheritedAttributes.iterator();
			while (iter.hasNext()) {
				Attribute attr = (Attribute) iter.next();
				if (numArgs > 0)
					line = line + ", \" + \n\t\t\t\"";
				line = line + attr.getName() + " = \\\"\" + get_" + attr.getName() + "() + \"\\\"";
				numArgs++;
			}
			line = line + "]\";";
			output.println(line);
			output.println("\t}");
		}
		output.println("");

		// Print method to identify which, if any, attribute is represented by a given constraint variable (package scope)
		// Returns -1 if no attribute is found
		if (isAbstract()) 
			output.println("\tabstract int getAttributeIdentifier(ConstraintVariable var);");
		else {
			output.println("\tint getAttributeIdentifier(ConstraintVariable var)");
			output.println("\t{");
			iter = getAllAttributes().iterator();
			while (iter.hasNext()) {
				Attribute attr = (Attribute) iter.next();
				if (attr.isConstraintVariable()) {
					output.println("\t\tif (var == getConstraintVariable_" + attr.getName() + "())");
					output.println("\t\t\treturn _att_" + attr.getName() + "_;");
				}
			}
			output.println("\t\treturn -1;");
			output.println("\t}");
		}
		output.println("");

		// Print method to return the name of an attribute (if there is one), given its static id number (package scope)
		if (isAbstract())
			output.println("\tabstract String getAttributeName(int identifier);");
		else {
			output.println("\tString getAttributeName(int identifier)");
			output.println("\t{");
			output.println("\t\tswitch(identifier) {");
			iter = getAllAttributes().iterator();
			while (iter.hasNext()) {
				Attribute attr = (Attribute) iter.next();
				output.println("\t\t\tcase _att_" + attr.getName() + "_: return \"" + attr.getName() + "\";");
			}
			output.println("\t\t\tdefault: return \"INVALID_ATTRIBUTE_IDENTIFIER\";");
			output.println("\t\t}");
			output.println("\t}");
		}
		output.println("");

		// Print method to return all constraint variables associated with this symbol (package scope)
		if (isAbstract()) 
			output.println("\tabstract Set getConstraintVariables();");
		else {
			output.println("\tSet getConstraintVariables()");
			output.println("\t{");
			output.println("\t\tSet vars = new HashSet();");
			iter = getAllAttributes().iterator();
			while (iter.hasNext()) {
				Attribute attr = (Attribute) iter.next();
				if (attr.isConstraintVariable())
					output.println("\t\tvars.add(getConstraintVariable_" + attr.getName() + "());");
			}
			output.println("\t\treturn vars;");
			output.println("\t}");
		}

		// End class and close file
		output.println("}");
		output.println("");
		output.close();
	}
}

