/*
    Copyright (C) 2003  Anthony R. Jansen.

    This file is part of the CIDER Toolkit.

    The CIDER Toolkit is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    The CIDER Toolkit is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with the CIDER Toolkit; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

// ----------------------------------------
// CIDER Toolkit: Compiler Package 
//
// Author:  Anthony R. Jansen
// Begun:   May 2003
// Class:   TextInputFile
//
// This class is used for processing a text
// input file.
// ----------------------------------------

package au.edu.monash.csse.tonyj.cider.compiler;

import java.io.RandomAccessFile;
import java.io.IOException;
import java.io.FileNotFoundException;

class TextInputFile {

	// I/O file handling variables
	private RandomAccessFile	inputFile;
	private	int			lineNumber;
	private	String			currentLine;
	private String			filename;

	// Constant
	public static final String	EOF = "end-of-file";

	// Constructor which accepts the input filename as
	// its argument, and opens up the file if possible
	public TextInputFile(String fname)
	{
		filename = fname;
		currentLine = "";
		lineNumber = 0;
		try {
			inputFile = new RandomAccessFile(filename, "r");
		}
		catch(FileNotFoundException e) {
			Error.die(this, "Could not open file: " + filename);
		}
	}

	// Returns the current line number
	public int getLineNumber()
	{
		return lineNumber;
	}

	// Returns the file name
	public String getFilename()
	{
		return filename;
	}

	// Method to read in a text line from the input file
	private String readLine()
	{
		String	str = "";
	
		try {
			str = inputFile.readLine();

			// If end-of-file, indicate in string
			if (str == null)
				str = EOF;
			else 
				lineNumber++;
		}
		catch(IOException e) {
			Error.die(this, "Could not read in line " + (lineNumber + 1) + " of file " + filename);
		}
		return str;
	}

	public boolean isEndOfFile()
	{
		// If no more tokens on this line (or a comment is present), get the next one
		while (currentLine.equals("") || currentLine.startsWith("#"))
			currentLine = readLine().trim();

		return currentLine.equals(EOF);
	}

	// Method reads in a token which may consist of letters, digits, _, -, +, . and @ characters
	public String readStandardToken()
	{
		String token = "";
		int index = 0;

		// If no more tokens on this line (or a comment is present), get the next one
		while (currentLine.equals("") || currentLine.startsWith("#"))
			currentLine = readLine().trim();

		try {
			for (int i = 0; i < currentLine.length(); i++) {
				if (Character.isLetterOrDigit(currentLine.charAt(i)) ||
					currentLine.charAt(i) == '_' ||
					currentLine.charAt(i) == '-' ||
					currentLine.charAt(i) == '+' ||
					currentLine.charAt(i) == '.' ||
					currentLine.charAt(i) == '@')
					index = i + 1;
				else
					break;
			}
			if (index > 0) {
				token = currentLine.substring(0, index);
				currentLine = currentLine.substring(index).trim();
			}
			else 
				Error.die(this, "Expecting a standard token on line " + lineNumber + " of file " + filename);
		}
		catch (IndexOutOfBoundsException e) {
			Error.die(this, "Error processing line " + lineNumber + " of file " + filename);
		}
		return token;
	}

	// Returns true if the next token is one specified
	public boolean isNextToken(String token)
	{
		// If no more tokens on this line (or a comment is present), get the next one
		while (currentLine.equals("") || currentLine.startsWith("#"))
			currentLine = readLine().trim();

		return currentLine.startsWith(token);
	}
	
	// Returns true if the next token is one specified, and processes that token
	public boolean readNextToken(String token)
	{
		// If no more tokens on this line (or a comment is present), get the next one
		while (currentLine.equals("") || currentLine.startsWith("#"))
			currentLine = readLine().trim();

		try {
			if (currentLine.startsWith(token)) {
				currentLine = currentLine.substring(token.length()).trim();
				return true;
			}
		}
		catch (IndexOutOfBoundsException e) {
			Error.die(this, "Error processing line " + lineNumber + " of file " + filename);
		}
		return false;
	}

	// Returns the contents of pair of quote marks (must be on a single line)
	// If quotes are between " characters, this character is escaped with \" in the string
	// If quotes are between ' characters, this character is escaped with \' in the string
	// <, >, \", \' and & are escaped with their XML references
	public String readQuoteString()
	{
		String token = "";
		int index;
		char quoteChar;
		
		// If no more tokens on this line (or a comment is present), get the next one
		while (currentLine.equals("") || currentLine.startsWith("#"))
			currentLine = readLine().trim();

		try {
			quoteChar = currentLine.charAt(0);
			if (!(quoteChar == '\"' || quoteChar == '\''))
				Error.die(this, "Expecting an opening quote mark on line " + lineNumber + " of file " + filename);
			currentLine = currentLine.substring(1);
			index = currentLine.indexOf(quoteChar);
			while (index > 0 && currentLine.charAt(index - 1) == '\\')
				index = currentLine.indexOf(quoteChar, index + 1);

			// Must have closing quote
			if (index == -1) 
				Error.die(this, "Expecting a closing quote mark on line " + lineNumber + " of file " + filename);
			token = currentLine.substring(0, index);
			currentLine = currentLine.substring(index+1).trim();
		}
		catch (IndexOutOfBoundsException e) {
			Error.die(this, "Error processing line " + lineNumber + " of file " + filename);
		}

		// Convert all XML references (must start with &)
		token = token.replaceAll("&", "&amp;");
		token = token.replaceAll("<", "&lt;");
		token = token.replaceAll(">", "&gt;");
		token = token.replaceAll("\\\'", "&apos;");
		token = token.replaceAll("\\\"", "&quot;");
		token = token.replaceAll("\'", "&apos;");
		token = token.replaceAll("\"", "&quot;");
		
		return token;
	}

	// Returns the contents of pair of brackets (must be on a single line)
	public String readBracketString()
	{
		String token = "";
		int index;
		
		// If no more tokens on this line (or a comment is present), get the next one
		while (currentLine.equals("") || currentLine.startsWith("#"))
			currentLine = readLine().trim();

		try {
			if (currentLine.charAt(0) != '(')
				Error.die(this, "Expecting an opening bracket on line " + lineNumber + " of file " + filename);
			currentLine = currentLine.substring(1);
			index = currentLine.indexOf(')');

			// Must have closing quote
			if (index == -1) 
				Error.die(this, "Expecting a closing bracket on line " + lineNumber + " of file " + filename);
			token = currentLine.substring(0, index);
			currentLine = currentLine.substring(index+1).trim();
		}
		catch (IndexOutOfBoundsException e) {
			Error.die(this, "Error processing line " + lineNumber + " of file " + filename);
		}
		return token;
	}

	// Close file before class is destroyed
	public void finalize()
	{
		try {
			if (inputFile != null)
				inputFile.close();
		}
		catch(IOException e) { ; }
	}
}

