/*
    Copyright (C) 2003  Anthony R. Jansen.

    This file is part of the CIDER Toolkit.

    The CIDER Toolkit is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    The CIDER Toolkit is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with the CIDER Toolkit; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

// ----------------------------------------
// CIDER Toolkit: Compiler Package 
//
// Author:  Anthony R. Jansen
// Begun:   June 2003
// Class:   TransformationConverter
//
// This class is used for converting a 
// cider transformation file into an XML 
// format.
// ----------------------------------------

package au.edu.monash.csse.tonyj.cider.compiler;

public class TransformationConverter extends BaseConverter {

	// Constructor
	public TransformationConverter(String inputFilename, String outputFilename)
	{
		super(inputFilename, outputFilename);
	}

	// Performs the conversion (should be called only once)
	public void convert()
	{
		// Output XML and DOCTYPE tags
		output.println("<?xml version=\"1.0\" encoding=\"UTF-8\" ?>");
		output.println("<!DOCTYPE Cider SYSTEM \"Cider-" + Compiler.VERSION + ".dtd\" >");
		output.println("");

		// Output cider and transformations start tags
		output.println("<Cider>");
		output.println("<Transformations>");
		output.println("");

		while (!input.isEndOfFile())
			processTransformationDefinition();

		// Output cider and transformations end tags
		output.println("</Transformations>");
		output.println("</Cider>");
		output.println("");
	}

	// Determines the type of transformation 
	private void processTransformationDefinition()
	{
		String name = input.readStandardToken();
		if (!input.readNextToken(":"))
			InputError("Expecting : to follow the transformation name");

		String type = input.readStandardToken();
		if (type.equals("transformation"))
			processTransformation(name);
		else if (type.equals("disjunction"))
			processDisjunction(name);
		else if (type.equals("compound_transformation"))
			processCompoundTransformation(name);
		else
			InputError("Unknown transformation type: " + type);
	}

	// Process a specific transformation
	private void processTransformation(String name)
	{
		if (!input.readNextToken("{"))
			InputError("Expecting { following a \"transformation\" declaration");

		output.println("\t<Transformation name=\"" + name + "\" >");
		
		// Get input symbols
		output.println("\t\t<InputSymbols>");
		if (!input.isNextToken("where")) {
			processSymbol("\t\t\t");
			while (input.readNextToken(","))
				processSymbol("\t\t\t");
		}
		output.println("\t\t</InputSymbols>");
		addLineComment();

		processConstraintSet("\t\t\t");
		processApplyActions();
		processSolverConstraints();

		if (!input.readNextToken("}"))
			InputError("Expecting } to conclude a \"transformation\" declaration");

		output.println("\t</Transformation>");
		addLineComment();
		output.println("");
	}

	// Process a specific disjunction
	private void processDisjunction(String name)
	{
		if (!input.readNextToken("{"))
			InputError("Expecting { following a \"disjunction\" declaration");

		output.println("\t<Disjunction name=\"" + name + "\" >");
		processComponent("\t\t");
		while (input.readNextToken("|"))
			processComponent("\t\t");

		if (!input.readNextToken(";"))
			InputError("Expecting ; to conclude disjunction list;");
		if (!input.readNextToken("}"))
			InputError("Expecting } to conclude a \"disjunction\" declaration");

		output.println("\t</Disjunction>");
		addLineComment();
		output.println("");
	}

	// Process a specific compound transformation
	private void processCompoundTransformation(String name)
	{
		if (!input.readNextToken("{"))
			InputError("Expecting { following a \"compound_transformation\" declaration");
		output.println("\t<CompoundTransformation name=\"" + name + "\" >");

		// Read in expression
		String exp = "";
		while (!input.readNextToken(";")) {
			if (input.isNextToken(TextInputFile.EOF))
				InputError("EOF reached in the \"compound_transformation\" declaration");

			if (input.readNextToken("("))
				exp += "( ";
			else if (input.readNextToken(")"))
				exp += ") ";
			else if (input.readNextToken("*"))
				exp += "* ";
			else if (input.readNextToken("|"))
				exp += "| ";
			else {
				exp += input.readStandardToken();
				if (input.readNextToken("!"))
					exp += "! ";
				else
					exp += " ";
			}
		}
		RegExpXMLWriter regExpWriter = new RegExpXMLWriter(exp.trim());
		regExpWriter.write(output);

		if (!input.readNextToken("}"))
			InputError("Expecting } to conclude a \"compound_transformation\" declaration");

		output.println("\t</CompoundTransformation>");
		addLineComment();
		output.println("");
	}

	// Process a component
	private void processComponent(String tabs)
	{
		String name = input.readStandardToken();
		if (input.readNextToken("!"))
			output.println(tabs + "<Component name=\"" + name + "\" parallel=\"true\" />");
		else
			output.println(tabs + "<Component name=\"" + name + "\" parallel=\"false\" />");
	}

	// Process a list of transformation actions
	private void processApplyActions()
	{
		// Return if not actions are present
		if (!input.readNextToken("apply")) {
			output.println("\t\t<ApplyActions />");
			return;
		}

		if (!input.readNextToken("{"))
			InputError("Expecting { following an \"apply\" declaration");
		output.println("\t\t<ApplyActions>");

		// Read in different actions
		while (!input.readNextToken("}")) {
			if (input.isNextToken(TextInputFile.EOF))
				InputError("EOF reached in the \"apply\" declaration");

			if (input.readNextToken("(nts)"))
				processNewTerminalSymbol();
			else if (input.readNextToken("(ap)"))
				processApplyProduction();
			else if (input.readNextToken("(set)"))
				processSetAttribute();
			else
				InputError("Unknown action listed in the \"apply\" declaration");
			addLineComment();
		}
		output.println("\t\t</ApplyActions>");
	}

	// Process a new terminal symbol declaration
	private void processNewTerminalSymbol()
	{
		String name = input.readStandardToken();
		if (!input.readNextToken(":"))
			InputError("Expecting : to follow symbol name in a \"(nts)\" declaration");
		String type = input.readStandardToken();
		if (!input.readNextToken("("))
			InputError("Expecting ( to follow symbol type in a \"(nts)\" declaration");

		output.println("\t\t\t<NewTerminalSymbol type=\"" + type + "\" name=\"" + name + "\" >");

		// Read in the arguments
		while (!input.readNextToken(")")) {
			if (input.isNextToken(TextInputFile.EOF))
				InputError("EOF reached in the \"(nts)\" declaration");
			output.println(getArgument("\t\t\t\t"));
			if (!(input.isNextToken(",") || input.isNextToken(")")))
				InputError("New terminal symbol argument must be followed by either a , or ) character");
			input.readNextToken(",");
		}
		if (!input.readNextToken(";"))
			InputError("Expecting ; to conclude a \"(nts)\" declaration");
		output.println("\t\t\t</NewTerminalSymbol>");
	}

	// Process an apply production declaration
	private void processApplyProduction()
	{
		String name = input.readStandardToken();
		output.println("\t\t\t<ApplyProduction name=\"" + name + "\" >");

		// See if there are any symbol mappings
		if (input.readNextToken("using")) {
			if (!input.readNextToken("("))
				InputError("Expecting ( to follow a \"using\" declaration");

			while (!input.readNextToken(")")) {
				if (input.isNextToken(TextInputFile.EOF))
					InputError("EOF reached in the \"using\" declaration");

				// Read in mapping
				String ruleName = input.readStandardToken();
				if (input.readNextToken("->")) {
					String prodName = input.readStandardToken();
					output.println("\t\t\t\t<SetSymbol name=\"" + prodName + "\" use=\"" + ruleName + "\" />");
				}
				else if (input.readNextToken("<-")) {
					String prodName = input.readStandardToken();
					output.println("\t\t\t\t<GetSymbol name=\"" + prodName + "\" label=\"" + ruleName + "\" />");
				}
				else
					InputError("Expecting -> or <- after symbol name in \"using\" declaration");

				if (!(input.isNextToken(",") || input.isNextToken(")")))
					InputError("Apply production \"using\" argument must be followed by either a , or ) character");
				input.readNextToken(",");
			}
		}
		if (!input.readNextToken(";"))
			InputError("Expecting ; to conclude an \"(ap)\" declaration");
		output.println("\t\t\t</ApplyProduction>");
	}

	// Process a set attribute declaration
	private void processSetAttribute()
	{
		String name = input.readStandardToken();
		if (!input.readNextToken(":="))
			InputError("Expecting := to follow attribute name in a \"(set)\" declaration");

		output.println("\t\t\t<SetAttribute name=\"" + name + "\" >");
		output.println(getArgument("\t\t\t\t"));

		if (!input.readNextToken(";"))
			InputError("Expecting ; to conclude a \"(set)\" declaration");
		output.println("\t\t\t</SetAttribute>");
	}

	// Process the solver constraints to be added
	private void processSolverConstraints()
	{
		// Return if no solver constraints are present
		if (!input.readNextToken("solver_constraints"))
			return;

		if (!input.readNextToken("{"))
			InputError("Expecting { following a \"solver_constraints\" declaration");
		output.println("\t\t<AddSolverConstraints>");

		// Read in constraints
		while (!input.readNextToken("}")) {
			if (input.isNextToken(TextInputFile.EOF))
				InputError("EOF reached in the \"solver_constraints\" declaration");
			processConstraint("\t\t\t");
		}
		output.println("\t\t</AddSolverConstraints>");
	}
}

