/*
    Copyright (C) 2003  Anthony R. Jansen.

    This file is part of the CIDER Toolkit.

    The CIDER Toolkit is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    The CIDER Toolkit is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with the CIDER Toolkit; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

// ----------------------------------------
// CIDER Toolkit: Compiler Package
//
// Author:  Anthony R. Jansen
// Begun:   February 2003
// Class:   TransformationDefinition
//
// This class is used to represent the
// information contained in a transformation
// definition, part of a diagrammatic
// language specification. The information
// is taken from an XML DOM tree.
// ----------------------------------------

package au.edu.monash.csse.tonyj.cider.compiler;

import org.w3c.dom.Node;
import org.w3c.dom.Element;
import org.w3c.dom.NodeList;
import org.w3c.dom.DOMException;
import java.io.PrintWriter;
import java.util.Iterator;
import java.util.Set;
import java.util.HashSet;
import java.util.List;
import java.util.LinkedList;
import java.util.ArrayList;
import java.util.Map;
import java.util.HashMap;
import java.util.NoSuchElementException;
import au.edu.monash.csse.tonyj.cider.constraints.ConstraintCompiler;

public class TransformationDefinition extends RuleDefinition {

	// Private variables
	private List		productionList;
	private LinkedList	inputSymbolsInOrder;

	// Symbols involved - maps name to a symbol definition
	private Map		newTerminalSymbols;

	// Constructor, which takes a number, name, a DOM Node for the Transformation and a list of symbol definitions and a list of productions
	public TransformationDefinition(int num, String gName, Element transformation, List sdList, List pList,
		String conClass, String conExpClass, ConstraintCompiler cc)
	{
		super(false, num, gName, transformation, sdList, conClass, conExpClass, cc);
		if (ruleName.equals(""))
			Error.die(this, transformation.getAttribute("_line_number_"), "Transformation element does not have a valid name");
		productionList = pList;
		inputSymbolsInOrder = new LinkedList();
		newTerminalSymbols = new HashMap();

		// Check node name
		if (!transformation.getTagName().equals("Transformation")) 
			Error.die(this, transformation.getAttribute("_line_number_"), "Expecting a <Transformation> element");

		// Get XML elements
		mainConstraintSetElement = getChildElement(transformation, "ConstraintSet");

		// Check that all the productions exist
		checkProductionsExist();

		// Read in the input, existential and output symbols (in that order)
		readInputSymbols();
		if (mainConstraintSetElement != null)
			readExistentialSymbols(mainConstraintSetElement, true, true);
		readOutputSymbols();

		/* No real reason for this error
		if (inputSymbols.isEmpty() && referableExistentialSymbols.isEmpty())
			Error.die(this,  transformation.getAttribute("_line_number_"), "Transformation does not have any input symbols or " +
				"referable existential symbols");
		*/

		// Parse all attribute strings
		parseAttributeStrings(ruleElement);
		checkSetAttributeSymbols();

		// Check all constraint attribute dependencies
		if (mainConstraintSetElement != null)
			checkVariables(mainConstraintSetElement, new HashMap());

		// Check that all attributes of new terminal symbols are initialized
		checkNewTerminalSymbols();

		// Check the validily of the symbols and attributes used to specify output symbols
		checkOutputsInOrder();

		// Check attribute dependencies for any solver constraints to be added
		checkSolverConstraintDependencies();

		// Move all referable symbol constraints to main constraint set
		relocateReferableSymbolConstraints();

		// Label all constraints
		if (mainConstraintSetElement != null) {
			labelConstraints(mainConstraintSetElement, "");
			labelConstraintExistsDependencies();
		}

		// Generate code for all the function references
		generateCodeForAllFunctions();
	}

	// Returns the production definition for a specified name, or null if it does not exist
	private ProductionDefinition getProduction(String name)
	{
		for (int i = 0; i < productionList.size(); i++) {
			ProductionDefinition p = (ProductionDefinition) productionList.get(i);
			if (p.getName().equals(name)) 
				return p;
		}
		return null;
	}

	// Checks that all the productions listed do exist
	private void checkProductionsExist()
	{
		NodeList nodes = ruleElement.getElementsByTagName("ApplyProduction");
                for (int i = 0; i < nodes.getLength(); i++) {
			Element applyNode = (Element) nodes.item(i);

			if (getProduction(applyNode.getAttribute("name")) == null)
				Error.die(this, applyNode.getAttribute("_line_number_"), "Transformation refers to a production rule "
					+ "which is not defined: " + applyNode.getAttribute("name"));
		}
	}

	// Reads in the input symbols
	private void readInputSymbols()
	{
		// Get input symbols
		Element inputs = getChildElement(ruleElement, "InputSymbols");
		if (inputs == null)
			Error.die(this, inputs.getAttribute("_line_number_"),
				"Expecting <Transformation> element to contain an <InputSymbols> element");

		// Read in input symbols and symbol sets
                NodeList children = inputs.getChildNodes();
                for (int i = 0; i < children.getLength(); i++) {
                        Element element = (Element) children.item(i);

			// Check node name
			if (element.getTagName().equals("Symbol")) {

				// Check for symbol name duplication
				if (inputSymbols.containsKey(element.getAttribute("name")))
					Error.die(this, element.getAttribute("_line_number_"), 
						"Transformation contains a symbol name duplication: " + element.getAttribute("name"));
				
				// Check for valid symbol type
				SymbolDefinition sDef = getSymbolDefinition(element.getAttribute("type"));
				if (sDef == null)
					Error.die(this, element.getAttribute("_line_number_"),
						"Transformation refers to undefined symbol type: " + element.getAttribute("type"));

				inputSymbols.put(element.getAttribute("name"), sDef);
				inputSymbolsInOrder.addLast(element.getAttribute("name"));
			}
			else 
				Error.die(this, element.getAttribute("_line_number_"),
					"Expecting <InputSymbols> in a transformation to contain only <Symbol> elements");
                }
	}

	// Recursively looks through a DOM subtree for nodes with a Symbol tag
	private void readExistentialSymbols(Node node, boolean referable, boolean positiveDependency)
	{
		NodeList children = node.getChildNodes();
		for (int i = 0; i < children.getLength(); i++) {
			Node child = children.item(i);
			if (child.getNodeName().equals("Symbol")) {
				Element element = (Element) child;

				// Check for symbol name duplication
				if (inputSymbols.containsKey(element.getAttribute("name")) || 
					existentialSymbols.containsKey(element.getAttribute("name")))
					Error.die(this, element.getAttribute("_line_number_"),
						"Transformation contains a symbol name duplication: " + element.getAttribute("name"));

				// Check for valid symbol type
				SymbolDefinition sDef = getSymbolDefinition(element.getAttribute("type"));
				if (sDef == null)
					Error.die(this, element.getAttribute("_line_number_"),
						"Transformation refers to undefined symbol type: " + element.getAttribute("type"));

				existentialSymbols.put(element.getAttribute("name"), sDef);
				if (referable)
					referableExistentialSymbols.put(element.getAttribute("name"), sDef);
				else if (positiveDependency)
					positiveExistentialSymbols.put(element.getAttribute("name"), sDef);
				else
					negativeExistentialSymbols.put(element.getAttribute("name"), sDef);
			}

			// Recursive call, adjusting the dependency type if needed
			if (child.getNodeName().equals("NotExists"))
				readExistentialSymbols(child, false, !positiveDependency);
			else
				readExistentialSymbols(child, referable, positiveDependency);
		}
	}

	// Read and check the names of output symbols defined by <NewTerminalSymbol> and <GetSymbol> tags
	private void readOutputSymbols()
	{
		// Check all new terminal symbols
		NodeList nodes = ruleElement.getElementsByTagName("NewTerminalSymbol");
		for (int i = 0; i < nodes.getLength(); i++) {
			Element element = (Element) nodes.item(i);

			// Check for symbol name duplication
			if (inputSymbols.containsKey(element.getAttribute("name")) || 
				existentialSymbols.containsKey(element.getAttribute("name")) ||
				newTerminalSymbols.containsKey(element.getAttribute("name")))
				Error.die(this, element.getAttribute("_line_number_"),
					"Transformation contains a symbol name duplication: " + element.getAttribute("name"));

			// Check for valid symbol type
			SymbolDefinition sDef = getSymbolDefinition(element.getAttribute("type"));
			if (sDef == null)
				Error.die(this, element.getAttribute("_line_number_"),
					"Transformation refers to undefined symbol type: " + element.getAttribute("type"));

			// Make sure symbol type is a terminal symbol
			if (!sDef.isTerminal()) 
				Error.die(this, element.getAttribute("_line_number_"),
					"Transformation tries to create a new terminal symbol that is not terminal: " + element.getAttribute("type"));

			newTerminalSymbols.put(element.getAttribute("name"), sDef);
		}

		// Check all output symbols produced by productions
		nodes = ruleElement.getElementsByTagName("GetSymbol");
		for (int i = 0; i < nodes.getLength(); i++) {
			Element element = (Element) nodes.item(i);
			Element applyNode = (Element) element.getParentNode();

			// Check that the production contains an output symbol of specified name
			ProductionDefinition pDef = getProduction(applyNode.getAttribute("name"));
			if (!pDef.getOutputSymbols().containsKey(element.getAttribute("name")))
				Error.die(this, element.getAttribute("_line_number_"), "<ApplyProduction> refers to an output symbol " +
					"whose name is not defined in the production: " + element.getAttribute("name"));

			// Check for symbol name duplication
			if (inputSymbols.containsKey(element.getAttribute("label")) || 
				existentialSymbols.containsKey(element.getAttribute("label")) ||
				newTerminalSymbols.containsKey(element.getAttribute("label")) ||
				outputSymbols.containsKey(element.getAttribute("label")))
				Error.die(this, element.getAttribute("_line_number_"),
					"Transformation contains a symbol name duplication: " + element.getAttribute("label"));

			outputSymbols.put(element.getAttribute("label"), pDef.getOutputSymbols().get(element.getAttribute("name")));
		}
	}

	// Check that all <SetAttribute> tags only apply to output symbols, referable existential symbols and new terminal symbols 
	private void checkSetAttributeSymbols()
	{		
		Element parentElement = getChildElement(ruleElement, "ApplyActions");
		NodeList nodes = parentElement.getElementsByTagName("SetAttribute");
		for (int i = 0; i < nodes.getLength(); i++) {
			Element element = (Element) nodes.item(i);
		
			if (!(newTerminalSymbols.containsKey(element.getAttribute("symbol_name")) ||
				outputSymbols.containsKey(element.getAttribute("symbol_name")) ||
				referableExistentialSymbols.containsKey(element.getAttribute("symbol_name"))))
				Error.die(this, element.getAttribute("_line_number_"), "<SetAttribute> symbol name not " +
					"an valid setable symbol for this transformation: " + element.getAttribute("symbol_name"));
		}
	}

	// Parses all string of the form [symbol_name].[attribute_name]
	private void parseAttributeStrings(Element element)
	{
		NodeList children = element.getChildNodes();
		for (int i = 0; i < children.getLength(); i++) {

			// Ignore non-element nodes
			if (children.item(i).getNodeType() != Node.ELEMENT_NODE)
				continue;
			Element child = (Element) children.item(i);

			// See if it is a SymbolValue element
			if (child.getTagName().equals("SymbolValue")) {

				// Check symbol of that symbol name exists
				String symbolName = child.getAttribute("name");
				SymbolDefinition sDef = (SymbolDefinition) inputSymbols.get(symbolName);
				if (sDef == null)
					sDef = (SymbolDefinition) outputSymbols.get(symbolName);
				if (sDef == null)
					sDef = (SymbolDefinition) existentialSymbols.get(symbolName);
				if (sDef == null)
					Error.die(this, child.getAttribute("_line_number_"),
						"Symbol name not defined for this production: " + symbolName);
			}

			// See if it is an SetAttribute, AttributeValue or Variable element
			if (child.getTagName().equals("SetAttribute") || child.getTagName().equals("AttributeValue") || 
				child.getTagName().equals("Variable")) {

				// Split name string from [symbol_name].[attribute_name] format
				String name = child.getAttribute("name");
				int dotIndex = name.indexOf('.');
				if (dotIndex == -1)
					Error.die(this, child.getAttribute("_line_number_"),
						"Invalid attribute reference format: dot missing in: " + name);
				String symbolName = name.substring(0, dotIndex);
				String attName = name.substring(dotIndex + 1);

				// Add name components to node
				child.setAttribute("symbol_name", symbolName);
				child.setAttribute("attribute_name", attName);

				// Check symbol of that symbol name exists
				SymbolDefinition sDef = (SymbolDefinition) inputSymbols.get(symbolName);
				if (sDef == null)
					sDef = (SymbolDefinition) outputSymbols.get(symbolName);
				if (sDef == null)
					sDef = (SymbolDefinition) newTerminalSymbols.get(symbolName);
				if (sDef == null)
					sDef = (SymbolDefinition) existentialSymbols.get(symbolName);
				if (sDef == null)
					Error.die(this, child.getAttribute("_line_number_"),
						"Symbol name not defined for this transformation: " + symbolName);

				// Check that symbol has specified attribute
				Attribute attr = sDef.getAttribute(attName);
				if (attr == null)
					Error.die(this, child.getAttribute("_line_number_"),
						"Attribute \'" + attName + "\' not defined for symbol \'" + symbolName + "\'");

				// Check that Variable nodes only contain constraint variable attributes
				if (child.getTagName().equals("Variable")) {
					if (!attr.getDataType().equals("ConstraintVariable"))
						Error.die(this, child.getAttribute("_line_number_"),
							"<Variable> element refers to an attribute which is not a constraint variable");
				}

				// Add symbol type and attribute datatype to node 
				child.setAttribute("symbol_type", sDef.getName());
				child.setAttribute("attribute_datatype", attr.getDataType());
			}
			parseAttributeStrings(child);
		}
	}

	// Recursively validates all variables in a DOM subtree
	private void checkVariables(Element element, Map validExistentialSymbols)
	{
		NodeList children = element.getChildNodes();
		for (int i = 0; i < children.getLength(); i++) {

			// Ignore non-element nodes
			if (children.item(i).getNodeType() != Node.ELEMENT_NODE)
				continue;
			Element child = (Element) children.item(i);

			// If it is an exists node, add to valid existential symbols (no need to recheck it)
			if (child.getTagName().equals("Exists") || child.getTagName().equals("NotExists")) {
				Element symbol = getChildElement(child, "Symbol");
				SymbolDefinition sDef = getSymbolDefinition(symbol.getAttribute("type"));
				validExistentialSymbols.put(symbol.getAttribute("name"), sDef);
			}

			// See if it is an AttributeValue or Variable element
			if (child.getTagName().equals("AttributeValue") || child.getTagName().equals("Variable")) {

				// Check symbol of a valid type
				String symbolName = child.getAttribute("symbol_name");
				if (!(inputSymbols.containsKey(symbolName) || validExistentialSymbols.containsKey(symbolName)))
					Error.die(this, child.getAttribute("_line_number_"),
						"Invalid symbol name for this part of the transformation: " + symbolName);
			}
			checkVariables(child, validExistentialSymbols);

			// Remove existential symbol if exiting exists node subtree
			if (child.getTagName().equals("Exists") || child.getTagName().equals("NotExists")) {
				Element symbol = getChildElement(child, "Symbol");
				validExistentialSymbols.remove(symbol.getAttribute("name"));
			}
		}
	}

	// Check that all attributes of new terminal symbols are initialized
	private void checkNewTerminalSymbols()
	{
		NodeList nodes = ruleElement.getElementsByTagName("NewTerminalSymbol");
		for (int i = 0; i < nodes.getLength(); i++) {
			Element element = (Element) nodes.item(i);
			String symbolName = element.getAttribute("name");
			SymbolDefinition sDef = (SymbolDefinition) newTerminalSymbols.get(symbolName);

			// Compare the number of child elements
			NodeList children = element.getChildNodes();
			if (children.getLength() != sDef.getAllAttributes().size())
				Error.die(this, element.getAttribute("_line_number_"), "Expected " + sDef.getAllAttributes().size() + 
					" attributes to be specified for New Terminal Symbol: " + symbolName);
		}
	}

	// This method checks the validity of symbol and attributes assignments
	// in the <ApplyActions> tag (if there is one), using the order of output operations
	private void checkOutputsInOrder()
	{
		Element osElement = getChildElement(ruleElement, "ApplyActions");
		if (osElement == null)
			return;

		Map validSymbols = new HashMap(referableExistentialSymbols);

		NodeList nodes = osElement.getChildNodes();
		for (int i = 0; i < nodes.getLength(); i++) {
			Element element = (Element) nodes.item(i);

			// Check variables used by element
			checkVariables(element, validSymbols);

			// Add any new valid symbols that may have been added
			if (element.getTagName().equals("NewTerminalSymbol")) {
				String symbolName = element.getAttribute("name");
				validSymbols.put(symbolName, newTerminalSymbols.get(symbolName));
			}
			else if (element.getTagName().equals("ApplyProduction")) {

				// Get production definition
				ProductionDefinition pDef = getProduction(element.getAttribute("name"));

				// Check elements relating to production 
				NodeList children = element.getChildNodes();
				for (int j = 0; j < children.getLength(); j++) {
					Element child = (Element) children.item(j);

					if (child.getTagName().equals("SetSymbol")) {

						// Check production contains a symbol of the specified name
						if (!(pDef.getInputSymbols().containsKey(child.getAttribute("name")) ||
							pDef.getReferableExistentialSymbols().containsKey(child.getAttribute("name"))))
							Error.die(this, child.getAttribute("_line_number_"), "<SetSymbol> refers to an " +
								"input or existential symbol whose name is not defined in the production: " + 
								child.getAttribute("name"));

						// Check type of symbol to be used is correct
						SymbolDefinition sDef = (SymbolDefinition) pDef.getInputSymbols().get(child.getAttribute("name"));
						if (sDef == null)
							sDef = (SymbolDefinition) pDef.getReferableExistentialSymbols().get(child.getAttribute("name"));
						if (!symbolNameMatchesSymbolType(child.getAttribute("use"), sDef.getName(), validSymbols))
							Error.die(this, child.getAttribute("_line_number_"), "<SetSymbol> tries to use " +
								"a symbol that is not of the correct type: " + child.getAttribute("use"));
					}
					else if (child.getTagName().equals("GetSymbol")) {
						validSymbols.put(child.getAttribute("label"), 
							pDef.getOutputSymbols().get(child.getAttribute("name")));
					}
				}
			}
		}
	}

	// Returns true iff the symbol of the specified name is of the specified type
	private boolean symbolNameMatchesSymbolType(String symbolName, String symbolType, Map validSymbols)
	{
		SymbolDefinition sDef = (SymbolDefinition) inputSymbols.get(symbolName);
		if (sDef == null)
			sDef = (SymbolDefinition) validSymbols.get(symbolName);
		if (sDef == null)
			return false;

		if (sDef.getName().equals(symbolType))
			return true;
		while (!sDef.getExtendsName().equals("")) {
			sDef = getSymbolDefinition(sDef.getExtendsName());
			if (sDef.getName().equals(symbolType))
				return true;
		}
		return false;
	}

	// Checks that all solver constraints to be added (if any) have valid attribute dependencies
	private void checkSolverConstraintDependencies()
	{
		Element ascElement = getChildElement(ruleElement, "AddSolverConstraints");
		if (ascElement == null)
			return;

		NodeList nodes = ascElement.getElementsByTagName("Variable");
		for (int i = 0; i < nodes.getLength(); i++) {
			Element element = (Element) nodes.item(i);

			// Check symbol of a valid type
			String symbolName = element.getAttribute("symbol_name");
			if (!(outputSymbols.containsKey(symbolName) || referableExistentialSymbols.containsKey(symbolName) ||
				newTerminalSymbols.containsKey(symbolName)))
				Error.die(this, element.getAttribute("_line_number_"),
					"Invalid symbol name for this part of the transformation: " + symbolName);
		}
	}

	// Produces a tranformation record class for this transformation
	public void createTransformationRecordClass()
	{
		int num, num2;
		Iterator iter;
		NodeList nodes;
		String constructor = "";
		String line;

		// Open file for the class
		PrintWriter output = Compiler.openTextFile(Compiler.PACKAGE_DIRECTORY + "TransformationRecord" + number + ".java");

		// Print class header details (package scope)
		output.println("");
		output.println(Compiler.PACKAGE_LINE);
		output.println("");
		output.println("import au.edu.monash.csse.tonyj.cider.constraints.Constraint;");
		output.println("");
		output.println("class TransformationRecord" + number + " implements TransformationRecord {");
		output.println("");

		// Print input symbol variables
		iter = inputSymbols.keySet().iterator();
		while (iter.hasNext()) {
			String name = (String) iter.next();
			SymbolDefinition sDef = (SymbolDefinition) inputSymbols.get(name);
			output.println("\tprivate " + sDef.getName() + " _" + name + ";");
			constructor += "\t\t_" + name + " = null;\n";
		}

		// Print existential symbol variables
		iter = existentialSymbols.keySet().iterator();
		while (iter.hasNext()) {
			String name = (String) iter.next();
			SymbolDefinition sDef = (SymbolDefinition) existentialSymbols.get(name);
			output.println("\tprivate " + sDef.getName() + " _" + name + ";");
			constructor += "\t\t_" + name + " = null;\n";
		}

		// Print new terminal symbol variables
		iter = newTerminalSymbols.keySet().iterator();
		while (iter.hasNext()) {
			String name = (String) iter.next();
			SymbolDefinition sDef = (SymbolDefinition) newTerminalSymbols.get(name);
			output.println("\tprivate " + sDef.getName() + " _" + name + ";");
			constructor += "\t\t_" + name + " = null;\n";
		}

		// Print output symbol variables
		iter = outputSymbols.keySet().iterator();
		while (iter.hasNext()) {
			String name = (String) iter.next();
			SymbolDefinition sDef = (SymbolDefinition) outputSymbols.get(name);
			output.println("\tprivate " + sDef.getName() + " _" + name + ";");
			constructor += "\t\t_" + name + " = null;\n";
		}

		// Print solver constraint variables for main constraint set
		if (mainConstraintSetElement != null) {
			nodes = mainConstraintSetElement.getElementsByTagName("SolverConstraint");
			for (int i = 0; i < nodes.getLength(); i++) {
				Element element = (Element) nodes.item(i);
				output.println("\tprivate Constraint constraint" + element.getAttribute("label") + ";");
			}
		}

		// Print solver constraint variables for add solver constraints
		Element ascElement = getChildElement(ruleElement, "AddSolverConstraints");
		if (ascElement != null) {
			nodes = ascElement.getElementsByTagName("SolverConstraint");
			for (int i = 0; i < nodes.getLength(); i++) {
				Element element = (Element) nodes.item(i);
				output.println("\tprivate Constraint addConstraint" + (i+1) + ";");
			}
		}

		// Print constructor
		output.println("");
		output.println("\tpublic TransformationRecord" + number + "()");
		output.println("\t{");
		output.print(constructor);
		output.println("\t}");

		// Print input symbol get and set methods
		iter = inputSymbols.keySet().iterator();
		while (iter.hasNext()) {
			String name = (String) iter.next();
			SymbolDefinition sDef = (SymbolDefinition) inputSymbols.get(name);
			output.println("");
			output.println("\tpublic " + sDef.getName() + " get_" + name + "()");
			output.println("\t{");
			output.println("\t\treturn _" + name + ";");
			output.println("\t}");
			output.println("");
			output.println("\tpublic void set_" + name + "(" + sDef.getName() + " value)");
			output.println("\t{");
			output.println("\t\t_" + name + " = value;");
			output.println("\t}");
		}

		// Print existential symbol get and set methods
		iter = existentialSymbols.keySet().iterator();
		while (iter.hasNext()) {
			String name = (String) iter.next();
			SymbolDefinition sDef = (SymbolDefinition) existentialSymbols.get(name);
			output.println("");
			output.println("\tpublic " + sDef.getName() + " get_" + name + "()");
			output.println("\t{");
			output.println("\t\treturn _" + name + ";");
			output.println("\t}");
			output.println("");
			output.println("\tpublic void set_" + name + "(" + sDef.getName() + " value)");
			output.println("\t{");
			output.println("\t\t_" + name + " = value;");
			output.println("\t}");
		}

		// Print new terminal symbol get and set methods
		iter = newTerminalSymbols.keySet().iterator();
		while (iter.hasNext()) {
			String name = (String) iter.next();
			SymbolDefinition sDef = (SymbolDefinition) newTerminalSymbols.get(name);
			output.println("");
			output.println("\tpublic " + sDef.getName() + " get_" + name + "()");
			output.println("\t{");
			output.println("\t\treturn _" + name + ";");
			output.println("\t}");
			output.println("");
			output.println("\tpublic void set_" + name + "(" + sDef.getName() + " value)");
			output.println("\t{");
			output.println("\t\t_" + name + " = value;");
			output.println("\t}");
		}

		// Print output symbol get and set methods
		iter = outputSymbols.keySet().iterator();
		while (iter.hasNext()) {
			String name = (String) iter.next();
			SymbolDefinition sDef = (SymbolDefinition) outputSymbols.get(name);
			output.println("");
			output.println("\tpublic " + sDef.getName() + " get_" + name + "()");
			output.println("\t{");
			output.println("\t\treturn _" + name + ";");
			output.println("\t}");
			output.println("");
			output.println("\tpublic void set_" + name + "(" + sDef.getName() + " value)");
			output.println("\t{");
			output.println("\t\t_" + name + " = value;");
			output.println("\t}");
		}

		// Print solver constraint get and set methods
		if (mainConstraintSetElement != null) {
			nodes = mainConstraintSetElement.getElementsByTagName("SolverConstraint");
			for (int i = 0; i < nodes.getLength(); i++) {
				Element element = (Element) nodes.item(i);
				output.println("");
				output.println("\tpublic Constraint getConstraint" + element.getAttribute("label") + "()");
				output.println("\t{");
				output.println("\t\treturn constraint" + element.getAttribute("label") + ";");
				output.println("\t}");
				output.println("");
				output.println("\tpublic void setConstraint" + element.getAttribute("label") + "(Constraint c)");
				output.println("\t{");
				output.println("\t\tconstraint" + element.getAttribute("label") + " = c;");
				output.println("\t}");
			}
		}
		if (ascElement != null) {
			nodes = ascElement.getElementsByTagName("SolverConstraint");
			for (int i = 0; i < nodes.getLength(); i++) {
				Element element = (Element) nodes.item(i);
				output.println("");
				output.println("\tpublic Constraint getAddConstraint" + (i+1) + "()");
				output.println("\t{");
				output.println("\t\treturn addConstraint" + (i+1) + ";");
				output.println("\t}");
				output.println("");
				output.println("\tpublic void setAddConstraint" + (i+1) + "(Constraint c)");
				output.println("\t{");
				output.println("\t\taddConstraint" + (i+1) + " = c;");
				output.println("\t}");
			}
		}

		// Print add constraints method
		output.println("");
		output.println("\tpublic boolean addSolverConstraints(Interpreter interpreter)");
		output.println("\t{");
		if (ascElement != null) {
			nodes = ascElement.getElementsByTagName("SolverConstraint");
			for (int i = 0; i < nodes.getLength(); i++) {
				output.println("\t\tif (!interpreter.addConstraint(addConstraint" + (i+1) + "))");
				output.println("\t\t\treturn false;");
			}
			if (nodes.getLength() > 0)
				output.println("\t\tinterpreter.solve()");
		}
		output.println("\t\treturn true;");
		output.println("\t}");

		// Print remove constraints method
		output.println("");
		output.println("\tpublic void removeSolverConstraints(Interpreter interpreter)");
		output.println("\t{");
		if (ascElement != null) {
			nodes = ascElement.getElementsByTagName("SolverConstraint");
			for (int i = 0; i < nodes.getLength(); i++) 
				output.println("\t\tinterpreter.removeConstraint(addConstraint" + (i+1) + ");");
			if (nodes.getLength() > 0)
				output.println("\t\tinterpreter.solve()");
		}
		output.println("\t}");

		// End class and close file
		output.println("}");
		output.println("");
		output.close();
	}

	// Produces a transformation rule class for this transformation 
	public void createTransformationRuleClass(DependencyGraph dependencyGraph)
	{
		String line;
		Iterator iter;
		NodeList nodes;

		// Open file for the class
		PrintWriter output = Compiler.openTextFile(Compiler.PACKAGE_DIRECTORY + "TransformationRule" + number + ".java");

		// Print class header details (package scope)
		output.println("");
		output.println(Compiler.PACKAGE_LINE);
		output.println("");
		output.println("import au.edu.monash.csse.tonyj.cider.constraints.ConstraintSolver;");
		output.println("import au.edu.monash.csse.tonyj.cider.constraints.ConstraintExpression;");
		output.println("import au.edu.monash.csse.tonyj.cider.constraints.Constraint;");
		output.println("import au.edu.monash.csse.tonyj.cider.constraints.ConstraintVariable;");
		output.println("import java.util.Vector;");
		output.println("import java.util.LinkedList;");
		output.println("import java.util.List;");
		output.println("import java.util.Map;");
		output.println("import java.util.Set;");
		output.println("import java.util.HashSet;");
		output.println("import java.util.Iterator;");
		output.println("");
		output.println("class TransformationRule" + number + " implements BasicTransformationRule {");
		output.println("");
		output.println("\tprivate ParseForest forest;");
		output.println("\tprivate ConstraintSolver csolver;");

		Map inputAndReferableSymbols = new HashMap(inputSymbols);
		inputAndReferableSymbols.putAll(referableExistentialSymbols);
		Set inputAndReferableTypes = new HashSet(inputAndReferableSymbols.values());

		// Print out lists and indexes needed for candidate nodes
		iter = inputAndReferableSymbols.keySet().iterator();
		while (iter.hasNext()) {
			String name = (String) iter.next();
			output.println("\tprivate List _" + name + "_list;"); 
			output.println("\tprivate int _" + name + "_index;"); 
		}
		output.println("");

		// Print simple constructor (with no candidate sets for input or referable existential symbols)
		output.println("\tpublic TransformationRule" + number + "(ParseForest f, ConstraintSolver cs)");
		output.println("\t{");
		output.println("\t\tforest = f;");
		output.println("\t\tcsolver = cs;");

		// Create lists for each symbol type (input and referable existential symbols)
		iter = inputAndReferableTypes.iterator();
		while (iter.hasNext()) {
			SymbolDefinition sDef = (SymbolDefinition) iter.next();
			output.println("\t\tList " + sDef.getName() + "_all_nodes_list = new Vector(((" + grammarName +
				"SymbolNodeSets) forest.getAllSymbolNodes()).get_" + sDef.getName() + "_set());");
		}

		iter = inputAndReferableSymbols.keySet().iterator();
		while (iter.hasNext()) {
			String name = (String) iter.next();
			SymbolDefinition sDef = (SymbolDefinition) inputAndReferableSymbols.get(name);
			output.println("\t\t_" + name + "_list = " + sDef.getName() + "_all_nodes_list;");
			output.println("\t\t_" + name + "_index = 0;"); 
		}
		output.println("\t}");
		output.println("");

		// Print constructor with optional candidate sets for input or referable existential symbols (if appropriate)
		// Optional sets are provided via a mapping of symbol name mapped with a Set of appropriate symbols
		output.println("\tpublic TransformationRule" + number + "(ParseForest f, ConstraintSolver cs, Map candidatesMap)");
		output.println("\t\tthrows IllegalArgumentException");
		output.println("\t{");
		output.println("\t\tthis(f, cs);");
		iter = inputAndReferableSymbols.keySet().iterator();
		while (iter.hasNext()) {
			String name = (String) iter.next();
			output.println("\t\tif (candidatesMap.containsKey(\"" + name + "\")) {");
			output.println("\t\t\tObject obj = candidatesMap.get(\"" + name + "\");");
			output.println("\t\t\tif (!(obj instanceof Set))");
			output.println("\t\t\t\tthrow new IllegalArgumentException(\"Map entry for \\\"" + name + "\\\" not a Set object\");");
			output.println("\t\t\t_" + name + "_list = new Vector((Set) obj);");
			output.println("\t\t}");
		}
		output.println("\t}");
		output.println("");

		// Create evaluate method
		output.println("\tpublic TransformationRecord evaluate()");
		output.println("\t{");
		output.println("\t\tSet usedSymbols = new HashSet();");
		output.println("\t\tTransformationRecord" + number + " record = new TransformationRecord" + number + "();");
		output.println("");

		// Fill symbols
		String tabs = "\t";
		String close = "";
		String lastIndex = "";
		int count = 1;
		List usedSymbolTypes = new LinkedList();
		iter = inputAndReferableSymbols.keySet().iterator();
		while (iter.hasNext()) {
			String name = (String) iter.next();
			SymbolDefinition sDef = (SymbolDefinition) inputAndReferableSymbols.get(name);
			tabs = "\t";
			lastIndex = "_" + name + "_index";
			for (int i = 0; i < count; i++)
				tabs += "\t";
			output.println(tabs + "for (; _" + name + "_index < _" + name + "_list.size(); _" + name + "_index++) {");
			output.println(tabs + "\t" + sDef.getName() + " s" + count + " = (" + sDef.getName() + ") ((SymbolNode) _" + name + 
				"_list.get(_" + name + "_index)).getSymbol();");

			// Check that there are no subtree issues with the symbol and already used symbols
			for (int i = 0; i < usedSymbolTypes.size(); i++) {
				SymbolDefinition usedType = (SymbolDefinition) usedSymbolTypes.get(i);
				if (dependencyGraph.hasCreationRelationship(usedType, sDef)) {
					Error.warn(this, "Transformation " + ruleName + ": symbol \"" + name + 
						"\" could be in the subtree of another input or existential symbol");
					// output.println(tabs + "\tif (forest.inSubTree(s" + count + ", s" + (i+1) + "))");
					// output.println(tabs + "\t\tcontinue;");
				}

				// Check reverse direction with other symbols
				if (dependencyGraph.hasCreationRelationship(sDef, usedType)) {
					Error.warn(this, "Transformation " + ruleName + ": symbol \"" + name + 
						"\" could contain another existential or input symbol in its subtree");
					// output.println(tabs + "\tif (forest.inSubTree(s" + (i+1) + ", s" + count + "))");
					// output.println(tabs + "\t\tcontinue;");
				}
			}
			output.println(tabs + "\tif (!usedSymbols.add(s" + count + "))");
			output.println(tabs + "\t\tcontinue;");
			output.println(tabs + "\trecord.set_" + name + "(s" + count + ");");
			output.println("");
			
			close = tabs + "\tusedSymbols.remove(s" + count + ");\n" + tabs + "\trecord.set_" + name + "(null);\n" + 
				tabs + "}\n" + tabs + "_" + name + "_index = 0;\n" + close;
			count++;
			usedSymbolTypes.add(sDef);
		}

		// Add solver constraints that do not depend on any non-referable symbols
		if (mainConstraintSetElement != null) {
			nodes = mainConstraintSetElement.getElementsByTagName("SolverConstraint");
			for (int i = 0; i < nodes.getLength(); i++) {
				Element constraint = (Element) nodes.item(i);
				if (constraint.getAttribute("dependsOn").equals("")) {

					// Create the constraint
					printConstraintVariableArray(constraint, tabs + "\t", output);
					String contents = Compiler.getTextContents(constraint);

					// See if constraint compiler can be used
					if (constraintCompiler == null) {
						output.println(tabs + "\tConstraint c" +  constraint.getAttribute("label") + " = new " + 
							constraintClass + "();");
						output.println(tabs + "\tc" + constraint.getAttribute("label") + ".initialize(\"" + contents + 
							"\", cvArray" + constraint.getAttribute("label") + ");");
					}
					else {
						String className = "Constraint_T" + number + "__" + constraint.getAttribute("label");
						constraintClasses.put(className, contents);
						output.println(tabs + "\tConstraint c" +  constraint.getAttribute("label") + " = (Constraint) new " + 
							className + "(cvArray" + constraint.getAttribute("label") + ");");
					}
					output.println(tabs + "\trecord.setConstraint" + constraint.getAttribute("label") + "(c" + 
						constraint.getAttribute("label") + ");");
				}
			}
			output.println("");
		}

		// Check symbols against constraints
		if (mainConstraintSetElement != null) {
			nodes = mainConstraintSetElement.getChildNodes();
			if (nodes.getLength() == 0) 
				output.println(tabs + "\tif (true) {");
			else {
				output.print(tabs + "\tif (");
				for (int i = 0; i < nodes.getLength(); i++) {
					Element child = (Element) nodes.item(i);
					if (i > 0) 
						output.print(" &&\n" + tabs + "\t\t");
					output.print("evaluateConstraint" + child.getAttribute("label") + "(record)");
				}
				output.println(") {");
			}
		}
		else 
			output.println(tabs + "\tif (true) {");

		// If evaluation successful, return the transformation record
		output.println("");
		output.println(tabs + "\t\tLog.getLogger().info(\"Transformation Rule " + number + " (" +
			ruleName + ") successfully evaluated\");");
		if (!lastIndex.equals(""))
			output.println(tabs + "\t\t" + lastIndex + "++;");
		output.println(tabs + "\t\treturn record;");
		output.println(tabs + "\t}");
		output.print(close);
		output.println("\t\treturn null;");
		output.println("\t}");
		output.println("");

		// Print the apply transformation method, which receives a record that has been successfully evaluated
		output.println("\tpublic boolean apply(TransformationRecord rec)");
		output.println("\t{");
		output.println("\t\tTransformationRecord" + number + " record = (TransformationRecord" + number + ") rec;");

		// Create undo point
		output.println("\t\tboolean tempACS;");
		output.println("\t\tboolean tempUE = forest.getInterpreter().isUndoEnabled();");
		output.println("\t\tforest.getInterpreter().enableUndo();");
		output.println("\t\tint undoIndex = forest.getInterpreter().getUndoIndex();");
		output.println("\t\tboolean success = true;");
		output.println("");

		// Remove input symbols from the forest 
		Element parentElement = getChildElement(ruleElement, "ApplyActions");
		iter = inputSymbolsInOrder.iterator();
		while (iter.hasNext()) {
			String name = (String) iter.next();
			output.println("\t\tforest.removeSubTree(record.get_" + name + "());");
		}

		nodes = parentElement.getChildNodes();

		// Define blocks of SetAttribute elements that contain constraint variables
		boolean inBlock = false;
		boolean blockHasCV = false;
		Element blockStartElement = null;

		for (int i = 0; i < nodes.getLength(); i++) {
			Element element = (Element) nodes.item(i);

			if (element.getTagName().equals("SetAttribute")) {

				// Check for block start
				if (!inBlock) {
					inBlock = true;
					blockHasCV = false;
					blockStartElement = element;
				}

				// Check if it is a constraint variable
				SymbolDefinition sDef = getSymbolDefinition(element.getAttribute("symbol_type"));
				Attribute attr = sDef.getAttribute(element.getAttribute("attribute_name"));
				if (attr.isConstraintVariable()) 
					blockHasCV = true;

				// Check for block end
				if (i < nodes.getLength() - 1) {
					Element nextElement = (Element) nodes.item(i+1);
					if (!nextElement.getTagName().equals("SetAttribute")) {
						if (blockHasCV) {
							blockStartElement.setAttribute("isBlockStart", "true");
							element.setAttribute("isBlockEnd", "true");
						}
						inBlock = false;
					}
				}
				else {
					if (blockHasCV) {
						blockStartElement.setAttribute("isBlockStart", "true");
						element.setAttribute("isBlockEnd", "true");
					}
					inBlock = false;
				}
			}
		}

		// Deal with each possible action
		for (int i = 0; i < nodes.getLength(); i++) {
			Element element = (Element) nodes.item(i);
		
			output.println("\t\tif (success) {");
			if (element.getTagName().equals("NewTerminalSymbol")) {

				String name = element.getAttribute("name");
				SymbolDefinition sDef = (SymbolDefinition) newTerminalSymbols.get(name);
				if (sDef.getDrawableClass().equals(""))
					line = "\t\t\t" + sDef.getName() + " _" + name + " = new " + sDef.getName() + "(";
				else
					line = "\t\t\t" + sDef.getName() + " _" + name + " = new " + sDef.getDrawableClass() + "(";

				// Go through attributes
				int numArgs = 0;
				List atts = sDef.getAllAttributes();
				Iterator iter2 = atts.iterator();
				while (iter2.hasNext()) {
					Attribute attr = (Attribute) iter2.next();
					if (numArgs > 0)
						line += ",\n\t\t\t";
					line += "initial_" + name + "_" + attr.getName() + "(record)";
					numArgs++;
				}
				line += ");";
				output.println(line);
				output.println("\t\t\trecord.set_" + name + "(_" + name + ");");
				output.println("\t\t\tsuccess = forest.addTerminalSymbol(_" + name + ");");
			}
			else if (element.getTagName().equals("ApplyProduction")) {

				// Set all candidate symbols that need setting
				Set symbolNames = new HashSet();
				NodeList children = element.getChildNodes();
				for (int j = 0; j < children.getLength(); j++) {
					Element child = (Element) children.item(j);

					if (child.getTagName().equals("SetSymbol")) {
						if (!symbolNames.contains(child.getAttribute("name"))) {
							output.println("\t\t\tSet p" + i + "_" + child.getAttribute("name") + 
								"_candidates = new HashSet();");
							symbolNames.add(child.getAttribute("name"));
						}
						output.println("\t\t\tp" + i + "_" + child.getAttribute("name") + 
							"_candidates.add(record.get_" + child.getAttribute("use") + "().getSymbolNode());");
					}
				}
				ProductionDefinition pDef = getProduction(element.getAttribute("name"));
				line = "\t\t\tProductionRecord" + pDef.getNumber() + " p" + i + "_record = ProductionRule" + pDef.getNumber() 
					+ ".evaluate(forest,\n" + "\t\t\t\tcsolver";
				iter = pDef.getInputSymbols().keySet().iterator();
				while (iter.hasNext()) {
					String name = (String) iter.next();
					if (symbolNames.contains(name))
						line += ", p" + i + "_" + name + "_candidates";
					else
						line += ", null";
				}
				iter = pDef.getReferableExistentialSymbols().keySet().iterator();
				while (iter.hasNext()) {
					String name = (String) iter.next();
					if (symbolNames.contains(name))
						line += ", p" + i + "_" + name + "_candidates";
					else
						line += ", null";
				}
				line += ");";
				output.println(line);

				// Deal with cases where the production fails
				output.println("\t\t\tif (p" + i + "_record == null)");
				output.println("\t\t\t\tsuccess = false;");

				// Assign output symbol labels
				if (element.getElementsByTagName("GetSymbol").getLength() > 0) {
					output.println("\t\t\telse {");
					for (int j = 0; j < children.getLength(); j++) {
						Element child = (Element) children.item(j);
						if (child.getTagName().equals("GetSymbol")) 
							output.println("\t\t\t\trecord.set_" + child.getAttribute("label") + "(p" + i + 
								"_record.get_" + child.getAttribute("name") + "());");
					}
					output.println("\t\t\t}");
				}
			}
			else if (element.getTagName().equals("SetAttribute")) {

				// See if element is at start of a block that contains constraint variables
				if (element.getAttribute("isBlockStart").equals("true")) {
					output.println("\t\t\ttempACS = forest.getInterpreter().isAutoConstraintSolvingOn();");
					output.println("\t\t\tforest.getInterpreter().setAutoConstraintSolving(false);");
				}

				// Set the attribute
				output.println("\t\t\trecord.get_" + element.getAttribute("symbol_name") + "().set_" +
					element.getAttribute("attribute_name") + "(valueOf_" + element.getAttribute("symbol_name") + "_" +
					element.getAttribute("attribute_name") + "(record));");

				// See if element is at end of a block that contains constraint variables
				if (element.getAttribute("isBlockEnd").equals("true")) {
					output.println("\t\t\tforest.getInterpreter().setAutoConstraintSolving(tempACS);");
					output.println("\t\t\tforest.getInterpreter().resolve();");
				}
			}
			output.println("\t\t\tforest.getInterpreter().checkForest();");
			output.println("\t\t}");
		}

		// Create the constraints to be added to the interpreter
		Element ascElement = getChildElement(ruleElement, "AddSolverConstraints");
		if (ascElement != null) {
			nodes = ascElement.getElementsByTagName("SolverConstraint");
			if (nodes.getLength() > 0) 
				output.println("\t\tif (success) {");

			for (int i = 0; i < nodes.getLength(); i++) {
				Element constraint = (Element) nodes.item(i);
				if (constraint.getAttribute("dependsOn").equals("")) {

					// Create the constraint
					printConstraintVariableArray(constraint, "\t", output);
					String contents = Compiler.getTextContents(constraint);

					// See if constraint compiler can be used
					if (constraintCompiler == null) {
						output.println("\t\t\tConstraint c" +  constraint.getAttribute("label") + " = new " + 
							constraintClass + "();");
						output.println("\t\t\tc" + constraint.getAttribute("label") + ".initialize(\"" + contents + 
							"\", cvArray" + constraint.getAttribute("label") + ");");
					}
					else {
						String className = "Constraint_T" + number + "__" + constraint.getAttribute("label");
						constraintClasses.put(className, contents);
						output.println("\t\t\tConstraint c" +  constraint.getAttribute("label") + 
							" = (Constraint) new " + className + "(cvArray" + constraint.getAttribute("label") + ");");
					}
					output.println("\t\t\trecord.setConstraint" + constraint.getAttribute("label") + "(c" + 
						constraint.getAttribute("label") + ");");
				}
			}

			// Add constraints to the interpreter
			if (nodes.getLength() > 0) {

				// Deal with case where add constraints fails
				output.println("\t\t\tif (!record.addSolverConstraints(forest.getInterpreter()))");
				output.println("\t\t\t\tsuccess = false;");
				output.println("\t\t}");
			}
		}

		// Check if transformation was completely successful
		output.println("\t\tif (success) {");
		output.println("\t\t\tif (!tempUE)");
		output.println("\t\t\t\tforest.getInterpreter().disableUndo();");
		output.println("\t\t\tLog.getLogger().info(\"Transformation Rule " + number + " (" +
			ruleName + ") successfully applied\");");
		output.println("\t\t\treturn true;");
		output.println("\t\t}");

		// If not successful, undo actions
		output.println("\t\tforest.getInterpreter().undoActions(undoIndex);");
		output.println("\t\tif (!tempUE)");
		output.println("\t\t\tforest.getInterpreter().disableUndo();");
		output.println("\t\tLog.getLogger().info(\"Transformation Rule " + number + " (" +
			ruleName + ") could NOT be successfully applied\");");
		output.println("\t\treturn false;");
		output.println("\t}");
		output.println("");

		// Print out the method to apply the transformation normally
		output.println("\tpublic boolean normalApply()");
		output.println("\t{");
		output.println("\t\tTransformationRecord record = evaluate();");

		// If no input of referable existentials, only deal with one record, otherwise multiple records
		if (inputAndReferableSymbols.isEmpty())
			output.println("\t\treturn apply(record);");
		else {
			output.println("\t\twhile (record != null) {");
			output.println("\t\t\tif (apply(record))");
			output.println("\t\t\t\treturn true;");
			output.println("\t\t\trecord = evaluate();");
			output.println("\t\t}");
			output.println("\t\treturn false;");
		}
		output.println("\t}");
		output.println("");

		// Print out the method to apply the transformation in parallel
		output.println("\tpublic boolean parallelApply()");
		output.println("\t{");
		output.println("\t\tTransformationRecord record = evaluate();");

		// If no input of referable existentials, only deal with one record, otherwise multiple records
		if (inputAndReferableSymbols.isEmpty())
			output.println("\t\treturn apply(record);");
		else {
			output.println("\t\tboolean success = false;");
			output.println("\t\tList recordList = new LinkedList();");
			output.println("\t\twhile (record != null) {");
			output.println("\t\t\trecordList.add(record);");
			output.println("\t\t\trecord = evaluate();");
			output.println("\t\t}");
			output.println("\t\tIterator iter = recordList.iterator();");
			output.println("\t\twhile (iter.hasNext()) {");
			output.println("\t\t\trecord = (TransformationRecord) iter.next();");
			output.println("\t\t\tif (apply(record))");
			output.println("\t\t\t\tsuccess = true;");
			output.println("\t\t}");
			output.println("\t\treturn success;");
		}
		output.println("\t}");

		// Print out the constraints
		printConstraintSet(mainConstraintSetElement, output);
		while (!constraintSets.isEmpty()) {
			try {
				Element cSet = (Element) constraintSets.removeFirst();
				printConstraintSet(cSet, output);
			}
			catch (NoSuchElementException e) { ; }
		}

		// Print out the attribute value calculations
		NodeList children = ruleElement.getElementsByTagName("NewTerminalSymbol");
		for (int i = 0; i < children.getLength(); i++) {

			Element ntsElement = (Element) children.item(i);
			SymbolDefinition sDef = (SymbolDefinition) newTerminalSymbols.get(ntsElement.getAttribute("name"));

			// Go through attributes
			NodeList children2 = ntsElement.getChildNodes();
			Iterator iter2 = sDef.getAllAttributes().iterator();
			for (int j = 0; j < children2.getLength(); j++) {

				// Get attribute and its corresponding node
				Element child = (Element) children2.item(j);
				Attribute attr = (Attribute) iter2.next();

				// Print method
				output.println("");
				output.println("\tpublic " + (attr.isConstraintVariable() ? "double" : attr.getDataType()) + " initial_" + 
					ntsElement.getAttribute("name") + "_" + attr.getName() + "(TransformationRecord" + number + " record)");
				output.println("\t{");
				output.println("\t\treturn " + printArgument(child) + ";");
				output.println("\t}");
			}
		}

		children = ruleElement.getElementsByTagName("SetAttribute");
		for (int i = 0; i < children.getLength(); i++) {
			Element setAttElement = (Element) children.item(i);
			SymbolDefinition sDef = null;
			sDef = (SymbolDefinition) outputSymbols.get(setAttElement.getAttribute("symbol_name"));
			if (sDef == null)
				sDef = (SymbolDefinition) inputSymbols.get(setAttElement.getAttribute("symbol_name"));
			if (sDef == null)
				sDef = (SymbolDefinition) newTerminalSymbols.get(setAttElement.getAttribute("symbol_name"));
			if (sDef == null)
				sDef = (SymbolDefinition) referableExistentialSymbols.get(setAttElement.getAttribute("symbol_name"));
			Attribute attr = sDef.getAttribute(setAttElement.getAttribute("attribute_name"));

			// Get <SetAttribute> child element
			NodeList children2 = setAttElement.getChildNodes();
			Element child = (Element) children2.item(0);

			// Print method
			String name = setAttElement.getAttribute("symbol_name") + "_" + setAttElement.getAttribute("attribute_name");
			output.println("");
			output.println("\tpublic " + (attr.isConstraintVariable() ? "double" : attr.getDataType()) + " valueOf_" + 
				name + "(TransformationRecord" + number + " record)");
			output.println("\t{");
			if (child.getTagName().equals("ConstraintExpression")) 
				Error.die(this, child.getAttribute("_line_number_"), "Cannot use a <ConstraintExpression> with a <SetAttribute>" +
					" element in a transformation");
			else
				output.println("\t\treturn " + printArgument(child) + ";");
			output.println("\t}");
		}

		// End class and close file
		output.println("}");
		output.println("");
		output.close();

		createConstraintClasses();
	}
}

