/*
    Copyright (C) 2003  Anthony R. Jansen.

    This file is part of the CIDER Toolkit.

    The CIDER Toolkit is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    The CIDER Toolkit is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with the CIDER Toolkit; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

// ----------------------------------------
// CIDER Toolkit: Solver Constraints Package
//
// Author:    Anthony R. Jansen
// Begun:     January 2003
// Inferface: ConstraintCompiler
//
// This interface defines static methods 
// allowing constraints to be precompiled
// into Java classes so that they do not
// have to be initialized with a string
// parse at run time.
// ----------------------------------------

package au.edu.monash.csse.tonyj.cider.constraints;

import java.io.PrintWriter;

/**
 * A constraint compiler allows solver constraint strings to constraint expression
 * strings to be parsed during a compile phase, so they will not need to be parsed
 * during the runtime of the application that uses them.
 * This interface defines how a constraint compiler is used. Classes that
 * implement this interface should also define a no-argument constructor.
 */
public interface ConstraintCompiler {

	/**
	 * Writes source code for a constraint class. 
	 * It first defines the package of the class (unless <code>classPackage</code> is the empty string),
	 * then the <code>import</code> statements that are needed, and then the class itself. The class should implement the {@link Constraint}
	 * interface, or extend another class that already implements this interface. 
	 * <p> Inside the class, only the constructor needs to be created. This constructor receives one argument, an array of 
	 * {@link ConstraintVariable}s, and also throws an <code>java.lang.ArrayIndexOutOfBoundsException</code> (which is thrown if an illegal 
	 * access to the array is attempted). Essentially, the constructor works the same as the {@link Constraint#initialize} method, except the 
	 * constructor will be able to create the constraint directly without having to parse a string. 
	 * @param className The name of the constraint class.
	 * @param classPackage The name of the package the constraint class belongs to.
	 * @param constraintStr A string that defines the constraint (this string needs to be parsed). 
	 * @param output The output stream where the class is to be written.
	 * @exception au.edu.monash.csse.tonyj.cider.constraints.CannotProcessConstraintException If string cannot be parsed.
	 * @exception java.lang.NumberFormatException If an illegal number format is found in the string being parsed.
	 */
	public void createConstraintClass(String className, String classPackage, String constraintStr, PrintWriter output)
		throws CannotProcessConstraintException, NumberFormatException;

	/**
	 * Writes source code for a constraint expression class. 
	 * It first defines the package of the class (unless <code>classPackage</code> is the empty string),
	 * then the <code>import</code> statements that are needed, and then the class itself. The class should implement the 
	 * {@link ConstraintExpression} interface, or extend another class that already implements this interface. 
	 * <p> Inside the class, only the constructor needs to be created. This constructor receives one argument, an array of 
	 * {@link ConstraintVariable}s, and also throws an <code>java.lang.ArrayIndexOutOfBoundsException</code> (which is thrown if an illegal 
	 * access to the array is attempted). Essentially, the constructor works the same as the {@link ConstraintExpression#initialize} method, 
	 * except the constructor will be able to create the constraint directly without having to parse a string. 
	 * @param className The name of the constraint expression class.
	 * @param classPackage The name of the package the constraint expression class belongs to.
	 * @param expressionStr A string that defines the constraint expression (this string needs to be parsed). 
	 * @param output The output stream where the class is to be written.
	 * @exception au.edu.monash.csse.tonyj.cider.constraints.CannotProcessConstraintException If string cannot be parsed.
	 * @exception java.lang.NumberFormatException If an illegal number format is found in the string being parsed.
	 */
	public void createConstraintExpressionClass(String className, String classPackage, String expressionStr, PrintWriter output)
		throws CannotProcessConstraintException, NumberFormatException;
}

