/*
    Copyright (C) 2003  Anthony R. Jansen.

    This file is part of the CIDER Toolkit.

    The CIDER Toolkit is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    The CIDER Toolkit is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with the CIDER Toolkit; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

// ----------------------------------------
// CIDER Toolkit: Token Canvas Package
//
// Author:    Anthony R. Jansen
// Begun:     November 2002
// Class:     InterpretedTokenCanvas
//
// This class extends the TokenCanvas
// class to define a Token Canvas that
// contains an IDE Interpreter for its
// tokens. 
// ----------------------------------------

package au.edu.monash.csse.tonyj.cider.canvas;

import au.edu.monash.csse.tonyj.cider.constraints.ConstraintSolver;
import au.edu.monash.csse.tonyj.cider.interpreter.Interpreter;
import au.edu.monash.csse.tonyj.cider.interpreter.GrammarSymbol;
import au.edu.monash.csse.tonyj.cider.interpreter.GrammarSymbolEvent;
import au.edu.monash.csse.tonyj.cider.interpreter.GrammarSymbolListener;
import java.awt.Dimension;
import java.io.PrintStream;
import java.util.Set;
import java.util.Iterator;

/**
 * This class extends the {@link TokenCanvas} class to be used with an {@link Interpreter}
 * for processing diagrammatic languages. 
 * <br><br><i><u>Important change from version 0.2.0</u>:</i> The <code>load()</code> and <code>save()</code>
 * methods have been removed. Loading and saving functionality must now be implemented by
 * the application developer. For example, to save, all terminal symbols (and their attribute values)
 * can be written to a file (in a format that the application developer is free to choose). Then, to
 * load again, each terminal symbol (and their attribute values) can be read in and added to an instance
 * of the Interpreter class, and the associated parse forest will be automatically generated.
 */
public class InterpretedTokenCanvas extends TokenCanvas implements GrammarSymbolListener {

	// Private variables
	private Interpreter	interpreter;

	/** 
	 * Creates a new InterpretedTokenCanvas object.
	 * @param d The dimensions of the new canvas.	
	 * @param inter The {@link Interpreter} object the canvas is to use.
	 */
	public InterpretedTokenCanvas(Dimension d, Interpreter inter)
	{
		super(d);
		interpreter = inter;
		interpreter.setJComponent(this);
		interpreter.addGrammarSymbolListener(this);

		// If interpreter already has drawable symbols, add them 
		Iterator iter = interpreter.getAllSymbols().iterator();
		while (iter.hasNext()) {
			GrammarSymbol symbol = (GrammarSymbol) iter.next();
			if (symbol instanceof DrawableToken)
				super.add((DrawableToken) symbol);
		}
	}

	/** 
	 * Creates a new InterpretedTokenCanvas object.
	 * @param width The width of the new canvas.	
	 * @param height The height of the new canvas.	
	 * @param inter The {@link Interpreter} object the canvas is to use.
	 */
	public InterpretedTokenCanvas(int width, int height, Interpreter inter)
	{
		this(new Dimension(width, height), inter);
	}

	/**
	 * This method is required by the {@link GrammarSymbolListener} interface.
	 * It should not be called directly.
	 */
	public void symbolAdded(GrammarSymbolEvent gse)
	{
		GrammarSymbol symbol = gse.getGrammarSymbol();
		if (symbol instanceof DrawableToken)
			super.add((DrawableToken) symbol);
	}

	/**
	 * This method is required by the {@link GrammarSymbolListener} interface.
	 * It should not be called directly.
	 */
	public void symbolRemoved(GrammarSymbolEvent gse)
	{
		GrammarSymbol symbol = gse.getGrammarSymbol();
		if (symbol instanceof DrawableToken)
			super.remove((DrawableToken) symbol);
	}

	/**
	 * This method is required by the {@link GrammarSymbolListener} interface.
	 * It should not be called directly.
	 */
	public void symbolModified(GrammarSymbolEvent gse) { ; }

	/**
	 * Overrides this method in the {@link TokenCanvas} class.
	 * @param t The token to be added. Must be a terminal symbol in the grammar.
	 * @return <code>true</code> if successful, or <code>false</code> if the 
	 * Interpreter would not accept the token as a valid terminal symbol.
	 */
	public boolean add(DrawableToken t)
	{
		// Make sure token is also a grammar symbol
		GrammarSymbol symbol;
		try {
			symbol = (GrammarSymbol) t;
		}
		catch (ClassCastException e) {
			return false;
		}
		return interpreter.addTerminalSymbol(symbol);
	}

	/**
	 * Overrides this method in the {@link TokenCanvas} class.
	 * @param t The token to be removed. Must be a terminal symbol in the grammar.
	 * @return <code>true</code> if successful, or <code>false</code> if the 
	 * token could not be removed from the Interpreter.
	 */
	public boolean remove(DrawableToken t)
	{
		// Make sure token is also a grammar symbol
		GrammarSymbol symbol;
		try {
			symbol = (GrammarSymbol) t;
		}
		catch (ClassCastException e) {
			return false;
		}
		return interpreter.removeTerminalSymbol(symbol);
	}

	/**
	 * Returns the {@link Interpreter} used by the canvas.
	 */
	public Interpreter getInterpreter()
	{
		return interpreter;
	}
}

