/*
    Copyright (C) 2003  Anthony R. Jansen.

    This file is part of the CIDER Toolkit.

    The CIDER Toolkit is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    The CIDER Toolkit is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with the CIDER Toolkit; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

// ----------------------------------------
// CIDER Toolkit: Token Canvas Package
//
// Author:    Anthony R. Jansen
// Begun:     November 2002
// Class:     TokenCanvas
//
// This class extends the swing JPanel
// class to define a Token Canvas that 
// contains a list of tokens that can be
// drawn in an given order (thought of as
// a stack, with tokens at the top of the
// stack drawn over tokens at the bottom
// of the stack - list index 0 is the top).
// ----------------------------------------

package au.edu.monash.csse.tonyj.cider.canvas;

import javax.swing.JPanel;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.Dimension;
import java.awt.Color;
import java.util.List;
import java.util.Vector;

/**
 * This class implements a token canvas which contains a collection of tokens in a stack.
 * When asked to draw itself, the token canvas goes through the tokens in the stack in order,
 * getting each one to draw itself (with tokens at the top of the stack drawn over tokens
 * at the bottom). The order of the tokens in the stack can be changed, and tokens can
 * be added to or removed from the canvas. 
 */
public class TokenCanvas extends JPanel {

	// Private variables
	private Color		backgroundColour;
	private List		stack;

	/** 
	 * Creates a new TokenCanvas object.
	 * @param d The dimensions of the new canvas.	
	 */
	public TokenCanvas(Dimension d)
	{
		super();
		backgroundColour = Color.WHITE;
		setPreferredSize(d);
		setOpaque(true);
		stack = new Vector();
	}

	/** 
	 * Creates a new TokenCanvas object.
	 * @param width The width of the new canvas.	
	 * @param height The height of the new canvas.	
	 */
	public TokenCanvas(int width, int height)
	{
		this(new Dimension(width, height));
	}

	/**
	 * Sets the background colour of the canvas.
	 */
	public void setBackground(Color c)
	{
		backgroundColour = c;
	}

	/**
	 * Returns the background colour of the canvas.
	 */
	public Color getBackground()
	{
		return backgroundColour;
	}

	/**
	 * Draws the current list of tokens in the order they appear on the stack. 
	 * Required by the Java Swing components. This method should not be called directly.
	 */
	public void paintComponent(Graphics g)
	{
		super.paintComponent(g);

		// Fill canvas with background colour first
		g.setColor(backgroundColour);
		g.fillRect(0, 0, getWidth(), getHeight());

		// Draw objects in reverse order 
		try {
			for (int i = stack.size() - 1; i >= 0; i--) 
				((DrawableToken) stack.get(i)).draw((Graphics2D) g);
		}
		catch (ArrayIndexOutOfBoundsException e) { ; }
	}

	/** 
	 * Adds a token to the bottom of the stack.
	 * @param t The token to be added.
	 * @return <code>true</code> if successful, or <code>false</code> if the 
	 * token is <code>null</code> or already in the stack.
	 */
	public boolean add(DrawableToken t)
	{
		if (t == null)
			return false;
		if (stack.contains(t))
			return false;

		stack.add(t);
		return true;
	}
	
	/** 
	 * Moves a token to the top of the stack.
	 * @param t The token to be moved.
	 * @return <code>true</code> if successful, or <code>false</code> if the 
	 * token is not in the stack.
	 */
	public boolean moveToTop(DrawableToken t)
	{
		return move(0, t);
	}

	/** 
	 * Moves a token to the bottom of the stack.
	 * @param t The token to be moved.
	 * @return <code>true</code> if successful, or <code>false</code> if the 
	 * token is not in the stack.
	 */
	public boolean moveToBottom(DrawableToken t)
	{
		return move(stack.size() - 1, t);
	}

	/** 
	 * Moves a token to a specific location in the stack.
	 * If an invalid index is specified the token will be placed at the bottom of the stack.
	 * @param index Location in the stack in the range 0 (top) to {@link #getNumTokens} - 1 (bottom).
	 * @param t The token to be moved.
	 * @return <code>true</code> if successful, or <code>false</code> if the 
	 * token is not in the stack.
	 */
	public boolean move(int index, DrawableToken t)
	{
		if (!stack.contains(t))
			return false;

		try {
			stack.remove(t);
			stack.add(index, t);
		}
		catch (ArrayIndexOutOfBoundsException e2) {
			stack.add(t);
		}
		return true;
	}

	/** 
	 * Removes a token from the stack.
	 * @param t The token to be removed.
	 * @return <code>true</code> if successful, or <code>false</code> if the 
	 * token is not in the stack.
	 */
	public boolean remove(DrawableToken t)
	{
		return stack.remove(t);
	}

	/** 
	 * Removes all tokens from the stack.
	 */
	public void clear()
	{
		stack.clear();
	}

	/**
	 * Returns the number of tokens in the stack
	 */
	public int getNumTokens()
	{
		return stack.size();
	}

	/**
	 * Returns the token at the top of the stack, or <code>null</code> if the stack is empty.
	 */
	public DrawableToken getTopToken()
	{
		return getToken(0);
	}

	/**
	 * Returns the token at the bottom of the stack, or <code>null</code> if the stack is empty.
	 */
	public DrawableToken getBottomToken()
	{
		return getToken(stack.size() - 1);
	}

	/**
	 * Returns the token at the specified location in the stack, or <code>null</code> if no token is at that index.
	 * @param index Location in the stack in the range 0 (top) to {@link #getNumTokens} - 1 (bottom).
	 */
	public DrawableToken getToken(int index)
	{
		try {
			return (DrawableToken) stack.get(index);
		}
		catch (ArrayIndexOutOfBoundsException e) {
			return null;
		}
	}

	/**
	 * Returns the location in the stack of the specified token, or -1 if it is not in the stack.
	 * @param t The token whose location is being sought.
	 * @return Location if token is found, -1 if token is not found.
	 */
	public int indexOf(DrawableToken t)
	{
		return stack.indexOf(t);
	}

	/**
	 * Searches the tokens to see if any contain the specified co-ordinates.
	 * @param x The x co-ordinate.
	 * @param y The y co-ordinate.
	 * @return A list of tokens that contain the specified co-ordinates
	 * in the order they appear on the stack (top at start of list).
	 */
	public List getTokensContaining(double x, double y)
	{
		List containsStack = new Vector();	

		// See what tokens contain the specified point
		try {
			for (int i = 0; i < stack.size(); i++) {
				DrawableToken token = (DrawableToken) stack.get(i);
				if (token.contains(x, y))
					containsStack.add(token);
			}
		}
		catch (ArrayIndexOutOfBoundsException e) { ; }

		return containsStack;
	}
}

