/*
    Copyright (C) 2003  Anthony R. Jansen.

    This file is part of the CIDER Toolkit.

    The CIDER Toolkit is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    The CIDER Toolkit is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with the CIDER Toolkit; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

// ----------------------------------------
// CIDER Toolkit: Solver Constraints Package
//
// Author:    Anthony R. Jansen
// Begun:     November 2002
// Interface: Constraint
//
// This is the interface that defines how a
// constraint implementation needs to be 
// accessed.
// ----------------------------------------

package au.edu.monash.csse.tonyj.cider.constraints;

import java.util.Set;

/**
 * A constraint essentially defines a mathematical relationship, 
 * such as <i>x</i> &lt; <i>y</i> + 3 (where each side of the &lt;
 * sign can be thought of as a {@link ConstraintExpression}).
 * This interface defines how a constraint is manipulated. Classes that
 * implement this interface should also define a no-argument constructor.
 */
public interface Constraint {

	/**
	 * Initializes the constraint from a string.
	 * @param constraint A string that defines the constraint (this string needs to be parsed). 
	 * @param variables An array of constraint variables that the <code>constraint</code> string refers to.
	 * @exception au.edu.monash.csse.tonyj.cider.constraints.CannotProcessConstraintException If string cannot be parsed.
	 * @exception java.lang.ArrayIndexOutOfBoundsException If an illegal access of the constraint variable array is attempted.
	 * @exception java.lang.NumberFormatException If an illegal number format is found in the string being parsed.
	 */
	public void initialize(String constraint, ConstraintVariable variables[])
		throws CannotProcessConstraintException, ArrayIndexOutOfBoundsException, NumberFormatException;

	/**
	 * Initializes a constraint of the form: <code>variable == expression</code>
	 * @param variable The variable to be used in defining the equals relationship.
	 * @param expression The expression to be used in defining the equals relationship.
	 */
	public void initializeWithEqualsRelationship(ConstraintVariable variable, ConstraintExpression expression);

	/**
	 * Initializes a constraint of the form: <code>variable == constant</code>
	 * @param variable The variable to be used in defining the equals relationship.
	 * @param constant The constant value.
	 */
	public void initializeConstant(ConstraintVariable variable, double constant);

	/**
	 * Returns <code>true</code> if and only if the solved values of the constraint variables 
	 * satisfy this constraint within the specified tolerance amount. This method must work
	 * regardless of whether the constraint has been added to a {@link ConstraintSolver} or not.
	 * If the constraint has not been initialized, behaviour is undefined.
	 * @param tolerance Tolerance amount for checking constraint satisfaction.
	 */
	public boolean isSatisfied(double tolerance);

	/**
	 * Returns a set containing all the constraint variables used by this constraint.
	 */
	public Set getConstraintVariables();

	/**
	 * Sets the weight of this constraint.
	 * @param weight Weight value.
	 */
	public void setWeight(double weight);

	/**
	 * Returns the weight of this constraint.
	 */
	public double getWeight();
}

