/*
    Copyright (C) 2003  Anthony R. Jansen.

    This file is part of the CIDER Toolkit.

    The CIDER Toolkit is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    The CIDER Toolkit is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with the CIDER Toolkit; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

// ----------------------------------------
// CIDER Toolkit: Solver Constraints Package
//
// Author:    Anthony R. Jansen
// Begun:     November 2002
// Interface: ConstraintExpression
//
// This is the interface that defines how a
// constraint expression implementation 
// needs to be accessed.
// ----------------------------------------

package au.edu.monash.csse.tonyj.cider.constraints;

import java.util.Set;

/**
 * A constraint expression is a component of a {@link Constraint}.
 * This interface defines how a constraint expression is manipulated. Classes that
 * implement this interface should also define a no-argument constructor.
 */
public interface ConstraintExpression {

	/**
	 * Initializes the constraint expression from a string.
	 * @param expression A string that defines the constraint expression (this string needs to be parsed). 
	 * @param variables An array of constraint variables that the <code>expression</code> string refers to.
	 * @exception au.edu.monash.csse.tonyj.cider.constraints.CannotProcessConstraintException If string cannot be parsed.
	 * @exception java.lang.ArrayIndexOutOfBoundsException If an illegal access of the constraint variable array is attempted.
	 * @exception java.lang.NumberFormatException If an illegal number format is found in the string being parsed.
	 */
	public void initialize(String expression, ConstraintVariable variables[])
		throws CannotProcessConstraintException, ArrayIndexOutOfBoundsException, NumberFormatException;

	/**
	 * Evaluates this expression using the solved values of the constraint variables, and returns the resulting value.
	 * If the constraint expression has not been initialized, behaviour is undefined.
	 * @return Value resulting from evaluation.
	 */
	public double evaluate();

	/**
	 * Returns a set containing all the constraint variables used by this constraint expression.
	 */
	public Set getConstraintVariables();
}

