/*
    Copyright (C) 2003  Anthony R. Jansen.

    This file is part of the CIDER Toolkit.

    The CIDER Toolkit is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    The CIDER Toolkit is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with the CIDER Toolkit; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

// ----------------------------------------
// CIDER Toolkit: Solver Constraints Package
//
// Author:  Anthony R. Jansen
// Begun:   November 2002
// Class:   ConstraintSolver
//
// This is the abstract base class that 
// needs to be used when implementing a 
// constraint solver. It contains several
// abstract methods that must be 
// implemented, as well as methods for 
// launching a constraint variable event.
// ----------------------------------------

package au.edu.monash.csse.tonyj.cider.constraints;

import java.util.Set;
import java.util.List;
import java.util.LinkedList;
import java.util.Iterator;

/**
 * This abstract base class allows a multi-way constraint solver to be implemented in the <b>CIDER</b> toolkit. 
 * Classes that extend this base class should also define a no-argument constructor.
 */
public abstract class ConstraintSolver {

	// Private variable
	private List	listeners;

	/**
	 * Adds a constraint variable to the solver. If the variable is already in the
	 * solver, this method does nothing.
	 * @param v The variable to be added.
	 */
	public abstract void addVariable(ConstraintVariable v);

	/**
	 * Removes a constraint variable from the solver if possible. If the variable is 
	 * currently being used by a constraint in the solver, it may not be possible to
	 * remove the variable.
	 * @param v The variable to be removed.
	 * @return <code>true</code> if the variable successfully removed, <code>false</code> otherwise.
	 */
	public abstract boolean removeVariable(ConstraintVariable v);

	/**
	 * Adds a constraint to the solver. 
	 * @param c The constraint to be added.
	 * @return <code>true</code> if successful, <code>false</code> if the constraint could not be added
	 * because it conflicts with the existing constraints in the solver.
	 */
	public abstract boolean addConstraint(Constraint c);

	/**
	 * Removes a constraint from the solver if possible. 
	 * @param c The constraint to be removed.
	 * @return <code>true</code> if the constraint successfully removed, <code>false</code> otherwise.
	 */
	public abstract boolean removeConstraint(Constraint c);

	/**
	 * Informs the solver that a new desired value is going to be suggested for a particular variable.
	 * Allows the solver to tag that variable as one that can be edited.
	 * @param v The variable to be edited.
	 */
	public abstract void addEditVariable(ConstraintVariable v);

	/**
	 * Returns a set containing all constraints in the solver.
	 */
	public abstract Set getConstraints();

	/**
	 * Returns a set containing all constraint variables in the solver (including, of course, all
	 * variables used by constraints in the solver).
	 */
	public abstract Set getConstraintVariables();

	/**
	 * Calls the solver to solve all constraints. This method should be called
	 * if constraints have been added to or removed from the solver.
	 * <p> This method must call the {@link #launchVariableModifiedEvent} method for
	 * each constraint variable whose value has been modified by this call.
	 */
	public abstract void solve();

	/**
	 * Calls the solver to re-solve the constraints. This method should be called
	 * if the suggested values of constraint variables have been modified. 
	 * <p> This method must call the {@link #launchVariableModifiedEvent} method for
	 * each constraint variable whose value has been modified by this call.
	 */
	public abstract void resolve();

	/**
	 * Adds an listener to listen for solver events (which are launched whenever a constraint
	 * variable is modified by the solver).
	 * @param cvl The listener to be added.
	 */
	public synchronized void addConstraintVariableListener(ConstraintVariableListener cvl)
	{
		if (listeners == null)
			listeners = new LinkedList();
		listeners.add(cvl);
	}

	/**
	 * Removes a listener that listens for solver events (which are launched whenever a constraint
	 * variable is modified by the solver).
	 * @param cvl The listener to be removed.
	 */
	public synchronized void removeConstraintVariableListener(ConstraintVariableListener cvl)
	{
		if (listeners != null)
			listeners.remove(cvl);
	}

	/**
	 * Launches an event which indicates that a constraint variable has been modified.
	 * All constraint variable listeners that have been added to the solver are informed of this event.
	 * @param cv The constraint variable that was modified.
	 * @param oldValue The old solved value of the constraint variable before the modificaton.
	 */
	public void launchVariableModifiedEvent(ConstraintVariable cv, Double oldValue)
	{
		if (listeners == null)
			return;

		ConstraintVariableEvent event = new ConstraintVariableEvent(this, cv, ConstraintVariableEvent.VARIABLE_MODIFIED, oldValue);
		Iterator iter = listeners.iterator();
		while (iter.hasNext())
			((ConstraintVariableListener) iter.next()).variableModified(event);
	}
}

