/*
    Copyright (C) 2003  Anthony R. Jansen.

    This file is part of the CIDER Toolkit.

    The CIDER Toolkit is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    The CIDER Toolkit is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with the CIDER Toolkit; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

// ----------------------------------------
// CIDER Toolkit: Interpreter Package
//
// Author:  Anthony R. Jansen
// Begun:   September 2002
// Class:   GrammarSymbol
//
// This is the abstract base class used by
// all the classes that define a symbol
// type, which are produced by the Compiler
// program.
// ----------------------------------------

package au.edu.monash.csse.tonyj.cider.interpreter;

import au.edu.monash.csse.tonyj.cider.constraints.ConstraintVariable;
import java.util.Iterator;
import java.util.Set;
import java.util.HashSet;

/**
 * This abstract base class is used by the <b>CIDER</b> compiler when creating grammar symbols that are
 * specific to a particular diagrammatic language.
 */
public abstract class GrammarSymbol {

	// Private variables
	private int			id_number;
	private boolean			terminal;
	private SymbolNode		node;

	// Static variable for ID numbers
	private static int ids = 1;

	// Constructor (package scope)
	GrammarSymbol(boolean term)
	{
		terminal = term;
		node = null;
		synchronized(this) {
			id_number = ids;
			ids++;
		}
	}

	/**
	 * Returns <code>true</code> if and only if the symbol is associated with an {@link Interpreter}.
	 */
	public boolean isRegistered()
	{
		return (node != null);
	}

	/**
	 * Returns <code>true</code> if the symbol is a terminal symbol, <code>false</code> otherwise
	 */
	public boolean isTerminal()
	{
		return terminal;
	}

	/**
	 * Returns a set containing all child symbols in the parse tree, or <code>null</code> if
	 * the symbol is not registered.
	 */
	public Set getChildren()
	{
		// If not in a parse tree, return null
		if (node == null)
			return null;

		Set children = new HashSet();

		// Check that children exist
		if (node.producedBy() != null) {
			Set childNodes = node.producedBy().getInputSymbolNodes();
			Iterator i = childNodes.iterator();
			while (i.hasNext()) 
				children.add(((SymbolNode) i.next()).getSymbol());
		}
		return children;
	}

	/**
	 * Returns a set containing all sibling symbols in the parse tree
	 * (that is, other symbols that were produced by the same production),
	 * or <code>null</code> if the symbol is not registered.
	 */
	public Set getSiblings()
	{
		// If not in a parse tree, return null
		if (node == null)
			return null;

		Set siblings = new HashSet();

		// Check that siblings exist
		if (node.producedBy() != null) {
			Set siblingNodes = node.producedBy().getOutputSymbolNodes();
			Iterator i = siblingNodes.iterator();
			while (i.hasNext()) 

				// Don't include current symbol with its siblings
				if (i.next() != node)
					siblings.add(((SymbolNode) i.next()).getSymbol());
		}
		return siblings;
	}

	/**
	 * Returns a set containing all parent symbols in the parse tree, or <code>null</code> if
	 * the symbol is not registered.
	 */
	public Set getParents()
	{
		// If not in a parse tree, return null
		if (node == null)
			return null;

		Set parents = new HashSet();

		// Check that parents exist
		if (node.usedBy() != null) {
			Set parentNodes = node.usedBy().getOutputSymbolNodes();
			Iterator i = parentNodes.iterator();
			while (i.hasNext()) 
				parents.add(((SymbolNode) i.next()).getSymbol());
		}
		return parents;
	}

	// Returns the parse tree node for this symbol
	SymbolNode getSymbolNode()
	{
		return node;
	}

	// Registers the symbol by setting the parse tree node 
	// and returning true iff the symbol is not currently registered
	boolean registerSymbol(SymbolNode n)
	{
		if (node != null)
			return false;
		node = n;
		return true;
	}

	// Deregisters the symbol, setting the parse tree node to null
	// and returning true iff the symbol is currently registered
	boolean deregisterSymbol()
	{
		if (node == null)
			return false;
		node = null;
		return true;
	}

	/**
	 * Returns the unique ID number associated with this grammar symbol.
	 */
	public int getIDNumber()
	{
		return id_number;
	}

	/**
	 * Overrides this method in the <code>Object</code> class. Primarily used for debugging purposes.
	 */
	public abstract String toString();

	/**
	 * Returns the name of this grammar symbol type.
	 */
	public abstract String getName();

	// The following methods must be implemented
	abstract int getAttributeIdentifier(ConstraintVariable var);
	abstract String getAttributeName(int identifier);
	abstract Set getConstraintVariables();
	abstract void restoreAttributeValue(int attribute, Object value);
}


