/*
    Copyright (C) 2003  Anthony R. Jansen.

    This file is part of the CIDER Toolkit.

    The CIDER Toolkit is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    The CIDER Toolkit is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with the CIDER Toolkit; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

// ----------------------------------------
// CIDER Toolkit: Interpreter Package
//
// Author:  Anthony R. Jansen
// Begun:   September 2002
// Class:   Interpreter
//
// This is the abstract base class used by
// the interpreter that is tailored to a
// specific diagrammatic language.
// ----------------------------------------

package au.edu.monash.csse.tonyj.cider.interpreter;

import au.edu.monash.csse.tonyj.cider.layout.ForceDirectedGraph;
import au.edu.monash.csse.tonyj.cider.constraints.ConstraintSolver;
import au.edu.monash.csse.tonyj.cider.constraints.Constraint;
import au.edu.monash.csse.tonyj.cider.constraints.ConstraintVariable;
import au.edu.monash.csse.tonyj.cider.constraints.ConstraintVariableListener;
import au.edu.monash.csse.tonyj.cider.constraints.ConstraintVariableEvent;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;
import java.util.HashSet;
import java.util.List;
import java.util.LinkedList;
import java.io.PrintStream;
import java.util.logging.Level;
import javax.swing.JComponent;

/**
 * The Interpreter is the main control class in the <b>CIDER</b> toolkit that maintains
 * a parse forest representation of a diagram, is responsible for applying productions,
 * and also controls the application of transformations. It as an abstract base class
 * that is extended by a grammar-specific class that is responsible for implementing
 * functionality that is grammar dependent. This grammar-specific class is produced
 * by the <b>CIDER</b> compiler. Note that when an instance of this class is created,
 * auto forest checking and auto constraint solving are initially turned on. 
 * <br><br><i><u>Important change from version 0.2.0</u>:</i> The <code>load()</code> and <code>save()</code>
 * methods have been removed. Loading and saving functionality must now be implemented by
 * the application developer. For example, to save, all terminal symbols (and their attribute values)
 * can be written to a file (in a format that the application developer is free to choose). Then, to 
 * load again, each terminal symbol (and their attribute values) can be read in and added to an instance 
 * of the Interpreter class, and the associated parse forest will be automatically generated.
 */
public abstract class Interpreter implements ConstraintVariableListener {

	// Private and protected variables
	protected ParseForest		forest;
	protected ConstraintSolver	solver;
	protected Class			solverClass;
	protected boolean 		autoForestChecking;
	protected boolean 		forestBeingChecked;
	private boolean			autoConstraintSolving;
	private Set			constraints;
	private Set			listeners;
	private JComponent		jComponent;
	private ForceDirectedGraph	forceDirectedGraph;

	// Undo variables
	private boolean		undoEnabled;
	private LinkedList	undoList;

	// Constant
	public static final String      VERSION = "0.3.0";

	// Constructor
	Interpreter()
	{
		constraints = new HashSet();
		autoForestChecking = true;
		forestBeingChecked = false;
		autoConstraintSolving = true;
		listeners = new HashSet();
		undoEnabled = false;
		undoList = new LinkedList();
		forceDirectedGraph = new ForceDirectedGraph();
		forceDirectedGraph.setParameters(25.0, 400.0, 0.5);
		addGrammarSymbolListener(forceDirectedGraph);
	}
	
	/**
	 * Invokes the force directed layout algorithm to be performed on the diagram.
	 * @param timeStep The time (in milliseconds) between animation steps. Note that this is the minimum time - 
	 * calculation of the next layout iteration may take longer than the value specified.
	 */
	public synchronized void applyForceDirectedLayout(int timeStep)
	{
		if (forceDirectedGraph != null) {
			Log.getLogger().info("Force Directed Layout called");
			forceDirectedGraph.applyVisualLayout(this, timeStep);
		}
	}

	/**
	 * This method is invoked by the constructor of an {@link au.edu.monash.csse.tonyj.cider.canvas.InterpretedTokenCanvas}. It should not
	 * be invoked at any other time.
	 */
	public synchronized void setJComponent(JComponent jc)
	{
		jComponent = jc;
	}

	/**
	 * Returns the {@link JComponent} associated with the visual representation of the diagram
	 * being stored by this Interpreter instance (or null if there is none). Typically, this should
	 * be an instance of an {@link au.edu.monash.csse.tonyj.cider.canvas.InterpretedTokenCanvas}.
	 */
	public synchronized JComponent getJComponent()
	{
		return jComponent;
	}

	/**
	 * This methods sets the parameters for force directed layout.
	 * @param springEquilibriumDistance The equilibrium distance that the spring force tries to achieve between graph nodes
	 * that are connected by a graph edge. Default value is 25.0
	 * @param particleRepulsionConstant Used to calculate the repulsive force between each pair of graph nodes (both connected and 
	 * unconnected), which drops of quadratically as the distance between nodes increases. Default value is 400.0
	 * <br> &nbsp; &nbsp; &nbsp; <code>Force = ParticleRepulsionConstant / (Distance * Distance)</code>
	 * @param springConstant Used to calculate spring force between connected graph nodes. Default value is 0.5
	 * <br> &nbsp; &nbsp; &nbsp; <code>Force = SpringConstant * | Distance - SpringEquilibriumDistance | </code>
	 */
	public synchronized void setForceDirectedLayoutParameters(double springEquilibriumDistance, double particleRepulsionConstant, 
		double springConstant)
	{
		if (forceDirectedGraph != null) 
			forceDirectedGraph.setParameters(springEquilibriumDistance, particleRepulsionConstant, springConstant);
	}

	/**
	 * Allows the logging level for the CIDER logger to be set. By default, the logging level is set to 
	 * <code>Level.INFO</code>. Logging can be turned off by using <code>Level.OFF</code>, and all messages 
	 * will be logged if <code>Level.ALL</code> is used.
	 * @param level The logging level to be set.
	 */
	public static void setLoggingLevel(Level level)
	{
		Log.getLogger().setLevel(level);
	}

	/**
	 * Explicitly calls the grammar-specific interpreter to evaluate the production rules.
	 */
	public abstract void evaluateProductionRules();

	/**
	 * Applies a specified transformation to the parse forest at the first valid location it finds.
	 * @param transformation The transformation to be applied.
	 * @return <code>true</code> if the transformation was successful, <code>false</code> otherwise.
	 * @exception java.lang.IllegalArgumentException If an unknown transformation is specified.
	 */
	public abstract boolean applyTransformation(int transformation) throws IllegalArgumentException;

	/**
	 * Applies a specified transformation to the parse forest in parallel. That is, it will find all
	 * the locations in the parse forest where the transformation can be applied, and then in a single
	 * step apply the transformation at all of these locations.
	 * @param transformation The transformation to be applied.
	 * @return <code>true</code> if the transformation was successful, <code>false</code> otherwise.
	 * @exception java.lang.IllegalArgumentException If an unknown transformation is specified, or a
	 * a transformation is specified that cannot be applied in parallel (such as a compound transformation).
	 */
	public abstract boolean applyTransformationInParallel(int transformation) throws IllegalArgumentException;

	/**
	 * Applies a specified transformation to the parse forest with a specified symbol mapping.
	 * @param transformation The transformation to be applied. Must be a basic transformation (not a disjunction or compound transformation).
	 * @param parallel <code>true</code> applies the transformation in parallel, <code>false</code> does not.
	 * @param candidatesMap Contains a mapping of <code>String</code> objects to <code>Set</code> objects.
	 * Each <code>String</code> object should be the name of an input or referable existential symbol in the transformation specified (as it
	 * appears in the input specification). The corresponding <code>Set</code> object should contain a collection of candidate symbols, 
	 * each of which must be of the correct type. That is, if the name corresponds to a symbol of type <code>Circle</code>, the <code>Set</code>
	 *  should only contain <code>Circle</code> objects. 
	 * <br> Any input or referable existential symbols not accounted for in the mapping will be chosen
	 * from the available symbols in the parse forest.
	 * @return <code>true</code> if the transformation was successful, <code>false</code> otherwise.
	 * @exception java.lang.IllegalArgumentException If an unknown transformation is specified, or the candidates mapping contains
	 * objects of the wrong type.
	 */
	public abstract boolean applyTransformationWithMapping(int transformation, boolean parallel, Map candidatesMap) 
		throws IllegalArgumentException;

	/**
	 * Returns a set containing all symbols in the parse forest.
	 */
	public synchronized Set getAllSymbols()
	{
		return forest.getAllSymbols();
	}

	/**
	 * Adds a terminal symbol to the parse forest. If auto forest checking is
	 * turned on, the forest will be updated as necessary with the symbols from
	 * invalid productions removed, and the symbols from any new valid productions added.
	 * If successful, an event detailing this action will be launched.
	 * @param symbol The symbol to be added.
	 * @return <code>true</code> if successful, <code>false</code> if the symbol
	 * is not a terminal symbol, or is already present in the parse forest of this or another Interpreter.
	 */
	public synchronized boolean addTerminalSymbol(GrammarSymbol symbol)
	{
		if (!symbol.isTerminal())
			return false;
		if (!forest.addTerminalSymbol(symbol))
			return false;
		if (autoForestChecking) {
			checkForest();
			evaluateProductionRules();
		}
		return true;
	}

	/**
	 * Removes a terminal symbol from the parse forest. If auto forest checking is
	 * turned on, the forest will be updated as necessary with the symbols from
	 * invalid productions removed, and the symbols from any new valid productions added.
	 * If successful, an event detailing this action will be launched.
	 * @param symbol The symbol to be removed.
	 * @return <code>true</code> if successful, <code>false</code> if the symbol
	 * is not a terminal symbol, or is not present in the parse forest of this Interpreter.
	 */
	public synchronized boolean removeTerminalSymbol(GrammarSymbol symbol)
	{
		if (!symbol.isTerminal())
			return false;

		boolean tempAFC = autoForestChecking;
		autoForestChecking = false;
		boolean success = forest.removeTerminalSymbol(symbol);
		autoForestChecking = tempAFC;
		if (!success)
			return false;

		if (autoForestChecking) {
			checkForest();
			evaluateProductionRules();
		}
		return true;
	}

	/**
	 * Removes from the parse forest the sub tree of this symbol. The method will find
	 * all terminal symbols in the parse forest that compose the symbol given, and
	 * removes them. Then, automatic parse forest checking will see all dependent
	 * non-terminal symbols removed as well. The event notification system will inform
	 * of all parse forest modifications. The method assumes that the symbol provided is
	 * in the parse forest.
	 * @param symbol The symbol to be removed.
	 * @return <code>true</code> if successful, <code>false</code> if the symbol
	 * is not present in the parse forest of this Interpreter.
	 */
	public synchronized boolean removeSubTree(GrammarSymbol symbol)
	{
		boolean tempAFC = autoForestChecking;
		autoForestChecking = false;
		boolean success = forest.removeSubTree(symbol);
		autoForestChecking = tempAFC;
		if (!success)
			return false;

		if (autoForestChecking) {
			checkForest();
			evaluateProductionRules();
		}
		return true;
	}

	/**
	 * Returns the multi-way constraint solver associated with the interpreter. 
	 * If no constraint solver is being used, <code>null</code> is returned.
	 */
	synchronized ConstraintSolver getConstraintSolver()
	{
		return solver;
	}

	/**
	 * Adds a constraint to the multi-way constraint solver.
	 * @param c The constraint to be added.
	 * @return <code>true</code> if successful, <code>false</code> if no constraint
	 * solver is present or the solver would not accept the constraint.
	 */
	synchronized boolean addConstraint(Constraint c)
	{
		// Make sure there is a solver
		if (solver == null)
			return false;

		if (solver.addConstraint(c)) {
			constraints.add(c);
			addUndoAction(new UndoAddConstraint(forest, c));
			return true;
		}
		return false;
	}

	/**
	 * Removes a constraint from the multi-way constraint solver.
	 * @param c The constraint to be removed.
	 * @return <code>true</code> if successful, <code>false</code> if no constraint
	 * solver is present or the solver could not remove the constraint.
	 */
	synchronized boolean removeConstraint(Constraint c)
	{
		// Make sure there is a solver
		if (solver == null)
			return false;

		if (constraints.contains(c)) {
			if (solver.removeConstraint(c)) {
				constraints.remove(c);
				addUndoAction(new UndoRemoveConstraint(forest, c));
				return true;
			}
		}
		return false;
	}

	// Returns the set of constraints (package scope)
	// Note, this does not include constraints used by grammar production rules
	Set getConstraints()
	{
		return constraints;
	}

	/**
	 * Calls the constraint solver resolve method. Waits for all symbol updates before checking forest.
	 * Note that it will only work if auto constraint checking is turned on. 
	 * <br> It is unlikely that this method will need to be called explicitly.
	 */
	public void resolve()
	{
		if (solver == null || !autoConstraintSolving)
			return;

		boolean tempAFC = autoForestChecking;
		autoForestChecking = false;
		solver.resolve();
		autoForestChecking = tempAFC;

		if (autoForestChecking) {
			checkForest();
			evaluateProductionRules();
		}
	}

	/**
	 * Calls the solver to solve all constraints. This method should be called
	 * if constraints have been added or removed. Note that it will only work if 
	 * auto constraint checking is turned on.
	 * <br> It is unlikely that this method will need to be called explicitly.
	 */
	public void solve()
	{
		if (solver == null || !autoConstraintSolving)
			return;

		boolean tempAFC = autoForestChecking;
		autoForestChecking = false;
		solver.solve();
		autoForestChecking = tempAFC;

		if (autoForestChecking) {
			checkForest();
			evaluateProductionRules();
		}
	}

	/**
	 * Prints the parse forest. Primarily for debugging purposes.
	 * @param stream The output stream where the text is sent to.
	 */
	public synchronized void printForest(PrintStream stream)
	{
		forest.print(stream);
		if (isUpToDate()) 
			stream.println("Parse Forest is up to date");
		else
			stream.println("Parse Forest is NOT up to date");
	}

	/**
	 * Returns <code>true</code> if the parse forest is up to date, <code>false</code> otherwise.
	 */
	public synchronized boolean isUpToDate()
	{
		Iterator iter = forest.getProductionNodes().iterator();
		while (iter.hasNext()) {
			if (!((ProductionNode) iter.next()).getRecord().isUpToDate())
				return false;
		}
		return true;
	}

	/**
	 * Adds an listener to listen for grammar symbol events.
	 * Events are launched when symbols are added to or removed from the parse forest,
	 * and when their attributes are modified.
	 * @param gsl The listener to be added.
	 */
	public synchronized void addGrammarSymbolListener(GrammarSymbolListener gsl)
	{
		listeners.add(gsl);
	}

	/**
	 * Removes a listener that listens for grammar symbol events.
	 * Events are launched when symbols are added to or removed from the parse forest,
	 * and when their attributes are modified.
	 * @param gsl The listener to be removed.
	 */
	public synchronized void removeGrammarSymbolListener(GrammarSymbolListener gsl)
	{
		listeners.remove(gsl);
	}

	/** 
	 * Returns <code>true</code> if auto forest checking is on, <code>false</code> otherwise.
	 * Auto forest checking means that each change that is made to the forest (via a symbol
	 * being added, removed or modified), will result in checks to ensure that the forest
	 * is in a valid state and up to date. 
	 */
	public synchronized boolean isAutoForestCheckingOn()
	{
		return autoForestChecking;
	}

	/** 
	 * Allows auto forest checking to be turned on or off.
	 * Auto forest checking means that each change that is made to the forest (via a symbol
	 * being added, removed or modified), will result in checks to ensure that the forest
	 * is in a valid state and up to date. It should normally be turned on.
	 * @param on <code>true</code> to turn on, <code>false</code> to turn off.
	 */
	public synchronized void setAutoForestChecking(boolean on)
	{
		autoForestChecking = on;
	}

	/** 
	 * Returns <code>true</code> if auto constraint solving is on, <code>false</code> otherwise.
	 * Auto constraint solving means that the multi-way constraint solver will automatically
	 * enforce any constraints that it is responsible for, ensuring that the parse forest remains in a valid state.
	 */
	public synchronized boolean isAutoConstraintSolvingOn()
	{
		return autoConstraintSolving;
	}

	/** 
	 * Allows auto constraint solving to be turned on or off.
	 * Auto constraint solving means that the multi-way constraint solver will automatically
	 * enforce any constraints that it is responsible for, ensuring that the parse forest remains in a valid state.
	 * It should normally be turned on.
	 * @param on <code>true</code> to turn on, <code>false</code> to turn off.
	 */
	public synchronized void setAutoConstraintSolving(boolean on)
	{
		autoConstraintSolving = on;
	}

	// Launches a symbol added event
	protected void launchSymbolAddedEvent(GrammarSymbol symbol)
	{
		GrammarSymbolEvent event = new GrammarSymbolEvent(this, symbol, GrammarSymbolEvent.SYMBOL_ADDED);
		Iterator iter = listeners.iterator();
		while (iter.hasNext())
			((GrammarSymbolListener) iter.next()).symbolAdded(event);
	}

	// Launches a symbol removed event
	protected void launchSymbolRemovedEvent(GrammarSymbol symbol)
	{
		GrammarSymbolEvent event = new GrammarSymbolEvent(this, symbol, GrammarSymbolEvent.SYMBOL_REMOVED);
		Iterator iter = listeners.iterator();
		while (iter.hasNext())
			((GrammarSymbolListener) iter.next()).symbolRemoved(event);
	}

	// Launches a symbol modified event
	protected void launchSymbolModifiedEvent(GrammarSymbol symbol)
	{
		GrammarSymbolEvent event = new GrammarSymbolEvent(this, symbol, GrammarSymbolEvent.SYMBOL_MODIFIED);
		Iterator iter = listeners.iterator();
		while (iter.hasNext())
			((GrammarSymbolListener) iter.next()).symbolModified(event);
	}

	/**
	 * Checks all productions and updates all attributes in the parse forest that require it.
	 * If auto forest checking is turned on, this method will be called automatically when needed.
	 */
	public synchronized void checkForest()
	{
		Iterator iter;
		boolean updateRecheck, reevaluateRecheck;

		// Make sure forest is not already being checked
		if (forestBeingChecked)
			return;

		forestBeingChecked = true;
		do {
			reevaluateRecheck = false;
			do {
				updateRecheck = false;

				// Check all attribute updates
				iter = forest.getProductionNodes().iterator();
				while (iter.hasNext()) {
					ProductionNode pNode = (ProductionNode) iter.next();
					if (pNode.getRecord().updateSymbolAttributes(this)) 
						updateRecheck = true;
				}
			} while (updateRecheck);

			// Check all reevaluations
			iter = forest.getProductionNodes().iterator();
			while (iter.hasNext()) {
				ProductionNode pNode = (ProductionNode) iter.next();
				if (!pNode.getRecord().reevaluateProduction(forest)) {
					forest.removeProduction(pNode.getRecord());
					reevaluateRecheck = true;
					break;
				}
			}
		} while (reevaluateRecheck);
		forestBeingChecked = false;
	}

	/**
	 * Reevaluates all productions in the parse forest.
	 */
	public synchronized void reevaluateForest()
	{
		Iterator iter = forest.getProductionNodes().iterator();
		while (iter.hasNext()) {
			ProductionNode pNode = (ProductionNode) iter.next();
			pNode.getRecord().fullRecheck();
		}
		checkForest();
		evaluateProductionRules();
	}

	// Informs parse forest that attributes have been modified (package scope)
	void symbolModified(GrammarSymbol symbol, int attribute, Object oldValue)
	{
		forest.symbolModified(symbol, attribute, oldValue);
		launchSymbolModifiedEvent(symbol);
		if (autoForestChecking) {
			checkForest();
			evaluateProductionRules();
		}
	}

	/**
	 * Implemented for an internally used interface.
	 * Listens for when a constraint variable has been modified by the multi-way constraint solver.
	 * This method should not be called directly.
	 */
	public synchronized void variableModified(ConstraintVariableEvent event)
	{
		ConstraintVariable var = event.getConstraintVariable();
		GrammarSymbol symbol = forest.getSymbol(var);

		// If symbol not relevant to the parse forest, ignore
		if (symbol == null)
			return;

		int identifier = symbol.getAttributeIdentifier(var);
		Log.getLogger().info("Symbol modified by constraint solver (attribute: " + symbol.getAttributeName(identifier) + "): " + 
			symbol.toString());
		symbolModified(symbol, identifier, event.getOldValue());
	}

	// Adds an action to the undo list (if undo is enabled)
	void addUndoAction(UndoRecord ur)
	{
		if (undoEnabled) 
			undoList.addLast(ur);
	}

	/**
	 * Returns <code>true</code> if the undo feature is currently enabled, <code>false</code> otherwise.
	 */
	public synchronized boolean isUndoEnabled()
	{
		return undoEnabled;
	}

	/**
	 * Turns on the ability to undo actions. Since storing user actions takes up 
	 * memory, the undo feature should only be enabled when necessary.
	 */
	public synchronized void enableUndo()
	{
		undoEnabled = true;
	}

	/**
	 * Turns off the ability to undo actions. 
	 */
	public synchronized void disableUndo()
	{
		undoEnabled = false;
		undoList.clear();
	}

	/**
	 * Returns the index to the most recent action recorded on the undo action list.
	 * Then, if further actions are carried out, it will possible to later undo all actions
	 * that occurred after the one whose index has been obtained, without going any further back
	 * (using the {@link #undoActions} method).
	 */
	public synchronized int getUndoIndex()
	{
		return undoList.size();
	}

	/**
	 * Will undo actions that occurred after a specific time, as indicated by the index value.
	 * Note that the undo feature allows for the fact that constraint variables may have been changed
	 * by the multi-way constraint solver.
	 * @param index The index in the undo action list (obtained from {@link #getUndoIndex}).
	 * @return <code>true</code> if successful, <code>false</code> if undo is not enabled,
	 * or an invalid index value is given.
	 */
	public synchronized boolean undoActions(int index)
	{
		if (!undoEnabled || index < 0) 
			return false;

		boolean tempAFC = autoForestChecking;
		boolean tempACS = autoConstraintSolving;
		autoForestChecking = false;
		autoConstraintSolving = false;
		undoEnabled = false;

		Log.getLogger().fine("About to commence undo actions");
		while (undoList.size() > index) {
			UndoRecord ur = (UndoRecord) undoList.removeLast();
			ur.undo();
		}
		Log.getLogger().fine("About to resolve after undo actions");
		if (solver != null)
			solver.resolve();
		Log.getLogger().fine("Completed undo actions");

		autoForestChecking = tempAFC;
		autoConstraintSolving = tempACS;
		undoEnabled = true;
		return true;
	}
}

