/*
    Copyright (C) 2003  Anthony R. Jansen.

    This file is part of the CIDER Toolkit.

    The CIDER Toolkit is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    The CIDER Toolkit is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with the CIDER Toolkit; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

// ----------------------------------------
// CIDER Toolkit: Interpreter Package
//
// Author:  Anthony R. Jansen
// Begun:   September 2002
// Class:   ParseForest
//
// This class represents a Parse Forest 
// which conatins sets of parse tree nodes,
// distinguishing between nodes that are
// root nodes in the parse tree, and those
// that have already been reduced by the 
// application of a production rule.
// ----------------------------------------

package au.edu.monash.csse.tonyj.cider.interpreter;

import au.edu.monash.csse.tonyj.cider.constraints.Constraint;
import au.edu.monash.csse.tonyj.cider.constraints.ConstraintVariable;
import au.edu.monash.csse.tonyj.cider.constraints.ConstraintSolver;
import java.io.PrintStream;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.Set;
import java.util.TreeSet;
import java.util.HashSet;
import java.util.Map;
import java.util.HashMap;

class ParseForest {

	// Sets for the symbol nodes
	private SymbolNodeSets	newRootSymbolNodes;
	private SymbolNodeSets	oldRootSymbolNodes;
	private SymbolNodeSets	allRootSymbolNodes;
	private SymbolNodeSets	newReducedSymbolNodes;
	private SymbolNodeSets	oldReducedSymbolNodes;
	private SymbolNodeSets	allReducedSymbolNodes;
	private SymbolNodeSets	allNewSymbolNodes;
	private SymbolNodeSets	allOldSymbolNodes;
	private SymbolNodeSets	allSymbolNodes;
	private Set		symbolsAddedSinceUpdate;
	private Set		symbolsRemovedSinceUpdate;
	private Set		productionNodes;
	private Map		constraintVariables;

	// Non-serializable variable
	transient private Interpreter	interpreter;

	// Constructor
	public ParseForest(Interpreter inter, Class snsClass)
	{
		interpreter = inter;
		try {
			newRootSymbolNodes = (SymbolNodeSets) snsClass.newInstance();
			oldRootSymbolNodes = (SymbolNodeSets) snsClass.newInstance();
			allRootSymbolNodes = (SymbolNodeSets) snsClass.newInstance();
			newReducedSymbolNodes = (SymbolNodeSets) snsClass.newInstance();
			oldReducedSymbolNodes = (SymbolNodeSets) snsClass.newInstance();
			allReducedSymbolNodes = (SymbolNodeSets) snsClass.newInstance();
			allNewSymbolNodes = (SymbolNodeSets) snsClass.newInstance();
			allOldSymbolNodes = (SymbolNodeSets) snsClass.newInstance();
			allSymbolNodes = (SymbolNodeSets) snsClass.newInstance();
		}
		catch (Exception e) {
			Log.getLogger().severe("Could not create SymbolNodeSets objects");
			Error.die("Could not create SymbolNodeSets objects");
		}
		productionNodes = new TreeSet(new ProductionNodeComparator());
		constraintVariables = new HashMap();
		symbolsAddedSinceUpdate = new HashSet();
		symbolsRemovedSinceUpdate = new HashSet();
	}

	// Sets the interpreter - used when a serialized object is restored
	public void setInterpreter(Interpreter inter)
	{
		interpreter = inter;
		allSymbolNodes.setInterpreter(interpreter);
	}

	// Returns the interpreter associated with the parse forest
	public Interpreter getInterpreter()
	{
		return interpreter;
	}

	// Returns a set containing all the symbols contained in the forest
	public Set getAllSymbols()
	{
		return allSymbolNodes.getSymbols();
	}

	// This method makes all nodes listed as old become listed as new, and
	// also clears the symbols added/removed since last update lists
	public void update()
	{
		oldRootSymbolNodes.addAll(newRootSymbolNodes);
		oldReducedSymbolNodes.addAll(newReducedSymbolNodes);
		allOldSymbolNodes.addAll(allNewSymbolNodes);
		newRootSymbolNodes.clear();
		newReducedSymbolNodes.clear();
		allNewSymbolNodes.clear();
		symbolsAddedSinceUpdate.clear();
		symbolsRemovedSinceUpdate.clear();
	}

	// Returns the symbols added since the last update
	public Set getSymbolsAddedSinceUpdate()
	{
		return symbolsAddedSinceUpdate;
	}

	// Returns the symbols removed since the last update
	public Set getSymbolsRemovedSinceUpdate()
	{
		return symbolsRemovedSinceUpdate;
	}

	// Get methods for all symbol node sets
	public SymbolNodeSets getNewRootSymbolNodes()
	{
		return newRootSymbolNodes;
	}
 
	public SymbolNodeSets getOldRootSymbolNodes()
	{
		return oldRootSymbolNodes;
	}
 
	public SymbolNodeSets getAllRootSymbolNodes()
	{
		return allRootSymbolNodes;
	}
 
	public SymbolNodeSets getNewReducedSymbolNodes()
	{
		return newReducedSymbolNodes;
	}
 
	public SymbolNodeSets getOldReducedSymbolNodes()
	{
		return oldReducedSymbolNodes;
	}
 
	public SymbolNodeSets getAllReducedSymbolNodes()
	{
		return allReducedSymbolNodes;
	}
 
	public SymbolNodeSets getAllNewSymbolNodes()
	{
		return allNewSymbolNodes;
	}
 
	public SymbolNodeSets getAllOldSymbolNodes()
	{
		return allOldSymbolNodes;
	}
 
	public SymbolNodeSets getAllSymbolNodes()
	{
		return allSymbolNodes;
	}
 
	// Returns the set of all production nodes
	public Set getProductionNodes()
	{
		return productionNodes;
	}

	// Empties the parse forest of all symbols
	public void clear()
	{
		newRootSymbolNodes.clear();
		oldRootSymbolNodes.clear();
		allRootSymbolNodes.clear();
		newReducedSymbolNodes.clear();
		oldReducedSymbolNodes.clear();
		allReducedSymbolNodes.clear();
		allNewSymbolNodes.clear();
		allOldSymbolNodes.clear();
		allSymbolNodes.clear();
		symbolsAddedSinceUpdate.clear();
		symbolsRemovedSinceUpdate.clear();
		productionNodes.clear();
		constraintVariables.clear();
	}

	// Returns the symbol associated with a specific constraint variable, or null if there is none
	public GrammarSymbol getSymbol(ConstraintVariable var)
	{
		return (GrammarSymbol) constraintVariables.get(var);
	}

	// This method receives the record of a production that has produced
	// new symbols to be added to the parse forest. It assumes that:
	// 1) the new symbols will be root symbols
	// 2) the new symbols are siblings (if there is more than one)
	// 3) any children of the new symbols are already in the parse forest
	// 4) any children of the new symbols are currently root symbols
	public boolean addProduction(ProductionRecord record)
	{
		// Add constraint variables to mapping
		Iterator i = record.getOutputSymbols().iterator();
		while (i.hasNext()) {
			GrammarSymbol symbol = (GrammarSymbol) i.next();
			Iterator i2 = symbol.getConstraintVariables().iterator();
			while (i2.hasNext())
				constraintVariables.put(i2.next(), symbol);
		}

		// Create required constraints
		if (!record.addSolverConstraints(interpreter)) {

			// If constraints could not be added, remove constraints,
			// remove constraint variables from mapping and return false
			record.removeSolverConstraints(interpreter);
			i = record.getOutputSymbols().iterator();
			while (i.hasNext()) {
				GrammarSymbol symbol = (GrammarSymbol) i.next();
				Iterator i2 = symbol.getConstraintVariables().iterator();
				while (i2.hasNext())
					constraintVariables.remove(i2.next());
			}
			return false;
		}

		// Determine parse tree level for new symbols
		int maxLevel = 0;
		Set childNodes = record.getInputSymbolNodes();
		i = childNodes.iterator();
		while (i.hasNext()) {
			SymbolNode child = (SymbolNode) i.next();
			if (child.getLevel() > maxLevel)
				maxLevel = child.getLevel();
		}

		// Create Production Node 
		ProductionNode pNode = new ProductionNode(record, maxLevel + 1);

		// Create Symbol Nodes for each new symbol
		Set newSymbols = record.getOutputSymbols();
		i = newSymbols.iterator();
		while (i.hasNext()) {
			GrammarSymbol symbol = (GrammarSymbol) i.next();
			SymbolNode sNode = new SymbolNode(symbol, pNode, maxLevel + 1, interpreter);

			// The node for each new symbol is added to the parse tree as a root node
			newRootSymbolNodes.add(sNode);
			allNewSymbolNodes.add(sNode);
			allRootSymbolNodes.add(sNode);
			allSymbolNodes.add(sNode);

			symbolsAddedSinceUpdate.add(symbol.getClass());
		}

		// Add production node as dependency for each existential symbol
		i = record.getReferableExistentialSymbolNodes().iterator();
		while (i.hasNext()) {
			SymbolNode sNode = (SymbolNode) i.next();
			sNode.addDependency(pNode);
		}

		// All child symbols of the new symbols are moved from 
		// being root symbols to being reduced symbols 
		i = childNodes.iterator();
		while (i.hasNext()) {
			SymbolNode child = (SymbolNode) i.next();
			child.setParent(pNode);
			if (newRootSymbolNodes.contains(child)) {
				newRootSymbolNodes.remove(child);
				newReducedSymbolNodes.add(child);
			}
			else {
				oldRootSymbolNodes.remove(child);
				oldReducedSymbolNodes.add(child);
			}
			allRootSymbolNodes.remove(child);
			allReducedSymbolNodes.add(child);
		}

		// Inform forest that new symbols have been added
		i = newSymbols.iterator();
		while (i.hasNext()) {
			GrammarSymbol s = (GrammarSymbol) i.next();
			Log.getLogger().info("Non terminal symbol added to parse forest: " + s.toString());
			interpreter.launchSymbolAddedEvent(s);
			symbolAdded(s);
		}
		productionNodes.add(pNode);

		// If undo is enabled, record this
		interpreter.addUndoAction(new UndoAddProduction(this, record));

		return true;
	}

	// This method adds a terminal symbol to the parse forest. It assumes that:
	// 1) The symbol being added is a terminal symbol 
	public boolean addTerminalSymbol(GrammarSymbol symbol)
	{
		// Make sure symbol is not already in a parse forest
		if (symbol.isRegistered())
			return false;

		// Create symbol node and add symbol as a root symbol 
		SymbolNode sNode = new SymbolNode(symbol, null, 0, interpreter);

		// Add constraint variables to mapping
		Iterator i = symbol.getConstraintVariables().iterator();
		while (i.hasNext())
			constraintVariables.put(i.next(), symbol);

		newRootSymbolNodes.add(sNode);
		allNewSymbolNodes.add(sNode);
		allRootSymbolNodes.add(sNode);
		allSymbolNodes.add(sNode);

		symbolsAddedSinceUpdate.add(symbol.getClass());
		Log.getLogger().info("Terminal symbol added to parse forest: " + symbol.toString());
		interpreter.launchSymbolAddedEvent(symbol);
		symbolAdded(symbol);

		// If undo is enabled, record this
		interpreter.addUndoAction(new UndoAddTerminalSymbol(this, symbol));

		return true;
	}

	// This method will remove a production from a parse tree, after (if required) 
	// recursively removing all ancestors of the production. It assumes that:
	// 1) The production to be removed is in the parse forest
	// 2) All symbols related to the production are in the parse forest
	// 3) All symbols related to the production are correctly categorized as either root or reduced
	public void removeProduction(ProductionRecord record)
	{
		ProductionNode pNode = record.getProductionNode();
		productionNodes.remove(pNode);

		// Remove constraints
		record.removeSolverConstraints(interpreter);
		record.reinitializeInputSymbols(interpreter);

		// Check each of the output symbols
		Iterator i = pNode.getOutputSymbolNodes().iterator();
		while (i.hasNext()) {
			SymbolNode sNode = (SymbolNode) i.next();

			// Remove each production that is a dependency via an existential symbol
			while (!sNode.getDependents().isEmpty()) {
				Iterator i2 = sNode.getDependents().iterator();
				ProductionNode dependentNode = (ProductionNode) i2.next();
				removeProduction(dependentNode.getRecord());
			}

			// If any of the output symbols are used in further productions, 
			// remove these productions and make the output symbols root symbols
			if (sNode.usedBy() != null) {
				removeProduction(sNode.usedBy().getRecord());
				sNode.setParent(null);
				if (newRootSymbolNodes.contains(sNode)) {
					newReducedSymbolNodes.remove(sNode);
					newRootSymbolNodes.add(sNode);
				}
				else {
					oldReducedSymbolNodes.remove(sNode);
					oldRootSymbolNodes.add(sNode);
				}
				allReducedSymbolNodes.remove(sNode);
				allRootSymbolNodes.add(sNode);
			}

			// Remove output symbols, which should now be root symbols
			if (newRootSymbolNodes.contains(sNode)) {
				newRootSymbolNodes.remove(sNode);
				allNewSymbolNodes.remove(sNode);
			}
			else {
				oldRootSymbolNodes.remove(sNode);
				allOldSymbolNodes.remove(sNode);
			}
			allRootSymbolNodes.remove(sNode);
			allSymbolNodes.remove(sNode);

			symbolsRemovedSinceUpdate.add(sNode.getSymbol().getClass());
			Log.getLogger().info("Non terminal symbol removed from parse forest: " + sNode.getSymbol().toString());
			interpreter.launchSymbolRemovedEvent(sNode.getSymbol());
			symbolRemoved(sNode.getSymbol());
			sNode.getSymbol().deregisterSymbol();

			// Remove constraint variables from mapping
			Iterator i2 = sNode.getSymbol().getConstraintVariables().iterator();
			while (i2.hasNext())
				constraintVariables.remove(i2.next());
		}

		// Remove production node as dependency for each existential symbol
		i = record.getReferableExistentialSymbolNodes().iterator();
		while (i.hasNext()) {
			SymbolNode sNode = (SymbolNode) i.next();
			sNode.removeDependency(pNode);
		}

		// Move all children from being reduced symbols to being
		// root symbols in the parse forest, with no parent symbol
		Set children = pNode.getInputSymbolNodes();
		i = children.iterator();
		while (i.hasNext()) {
			SymbolNode sNode = (SymbolNode) i.next();
			sNode.setParent(null);
			if (newRootSymbolNodes.contains(sNode)) {
				newReducedSymbolNodes.remove(sNode);
				newRootSymbolNodes.add(sNode);
			}
			else {
				oldReducedSymbolNodes.remove(sNode);
				oldRootSymbolNodes.add(sNode);
			}
			allReducedSymbolNodes.remove(sNode);
			allRootSymbolNodes.add(sNode);
		}

		// If undo is enabled, record this
		interpreter.addUndoAction(new UndoRemoveProduction(this, record));
	}

	// This method will remove a terminal symbol from a parse tree, after (if required) 
	// recursively removing all of its ancestors. It assumes that:
	// 1) All ancestors of the symbol are in the parse forest
	// 2) All ancestors are correctly categorized as either root or reduced
	// 3) The symbol being removed is a terminal symbol
	public boolean removeTerminalSymbol(GrammarSymbol symbol)
	{
		// Check if symbol is in this parse forest
		if (!(symbol.isRegistered() && symbol.getSymbolNode().getInterpreter() == interpreter))
			return false;

		// Remove each production that is a dependency via an existential symbol
		while (!symbol.getSymbolNode().getDependents().isEmpty()) {
			Iterator i2 = symbol.getSymbolNode().getDependents().iterator();
			ProductionNode dependentNode = (ProductionNode) i2.next();
			removeProduction(dependentNode.getRecord());
		}

		// If a symbol is used in a production, remove this first
		if (symbol.getSymbolNode().usedBy() != null)
			removeProduction(symbol.getSymbolNode().usedBy().getRecord());

		// The symbol node should now be a root node
		// Remove symbol node from the parse forest
		if (newRootSymbolNodes.contains(symbol.getSymbolNode())) {
			newRootSymbolNodes.remove(symbol.getSymbolNode());
			allNewSymbolNodes.remove(symbol.getSymbolNode());
		}
		else {
			oldRootSymbolNodes.remove(symbol.getSymbolNode());
			allOldSymbolNodes.remove(symbol.getSymbolNode());
		}
		allRootSymbolNodes.remove(symbol.getSymbolNode());
		allSymbolNodes.remove(symbol.getSymbolNode());

		symbolsRemovedSinceUpdate.add(symbol.getClass());
		Log.getLogger().info("Terminal symbol removed from parse forest: " + symbol.toString());
		interpreter.launchSymbolRemovedEvent(symbol);
		symbolRemoved(symbol);
		symbol.deregisterSymbol();

		// If undo is enabled, record this
		interpreter.addUndoAction(new UndoRemoveTerminalSymbol(this, symbol));

		// Remove constraint variables from mapping
		Iterator i = symbol.getConstraintVariables().iterator();
		while (i.hasNext())
			constraintVariables.remove(i.next());

		return true;
	}

	// This method removes a subtree from the parse forest by finding
	// the leaf nodes of the given symbol, and removing them. It assumes that:
	// 1) All ancestors and descendants of the symbol are in the parse forest
	public boolean removeSubTree(GrammarSymbol symbol)
	{
		// Check if symbol is in this parse forest
		if (!(symbol.isRegistered() && symbol.getSymbolNode().getInterpreter() == interpreter))
			return false;

		// If a terminal symbol, remove and return
		if (symbol.isTerminal())
			return removeTerminalSymbol(symbol);

		// Otherwise, find child symbols
		Set childSymbols = new HashSet();
		ProductionNode pNode = symbol.getSymbolNode().producedBy();
		Iterator iter = pNode.getInputSymbolNodes().iterator();
		while (iter.hasNext()) {
			SymbolNode childNode = (SymbolNode) iter.next();
			childSymbols.add(childNode.getSymbol());
		}

		// Remove child symbol subtrees
		iter = childSymbols.iterator();
		while (iter.hasNext()) {
			GrammarSymbol child = (GrammarSymbol) iter.next();
			removeSubTree(child);
		}
		return true;
	}

	// This method returns true iff the candidate symbol is in the subtree of the subtree symbol 
	// It assumes that both symbols are in the parse forest
	public boolean inSubTree(GrammarSymbol candidateSymbol, GrammarSymbol subtreeSymbol)
	{
		// Check if the same symbol
		if (candidateSymbol.equals(subtreeSymbol))
			return true;

		// Check for terminal symbol
		if (subtreeSymbol.isTerminal())
			return false;

		// Find child symbols of subtree symbol
		ProductionNode pNode = subtreeSymbol.getSymbolNode().producedBy();
		Iterator iter = pNode.getInputSymbolNodes().iterator();
		while (iter.hasNext()) {
			SymbolNode childNode = (SymbolNode) iter.next();

			// Recursively check for each child of the subtree symbol
			if (inSubTree(candidateSymbol, childNode.getSymbol()))
				return true;
		}
		return false;
	}

	// Prints out all the productions, listing which symbols were involved
	public void print(PrintStream stream)
	{
		stream.println("*** Parse Forest State ***");
		stream.println("");
		stream.println("Root Symbols");
		stream.println("------------");
		allRootSymbolNodes.print(stream);
		stream.println("");
		stream.println("Reduced Symbols");
		stream.println("---------------");
		allReducedSymbolNodes.print(stream);
		stream.println("");
		stream.println("Productions");
		stream.println("-----------");
		Iterator iter = getProductionNodes().iterator();
		while (iter.hasNext()) {
			ProductionNode pNode = (ProductionNode) iter.next();
			pNode.getRecord().print(stream);
			stream.println("-----------");
		}
	}

	// Informs all production nodes that a symbol has been added to the forest
	public void symbolAdded(GrammarSymbol symbol)
	{
		Iterator iter = productionNodes.iterator();
		while (iter.hasNext()) {
			ProductionNode pNode = (ProductionNode) iter.next();
			pNode.getRecord().symbolAdded(symbol);
		}
	}

	// Informs all production nodes that a symbol has been removed from the forest
	// Also removes any other interpreter constraints that depend on one of the
	// constraint variables used by this symbol
	public void symbolRemoved(GrammarSymbol symbol)
	{
		Set toBeRemoved = new HashSet();

		Iterator iter = interpreter.getConstraints().iterator();
		while (iter.hasNext()) {
			Constraint c = (Constraint) iter.next();

			Iterator iter2 = symbol.getConstraintVariables().iterator();
			while (iter2.hasNext()) {
				if (c.getConstraintVariables().contains(iter2.next())) {
					toBeRemoved.add(c);
					break;
				}
			}
		}
		iter = toBeRemoved.iterator();
		while (iter.hasNext())
			interpreter.removeConstraint((Constraint) iter.next());

		iter = productionNodes.iterator();
		while (iter.hasNext()) {
			ProductionNode pNode = (ProductionNode) iter.next();
			pNode.getRecord().symbolRemoved(symbol);
		}
	}

	// Informs all production nodes that a symbol in the forest has been modified
	public void symbolModified(GrammarSymbol symbol, int attribute, Object oldValue)
	{
		// If undo is enabled, record this
		interpreter.addUndoAction(new UndoAttributeModification(symbol, attribute, oldValue));

		Iterator iter = productionNodes.iterator();
		while (iter.hasNext()) {
			ProductionNode pNode = (ProductionNode) iter.next();
			pNode.getRecord().symbolModified(symbol, attribute);
		}
	}
}

