/*
    Copyright (C) 2003  Anthony R. Jansen.

    This file is part of the CIDER Toolkit.

    The CIDER Toolkit is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    The CIDER Toolkit is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with the CIDER Toolkit; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

// ----------------------------------------
// CIDER Toolkit: Interpreter Package
//
// Author:  Anthony R. Jansen
// Begun:   September 2002
// Class:   ProductionRecord
//
// This is the abstract base class used by
// all the classes that define a Production
// Record for a specific production rule.
// The specific production rule classes are
// produced by the Compiler program.
// ----------------------------------------

package au.edu.monash.csse.tonyj.cider.interpreter;

import java.io.PrintStream;
import java.util.Set;

abstract class ProductionRecord {

	// The associated production node in the parse tree
	private ProductionNode	node;
	private int		id_number;

	// Static variable for ID numbers
	private static int ids = 1;

	// Constructor
	public ProductionRecord()
	{
		node = null;
		synchronized(this) {
			id_number = ids;
			ids++;
		}
	}

	// Returns the id number
	int getIDNumber()
	{
		return id_number;
	}

	// Sets the production node
	public void setProductionNode(ProductionNode n)
	{	
		node = n;
	}

	// Returns the production node
	public ProductionNode getProductionNode()
	{
		return node;
	}

	// Abstract methods that must be implemented by sub-classes
	public abstract boolean reevaluateProduction(ParseForest forest);
	public abstract boolean updateSymbolAttributes(Interpreter interpreter);
	public abstract boolean isUpToDate();
	public abstract void fullRecheck();
	public abstract void print(PrintStream stream);
	public abstract Set getInputSymbolNodes();
	public abstract Set getReferableExistentialSymbolNodes();
	public abstract Set getOutputSymbolNodes();
	public abstract Set getOutputSymbols();
	public abstract void symbolAdded(GrammarSymbol symbol);
	public abstract void symbolRemoved(GrammarSymbol symbol);
	public abstract void symbolModified(GrammarSymbol symbol, int attribute);
	public abstract boolean addSolverConstraints(Interpreter interpreter);
	public abstract void removeSolverConstraints(Interpreter interpreter);
	public abstract void reinitializeInputSymbols(Interpreter interpreter);
}

