/*
    Copyright (C) 2003  Anthony R. Jansen.

    This file is part of the CIDER Toolkit.

    The CIDER Toolkit is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    The CIDER Toolkit is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with the CIDER Toolkit; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

// ----------------------------------------
// CIDER Toolkit: Interpreter Package
//
// Author:  Anthony R. Jansen
// Begun:   September 2002
// Class:   SymbolNode
//
// This class represents a symbol node in
// the parse forest. Symbol nodes link to
// production nodes in a parse tree, which
// in turn link to other symbol nodes.
// ----------------------------------------

package au.edu.monash.csse.tonyj.cider.interpreter;

import java.util.Set;
import java.util.HashSet;

class SymbolNode {

	// Private variables
	private ProductionNode	parent;
	private ProductionNode	child;
	private Set		dependents;
	private GrammarSymbol	symbol;
	private int		level;

	// Non-serializable variable
	transient private Interpreter	interpreter;

	// Constructor
	public SymbolNode(GrammarSymbol s, ProductionNode node, int l, Interpreter inter)
	{
		interpreter = inter;
		symbol = s;
		child = node;
		parent = null;
		dependents = new HashSet();
		level = l;
		symbol.registerSymbol(this);
	}

	// Returns the symbol associated with this node
	public GrammarSymbol getSymbol()
	{
		return symbol;
	}

	// Allows the parent variable to be set
	public void setParent(ProductionNode node)
	{
		parent = node;
	}

	// Returns the production node that the symbol in this node was produced by
	public ProductionNode producedBy()
	{
		return child;
	}

	// Returns the production node that uses the symbol in the node to create another symbol
	public ProductionNode usedBy()
	{
		return parent;
	}

	// Adds a production node that depends on this symbol as an existential member
	public void addDependency(ProductionNode pNode)
	{
		dependents.add(pNode);
	}

	// Removes a production node that no longer depends on this symbol as an existential member
	public void removeDependency(ProductionNode pNode)
	{
		dependents.remove(pNode);
	}

	// Returns the set of production nodes that depend on this symbol as an existential member
	public Set getDependents()
	{
		return dependents;
	}

	// Returns the level of the node from its furthest terminal node
	public int getLevel()
	{
		return level;
	}

	// Sets the interpreter - used when a serialized object is restored
	public void setInterpreter(Interpreter inter)
	{
		interpreter = inter;
	}

	// Returns the interpreter
	public Interpreter getInterpreter()
	{
		return interpreter;
	}

	// Log message when finalized
	public void finalize() throws Throwable
	{
		Log.getLogger().finest("Symbol Node finalized");
	}
}

