/*
    Copyright (C) 2003  Anthony R. Jansen.

    This file is part of the CIDER Toolkit.

    The CIDER Toolkit is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    The CIDER Toolkit is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with the CIDER Toolkit; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

// ----------------------------------------
// CIDER Toolkit: Layout Package
//
// Author:  Anthony R. Jansen
// Begun:   February 2004
// Class:   ForceDirectedGraph
//
// The class that manipulates the force
// directed graph used in laying out the
// diagram.
// ----------------------------------------

package au.edu.monash.csse.tonyj.cider.layout;

import java.util.Vector;
import java.util.Iterator;
import javax.swing.JComponent;
import java.awt.event.ActionListener;
import java.awt.event.ActionEvent;
import au.edu.monash.csse.tonyj.cider.interpreter.Interpreter;
import au.edu.monash.csse.tonyj.cider.interpreter.GrammarSymbol;
import au.edu.monash.csse.tonyj.cider.interpreter.GrammarSymbolListener;
import au.edu.monash.csse.tonyj.cider.interpreter.GrammarSymbolEvent;

public class ForceDirectedGraph implements GrammarSymbolListener {

	// Private variables
	private	int			count;
	private ForceDirectedSystem	layoutSystem;

	private static final int MAX_COUNT = 200;

	// Constructor
	public ForceDirectedGraph()
	{
		layoutSystem = new ForceDirectedSystem();
	}

	// Set the parameters for the system
	public void setParameters(double sdd, double prc, double sc)
	{
		layoutSystem.setParameters(sdd, prc, sc);
	}

	// Clears the system of all nodes and edges
	public void clear()
	{
		layoutSystem.clear();
	}

	// Fully adjusts the graph without any visual manipulation
	public void applyNonVisualLayout()
	{
		count = 0;
		while (layoutSystem.adjustNodes() && count < MAX_COUNT)
			count++; 
	}

	// Adjusts the system incrementally, visually updating as it goes
	// The visual updating is all done within the one thread
	public void applyVisualLayout(Interpreter interpreter, int timeStep)
	{
		JComponent jc = interpreter.getJComponent();

		// Check JComponent is not null
		if (jc == null) {
			applyNonVisualLayout();
			return;
		}

		count= 0;
		long nextTime = System.currentTimeMillis() + timeStep;

		boolean tempACS = interpreter.isAutoConstraintSolvingOn();
		while (count < MAX_COUNT) {

			interpreter.setAutoConstraintSolving(false);
			boolean adjust = layoutSystem.adjustNodes();
			interpreter.setAutoConstraintSolving(true);
			interpreter.resolve();
			jc.paintImmediately(0, 0, jc.getWidth(), jc.getHeight());
			if (!adjust)
				break;

			count++;
			nextTime = Math.max(System.currentTimeMillis(), nextTime + timeStep);
			while (System.currentTimeMillis() < nextTime)
				Thread.yield();
		}
		interpreter.setAutoConstraintSolving(tempACS);
	}

	// Needed to listen for when GraphNodeInts and GraphEdgeInts are added
	public void symbolAdded(GrammarSymbolEvent gse)
	{
		GrammarSymbol symbol = gse.getGrammarSymbol();

		// See if new node has been added
		if (symbol instanceof GraphNodeInt) {
			GraphNodeInt node = (GraphNodeInt) symbol;
			layoutSystem.addNode(new GNLayoutNode(node, true));
		}

		// See if new edge has been added
		else if (symbol instanceof GraphEdgeInt) {
			GraphEdgeInt edge = (GraphEdgeInt) symbol;
			layoutSystem.addNode(new GELayoutNode(edge, true));
			layoutSystem.addConnection(getLayoutNode(layoutSystem, edge.get_node1()), getLayoutNode(layoutSystem, edge));
			layoutSystem.addConnection(getLayoutNode(layoutSystem, edge), getLayoutNode(layoutSystem, edge.get_node2()));
		}
	}

	// Needed to listen for when GraphNodeInts and GraphEdgeInts are removed
	public void symbolRemoved(GrammarSymbolEvent gse)
	{
		GrammarSymbol symbol = gse.getGrammarSymbol();

		// See if node has been removed (assume all relevant edges are already gone)
		if (symbol instanceof GraphNodeInt) {
			GraphNodeInt node = (GraphNodeInt) symbol;
			layoutSystem.removeNode(getLayoutNode(layoutSystem, node));
		}

		// See if edge has been removed
		else if (symbol instanceof GraphEdgeInt) {
			GraphEdgeInt edge = (GraphEdgeInt) symbol;
			layoutSystem.removeConnection(getLayoutNode(layoutSystem, edge.get_node1()), getLayoutNode(layoutSystem, edge));
			layoutSystem.removeConnection(getLayoutNode(layoutSystem, edge), getLayoutNode(layoutSystem, edge.get_node2()));
			layoutSystem.removeNode(getLayoutNode(layoutSystem, edge));
		}
	}

	// Not needed
	public void symbolModified(GrammarSymbolEvent gse) { ; }

	// Returns the layout node in a layout system that contains
	// the specified graph node, or null if no such node is present
	private LayoutNode getLayoutNode(ForceDirectedSystem fds, GraphNodeInt node)
	{
		Iterator iter = fds.getNodes().iterator();
		while (iter.hasNext()) {
			LayoutNode ln = (LayoutNode) iter.next();
			if (ln instanceof GNLayoutNode) {
				if (((GNLayoutNode) ln).getGraphNodeInt() == node)
					return ln;
			}
		}
		return null;
	}

	// Returns the layout node in a layout system that contains
	// the specified graph edge, or null if no such node is present
	private LayoutNode getLayoutNode(ForceDirectedSystem fds, GraphEdgeInt edge)
	{
		Iterator iter = fds.getNodes().iterator();
		while (iter.hasNext()) {
			LayoutNode ln = (LayoutNode) iter.next();
			if (ln instanceof GELayoutNode) {
				if (((GELayoutNode) ln).getGraphEdgeInt() == edge)
					return ln;
			}
		}
		return null;
	}
}

