/*
    Copyright (C) 2003  Anthony R. Jansen.

    This file is part of the CIDER Toolkit.

    The CIDER Toolkit is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    The CIDER Toolkit is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with the CIDER Toolkit; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

// ----------------------------------------
// CIDER Toolkit: Layout Package
//
// Author:  Anthony R. Jansen
// Begun:   February 2004
// Class:   ForceDirectedSystem
//
// This class controls the manipulation of
// a force directed layout system.
// ----------------------------------------

package au.edu.monash.csse.tonyj.cider.layout;

import java.util.Vector;
import java.util.Iterator;

class ForceDirectedSystem {

	private Vector	nodes;
	private double	springDesiredDistance = 25.0;
	private double 	particleRepulsionConstant = 400.0;
	private double 	springConstant = 0.5;

	// Constants
	private static final double MINIMUM_CUTOFF_FORCE = 2.0;
	private static final double MAXIMUM_FORCE = 1.5;

	// Constructor
	ForceDirectedSystem()
	{
		nodes = new Vector();
	}

	void setParameters(double sdd, double prc, double sc)
	{
		springDesiredDistance = sdd;
		particleRepulsionConstant = prc;
		springConstant = sc;
	}

	void addNode(LayoutNode node)
	{
		// Check for duplication and clear connections
		if (!nodes.contains(node)) {
			node.connectedTo.clear();
			nodes.add(node);
		}
	}

	void removeNode(LayoutNode node)
	{
		nodes.remove(node);
	}

	void addConnection(LayoutNode node1, LayoutNode node2)
	{
		// Make sure nodes are not the same
		if (node1 == node2)
			return;

		// Make sure both nodes are present
		if (!nodes.contains(node1) || !nodes.contains(node2))
			return;

		// Do not duplicate entries
		if (!node1.connectedTo.contains(node2))
			node1.connectedTo.add(node2);
		if (!node2.connectedTo.contains(node1))
			node2.connectedTo.add(node1);
	}

	void removeConnection(LayoutNode node1, LayoutNode node2)
	{
		if (node1 != null && node2 != null) {
			node1.connectedTo.remove(node2);
			node2.connectedTo.remove(node1);
		}
	}

	void clear()
	{
		nodes.clear();
	}

	Vector getNodes()
	{
		return nodes;
	}

	// Adjusts the node positions based on forces
	boolean adjustNodes()
	{
		forceReset();
		calculateNodePairForces();
		double max = calculateMaximumForce();

		// If close to equilibrium, stop
		if (max < MINIMUM_CUTOFF_FORCE)
			return false;

		// Prevent movement which is too extreme
		// Only count small movements
		if (max > MAXIMUM_FORCE) 
			forceScale(MAXIMUM_FORCE / max);

		Iterator iter = nodes.iterator();
		while (iter.hasNext()) {
			LayoutNode node = (LayoutNode) iter.next();
			if (node.isModifiable())
				node.adjustPosition();
		}
		return true;
	}

	// Calculates the maximum force, only checking modifiable nodes
	private double calculateMaximumForce()
	{
		double max = 0.0;

		Iterator iter = nodes.iterator();
		while (iter.hasNext()) {
			LayoutNode node = (LayoutNode) iter.next();

			// Check modifiability
			if (!node.isModifiable())
				continue;

			ForceVector v = node.v;
			double force = Math.sqrt((v.dx * v.dx) + (v.dy * v.dy));
			if (force > max)
				max = force;
		}
		return max;
	}

	// Scales the force vector for all nodes
	private void forceScale(double s)
	{
		Iterator iter = nodes.iterator();
		while (iter.hasNext())
			((LayoutNode) iter.next()).v.scale(s);
	}

	// Resets the force vector for all nodes
	private void forceReset()
	{
		Iterator iter = nodes.iterator();
		while (iter.hasNext())
			((LayoutNode) iter.next()).v.reset();
	}

	private void calculateNodePairForces()
	{
		for (int i = 0; i < nodes.size(); i++) {
			LayoutNode node1 = (LayoutNode) nodes.elementAt(i);
			for (int j = i+1; j < nodes.size(); j++) {
				LayoutNode node2 = (LayoutNode) nodes.elementAt(j);
				double dx = node2.getX() - node1.getX(); 
				double dy = node2.getY() - node1.getY(); 
				double sdx = Math.abs(dx) - (0.5 * (node2.getWidth() + node1.getWidth()));
				double sdy = Math.abs(dy) - (0.5 * (node2.getHeight() + node1.getHeight()));
				if (sdx < 0.0)
					sdx = 0.0;
				if (sdy < 0.0)
					sdy = 0.0;
				double distance = Math.sqrt((sdx * sdx) + (sdy * sdy));
				if (distance <= 0.1)
					distance = 0.1;
				if (dx + dy == 0.0)
					dx = 0.1;

				// Calculate repulsion
				setParticleRepulsionForce(node1, node2, distance, dx, dy);

				// Calculate attraction
				if (node1.connectedTo.contains(node2))
					setSpringAttractionForce(node1, node2, distance, dx, dy);
			}
		}
	}	

	// Calculate particle repulsion force 
	private void setParticleRepulsionForce(LayoutNode node1, LayoutNode node2, double distance, double dx, double dy)
	{
		double force = particleRepulsionConstant / (distance * distance);
		ForceVector v = new ForceVector(dx * force / distance, dy * force / distance);
		node2.v.add(v);
		v.negate();
		node1.v.add(v);
	}

	// Calculate spring attraction force
	private void setSpringAttractionForce(LayoutNode node1, LayoutNode node2, double distance, double dx, double dy)
	{
		// Calculate equilibrium distance
		double eqDist = springDesiredDistance;
		double force = springConstant * (distance - eqDist);
		ForceVector v = new ForceVector(dx * force / distance, dy * force / distance);
		node1.v.add(v);
		v.negate();
		node2.v.add(v);
	}

}

