/*
    Copyright (C) 2003  Anthony R. Jansen.

    This file is part of the CIDER Toolkit.

    The CIDER Toolkit is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    The CIDER Toolkit is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with the CIDER Toolkit; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

// ----------------------------------------
// CIDER Toolkit: QOCA Implementation
//
// Author:    Anthony R. Jansen
// Begun:     December 2002
// Class:     QocaConstraint
//
// This class implements the Qoca 
// constraint using the constraints package 
// interface.
// ----------------------------------------

package au.edu.monash.csse.tonyj.cider.qoca;

import au.edu.monash.csse.tonyj.cider.constraints.*;
import java.util.Iterator;
import java.util.Set;
import java.util.Map;
import java.util.HashMap;

public class QocaConstraint implements Constraint {

	// Protected variables
	protected double		constraintConstant;
	protected int 			operation;
	protected Map			variableCoefficients;

	// Non-serializable variable
	transient protected qoca.QcConstraint	constraint;

	// Constructor
	public QocaConstraint()
	{
		constraintConstant = 0.0;
		constraint = null;
		variableCoefficients = new HashMap();
	}
	
	// Initializes the constraint from the string
	public void initialize(String constraintStr, ConstraintVariable variables[])
		throws CannotProcessConstraintException, ArrayIndexOutOfBoundsException, NumberFormatException
	{
		int index;

		// Get type of operation
		index = constraintStr.indexOf("<=");	
		operation = qoca.QcConstraint.LE;
		if (index == -1) {
			index = constraintStr.indexOf(">=");	
			operation = qoca.QcConstraint.GE;
			if (index == -1) {
				index = constraintStr.indexOf("==");	
				operation = qoca.QcConstraint.EQ;
				if (index == -1)
					throw new CannotProcessConstraintException("Missing operation, ==, <= or >=");
			}
		}

		// Get polynomial and constant components of the string
		String poly = constraintStr.substring(0, index);
		index = constraintStr.indexOf(' ', index);
		String constant = constraintStr.substring(index).trim();
		constraintConstant = Double.parseDouble(constant);

		// Create polynomial 
		qoca.QcLinPoly qocaPoly = new qoca.QcLinPoly();

		// Get polynomial terms
		String terms[] = poly.split("\\u002B");
		for (int i = 0; i < terms.length; i++) {
			terms[i] = terms[i].trim();
			
			// Get coefficient and variable for each term
			index = terms[i].indexOf('[');
			if (index == -1)
				throw new CannotProcessConstraintException("Missing [ symbol in polynomial term");
			String coefficient = terms[i].substring(0, index).trim();
			if (coefficient.equals(""))
				coefficient = "1.0";
			else if (coefficient.equals("-"))
				coefficient = "-1.0";
			String variable = terms[i].substring(index).trim();
			if (!(variable.startsWith("[") && variable.endsWith("]")))
				throw new CannotProcessConstraintException("Expecting each variable designation to be within square brackets []");

			// Add terms to polynomial and update constraint variables set
			ConstraintVariable var = variables[i];
			if (variableCoefficients.containsKey(var))
				throw new CannotProcessConstraintException("Each variable may appear only once per constraint");
			variableCoefficients.put(var, new Double(coefficient));
			qocaPoly.addUniqueTerm(Double.parseDouble(coefficient), ((QocaConstraintVariable) var).getQcFloat());
		}

		// Initialize constraint
		constraint = new qoca.QcConstraint("c", qocaPoly, operation, constraintConstant);
	}

	// Initializes a constraint of the form:  variable == expression
	public void initializeWithEqualsRelationship(ConstraintVariable variable, ConstraintExpression expression)
	{
		double coefficient;

		// Get constraint terms
		constraintConstant = -1.0 * ((QocaConstraintExpression) expression).getConstant();

		// Create new polynomial
		qoca.QcLinPoly qocaPoly = new qoca.QcLinPoly();
		Iterator iter = ((QocaConstraintExpression) expression).getConstraintVariables().iterator();
		while (iter.hasNext()) {
			ConstraintVariable var = (ConstraintVariable) iter.next();
			coefficient = (((QocaConstraintExpression) expression).getCoefficient(var)).doubleValue();
			variableCoefficients.put(var, new Double(coefficient));
			qocaPoly.addUniqueTerm(coefficient, ((QocaConstraintVariable) var).getQcFloat());
		}

		// Add extra variable for equals relationship
		coefficient = -1.0;
		qocaPoly.addUniqueTerm(coefficient, ((QocaConstraintVariable) variable).getQcFloat());
		if (variableCoefficients.containsKey(variable))
			coefficient += (((QocaConstraintExpression) expression).getCoefficient(variable)).doubleValue();
		variableCoefficients.put(variable, new Double(coefficient));

		// Initialize constraint
		operation = qoca.QcConstraint.EQ;
		constraint = new qoca.QcConstraint("c", qocaPoly, operation, constraintConstant);
	}

	// Initializes a constraint of the form:  variable == constant
	public void initializeConstant(ConstraintVariable variable, double constant)
	{
		// Set constant term
		constraintConstant = constant;

		// Create new polynomial and add variable
		qoca.QcLinPoly qocaPoly = new qoca.QcLinPoly();
		qocaPoly.addUniqueTerm(1.0, ((QocaConstraintVariable) variable).getQcFloat());
		variableCoefficients.put(variable, new Double(1.0));

		// Initialize constraint
		operation = qoca.QcConstraint.EQ;
		constraint = new qoca.QcConstraint("c", qocaPoly, operation, constraintConstant);
	}

	// Returns true iff the current constraint variable values 
	// satisfy this constraint within the specified tolerance amount
	public boolean isSatisfied(double tolerance)
	{
		double lhs = 0.0;
		double rhs = constraint.getRHS();
		qoca.QcLinPoly qocaPoly = constraint.getLinPoly();

		// Calculate lhs of relationship
		Iterator iter = qocaPoly.fTerms.iterator();
		while (iter.hasNext()) {
			qoca.QcLinPolyTerm qocaTerm = (qoca.QcLinPolyTerm) iter.next();
			lhs += qocaTerm.getValue();
		}

		// Check which operator was used
		if (constraint.getOp() == qoca.QcConstraint.GE)
			return (lhs >= (rhs - tolerance));
		else if (constraint.getOp() == qoca.QcConstraint.LE)
			return (lhs <= (rhs + tolerance));
		else
			return ((lhs >= (rhs - tolerance)) && (lhs <= (rhs + tolerance)));
	}

	// Returns a set containing all constraint variables
	public Set getConstraintVariables()
	{
		return variableCoefficients.keySet();
	}

	// Returns the qoca constraint
	public qoca.QcConstraint getQcConstraint()
	{
		return constraint;
	}

	// Set the weight of this constraint
	public void setWeight(double weight) { ; }

	// Get the weight of this constraint
	public double getWeight() { return 0.0; }
}

