/*
    Copyright (C) 2003  Anthony R. Jansen.

    This file is part of the CIDER Toolkit.

    The CIDER Toolkit is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    The CIDER Toolkit is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with the CIDER Toolkit; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

// ----------------------------------------
// CIDER Toolkit: QOCA Implementation
//
// Author:    Anthony R. Jansen
// Begun:     January 2003
// Class:     QocaConstraintCompiler
//
// This class implements static methods for
// allowing constraints to be precompiled
// into Java classes so that they do not
// have to be initialized with a string
// parse at run time.
// ----------------------------------------

package au.edu.monash.csse.tonyj.cider.qoca;

import au.edu.monash.csse.tonyj.cider.constraints.*;
import java.io.PrintWriter;
import java.util.Set;
import java.util.HashSet;

public class QocaConstraintCompiler implements ConstraintCompiler {

	// Constant
	public static final String      ORIGIN_LINE = "// This class was automatically generated by the Qoca constraint compiler";

	// Empty constructor
	public QocaConstraintCompiler() { }

	// Method to create a constraint class
	public void createConstraintClass(String className, String classPackage, String constraintStr, PrintWriter output)
		throws CannotProcessConstraintException, NumberFormatException
	{
		Set usedIndexes = new HashSet();
		String operation = "";
		int index;

		// Print header stuff
		output.println(ORIGIN_LINE);
		output.println("");
		if (!classPackage.equals("")) {
			output.println("package " + classPackage + ";");
			output.println("");
		}
		output.println("import au.edu.monash.csse.tonyj.cider.qoca.*;");
		output.println("import au.edu.monash.csse.tonyj.cider.constraints.*;");
		output.println("");

		// Begin class
		output.println("class " + className + " extends QocaConstraint {");
		output.println("");

		// Constructor
		output.println("\tpublic " + className + "(ConstraintVariable variables[]) throws ArrayIndexOutOfBoundsException");
		output.println("\t{");
		output.println("\t\tsuper();");

		// Get type of operation
		index = constraintStr.indexOf("<=");	
		operation = "qoca.QcConstraint.LE";
		if (index == -1) {
			index = constraintStr.indexOf(">=");	
			operation = "qoca.QcConstraint.GE";
			if (index == -1) {
				index = constraintStr.indexOf("==");	
				operation = "qoca.QcConstraint.EQ";
				if (index == -1)
					throw new CannotProcessConstraintException("Missing operation, ==, <= or >=");
			}
		}

		// Get polynomial and constant components of the string
		String poly = constraintStr.substring(0, index);
		index = constraintStr.indexOf(' ', index);
		String constant = constraintStr.substring(index).trim();
		output.println("\t\tconstraintConstant = " + Double.parseDouble(constant) + ";");
		output.println("\t\toperation = " + operation + ";");

		// Create polynomial 
		output.println("\t\tqoca.QcLinPoly qocaPoly = new qoca.QcLinPoly();");

		// Get polynomial terms
		String terms[] = poly.split("\\u002B");
		for (int i = 0; i < terms.length; i++) {
			terms[i] = terms[i].trim();
			
			// Get coefficient and variable for each term
			index = terms[i].indexOf('[');
			if (index == -1)
				throw new CannotProcessConstraintException("Missing [ symbol in polynomial term");
			String coefficient = terms[i].substring(0, index).trim();
			if (coefficient.equals(""))
				coefficient = "1.0";
			else if (coefficient.equals("-"))
				coefficient = "-1.0";
			String variable = terms[i].substring(index).trim();
			if (!(variable.startsWith("[") && variable.endsWith("]")))
				throw new CannotProcessConstraintException("Expecting each variable designation to be within square brackets []");

			// Make sure variable has not been used before
			if (!usedIndexes.add(variable))
				throw new CannotProcessConstraintException("Each variable may appear only once per constraint");
				
			// Add terms to polynomial and update constraint variables set
			output.println("\t\tqocaPoly.addUniqueTerm(" + Double.parseDouble(coefficient) + 
				", ((QocaConstraintVariable) variables[" + i + "]).getQcFloat());");
			output.println("\t\tvariableCoefficients.put(variables[" + i + "], new Double(" + Double.parseDouble(coefficient) + "));");
		}

		// Initialize constraint
		output.println("\t\tconstraint = new qoca.QcConstraint(\"c\", qocaPoly, operation, constraintConstant);");
		output.println("\t}");

		// End of class
		output.println("}");
		output.println("");
	}	

	// Method to create a constraint expression class
	public void createConstraintExpressionClass(String className, String classPackage, String expression, PrintWriter output)
		throws CannotProcessConstraintException, NumberFormatException
	{
		Set usedIndexes = new HashSet();

		// Print header stuff
		output.println(ORIGIN_LINE);
		output.println("");
		if (!classPackage.equals("")) {
			output.println("package " + classPackage + ";");
			output.println("");
		}
		output.println("import au.edu.monash.csse.tonyj.cider.qoca.*;");
		output.println("import au.edu.monash.csse.tonyj.cider.constraints.*;");
		output.println("");

		// Begin class
		output.println("class " + className + " extends QocaConstraintExpression {");
		output.println("");

		// Constructor
		output.println("\tpublic " + className + "(ConstraintVariable variables[]) throws ArrayIndexOutOfBoundsException");
		output.println("\t{");
		output.println("\t\tsuper();");

		// Get polynomial terms
		String terms[] = expression.split("\\u002B");
		for (int i = 0; i < terms.length; i++) {
			terms[i] = terms[i].trim();
			
			// Deal with constants
			int index = terms[i].indexOf('[');
			if (index == -1) {
				output.println("\t\texpConstant += " + Double.parseDouble(terms[i].trim()) + ";");
				continue;
			}

			// Get coefficient and variable for each term
			String coefficient = terms[i].substring(0, index).trim();
			if (coefficient.equals(""))
				coefficient = "1.0";
			else if (coefficient.equals("-"))
				coefficient = "-1.0";
			String variable = terms[i].substring(index).trim();
			if (!(variable.startsWith("[") && variable.endsWith("]")))
				throw new CannotProcessConstraintException("Expecting each variable designation to be within square brackets []");

			// Make sure variable has not been used before
			if (!usedIndexes.add(variable))
				throw new CannotProcessConstraintException("Each variable may appear only once per expression");
				
			// Update constraint variables set
			output.println("\t\tvariableCoefficients.put(variables[" + i + "], new Double(" + Double.parseDouble(coefficient) + "));");
		}
		output.println("\t}");

		// End of class
		output.println("}");
		output.println("");
	}	
}

