/*
    Copyright (C) 2003  Anthony R. Jansen.

    This file is part of the CIDER Toolkit.

    The CIDER Toolkit is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    The CIDER Toolkit is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with the CIDER Toolkit; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

// ----------------------------------------
// CIDER Toolkit: QOCA Implementation
//
// Author:    Anthony R. Jansen
// Begun:     December 2002
// Class:     QocaConstraintExpression
//
// This class implements the Qoca 
// linear polynomial and a constant value
// using the constraints package interface.
// ----------------------------------------

package au.edu.monash.csse.tonyj.cider.qoca;

import au.edu.monash.csse.tonyj.cider.constraints.*;
import java.util.Iterator;
import java.util.Map;
import java.util.HashMap;
import java.util.Set;

public class QocaConstraintExpression implements ConstraintExpression {

	// Protected variables
	protected double		expConstant;
	protected Map			variableCoefficients;

	// Constructor
	public QocaConstraintExpression()
	{
		expConstant = 0.0;
		variableCoefficients = new HashMap();
	}
	
	// Initializes the constraint expression from the string
	public void initialize(String expression, ConstraintVariable variables[])
		throws CannotProcessConstraintException, ArrayIndexOutOfBoundsException, NumberFormatException
	{
		// Get polynomial terms
		String terms[] = expression.split("\\u002B");
		for (int i = 0; i < terms.length; i++) {
			terms[i] = terms[i].trim();

			// Deal with constants
			int index = terms[i].indexOf('[');
			if (index == -1) {
				expConstant += Double.parseDouble(terms[i].trim());
				continue;
			}

			// Get coefficient and variable for each term
			String coefficient = terms[i].substring(0, index).trim();
			if (coefficient.equals(""))
				coefficient = "1.0";
			else if (coefficient.equals("-"))
				coefficient = "-1.0";
			String variable = terms[i].substring(index).trim();
			if (!(variable.startsWith("[") && variable.endsWith("]")))
				throw new CannotProcessConstraintException("Expecting each variable designation to be within square brackets []");

			// Add terms to polynomial and update constraint variables set
			ConstraintVariable var = variables[i];
			if (variableCoefficients.containsKey(var))
				throw new CannotProcessConstraintException("Each variable may appear only once per expression");
			variableCoefficients.put(var, new Double(coefficient));
		}
	}

	// Evaluates the expression and returns the value
	public double evaluate()
	{
		double value = expConstant;

		Iterator iter = variableCoefficients.keySet().iterator();
		while (iter.hasNext()) {
			ConstraintVariable var = (ConstraintVariable) iter.next();
			double coefficient = ((Double) variableCoefficients.get(var)).doubleValue();
			value += coefficient * var.getSolvedValue();
		}
		return value;
	}

	// Returns a set containing all constraint variables
	public Set getConstraintVariables()
	{
		return variableCoefficients.keySet();
	}

	// Returns the constant value
	public double getConstant()
	{
		return expConstant;
	}

	// Returns the coefficent of the specifed constraintVariable (or null if not part of this expression)
	public Double getCoefficient(ConstraintVariable var)
	{
		return (Double) variableCoefficients.get(var);
	}
}

