/*
    Copyright (C) 2003  Anthony R. Jansen.

    This file is part of the CIDER Toolkit.

    The CIDER Toolkit is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    The CIDER Toolkit is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with the CIDER Toolkit; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

// ----------------------------------------
// CIDER Toolkit: QOCA Implementation
//
// Author:  Anthony R. Jansen
// Begun:   December 2002
// Class:   QocaConstraintSolver
//
// This class implements the Qoca 
// constraint solver using the constraints
// package interface.
// ----------------------------------------

package au.edu.monash.csse.tonyj.cider.qoca;

import au.edu.monash.csse.tonyj.cider.constraints.*;
import java.util.Iterator;
import java.util.Set;
import java.util.HashSet;
import java.util.Map;
import java.util.HashMap;

import java.io.PrintStream;
import java.io.FileOutputStream;
import java.io.FileNotFoundException;

public class QocaConstraintSolver extends ConstraintSolver {

	// Private variable
	private qoca.QcCassSolver	csolver;
	private Set			constraints;
	private Set			constraintVariables;

	// Constructor
	public QocaConstraintSolver()
	{
		csolver = new qoca.QcCassSolver();
		constraints = new HashSet();
		constraintVariables = new HashSet();
	}

	// Add a variable to the constraint solver
	public synchronized void addVariable(ConstraintVariable v)
	{
		csolver.addVar(((QocaConstraintVariable) v).getQcFloat());
		constraintVariables.add(v);
	}

	// Remove a variable from the constraint solver if it is free
	public synchronized boolean removeVariable(ConstraintVariable v)
	{
		boolean success = csolver.removeVar(((QocaConstraintVariable) v).getQcFloat());
		if (success)
			constraintVariables.remove(v);
		return success;
	}

	// Add a constraint
	public synchronized boolean addConstraint(Constraint c)
	{
		boolean success = csolver.addConstraint(((QocaConstraint) c).getQcConstraint());
		if (success) {
			constraints.add(c);
			constraintVariables.addAll(c.getConstraintVariables());
		}
		return success;
	}

	// Remove a constraint 
	public synchronized boolean removeConstraint(Constraint c)
	{
		boolean success = csolver.removeConstraint(((QocaConstraint) c).getQcConstraint());
		if (success)
			constraints.remove(c);
		return success;
	}

	// Add a variable to be edited
	public synchronized void addEditVariable(ConstraintVariable v)
	{
		if (!csolver.isEditVar(((QocaConstraintVariable) v).getQcFloat()))
			csolver.addEditVar(((QocaConstraintVariable) v).getQcFloat());
	}

	public synchronized void solve()
	{
		// Get all current variable values
		Map variableValues = new HashMap();
		Iterator iter = constraintVariables.iterator();
		while (iter.hasNext()) {
			ConstraintVariable var = (ConstraintVariable) iter.next();
			variableValues.put(var, new Double(var.getSolvedValue()));
		}

		// Call solve method
		csolver.solve();
		csolver.endEdit();

		// See what variable values have changed
		iter = variableValues.keySet().iterator();
		while (iter.hasNext()) {
			ConstraintVariable var = (ConstraintVariable) iter.next();
			Double oldValue = (Double) variableValues.get(var);
			if (var.getSolvedValue() != oldValue.doubleValue())
				launchVariableModifiedEvent(var, oldValue);
		}
	}

	public synchronized void resolve()
	{
		// Get all current variable values
		Map variableValues = new HashMap();
		Iterator iter = constraintVariables.iterator();
		while (iter.hasNext()) {
			ConstraintVariable var = (ConstraintVariable) iter.next();
			variableValues.put(var, new Double(var.getSolvedValue()));
		}

		// Call resolve method
		csolver.resolve();
		csolver.endEdit();

		// See what variable values have changed
		iter = variableValues.keySet().iterator();
		while (iter.hasNext()) {
			ConstraintVariable var = (ConstraintVariable) iter.next();
			Double oldValue = (Double) variableValues.get(var);
			if (var.getSolvedValue() != oldValue.doubleValue())
				launchVariableModifiedEvent(var, oldValue);
		}
	}

	public synchronized Set getConstraints()
	{
		return constraints;
	}

	public synchronized Set getConstraintVariables()
	{
		return constraintVariables;
	}
}

