// Created Sep 19, 2005 by Aidan Lane
// This class provides utility methods that help reduce the amount of JNI code
// that needs to be written in C++ and improves performance, while not
// contaminating CiderInterface with FSA specific code.
// Benefits:
// - Reduces the number of classes that the C++ needs to find
//   (doesn't need to find FSAInterpreter, DrawableArrow, DrawableCircle or Color)
// - Leaves almost all memory (de)allocation to the Java side
//   (only FSACiderInterface needs to be allocated, not FSAInterpreter,
//    DrawableArrow, DrawableCircle, etc).
// - Uses primitives in parameters
// - The number of parameters has been reduced
// - Most adapting/wrapping code is here - in Java

package fsa_editor;

// TODO: remove what is not needed!
import au.edu.monash.csse.tonyj.cider.canvas.DrawableToken;
import au.edu.monash.csse.tonyj.cider.interpreter.*;


public class FSACiderInterface extends CiderInterface {

    // Note: "cider" is explicit, as Digest elements have their own IDs
    private static native void
	native_arrowPointsModified(int symbolId,
				   double start_x, double start_y,
				   double mid_x, double mid_y,
				   double end_x, double end_y);
    private static native void
	native_circleModified(int symbolId,
			      double radius, double mid_x, double mid_y);
    private static native void
	native_textRectModified(int symbolId,
				double mid_x, double mid_y,
				double haldwidth, double halfheight);


    FSACiderInterface() {
	super(new FSAInterpreter());
    }

    public void symbolModified(GrammarSymbolEvent gse)
    {
	GrammarSymbol s = gse.getGrammarSymbol();
	if (s instanceof DrawableArrow) {
	    DrawableArrow d = (DrawableArrow)s;
	    native_arrowPointsModified(d.getIDNumber(), d.get_start_x(), d.get_start_y(),
				       d.get_mid_x(), d.get_mid_y(),
				       d.get_end_x(), d.get_end_y());
	}
	else if (s instanceof DrawableCircle) {
	    DrawableCircle d = (DrawableCircle)s;
	    native_circleModified(d.getIDNumber(), d.get_radius(),
				  d.get_mid_x(), d.get_mid_y());
	}
	else if (s instanceof DrawableText) {
	    DrawableText d = (DrawableText)s;
	    native_textRectModified(d.getIDNumber(), d.get_mid_x(), d.get_mid_y(),
				    d.get_halfwidth(), d.get_halfheight());
	}
    }

    /**
     * Creates a new DrawableArrow object, but does not add it to the forest.
     */
    public DrawableArrow createDrawableArrow(double start_x, double start_y,
					     double mid_x, double mid_y,
					     double end_x, double end_y,
					     double start_radius,
					     double end_radius) {
	return new DrawableArrow(start_x, start_y, mid_x, mid_y,
				 end_x, end_y, start_radius, end_radius,
				 false, new SimpleColor(128,128,128));
    }

    /**
     * Creates a new DrawableCircle object, but does not add it to the forest.
     */
    public DrawableCircle createDrawableCircle(double radius,
					       double mid_x, double mid_y) {
	return new DrawableCircle(radius, mid_x, mid_y, false,
				  new SimpleColor(128,128,128));
    }

    /**
     * Creates a new DrawableText object, but does not add it to the forest.
     */
    public DrawableText createDrawableText(String label,
					   double mid_x, double mid_y,
					   double halfwidth, double halfheight) {
	return new DrawableText(label, mid_x, mid_y,
				halfwidth, halfheight, false,
				new SimpleColor(128,128,128));
    }

    /**
     * Sets the given DrawableArrow's start, middle and end points.
     *
     * @return true on success (which implies that the ciderSymbolId refers to
     * valid DrawableArrow).
     */
    public void setArrowPoints(DrawableArrow s,
			       double start_x, double start_y,
			       double mid_x, double mid_y,
			       double end_x, double end_y) {
	getInterpreter().setAutoForestChecking(false);
	s.set_start_x(start_x);
	s.set_start_y(start_y);
	s.set_mid_x(mid_x);
	s.set_mid_y(mid_y);
	s.set_end_x(end_x);
	getInterpreter().setAutoForestChecking(true);
	s.set_end_y(end_y); // last set - MUST be after setAutoForestChecking(true)
    }

    /**
     * Set the given DrawableCircle's position.
     * @return true on success.
     */
    public void setCirclePos(DrawableCircle s, double mid_x, double mid_y) {
	getInterpreter().setAutoForestChecking(false);
	s.set_mid_x(mid_x);
	getInterpreter().setAutoForestChecking(true);
	s.set_mid_y(mid_y); // last set - MUST be after setAutoForestChecking(true)
    }

    /**
     * Set the given DrawableCircle's radius.
     * @return true on success.
     */
    public void setCircleRadius(DrawableCircle s, double radius) {
	s.set_radius(radius);
    }

    /**
     * Set the given DrawableText's position.
     * @return true on success.
     */
    public void setTextPos(DrawableText s, double mid_x, double mid_y) {
	getInterpreter().setAutoForestChecking(false);
	s.set_mid_x(mid_x);
	getInterpreter().setAutoForestChecking(true);
	s.set_mid_y(mid_y); // last set - MUST be after setAutoForestChecking(true)
    }

    /**
     * Set the given DrawableText's half-dimensions.
     * @return true on success.
     */
    public void setTextHalfDimensions(DrawableText s,
				      double halfwidth, double halfheight) {
	getInterpreter().setAutoForestChecking(false);
	s.set_halfwidth(halfwidth);
	getInterpreter().setAutoForestChecking(true);
	s.set_halfheight(halfheight); // last set - MUST be after setAutoForestChecking(true)
    }
}