/*
 * Imported in to Digest by Aidan Lane on Tue Sep 13 2005.
 */

// Class to convert an NFA to a DFA

package fsa_editor;

import au.edu.monash.csse.tonyj.cider.interpreter.*;
import au.edu.monash.csse.tonyj.cider.canvas.*;
import java.util.Set;
import java.util.TreeSet;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Vector;
import java.util.LinkedList;
import java.util.NoSuchElementException;

public class NFAConverter implements GrammarSymbolListener {

	// Private variables
	private Set		stateSet;
	private Set		transitionSet;
	private Set		emptyTransitionSet;
	private Vector		alphabet;

	private StateGroup	startGroup;
	private StateGroup	currentGroup;
	private int		currentIndex;
	private StateGroup	targetGroup;
	private	Transition	currentTransition;

	private	Set		allGroups;
	private	LinkedList	unprocessedGroups;

	private InterpretedTokenCanvas	canvas;

	private	double		xValue;
	private	double		yValue;

	private SimpleColor	fromColour;
	private SimpleColor	toColour;

	// Constructor
	public NFAConverter(InterpretedTokenCanvas c, int tag)
	{
		canvas = c;
		canvas.getInterpreter().addGrammarSymbolListener(this);

		fromColour = SimpleColor.magenta;
		toColour = SimpleColor.green;

		startGroup = new StateGroup();
		stateSet = new HashSet();
		transitionSet = new HashSet();
		emptyTransitionSet = new HashSet();
		alphabet = new Vector();
		allGroups = new HashSet();
		unprocessedGroups = new LinkedList();

		initialize(tag);
		currentGroup = startGroup;
		currentIndex = 0;
		targetGroup = startGroup;

		// Draw the start group and start arrow
		xValue = 80.0;
		canvas.add(new DrawableArrow(20.0, yValue, 50.0, yValue, 80.0, yValue, 0.0, 0.0, false, null));
		createState(xValue, yValue);

		allGroups.add(startGroup);
		colourStateGroup(startGroup, fromColour);
	}

	// Finalize method
	public void finalize() throws Throwable
	{
		canvas.getInterpreter().removeGrammarSymbolListener(this);
	}

	// Get all transition and state symbols from the parse forest with the specified tag
	// Determines the FSA alphabet from the transitions
	// (will treat regular expressions as unique alphabet members)
	// Also determines start states
	private void initialize(int tag)
	{
		Set allStatesSet = new HashSet();

		Iterator iter = canvas.getInterpreter().getAllSymbols().iterator();
		while (iter.hasNext()) {
			Object obj = iter.next();
			if (obj instanceof State)
				allStatesSet.add(obj);
			if (obj instanceof State && ((State) obj).get_tag() == tag)
				stateSet.add(obj);
			else if (obj instanceof Transition && ((Transition) obj).get_tag() == tag)
				transitionSet.add(obj);
			else if (obj instanceof StartArc && ((StartArc) obj).get_state().get_tag() == tag)
				startGroup.stateSet.add(((StartArc) obj).get_state());
		}
	
		// Calculate the alphabet
		TreeSet alphabetSet = new TreeSet();
		iter = transitionSet.iterator();
		while (iter.hasNext()) {
			Transition transition = (Transition) iter.next();
			if (transition.get_transition().equals(""))
				emptyTransitionSet.add(transition);
			else 
				alphabetSet.add(transition.get_transition());
		}
		alphabet.addAll(alphabetSet);

		// Add empty link states to start states
		addEmptyTransitionStates(startGroup);

		// Find lowest y co-ordinate
		iter = allStatesSet.iterator();
		while (iter.hasNext()) {
			State state = (State) iter.next();
			if (state.get_mid_y() + state.get_radius() > yValue)
				yValue = state.get_mid_y() + state.get_radius();
		}
		yValue += 80.0;
	}

	// Add states that link from the current states via an empty link
	private void addEmptyTransitionStates(StateGroup sg)
	{
		boolean flag;

		do {
			flag = false;
			Iterator iter = emptyTransitionSet.iterator();
			while (iter.hasNext()) {
				Transition transition = (Transition) iter.next();
				if (sg.stateSet.contains(transition.get_from()) && !sg.stateSet.contains(transition.get_to())) {
					sg.stateSet.add(transition.get_to());
					flag = true;
				}
			}
		} while (flag);
	}

	// Creates a new state group that contains states that are linked to from the state group passed to it
	private StateGroup calculateToStateGroup(StateGroup fromGroup, String transitionStr)
	{
		StateGroup toGroup = new StateGroup();

		Iterator iter = transitionSet.iterator();
		while (iter.hasNext()) {
			Transition transition = (Transition) iter.next();
			if (transition.get_transition().equals(transitionStr) && fromGroup.stateSet.contains(transition.get_from()))
					toGroup.stateSet.add(transition.get_to());
		}

		// Add empty link states 
		addEmptyTransitionStates(toGroup);
		return toGroup;
	}

	// Processes the current state group for the current transition index
	// Returns false if there is nothing left to process, otherwise returns true
	public boolean processNextStep()
	{
		// Clear colour for all groups and transitions
		Iterator iter = allGroups.iterator();
		while (iter.hasNext()) 
			colourStateGroup((StateGroup) iter.next(), null);
		iter = transitionSet.iterator();
		while (iter.hasNext()) 
			((Transition) iter.next()).set_colour(null);
		if (currentTransition != null)
			currentTransition.set_colour(null);

		if (currentGroup == null)
			return false;

		// Get current transition string
		String transitionStr = "";
		try {
			transitionStr = (String) alphabet.elementAt(currentIndex);
		}
		catch (ArrayIndexOutOfBoundsException e1) {
			return false;
		}

		StateGroup toGroup = calculateToStateGroup(currentGroup, transitionStr);

		// See if the to group matches an existing group
		boolean flag = false;
		iter = allGroups.iterator();
		while (iter.hasNext()) {
			StateGroup group = (StateGroup) iter.next();
			if (group.equals(toGroup)) {
				toGroup = group;
				flag = true;
				break;
			}
		}

		// If does not already exist, create group
		if (!flag) {
			targetGroup = toGroup;
			createState(xValue, yValue);
			allGroups.add(toGroup);
			unprocessedGroups.addLast(toGroup);
		}
	 
		// Add transition
		double mid_x, mid_y;
		if (currentGroup == toGroup) {
			mid_x = currentGroup.dfaState.get_mid_x();
			mid_y = 60.0 + currentGroup.dfaState.get_mid_y();
		}
		else {
			mid_x = (currentGroup.dfaState.get_mid_x() + toGroup.dfaState.get_mid_x()) / 2.0;
			mid_y = (currentGroup.dfaState.get_mid_y() + toGroup.dfaState.get_mid_y()) / 2.0;
		} 

		// Draw to canvas
		canvas.add(new DrawableArrow(currentGroup.dfaState.get_mid_x(), currentGroup.dfaState.get_mid_y(), 
			mid_x, mid_y, toGroup.dfaState.get_mid_x(), toGroup.dfaState.get_mid_y(), 0.0, 0.0, false, null));
		canvas.add(new DrawableText(transitionStr, mid_x, mid_y, 40, 40, false, null));

		// Colour elements
		colourStateGroup(toGroup, toColour);
		colourStateGroup(currentGroup, fromColour);
		colourTransitions(currentGroup, transitionStr, toColour);
		colourTransitions(toGroup, "", toColour);
		if (currentTransition != null)
			currentTransition.set_colour(toColour);

		// Go to next index, or group if appropriate
		currentIndex++;
		if (currentIndex >= alphabet.size()) {
			currentIndex = 0;
			try { 
				currentGroup = (StateGroup) unprocessedGroups.removeFirst();
			}
			catch (NoSuchElementException e2) {
				currentGroup = null;
			}
		}
		return true;
	}

	// Colours all specified transitions from the given state group
	private void colourTransitions(StateGroup fromGroup, String transitionStr, SimpleColor colour)
	{
		Iterator iter = transitionSet.iterator();
		while (iter.hasNext()) {
			Transition transition = (Transition) iter.next();
			if (transition.get_transition().equals(transitionStr) && fromGroup.stateSet.contains(transition.get_from()))
					transition.set_colour(colour);
		}
	}

	// Colours all states defined in a state group
	private void colourStateGroup(StateGroup group, SimpleColor colour)
	{
		Iterator iter = group.stateSet.iterator();
		while (iter.hasNext()) {
			State state = (State) iter.next();
			state.set_colour(colour);
		}
		if (group.dfaState != null)
			group.dfaState.set_colour(colour);
	}

	// Create state in the canvas from the target state group at the specified co-ordinates
	private void createState(double x, double y)
	{
		if (targetGroup == null)
			return;

		canvas.add(new DrawableCircle(20.0, x, y, false, null));
		if (targetGroup.hasFinal())
			canvas.add(new DrawableCircle(24.0, x, y, false, null));
		canvas.add(new DrawableText(targetGroup.getName(), x, y, 40, 40, false, null));

		xValue += 80.0;
	}

	// Needed for Grammar Symbol Listener
	public void symbolModified(GrammarSymbolEvent gse) { ; }
	public void symbolRemoved(GrammarSymbolEvent gse) { ; }

	public void symbolAdded(GrammarSymbolEvent gse)
	{
		if (targetGroup != null && gse.getGrammarSymbol() instanceof State)
			targetGroup.dfaState = (State) gse.getGrammarSymbol();
		else if (gse.getGrammarSymbol() instanceof Transition)
			currentTransition = (Transition) gse.getGrammarSymbol();
	}
}

