/*  -*- c++ -*-  (for Emacs)
 *
 *  fsadiagramcontroller.cpp
 *  Digest
 * 
 *  Created by Aidan Lane on Fri Sep 30 2005.
 *  Copyright (c) 2005 Optimisation and Constraint Solving Group,
 *  Monash University. All rights reserved.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#include "fsadiagramcontroller.h"

#include "digest.h"

// TODO: remove what is not needed
#include <QCoreApplication>
#include <QDebug> // TODO: remove me
#include <QHash>
#include <QPointF>
#include <QPolygonF>
#include <QReadWriteLock>
#include <QRectF>
#include <QSizeF>

// TODO: remove what is not needed
#include "MvcDiagram/diagram.h"
#include "MvcDiagram/diagramcontroller.h"
#include "MvcDiagram/abstractdiagramelement.h"
#include "MvcDiagram/diagramelementshape.h"


#define GRAMMARSYMBOL_SIG "Lau/edu/monash/csse/tonyj/cider/interpreter/GrammarSymbol;"


static QHash<JNIEnv*, FsaDiagramController*> envToController;
static QReadWriteLock envToControllerLock; // needed - controllers run in separate threads


void JNICALL Java_CiderInterface_native_symbolAdded(JNIEnv* env, jclass,
						    jint symbolId) {
  envToControllerLock.lockForRead();
  Q_ASSERT( envToController.contains(env) );
  envToController.value(env)->onSymbolAdded( symbolId );
  envToControllerLock.unlock();
}

void JNICALL Java_CiderInterface_native_symbolRemoved(JNIEnv* env, jclass,
						      jint symbolId) {
  envToControllerLock.lockForRead();
  Q_ASSERT( envToController.contains(env) );
  envToController.value(env)->onSymbolRemoved( symbolId );
  envToControllerLock.unlock();
}

void JNICALL Java_FSACiderInterface_native_arrowPointsModified(JNIEnv* env, jclass,
							       jint symbolId,
							       jdouble start_x, jdouble start_y,
							       jdouble mid_x, jdouble mid_y,
							       jdouble end_x, jdouble end_y) {
  // Note: ElementDocController::elementAttributeChangeDataEvent() filters redundant updates
  QPolygonF polygon;
  polygon += QPointF( 0.0, 0.0 );
  polygon += QPointF( mid_x - start_x, mid_y - start_y );
  polygon += QPointF( end_x - start_x, end_y - start_y );
  envToControllerLock.lockForRead();
  Q_ASSERT( envToController.contains(env) );
  envToController.value(env)
    ->setElementAttributeData( symbolId,
			       MvcDiagram::DEA_Origin,
			       QPointF(start_x, start_y) );
  envToController.value(env)
    ->setElementAttributeData( symbolId,
			       MvcDiagram::DEA_Polygon,
			       QVariant::fromValue(polygon) );
  envToControllerLock.unlock();
}

void JNICALL Java_FSACiderInterface_native_circleModified(JNIEnv* env, jclass,
							  jint symbolId,
							  jdouble radius,
							  jdouble mid_x, jdouble mid_y) {
  // Note: ElementDocController::elementAttributeChangeDataEvent() filters redundant updates
  envToControllerLock.lockForRead();
  Q_ASSERT( envToController.contains(env) );
  envToController.value(env)
    ->setElementAttributeData( symbolId,
			       MvcDiagram::DEA_Origin,
			       QPointF(mid_x-radius, mid_y-radius) );
  envToController.value(env)
    ->setElementAttributeData( symbolId,
			       MvcDiagram::DEA_Size,
			       QSizeF(radius*2.0, radius*2.0) );
  envToControllerLock.unlock();
}

void JNICALL Java_FSACiderInterface_native_textRectModified(JNIEnv* env, jclass,
							    jint symbolId,
							    jdouble mid_x, jdouble mid_y,
							    jdouble halfwidth, jdouble halfheight) {
  // Note: ElementDocController::elementAttributeChangeDataEvent() filters redundant updates
  envToControllerLock.lockForRead();
  Q_ASSERT( envToController.contains(env) );
  envToController.value(env)
    ->setElementAttributeData( symbolId,
			       MvcDiagram::DEA_Origin,
			       QPointF(mid_x-halfwidth, mid_y-halfheight) );
  envToController.value(env)
    ->setElementAttributeData( symbolId,
			       MvcDiagram::DEA_Size,
			       QSizeF(halfwidth*2.0, halfheight*2.0) );
  envToControllerLock.unlock();
}


/*!
 * Constructs a diagram controller that contains CIDER FSA specific code.
 */
// TODO: DOC ME!
// WARNING: FsaDiagramController must live inside CiderView's worker thread
//          -> it MUST NOT be a child of CiderView (which lives inside the GUI thread).
//          The "ciderView" is only taken because the worker needs to know the ciderview's
//          current controller.
FsaDiagramController::FsaDiagramController( JavaVM* jvm, QObject* parent )
  : GuiDiagramController(jvm, parent),
    m_jvm(jvm),
    env(0),
    fsaCiderInterface_cls(0),
    fsaCiderInterface_gref(0),
    fsaCiderInterfaceConstructor_mid(0),
    add_mid(0),
    remove_mid(0),
    getIDNumber_mid(0),
    printForest_mid(0),
    createDrawableArrow_mid(0),
    createDrawableCircle_mid(0),
    createDrawableText_mid(0),
    setArrowPoints_mid(0),
    setCirclePos_mid(0),
    setCircleRadius_mid(0),
    setTextPos_mid(0),
    setTextHalfDimensions_mid(0)
{
  // TODO: cleanup code!

  jint res = 0;

  // Must pass NULL as the third argument
  res = m_jvm->AttachCurrentThread( (void**)&env, 0 );
  if ( res < 0 ) {
    qDebug( "Thread: attach failed" ); // TODO: handle me properly!
    return;
  }

  // Update envToController - used for the native callbacks
  envToControllerLock.lockForWrite();
  Q_ASSERT( ! envToController.contains(env) ); // sanity check - can't be duplicates
  envToController.insert( env, this );
  envToControllerLock.unlock();

  // WARNING!!! Yes, the following is meant to use "/" to separate the package
  //            name from the class name - this took a while to figure out :-/
  fsaCiderInterface_cls = env->FindClass( "fsa_editor/FSACiderInterface" );
  if ( fsaCiderInterface_cls == 0 ) {
    qDebug( "Thread: Can't find the fsa_editor/FSACiderInterface class" ); // TODO: handle me properly!
    cleanup();
    return;
  }

  fsaCiderInterfaceConstructor_mid = tryGetMethodID( fsaCiderInterface_cls,
						     "<init>",
						     "()V" );
  add_mid                    = tryGetMethodID( fsaCiderInterface_cls,
					       "add",
					       "("GRAMMARSYMBOL_SIG")Z" );
  remove_mid                 = tryGetMethodID( fsaCiderInterface_cls,
					       "remove",
					       "("GRAMMARSYMBOL_SIG")Z" );
  getIDNumber_mid            = tryGetMethodID( fsaCiderInterface_cls,
					       "getIDNumber",
					       "("GRAMMARSYMBOL_SIG")I" );
  printForest_mid            = tryGetMethodID( fsaCiderInterface_cls,
					       "printForest", "()V" );
  createDrawableArrow_mid    = tryGetMethodID( fsaCiderInterface_cls,
					       "createDrawableArrow",
					       "(DDDDDDDD)Lfsa_editor/DrawableArrow;" );
  createDrawableCircle_mid   = tryGetMethodID( fsaCiderInterface_cls,
					       "createDrawableCircle",
					       "(DDD)Lfsa_editor/DrawableCircle;" );
  createDrawableText_mid     = tryGetMethodID( fsaCiderInterface_cls,
					       "createDrawableText",
					       "(Ljava/lang/String;DDDD)"
					       "Lfsa_editor/DrawableText;" );
  setArrowPoints_mid         = tryGetMethodID( fsaCiderInterface_cls,
					       "setArrowPoints",
					       "(Lfsa_editor/DrawableArrow;DDDDDD)V" );
  setCirclePos_mid           = tryGetMethodID( fsaCiderInterface_cls,
					       "setCirclePos",
					       "(Lfsa_editor/DrawableCircle;DD)V" );
  setCircleRadius_mid        = tryGetMethodID( fsaCiderInterface_cls,
					       "setCircleRadius",
					       "(Lfsa_editor/DrawableCircle;D)V" );
  setTextPos_mid             = tryGetMethodID( fsaCiderInterface_cls,
					       "setTextPos",
					       "(Lfsa_editor/DrawableText;DD)V" );
  setTextHalfDimensions_mid  = tryGetMethodID( fsaCiderInterface_cls,
					       "setTextHalfDimensions",
					       "(Lfsa_editor/DrawableText;DD)V" );

  // TODO: rename me!
  jobject fsaCiderInterface_lref
    = env->NewObject( fsaCiderInterface_cls, fsaCiderInterfaceConstructor_mid );
  if ( fsaCiderInterface_lref == 0 ) {
    qDebug( "Thread: Can't create instance of fsa_editor/FSACiderInterface" ); // TODO: handle me properly!
    cleanup();
    return;
  }
  fsaCiderInterface_gref = env->NewGlobalRef( fsaCiderInterface_lref );
  if ( fsaCiderInterface_gref == 0 ) {
    qDebug( "Thread: Can't create global ref of fsa_editor/FSACiderInterface object" ); // TODO: handle me properly!
    cleanup();
    return;
  }
  
  JNINativeMethod nativeMethods[5];
  nativeMethods[0].name = "native_symbolAdded";
  nativeMethods[0].signature = "(I)V";
  nativeMethods[0].fnPtr = (void*)Java_CiderInterface_native_symbolAdded;
  nativeMethods[1].name = "native_symbolRemoved";
  nativeMethods[1].signature = "(I)V";
  nativeMethods[1].fnPtr = (void*)Java_CiderInterface_native_symbolRemoved;
  nativeMethods[2].name = "native_arrowPointsModified";
  nativeMethods[2].signature = "(IDDDDDD)V";
  nativeMethods[2].fnPtr = (void*)Java_FSACiderInterface_native_arrowPointsModified;
  nativeMethods[3].name = "native_circleModified";
  nativeMethods[3].signature = "(IDDD)V";
  nativeMethods[3].fnPtr = (void*)Java_FSACiderInterface_native_circleModified;
  nativeMethods[4].name = "native_textRectModified";
  nativeMethods[4].signature = "(IDDDD)V";
  nativeMethods[4].fnPtr = (void*)Java_FSACiderInterface_native_textRectModified;
  if ( env->RegisterNatives(fsaCiderInterface_cls, nativeMethods, 5) != 0 ) { // 0 -> success
    qDebug( "Thread: Couldn't register natives" ); // TODO: handle me properly!
    cleanup();
    return;
  }
}


FsaDiagramController::~FsaDiagramController()
{
  cleanup();
}


void FsaDiagramController::cleanup()
{
  foreach ( jobject s, m_elementPtrToSymbol )
    env->DeleteGlobalRef( s );
  m_elementPtrToSymbol.clear();
  m_symbolIdToElementPtr.clear();

  // TODO: move me elsewhere?
  env->DeleteGlobalRef( fsaCiderInterface_gref );
  env->DeleteGlobalRef( fsaCiderInterface_cls );
  fsaCiderInterface_gref = 0;
  fsaCiderInterface_cls = 0;

  // Update envToController - used for the native callbacks
  envToControllerLock.lockForWrite();
  Q_ASSERT( envToController.contains(env) ); // sanity check - must contain us
  envToController.remove( env );
  envToControllerLock.unlock();

  if ( env->ExceptionOccurred() )
    env->ExceptionDescribe(); // TODO: handle me properly!
  m_jvm->DetachCurrentThread();
  env = 0;
}


void FsaDiagramController::elementAddEvent( CElementAddEvent* event )
{
  Q_ASSERT( event != 0 );
  if ( ! tryCiderElementAdd(event->element()) )
    GuiDiagramController::elementAddEvent( event );
}

void FsaDiagramController::elementRemoveEvent( CElementRemoveEvent* event )
{
  Q_ASSERT( event != 0 );
  if ( ! (m_elementPtrToSymbol.contains(event->element())
	  && tryCiderElementRemove(event->element())) )
    GuiDiagramController::elementRemoveEvent( event );
}

void FsaDiagramController::elementAttributeAddEvent( CElementAttributeAddEvent* event )
{
  // TODO: fixme: use this event properly!
  Q_ASSERT( event != 0 );
   if ( ! (m_elementPtrToSymbol.contains(event->element())
  	  && tryCiderElementModify(event->element(), event->attribute(), QVariant())) )//remove variant hack
  GuiDiagramController::elementAttributeAddEvent( event );
}

void FsaDiagramController::elementAttributeRemoveEvent( CElementAttributeRemoveEvent* event )
{
  // TODO: fixme: use this event properly!
  Q_ASSERT( event != 0 );
    if ( ! (m_elementPtrToSymbol.contains(event->element())
	    && tryCiderElementModify(event->element(), event->attribute(), QVariant())) )//remove variant hack
      GuiDiagramController::elementAttributeRemoveEvent( event );
}

void FsaDiagramController::elementAttributeChangeDataEvent( CElementAttributeChangeDataEvent* event )
{
  // FIX PROBLEM: as attributes are modified (e.g. shape: circle->square->circle) cider may need to add, remove!
  //              make circle just a shape in cider with a type as a param?
  Q_ASSERT( event != 0 );
  if ( ! (m_elementPtrToSymbol.contains(event->element())
  	  && tryCiderElementModify(event->element(), event->attribute(), event->data())) )
    GuiDiagramController::elementAttributeChangeDataEvent( event );
}


/*!
 * This should only be called by CiderInterface (in Java,
 * via Java_CiderInterface_native_symbolAdded).
 *
 * \b Note: "cider" is explicit, as Digest elements have their own IDs.
 */
void FsaDiagramController::onSymbolAdded( jint symbolId )
{
  // TODO: make assertions
  AbstractElement* e = m_symbolIdToElementPtr.value( symbolId );
  Q_ASSERT( e != 0 );
  CElementAddEvent event( e, this, false );
  GuiDiagramController::elementAddEvent( &event );
}

/*!
 * This should only be called by CiderInterface (in Java,
 * via Java_CiderInterface_native_symbolRemoved).
 */
void FsaDiagramController::onSymbolRemoved( jint symbolId )
{
  // TODO: make assertions
  AbstractElement* e = m_symbolIdToElementPtr.value( symbolId );
  jobject symbol = m_elementPtrToSymbol.value( e );
   Q_ASSERT( e != 0 );
  CElementRemoveEvent event( e, this, false );
  GuiDiagramController::elementRemoveEvent( &event );
  m_symbolIdToElementPtr.remove( symbolId );
  m_elementPtrToSymbol.remove( e ); 
  env->DeleteGlobalRef( symbol );
}


/*!
 * This should only be called by FsaDiagramController (in Java
 * via native_arrowPointsModified, native_circleModified, etc...)
 */
void FsaDiagramController::setElementAttributeData( jint symbolId,
						    int attributeType,
						    const QVariant& newAttributeData )
{
  AbstractElement* e = m_symbolIdToElementPtr.value( symbolId );
  Q_ASSERT( e != 0 );
  AbstractElementAttribute* a = e->attribute( attributeType );
  if ( a != 0 ) { // better & safer not to assert this
    CElementAttributeChangeDataEvent event( e, a, newAttributeData, this, false );
    GuiDiagramController::elementAttributeChangeDataEvent( &event );
  }
}



bool FsaDiagramController::tryCiderElementAdd( AbstractElement* element )
{
  Q_ASSERT( env != 0 );
  Q_ASSERT( fsaCiderInterface_gref != 0 );
  Q_ASSERT( createDrawableArrow_mid != 0 );
  Q_ASSERT( createDrawableCircle_mid != 0 );
  Q_ASSERT( createDrawableText_mid != 0 );
  Q_ASSERT( printForest_mid != 0 );

  Q_ASSERT( element != 0 );

  bool     success = false;
  jobject  symbol_lref = 0;

  // Init the variables using the element's attributes
  //StrokeList ink;
  QString    label;
  QPointF    origin;
  QPolygonF  polygon;
  QSizeF     size;
  DiagramElementShape::Type shape = DiagramElementShape::Type(0);
  foreach ( const AbstractElementAttribute* a, element->attributes() )
    {
      Q_ASSERT( a != 0 );
      switch ( a->type() ) {
	//  case MvcDiagram::DEA_Ink:
	//	ink = a->toVariant().value<StrokeList>(); break;
      case MvcDiagram::DEA_Label:
	label = a->toVariant().toString(); break;
      case MvcDiagram::DEA_Polygon:
	polygon = a->toVariant().value<QPolygonF>(); break;
      case MvcDiagram::DEA_Origin:
	origin = a->toVariant().toPointF(); break;
      case MvcDiagram::DEA_Shape:
	shape = DiagramElementShape::Type(a->toVariant().toInt()); break;
      case MvcDiagram::DEA_Size:
	size = a->toVariant().toSizeF(); break;
      default:
	break;
      }
    }

  switch ( element->type() )
    {
    case MvcDiagram::DE_Polyline:
      while ( polygon.size() < 3 ) polygon.append( QPointF(0.0,0.0) ); // TODO: this VERY UGLY hack!
      if ( polygon.size() >= 3 ) {
	// TODO: ensure that this does not get garbaged collected yet!
	polygon.translate( origin );
	symbol_lref = env->CallObjectMethod( fsaCiderInterface_gref,
					     createDrawableArrow_mid,
					     polygon.at(0).x(), polygon.at(0).y(),
					     polygon.at(1).x(), polygon.at(1).y(),
					     polygon.at(2).x(), polygon.at(2).y() );
      }
      break;

    case MvcDiagram::DE_Shape:
      {
	float radius = size.width() / 2.0;
	if ( shape == DiagramElementShape::Circle )
	  // TODO: ensure that this does not get garbaged collected yet!
	  symbol_lref = env->CallObjectMethod( fsaCiderInterface_gref,
					       createDrawableCircle_mid, radius,
					       origin.x() + radius,
					       origin.y() + radius );
      }
      break;

    case MvcDiagram::DE_Text:
      {
	qreal hW = size.width() / 2.0;
	qreal hH = size.height() / 2.0;
	// TODO: ensure that this does not get garbaged collected yet!
	symbol_lref = env->CallObjectMethod( fsaCiderInterface_gref,
					     createDrawableText_mid,
					     env->NewStringUTF(label.toUtf8().constData()),
					     origin.x() + hW, origin.y() + hH,
					     hW, hH );
	break;
      }
    }

  if ( symbol_lref != 0 ) {
    jobject symbol = env->NewGlobalRef( symbol_lref );
    if ( symbol == 0 ) {
      qDebug( "Thread: Can't create global ref of object" ); // TODO: handle me properly!
      cleanup();
      return false;
    }
    jint symbolId = env->CallIntMethod( fsaCiderInterface_gref, getIDNumber_mid, symbol );
    Q_ASSERT( ! m_symbolIdToElementPtr.contains(symbolId) );  // no duplicates
    Q_ASSERT( ! m_elementPtrToSymbol.contains(element) );    // no duplicates
    m_symbolIdToElementPtr.insert( symbolId, element );
    m_elementPtrToSymbol.insert( element, symbol );
    success = env->CallBooleanMethod( fsaCiderInterface_gref, add_mid, symbol );
  }

  //env->CallVoidMethod( fsaCiderInterface_gref, printForest_mid ); // TODO: remove me

  return success;
}


/*!
 * This should only be called via a queued signal-slot connection by CiderView.
 */
bool FsaDiagramController::tryCiderElementRemove( AbstractElement* element )
{
  Q_ASSERT( element != 0 );

  Q_ASSERT( env != 0 );
  Q_ASSERT( fsaCiderInterface_gref != 0 );
  Q_ASSERT( remove_mid != 0 );

  bool success = false;
  jobject symbol = m_elementPtrToSymbol.value( element );

  // Note: CIDER does not interact with all elements - ignore if we have no record
  if ( symbol != 0 )
    success = env->CallBooleanMethod( fsaCiderInterface_gref, remove_mid, symbol );

  return success;
}


// NOTE: the data must come from the \em newAttributeData parameter, as the attribute won't have been updated yet
//       -> this is intentional! as CIDER may want to set it to a diff value -> prevent useless updates!
bool FsaDiagramController::tryCiderElementModify( AbstractElement* element,
						  AbstractElementAttribute* attribute,
						  const QVariant& newAttributeData )
{
  Q_ASSERT( element != 0 );
  Q_ASSERT( attribute != 0 );

  Q_ASSERT( env != 0 );
  Q_ASSERT( fsaCiderInterface_gref != 0 );
  Q_ASSERT( setArrowPoints_mid != 0 );
  Q_ASSERT( setCirclePos_mid != 0 );
  Q_ASSERT( setCircleRadius_mid != 0 );
  Q_ASSERT( setTextPos_mid != 0 );
  Q_ASSERT( setTextHalfDimensions_mid != 0 );
  Q_ASSERT( printForest_mid != 0 );

  bool success = false;
  jobject symbol = m_elementPtrToSymbol.value( element );
  int attributeType = attribute->type();


  // Important: CIDER does not interact with all elements - ignore if we have no record
  if ( symbol == 0 ) return false;


  // Init the variables using the element's attributes
  //StrokeList ink;
  // TODO: remove what's not used
  QString    label;
  QPointF    origin;
  QPolygonF  polygon;
  QSizeF     size;
  DiagramElementShape::Type shape = DiagramElementShape::Type(0);

  foreach ( const AbstractElementAttribute* a, element->attributes() )
    {
      Q_ASSERT( a != 0 );
      if ( a == attribute ) continue; // don't waste time on it, we use newAttributeData
      switch ( a->type() ) {
	//  case MvcDiagram::DEA_Ink:
	//	ink = a->toVariant().value<StrokeList>(); break;
      case MvcDiagram::DEA_Label:
	label = a->toVariant().toString(); break;
      case MvcDiagram::DEA_Polygon:
	polygon = a->toVariant().value<QPolygonF>(); break;
      case MvcDiagram::DEA_Origin:
	origin = a->toVariant().toPointF(); break;
      case MvcDiagram::DEA_Shape:
	shape = DiagramElementShape::Type(a->toVariant().toInt()); break;
      case MvcDiagram::DEA_Size:
	size = a->toVariant().toSizeF(); break;
      default:
	break;
      }
    }

  switch ( element->type() )
    {
    case MvcDiagram::DE_Polyline:
      {
	if ( attributeType == MvcDiagram::DEA_Origin )
	  origin = newAttributeData.toPointF();
	else if ( attributeType == MvcDiagram::DEA_Polygon )
	  polygon = newAttributeData.value<QPolygonF>();

	if ( polygon.size() >= 3 ) {
	  polygon.translate( origin );
	  env->CallVoidMethod( fsaCiderInterface_gref,
			       setArrowPoints_mid, symbol,
			       polygon.at(0).x(), polygon.at(0).y(),
			       polygon.at(1).x(), polygon.at(1).y(),
			       polygon.at(2).x(), polygon.at(2).y() );
	  success = true;
	}
      }
      break;

    case MvcDiagram::DE_Shape:
      {
	if ( shape == DiagramElementShape::Circle )
	  {
	    if ( attributeType == MvcDiagram::DEA_Origin ) {
	      origin = newAttributeData.toPointF();
	      qreal radius = size.width() / 2.0;
	      env->CallVoidMethod( fsaCiderInterface_gref,
				   setCirclePos_mid, symbol,
				   origin.x() + radius,
				   origin.y() + radius );
	      success = true;
	    }
	    else if ( attributeType == MvcDiagram::DEA_Size ) {
	      size = newAttributeData.toSizeF();
	      env->CallVoidMethod( fsaCiderInterface_gref,
				   setCircleRadius_mid, symbol,
				   size.width() / 2.0 );
	      success = true;
	    }
	  }
      }
      break;

    case MvcDiagram::DE_Text:
      {
	// TODO: handle a label change
	if ( attributeType == MvcDiagram::DEA_Origin ) {
	  origin = newAttributeData.toPointF();
	  env->CallVoidMethod( fsaCiderInterface_gref,
			       setTextPos_mid, symbol,
			       origin.x() + (size.width() / 2.0),
			       origin.y() + (size.height() / 2.0) );
	  success = true;
	}
	else if ( attributeType == MvcDiagram::DEA_Size ) {
	  size = newAttributeData.toSizeF();
	  env->CallVoidMethod( fsaCiderInterface_gref,
			       setTextHalfDimensions_mid, symbol,
			       size.width() / 2.0,
			       size.height() / 2.0 );
	  success = true;
	}
      }
      break;
    }

  return success;
}


jmethodID FsaDiagramController::tryGetMethodID( jclass cls, const char* name, const char* sig )
{
  jmethodID mid = env->GetMethodID( cls, name, sig );
  if ( mid == 0 ) {
    qDebug( "Thread: Can't find method: %s", name );
    exit( 1 ); // TODO: handle me properly!
  }
  return mid;
}
