/*
 *  digestapplication.cpp
 *  Nodal
 *
 *  Imported into Nodal by Aidan Lane on Thu Feb 24 2005.
 *  Copyright (c) 2005 CEMA, Monash University. All rights reserved.
 *
 *  Original file:
 *
 *    digestapplication.cpp
 *    EverGreen
 *
 *    Created by Aidan Lane on Mon Jul 19 2004.
 *    Copyright (c) 2004 Aidan Lane. All rights reserved.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

//TODO: if we add a grey background in for Windows and Linux, then add the Window menu back for them!

#include "digestapplication.h"

#include <QAction>
#include <QActionGroup>
#include <QDebug> // TODO: remove me
#include <QDesktopWidget>
#include <QDir> // TODO: remove me!
#include <QEventLoop>
#include <QFileDialog>
#include <QFileOpenEvent>
#include <QMenuBar>
#include <QMessageBox>
#include <QSqlQuery>

#ifdef Q_WS_MAC
#include <Carbon/Carbon.h> // mainly for CFBundleCopyBundleURL & CFURLCopyFileSystemPath
#endif

#include "MvcDiagram/diagram.h"
#include "MvcDigestDb/digestdbmodel.h"
#include "MvcDigestDb/digestdbcontroller.h"
#include "MvcSettings/settingsmodel.h"
#include "MvcSettings/settingscontroller.h"

#include "digestactionmanager.h"
#include "digestcontrollerthreadhost.h"
#include "elementinspector.h"
#include "fsadiagramcontroller.h"
#include "mdidiagramwindow.h"

#include "ui_aboutboxform.h"


// TODO: allow the following to be modified
#define DATABASE_DRIVER_STR       "QSQLITE"

#define DEFAULT_FILENAME_BASE  tr("Untitled") // TODO: move me to the digest.h
#define FILE_FILTER            (Digest::appName + " Diagram (*." \
				+ Digest::filenameExtension + tr(")")) // TODO: move me to the digest.h

#define DEFAULT_ELEMENT_INSPECTOR_WIDTH   280
#define DEFAULT_ELEMENT_INSPECTOR_HEIGHT  500



DigestApplication::DigestApplication( int& argc, char** argv )
  : MDIApplication(argc, argv),
    m_jvm(0),
    m_dbModel(0),
    m_dbController(0),
    m_settingsModel(0),
    m_settingsController(0)
{
  // General initialisation
  init();
}


DigestApplication::~DigestApplication()
{
  /* TODO: INVESTIGATE TO SEE IF THE FOLLOWING IS STILL TRUE!
   * All MVC sets (diagrams, diagram controllers and diagram windows) will
   * automatically be deleted, as the diagram windows will be closed by appQuit().
   * Each close call will itself call our childWindowCloseRequest() method, which
   * looks after all memory de-allocation.
   */

  // The Element Inspector and the Gesture and SQL Browsers are **NOT** children
  // of us -> they need to be deleted explicitly!
  delete m_elementInspector;

  Q_ASSERT( m_jvm != 0 );
  jint res = m_jvm->DestroyJavaVM(); // yes, it's an odd way to destroy it!
  if ( res < 0 ) {
    qDebug( "Couldn't destroy the Java VM" );
    exit( 1 ); // TODO: handle me properly
  }

  /* Note: THIS IS EXTREMELY IMPORTANT!!!
   *       If we don't do this, we get intermittent spurts of
   *       "database table is locked Unable to fetch row" from lastError().
   *
   * Also, from the Qt 4.0.0 doc:
   * close() "Closes the database connection, freeing any resources acquired.
   * This will also affect copies of this QSqlDatabase object."
   * and:
   * removeDatabase() "Removes the database connection connectionName from the
   * list of database connections. Warning: There should be no open queries on
   * the database connection when this function is called, otherwise a resource
   * leak will occur."
   */
  m_db.close();
  QSqlDatabase::removeDatabase( m_dbConnectionName );
}


/*!
 * Remember that we are not a derivative of QWidget, so we must explictly tell
 * each of a children to be shown if we want them to be so.
 */
// TODO: rememeber settings and only ceate a new diagram if the settings tell it to.
void DigestApplication::init()
{
  // TODO: CLEANUP!
  // TODO: CLEANUP!
  // TODO: CLEANUP!


  /*
   * INIT DATABASE
   */
  m_db = QSqlDatabase::addDatabase( DATABASE_DRIVER_STR ); // TODO: make the driver a pref!

  QString dataDir = QDir::homePath() + "/Library/Digest/"; // TODO: keep me in the app's settings!
  QDir dir;
  if ( ! dir.exists(dataDir) )
    dir.mkdir( dataDir );
  m_dbConnectionName = dataDir + "/database.dat";//:memory:" );      // TODO: perhaps make the database name a parameter!?!?

  m_db.setDatabaseName( m_dbConnectionName );
  if ( ! m_db.open() ) {
    qDebug("ERROR OPENING DATABASE!");
    /*
      TODO: emit a error signal! - catch that in the gui code to report it!
    // TODO: improve error message
    QMessageBox::critical(0, qApp->tr("Cannot open database"),
			  qApp->tr("Unable to establish a database connection.\n\n"
				   "Click Cancel to exit."), QMessageBox::Cancel,
			  QMessageBox::NoButton);
    */
    return;
  }
  DigestDbModel::createTables( m_db );


  /*
   * INIT DATABASE MVC SYSTEM
   */
  m_dbModel = new DigestDbModel( m_db, this );
  m_dbController = new DigestDbController( m_dbModel, this );


  /*
   * INIT SETTINGS MVC SYSTEM
   */
  m_settingsModel = new SettingsModel( Digest::orgName, Digest::appName, this );
  m_settingsController = new SettingsController( m_settingsModel, this );


  /*
   * INIT (GLOBAL) JAVA VIRTUAL MACHINE
   */
  initJvm();

  /*
   * INIT USER INTERFACE
   */
  setupUi();

  /* If the user passed filenames then try to open them
   * Translate the command line arguments into QFileOpenEvents, which we
   * already handle (that method also directly handles Mac OS X Finder's
   * file open events).
   */
  for ( int i=1; i < MDIApplication::argc(); ++i ) { // note: argv[0] == app name, not real arg
#ifdef Q_WS_MAC // ignore the process number flag that the Finder passes
    if( QString(argv()[i]).startsWith("-psn") ) continue;
#endif
    postEvent( this, new QFileOpenEvent(argv()[i]) );
  }
}


void DigestApplication::initJvm()
{
  // TODO: cleanup!

  Q_ASSERT( m_jvm == 0 ); // sanity check - shouldn't get here more than once

  // Keep these alive - just in case they need to be accessed later (via pointers)
  static JNIEnv* env;
  static JavaVMInitArgs vm_args;
  static char classPathOption[4096]; // the app's path may be long and we repeat it 3x

  vm_args.version = JNI_VERSION_1_4;

  JNI_GetDefaultJavaVMInitArgs( &vm_args );

  // TODO: support class paths for other OSs!
#ifdef Q_WS_MAC
  CFURLRef    bundleRef = CFBundleCopyBundleURL( CFBundleGetMainBundle() );
  CFStringRef macPath   = CFURLCopyFileSystemPath( bundleRef,
						   kCFURLPOSIXPathStyle );
  const char* macPathPtr
    = CFStringGetCStringPtr( macPath, CFStringGetSystemEncoding() );
  snprintf( classPathOption, 4096,
	    "-Djava.class.path="
	    "%s/Contents/Libraries/"
	    ":%s/Contents/Libraries/GestureRecognition.jar"
	    ":%s/Contents/Libraries/CiderCore-0.3.0.jar"
	    ":%s/Contents/Libraries/fsa_editor.jar",
	    macPathPtr, macPathPtr, macPathPtr, macPathPtr );
  CFRelease( bundleRef );
  CFRelease( macPath );
#else
  // TODO: finish me!
  snprintf( classPathOption, 4096,
	    "-Djava.class.path=.:GestureRecognition.jar:CiderCore-0.3.0.jar:fsa_editor.jar" );
#endif

  // Append OUR class path to the end of default system class path
  JavaVMOption options[1];
  vm_args.nOptions = 1;
  options[0].optionString = classPathOption;
  //  DOESN'T WORK STILL!!! options[1].optionString = "-Djava.awt.headless=true"; // REQUIRES JAVA 1.4 - TODO: DOC WHY THIS IS SO EXTREAMLY IMPORTANT!!! (WILL CRASH OTHERWISE ON MACOSX)
  vm_args.options = options;
  vm_args.ignoreUnrecognized = JNI_TRUE;

  // Create the Java VM
  jint res = JNI_CreateJavaVM( &m_jvm,(void**)&env, &vm_args );
  if ( res < 0 ) {
    qDebug( "Can't create Java VM" );
    exit( 1 ); // TODO: handle me properly
  }
}


bool DigestApplication::event( QEvent* e )
{
  Q_ASSERT( e != 0 );

  bool eaten = false;

  switch ( e->type() )
    {
    case QEvent::FileOpen: {
      // Note: Qt should post this event BEFORE the ApplicationActivated event
      // Note: Mac OS X's Finder does NOT open files using command line arguments,
      //       instead it passes an event, which Qt forwards to us as a QFileOpenEvent.
      eaten = true;

      QFileOpenEvent* fe = static_cast<QFileOpenEvent*>( e );

      // Create a new Model-View-Controler managed diagram.
      // We call with createFirstPlayer=false, as were about to CFileEvent::FileInitFrom.
      MDIDiagramWindow* view = createNewMvcSet();
      if ( m_managedDiagrams.value(view).model == 0 ) {
	closeDiagram( view ); // TODO: FIXME!
	return eaten;
      }

      postEvent( m_managedDiagrams.value(view).cached_controller,
		 new CFileEvent(CFileEvent::FileInitFrom, fe->file()) );

      QFileInfo fileInfo( fe->file() );
      m_prevPath = fileInfo.absolutePath();

       // Change to the most useful tool for an EXISTING diagram
      Q_ASSERT( m_actionManager != 0 );
      Q_ASSERT( m_actionManager->selectionToolAction() != 0 );
      m_actionManager->selectionToolAction()->trigger();

      break;
    }

    case QEvent::ApplicationActivated:
      // Note: Qt should post this event AFTER any FileOpen event
      // If there are no managed diagrams yet (i.e. there was no FileOpen event,
      // or there was but it failed) then create a newDiagram.
      if ( m_managedDiagrams.isEmpty() ) {
	newDiagram();
	// TODO: possibly re-enable me:
#if 0
	// Change to the most useful tool for a NEW diagram
	Q_ASSERT( m_actionManager != 0 );
	Q_ASSERT( m_actionManager->shapeToolAction() != 0 );
	m_actionManager->shapeToolAction()->trigger();
#endif
      }
      break;

    default:
      eaten = MDIApplication::event( e );
      break;
    }

  return eaten;
}



/*!
 * Create the user interface. This should only be called by the constructor.
 *
 * This will make a call to createMenuBar().
 */
void DigestApplication::setupUi()
{
  m_actionManager = new DigestActionManager( this );


  createMenuBar();

   // WARNING! We are using avaliableGeometry, as we want to account for docks, global menu bar, task bars, etc...
  const QDesktopWidget* dw = MDIApplication::desktop();
  QRect availableGeometry = dw->availableGeometry();

  /*
   * Create the Element Inspector
   */
  // TODO: clean-up!
  m_elementInspector = new ElementInspector( 0, Qt::Tool ); // TODO: use the "magical parent"
  // TODO: remove me: int totalFrameWidth = m_elementInspector->frameGeometry().width() - m_elementInspector->width();
  int frameAndTitleHeight = m_elementInspector->frameGeometry().height() - m_elementInspector->height();
  m_elementInspector->resize( DEFAULT_ELEMENT_INSPECTOR_WIDTH, DEFAULT_ELEMENT_INSPECTOR_HEIGHT % (availableGeometry.height() - frameAndTitleHeight) );
  m_elementInspector->move( availableGeometry.right() - m_elementInspector->frameGeometry().width(), availableGeometry.top() // top-right hand corner - TODO: save and restore this pos!
#ifdef Q_WS_MAC
			    + 4 // TODO: remove this hack -> currently required (snapshot-20050411) so that the top of the element inspector is not hidden by the Mac menu bar
#endif
			    );
  connect( m_elementInspector,
	   SIGNAL(visibilityToggled()),
	   SIGNAL(elementInspectorVisibilityToggled()) ); // pass it on...

 
  // TODO: remember if these should be shown or not!
  Q_ASSERT( ! m_elementInspector.isNull() );
  m_elementInspector->show();
}



/*!
 * The action manager must have been created before calling this.
 * Asserts that the action manager is non-null.
 */
void DigestApplication::createMenuBar() // TODO: dissolve me
{
  Q_ASSERT( m_actionManager != 0 );

  // TODO: tune for Windows and Linux!

  // On Mac OS X, prevent the any icons (for actions) from appearing in the menubar
  // TODO: remove this if & when QMenu::setIconsVisible() is implemented by Trolltech
#ifdef Q_WS_MAC // _IS_ Mac
  extern void qt_mac_set_menubar_icons(bool b); // from qmenu_mac.cpp
  qt_mac_set_menubar_icons( false );
#endif


  m_globalMenuBar = new QMenuBar(); // TODO: DOC ME: intensionally parent-LESS -> need another version for windows...
  Q_ASSERT( m_globalMenuBar != 0 );
  
  m_fileMenu = m_globalMenuBar->addMenu(tr("&File"));
  foreach (QAction *action, m_actionManager->fileActions()->actions()) {
    m_fileMenu->addAction( action );
    if ( action->text() == DigestActionManager::tr("&Open...") ) {
      m_recentFilesMenu = m_fileMenu->addMenu( tr("Open &Recent") );
      // Pop the "Recent Files" stuff in here.
      //TODO: re-enable me:      
//      foreach(QAction *recentAction, m_actionManager->recentFilesActions()->actions())
	//	recentFilesMenu->addAction(recentAction);
      }
  }
  
  m_editMenu = m_globalMenuBar->addMenu( tr("&Edit") );
  foreach ( QAction *action, m_actionManager->editActions()->actions() )
    m_editMenu->addAction( action );

  m_viewMenu = m_globalMenuBar->addMenu( tr("&View") );
  foreach ( QAction *action, m_actionManager->viewActions()->actions() )
    m_viewMenu->addAction( action );

  m_toolMenu = m_globalMenuBar->addMenu( tr("&Tool") );
  foreach ( QAction *action, m_actionManager->toolActions()->actions() )
    m_toolMenu->addAction( action );

  m_windowMenu = m_globalMenuBar->addMenu( tr("&Window") );
  foreach ( QAction *action, m_actionManager->windowActions()->actions() )
    m_windowMenu->addAction( action );

#if 0
  m_helpMenu = m_globalMenuBar->addMenu( tr("&Help") );
  foreach ( QAction *action, m_actionManager->helpActions()->actions() )
    m_helpMenu->addAction( action );
#else
  
#ifdef Q_WS_MAC
  // We can put it anywhere, as Qt will move / merge it into the app menu
  m_fileMenu->addAction( m_actionManager->appAboutAction() );
#else
  m_helpMenu = m_globalMenuBar->addMenu( tr("&Help") );
  m_helpMenu->addAction( m_actionManager->appAboutAction() );
#endif

#endif
}



/*!
 * This is all that external objects need to know about the element inspector.
 *
 * We don't expose its class to anyone else, nor the burden of having to know how
 * to manage it.
 *
 * If our pointer to the element inspector is null (and since we use a guarded
 * pointer, that would mean that it's not "alive"), this will return false
 * (valid scenario).
 */
bool DigestApplication::isElementInspectorVisible() const {
  return ( (! m_elementInspector.isNull())
	   && m_elementInspector->isVisible() );
}


/*!
 * Re-implemented form MDIApplication.
 *
 * Looks after the actual clean-up of the diagram (unlike closeDiagram(),
 * which just initiates the close).
 *
 * Returns false if the request is rejected, otherwise it returns true.
 */
bool DigestApplication::childWindowCloseRequest( MDIChildBase* child )
{
  Q_ASSERT( child != 0 );
  MDIDiagramWindow* win = static_cast<MDIDiagramWindow*>( child ); // we only put MDIDiagramWindows into it

  Diagram* diagram = m_managedDiagrams.value(win).model;
  Q_ASSERT( diagram != 0 ); // window exists -> so should its model!

  
  // Ask the user if they want to save any unsaved-changes first!
  if ( diagram->isModified() )
    {
      /* Note: We're not using QMessageBox::question(), as we want the message
       *       box to be presented as a "sheet" on Mac OS X.
       *       Also, Mac OS X uses a different button ordering than KDE on Linux
       *       and Windows. Lastly, we use the caption for the short question text
       *       on the Mac, as it's not a dialog, but instead a sheet.
       */
       QMessageBox* mb = new QMessageBox(
#ifdef Q_WS_MAC // _IS_ Mac OS X
					 tr(""), // note: the caption would give bold text, but the dist b/n it and the main text is too large.
					 tr("<html>"
					    "<span style=\"font-size:13pt; font-weight:600;\">"
					    "Do you want to save changes to this diagram\n"
					    " before closing?"
					    "</span>"
					    "<br/><br/>" // note: <br> provides a small gap, as to match Mac OS X's
					    "<span style=\"font-size:11pt\">"
					    "If you don't save, your changes will be lost."
					    "</span>"
					    "</html>"),
					 QMessageBox::Question,
					 QMessageBox::No,
					 QMessageBox::Cancel | QMessageBox::Escape,
					 QMessageBox::Yes | QMessageBox::Default, 
					 win, Qt::Sheet
#else
					 Digest::appName,
					 tr("Do you want to save changes to this diagram\n"
					    " before closing?"),
					 QMessageBox::Question,
					 QMessageBox::Yes | QMessageBox::Default, 
					 QMessageBox::No,
					 QMessageBox::Cancel | QMessageBox::Escape,
					 win, Qt::Dialog | Qt::MSWindowsFixedSizeDialogHint
#endif
					);
#ifdef Q_WS_MAC // _IS_ Mac OS X
       mb->setIconPixmap( QPixmap(":/images/Digest64.png") );
       // TODO: remove the need for the following
       mb->setMinimumSize( 420, 150 );
       mb->setMaximumSize( 420, 150 );
#endif

      // Set button names to ones that don't require the user to really read the
      // message box's text - safer and more intuitive.
      mb->setButtonText( QMessageBox::Yes,    tr("&Save") );
      mb->setButtonText( QMessageBox::No,     tr("&Don't Save") );
      mb->setButtonText( QMessageBox::Cancel, tr("&Cancel") );

      switch( mb->exec() ) {
      case QMessageBox::Yes: // Save clicked or Alt+S pressed or Enter pressed.
	// Save and close document
	if ( !saveDiagram(child) ) return false; // if save failed then reject close request
	break;
      case QMessageBox::No: // Discard clicked or Alt+D pressed
	// Don't save BUT do close document
	// WARNING: Don't return yet, we still need to close the document
	break;
      case QMessageBox::Cancel: // Cancel clicked or Escape pressed
	// Don't save AND don't close document
	return false; // REJECT close request
      }
    }



  /*
   *  WARNING WARNING WARNING!
   *
   * We MUST make sure that the element inspector does not refer to any diagram
   * controller before we do ANYTHING! Otherwise, if there are 2+ diagram windows
   * open, then the program will crash (if there is only one, then setActiveChild(0)
   * will be called when the child window is removed, which will make a similar call).
   */
  Q_ASSERT( ! m_elementInspector.isNull() );
  if ( m_elementInspector->controller() != 0 )
    m_elementInspector->controller()->detachView( m_elementInspector );

  /*
   * VERY IMPORTANT:
   * We need to make sure that all posted events have been handled before we
   * close the diagram! Otherwise, if for example we've just posted a FileSaveTo
   * event, it may not have been processed yet. Hence, we explicitly ask for all
   * events (other than new user input events) to processed immediately.
   */
  MDIApplication::processEvents( QEventLoop::ExcludeUserInputEvents );

  // CLOSE THE DIAGRAM ***VIEW***
  MDIApplication::childWindowCloseRequest( child ); // deletes the diagram window view

  QMutableHashIterator<MDIDiagramWindow*, DiagramMvcStruct> i(m_managedDiagrams);
  while ( i.hasNext() )
    {
      i.next();
      if ( i.key() == child )
	{
	  // Free the memory used by the MVC components in an orderly fashion
	  // UNNEEDED: delete it.key - MDIApplication::childWindowCloseRequest does this for us!	  
	  delete i.value().model;
	  delete i.value().controllerThreadHost; // - MUST be deleted AFTER its model and its views
	  i.remove(); // remove any records of it
	  break;      // it was found -> we're finished now!
	}
    }

  return true; // ACCEPT close request
}


/*!
 * Re-implemented form MDIApplication.
 *
 * Emits activeChildChanged  EVEN IF the child is null
 * (it is important to know if there are no diagram windows!).
 */
void DigestApplication::setActiveChild( MDIChildBase* child )
{
  MDIApplication::setActiveChild( child );

  // TODO: cleanup!

  // Set the appropriate diagram controller for the element inspector
  // - if there is no active child, then we pass 0, as to clear&disable the inspector
  Q_ASSERT( ! m_elementInspector.isNull() );
  if ( m_elementInspector->controller() != 0 )
    m_elementInspector->controller()->detachView( m_elementInspector );
  if ( child != 0 // we only create MDIDiagramWindow children:
       && m_managedDiagrams.contains(static_cast<MDIDiagramWindow*>(child))
       && m_managedDiagrams.value(static_cast<MDIDiagramWindow*>(child)).cached_controller != 0 )
    {
      m_managedDiagrams.value(static_cast<MDIDiagramWindow*>(child))
	.cached_controller->attachView( m_elementInspector );
    }

  emit activeChildChanged( child );
}



/*!
 * Create new diagram and set it as active.
 */
void DigestApplication::newDiagram()
{
  createNewMvcSet();
}


void DigestApplication::openDiagram()
{
  // If the user hasn't opened or saved a file yet, then use a "sane" path
  if ( m_prevPath.isEmpty() )
    m_prevPath = saneDefaultPath();

  // Note: WE DON'T WANT TO BE THE OWNER! - we don't want Mac OS X to create a
  // "drop-down sheet" for an OPEN dialog (unlike "Save As")!
  QString filename = QFileDialog::getOpenFileName( 0, tr("Open"), m_prevPath,
						   FILE_FILTER );

  // Note: the following also covers the case of when the user clicks cancel
  if ( filename.isEmpty() )
    return;

  // Create a new Model-View-Controler managed diagram.
  MDIDiagramWindow* view = createNewMvcSet();
  if ( m_managedDiagrams.value(view).model == 0 ) {
    closeDiagram( view ); // TODO: FIXME!
    return;
  }

  postEvent( m_managedDiagrams.value(view).cached_controller,
	     new CFileEvent(CFileEvent::FileInitFrom, filename) );

  QFileInfo fileInfo( filename );
  m_prevPath = fileInfo.absolutePath();
}


/*!
 * Tells the active child DOCUMENT window to close, if there is one.
 *
 * If "child" is null, then activeChildWindow() will be used.
 *
 * Note: This differs from closeActiveWindow() in that this method only deals
 *       with the document windows, not others such as the preferences and
 *       about box dialogs.
 *       On File->Close, closeActiveWindow() should get called, not this.
 */
void DigestApplication::closeDiagram( MDIChildBase* child )
{
  if ( child == 0 ) child = activeChildWindow();
  Q_ASSERT( child != 0 );

  // WARNING!!! childWindowCloseRequest() looks after the actual clean-up
  static_cast<MDIDiagramWindow*>(child)->close();
}


/*!
 * This will be dispatched to the active child window, if there is one. 
 *
 * If "child" is null, then activeChildWindow() will be used.
 *
 * If the user hasn't save the file before, then they will be presented with the
 * "Save As" dialog, by saveDiagramAs().
 *
 * Returns false IFF the user was presented with a "Save As" dialog and then
 * clicked the "Cancel" button or escaped the dialog in any other way.
 * Hence, if the user isn't presented with the "Save As" dialog but the save
 * process actually failed, then this WILL return true. It's the responsibility
 * of the diagram controller to handle those types of errors.
 */
bool DigestApplication::saveDiagram( MDIChildBase* child )
{
  if ( child == 0 ) child = activeChildWindow();
  Q_ASSERT( child != 0 );

  MDIDiagramWindow* win = static_cast<MDIDiagramWindow*>( child ); // we only put MDIDiagramWindows into it

  Diagram* diagram = m_managedDiagrams.value(win).model;
  Q_ASSERT( diagram != 0 ); // window exists -> so should its model!

  const QString& currentFilename = diagram->filename();
  if ( currentFilename.isEmpty() ) {
    if ( ! saveDiagramAs(child) ) // the filename hasn't beeen set, so run "Save As" instead
      return false;
  }
  else
    postEvent( win->diagramController(),
	       new CFileEvent(CFileEvent::FileSaveTo, currentFilename) );

  QFileInfo fileInfo( currentFilename );
  m_prevPath = fileInfo.absolutePath();

  return true;
}


/*!
 * This will be dispatched to the active child window, if there is one.
 *
 * If "child" is null, then activeChildWindow() will be used.
 *
 * Returns false if the use cancels the save dialog or if the filename is empty,
 * else returns true.
 */
bool DigestApplication::saveDiagramAs( MDIChildBase* child )
{
  if ( child == 0 ) child = activeChildWindow();
  Q_ASSERT( child != 0 );

  MDIDiagramWindow* win = static_cast<MDIDiagramWindow*>( child ); // we only put MDIDiagramWindows into it

  /* Set the diagram window as THE active window, as this (saveDiagramAs) may
   * have been called by a method such as closeDiagram() or saveAllDiagrams(),
   * where the child being requested to be saved may not be focused and hence,
   * the save dialog would also not be focused.
   */
  win->activateWindow();
  win->raise();
  
  // If the user hasn't opened or saved a file yet, then use a "sane" path
  if ( m_prevPath.isEmpty() )
    m_prevPath = saneDefaultPath();

  // Get a "new" filename for the diagram
  Diagram* diagram = m_managedDiagrams.value(win).model;
  Q_ASSERT( diagram != 0 ); // window exists -> so should its model!
  QString filename = ( diagram->filename().isEmpty()
		       ? (m_prevPath + "/"
			  + diagram->name() + "." + Digest::filenameExtension)
		       : diagram->filename() );

  // Get new filename
  // Note: Qt looks after the case of when a user chooses an existing file,
  //       asking them for confirmation when they do so.
  filename = QFileDialog::getSaveFileName(win, tr("Save As"),
					  filename, FILE_FILTER );

  // Note: the following also covers the case of when the user clicks cancel
  if ( filename.isEmpty() )
    return false;

  postEvent( win->diagramController(),
	     new CFileEvent(CFileEvent::FileSaveTo,filename) );

  QFileInfo fileInfo( filename );
  m_prevPath = fileInfo.absolutePath();

  return true;
}


/*!
 * Calls saveDiagram() on all childWindows().
 *
 * Returns false IFF the user cancels any of the "Save As" dialogs (should they
 * be shown one), in which case this method will also be immediately stopped.
 */
bool DigestApplication::saveAllDiagrams()
{
  foreach ( MDIChildBase* child, childWindows() ) {
      Q_ASSERT( child != 0 );
      if ( ! saveDiagram(child) ) return false;
    }
  return true;
}


/*!
 * Note: This differs from closeNetwork() in that this method accounts for not
 *       only document windows but also for the preference & about boxes
 *       (BTW: these are non-modal on Mac OS X).
 *       This is what the action manager should call on File->Close, as to keep
 *       the app consistent with other apps on Mac OS X (i.e. Command+W closes
 *       any window other than tool windows). This may need altering for other
 *       windowing environments.
 */
void DigestApplication::closeActiveWindow()
{
  QWidget* w = activeWindow(); // cache it just in case it's expensive
  if ( w != 0 )
    w->close();
}


/*!
 * This should be called by the element inspector when it is made visible or
 * explicitly hidden. It should also be called by an appropriate menu action.
 *
 * Will emit elementInspectorVisibilityToggled() before returning.
 */
void DigestApplication::setElementInspectorVisible( bool visible )
{
  Q_ASSERT( ! m_elementInspector.isNull() );
  m_elementInspector->setVisible( visible );
  m_elementInspector->raise();
  emit elementInspectorVisibilityToggled();
}


/*!
 * Shows the non-modal preferences box dialog.
 *
 * If the preferences dialog is already open (but possibly behind another window),
 * then it will be brought to the foreground.
 *
 * The dialog is set to automatically delete itself when it is closed, which
 * will null-ify our guarded pointer, allowing a new one to be created later.
 */
void DigestApplication::showAppPrefs()
{
  if ( ! prefsDlg.isNull() )
    {
      // Already exists - bring it to the front
      prefsDlg->activateWindow();
      prefsDlg->raise();
    }
  else
    {
      // Create a new dialog and show it.
      // The dialog should be fixed in size and only have a border, tile-bar and close button.
      prefsDlg = new DigestPrefsDialog( m_settingsController,
					m_dbController, 0,
					Qt::Window
					| Qt::WindowTitleHint
					| Qt::WindowSystemMenuHint ); //TODO: use "magical" parent
      QRect r = MDIApplication::desktop()->availableGeometry();
      prefsDlg->move( r.width()/2 - prefsDlg->width()/2,
			 (int)((float)(r.height()/2 - prefsDlg->height()/2) * 0.5) ); // get centre then halve it
      prefsDlg->setAttribute( Qt::WA_DeleteOnClose );
      prefsDlg->show();
    }
}


// TODO: implement me!
void DigestApplication::showAppHelp()
{
  qDebug("DigestApplication::showAppHelp");
}


/*!
 * Shows the non-modal about box dialog. It looks and behaves according to Apple HIG.
 *
 * If the about box is already open (but possibly behind another window),
 * then it will be brought to the foreground.
 *
 * The dialog is set to automatically delete itself when it is closed, which
 * will null-ify our guarded pointer, allowing a new one to be created later.
 */
void DigestApplication::showAppAbout()
{
  if ( !aboutBoxDlg.isNull() )
    {
      // Already exists - bring it to the front
      aboutBoxDlg->activateWindow();
      aboutBoxDlg->raise();
    }
  else
    {
      // Create a new dialog and show it.
      // The dialog should be fixed in size and only have a border, tile-bar and close button.
      aboutBoxDlg = new QDialog( 0, Qt::Window | Qt::WindowTitleHint | Qt::WindowSystemMenuHint ); //TODO: use "magical" parent
      Ui::AboutBoxForm aboutBoxUi;
      aboutBoxUi.setupUi( aboutBoxDlg );
      QRect r = MDIApplication::desktop()->availableGeometry();
      aboutBoxDlg->move( r.width()/2 - aboutBoxDlg->width()/2,
			 (int)((float)(r.height()/2 - aboutBoxDlg->height()/2) * 0.5) ); // get centre then halve it
      aboutBoxDlg->setAttribute( Qt::WA_DeleteOnClose );
      aboutBoxDlg->show();
    }
}

/*!
 * Try to quit the application immediately.
 *
 * This does NOT depend on this being the last window to close.
 *
 * All MVC sets (diagrams, diagram controllers and diagram windows) will be
 * deleted, as WE will close all of the diagram windows.
 *
 * This allows the user to click "Cancel" in any "Save As" dialog or escape them
 * in any other way, which will terminate the quit process.
 */
void DigestApplication::appQuit()
{
  foreach ( MDIChildBase* child, childWindows() ) {
    if ( ! childWindowCloseRequest(child) ) return; // request failed -> stop quit process
  }

  MDIApplication::exit( 0 );
}



QString DigestApplication::saneDefaultPath()
{
  // WARNING! Don't cache the path, as the path may be deleted or a "better" one
  //          added between calls to this method.

  QString path = QDir::homePath();

#if defined( Q_WS_WIN ) // _IS_ Windows
  if ( QFile::exists(QDir::homePath() + "/My Documents") )
    path = QDir::homePath() + "/My Documents";
#endif
  /* All other OSs Mac OS X, Linux, etc... (use better folder names :-)
   * AND if the My* failed for Windows
   * (apparently My* will be dropped for Windows Longhorn anyway)
   */
  if ( QFile::exists(QDir::homePath() + "/Documents") )
    path = QDir::homePath() + "/Documents";

  return path;
}


/*!
 * Create new diagram and set it as active.
 *
 * This is for internal use only. It's currenly used by newDiagram() and
 * openDiagram(). It may be re-implemented.
 */
MDIDiagramWindow* DigestApplication::createNewMvcSet()
{
  // Generate a unique diagram name (unique to currently open diagrams)
  unsigned num = 1;
  foreach ( const DiagramMvcStruct& mvc, m_managedDiagrams )
    {
      if ( num <= 1
	   && mvc.model->name() == DEFAULT_FILENAME_BASE )
	++num;
      else if ( num > 1
		&& mvc.model->name() == DEFAULT_FILENAME_BASE
		+ " " + QString::number(num) )
	++num;
    }
  QString diagramName = ( (num <= 1)
			  ? DEFAULT_FILENAME_BASE
			  : DEFAULT_FILENAME_BASE + " " + QString::number(num) );

  /* Create the model-view-controller set.
   * We must be the owner of everything, so that when we're destroyed,
   * all models, views and controllers will be destroyed with us.
   */
  DiagramMvcStruct s;

  s.model = new Diagram( diagramName, this );

  DigestControllerThreadHost<FsaDiagramController>* host
    = new DigestControllerThreadHost<FsaDiagramController>( s.model, m_jvm, this );
  host->start();
  // Wait for the thread to start and the controller to be created.
  // Note: this is not the proper use of wait(), but give the thread some time.
  while ( host->controller() == 0 )
    host->wait( 10 ); // keep me small - we SHOULD have to wait
  s.controllerThreadHost = host;
  s.cached_controller = host->controller();

  MDIDiagramWindow* view // WARNING! this automatically calls addChildWindow()
    = new MDIDiagramWindow( s.cached_controller,
			    m_dbController,
			    m_settingsController,
			    this, m_jvm );

  /*
   * Set defaults for the view
   */
  view->setCurrentTool( Digest::SelectionTool );

  // TODO: fix the following hack! - it's not the usful anyway!
  // Try to fill then screen vertically and then derive the width from that...
  Q_ASSERT( MDIApplication::desktop() != 0 );
  QRect availableGeometry = MDIApplication::desktop()->availableGeometry();
  QRect frameGeometry = view->frameGeometry();
  int newHeight = availableGeometry.height();
  //int newHeight = availableGeometry.bottom() - frameGeometry.top();
  int newWidth = (int)(newHeight * 0.707070); // FIXME! rough approx. of A4 ratio - this is too limiting!
  /*
  if ( frameGeometry.left()+newWidth > availableGeometry.right() ) {
    newWidth = availableGeometry.right() - frameGeometry.left();
    newHeight = (int)(newWidth / 0.707070);
  }
  */
  // TODO: use pixel metrics to simplify the following!
  view->resize( newWidth - (frameGeometry.width()-view->width()),
		newHeight - (frameGeometry.height()-view->height()) );


  // Manage it
  // WARNING! Don't call addChildWindow(view),
  //          as it automatically calls addChildWindow() itself!
  m_managedDiagrams.insert( view, s );
  view->show();


  return view;
}
