/*
 *  mdiapplication.cpp
 *  Nodal
 *
 *  Imported into Nodal by Aidan Lane on Thu Feb 24 2005.
 *  Copyright (c) 2005 CEMA, Monash University. All rights reserved.
 *
 *  Original file:
 *
 *    mdiapplication.h
 *    EverGreen
 *
 *    Created by Aidan Lane on Mon Jul 19 2004.
 *    Copyright (c) 2004 Aidan Lane. All rights reserved.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

/*
 * Multi-Document-Interface Manager - manage "root-less" windows
 * This is similar at an abstact level to a widget layout manager:
 * - We layout QWidget MDI "children" (as "root-less" windows - e.g. UNIX or MacOS style)
 * - We layout QWidget "tool windows" and connect them to the active/focused MDI children
 *
 * An MDIApplication object instance does NOT show itself - TODO: revise this
 *
 * Note: MDIApplication only shown when there are no MDIChild objects that it is managing
 */
 
/* From the old MDIParent system:
 * I wrote this so that MDI could be supported by Qt on Mac OS X properly.
 * MDIChild objects are NOT children of MDIParent in the eyes of Qt, as they
 * need to be "root-less" work properly (instead we maintain track of it here).
 * Using addToolwindow and removeToolwindow, you can set which tool windows the
 * MDIChild objects are to "own" when they are focused.
 * Note: these tool windows can be ANY widget - but ideally of type QDialog.
 */


#ifndef MDIAPPLICATION_H
#define MDIAPPLICATION_H


#include <QApplication>
#include <QDesktopWidget>
#include <QStyle>


// TODO: make the following settable members of MDIApplication
#define CASCADE_BASE_X 50
#define CASCADE_BASE_Y 00


class MDIChildBase {
 public:
  MDIChildBase() {}
  virtual ~MDIChildBase() {}
  virtual QWidget* widgetPtr() const = 0;
};



/*!
 */
class MDIApplication : public QApplication {

 public:
  MDIApplication(int& argc, char** argv )
    : QApplication(argc, argv) { m_activeChildWindow = 0; }
  virtual ~MDIApplication() {}

  virtual QWidget* topLevel() const { return 0; } // i.e. no parent, but could be used on Windows to provide a common workspace - FIXME: implement this system!
  QList<MDIChildBase*> childWindows() const { return m_childWindows; } // QList is implicitly shared, so we don't need to return a "const ref"
  MDIChildBase* activeChildWindow() const { return m_activeChildWindow; }

  /*!
   * Designed to be called by child windows when they want to be closed.
   *
   * \b Warning: \em This class is in charge of physically closing
   * de-allocating the child.
   *
   * If you want to close a child window, use its close() method, which will
   * still call this (indirectly), but will also execute any other code that
   * it has added to its close event.
   *
   * Re-implement this if you need to ask the user something before closing the
   * window (e.g. "Save changes before closing?) or perform some other
   * processing before it's closed.
   * However, you should still call this version once you have finished, in
   * order to perform the housekeeping that it provides.
   *
   * This implementation calls simply deletes the child.
   *
   * Returns true if the window was closes, false if the request was denied.
   *
   * Asserts that the \em child window is non-null.
   */
  virtual bool childWindowCloseRequest( MDIChildBase* child )
  {
    Q_ASSERT( child != 0 );
    // WARNING!!! don't call close() on the child -> we would loop infinitely!
    // WARNING!!! MDIChild's destructor will call our removeChildWindow() to remove itself
    delete child;
    return true;
  }
	
	
  // TODO: implement better window positioning - CURRENTLY CASCADING
  //
  // Asserts that the child is non-null.
  virtual void addChildWindow( MDIChildBase* child )
  { 
    Q_ASSERT( child != 0 );
		
    m_childWindows.append( child );

    QWidget* w = child->widgetPtr();

    Q_ASSERT( QApplication::desktop() != 0 );
    QRect availableGeometry = QApplication::desktop()->availableGeometry();
		
    // TODO: implement better window positioning - CURRENTLY CASCADING
    if ( m_childWindows.count() == 1 )
      {
	w->move( availableGeometry.left() + CASCADE_BASE_X,
		 availableGeometry.top() + CASCADE_BASE_Y );
      }
    else // note: we assume that m_childWindows.count() > 1 (we append one just above and it !=1)
      {
	QPoint secLast = (m_childWindows[m_childWindows.count()-2]->widgetPtr())->pos(); // and get the SECOND LAST element
	Q_ASSERT( QApplication::style() != 0 );
	int cascadeOffset = QApplication::style()->pixelMetric( QStyle::PM_TitleBarHeight, 0, w );
	if ( cascadeOffset <= 0 ) cascadeOffset = 22; // TODO: remove the need for this?
	int tryX = secLast.x() + cascadeOffset;
	int tryY = secLast.y() + cascadeOffset;

	if ( (tryX + w->width()) <= availableGeometry.right()
	     && (tryY + w->height()) <= availableGeometry.bottom() )
	  w->move( tryX, tryY );
	else
	  w->move( availableGeometry.left() + CASCADE_BASE_X,
		   availableGeometry.top() + CASCADE_BASE_Y );
      }
  }


  // TODO: UPDATE DOC!!!
  // If no child windows are left, then make ourself the active window so that
  // the active menu bar is ours.
  // If "w" == 0, the we return without doing anything
  // Calls setActiveChild(0) if no children are left.
  // Note that if 1+ still exist, then one will automatically become active, automatically calling setActiveWindow for us.
  //
  // Asserts that the child is non-null.
  virtual void removeChildWindow( MDIChildBase* child )
  {
    Q_ASSERT( child != 0 );
    m_childWindows.removeAll( child );
    if ( m_childWindows.isEmpty() ) setActiveChild(0);
  }


  // TODO: update my doc!
  // This "reparents" the child windows to "belong" to the MDI child
  // Won't reparent the tool window if the "new" MDI child is already its parent
  // Warning: this should only be called by the QWidget object themselves - don't touch!
  // Returns true iff OUR child was CHNAGED
  // If the child is not registed with us or the child is already the active one, then this will return w/o doing anything.
  // It IS valid to pass 0 in as the child, the the case where no child is active
  virtual void setActiveChild( MDIChildBase* child )
  {
    m_activeChildWindow = child; // ALWAYS updates this!
  }


 private:
  QList<MDIChildBase*> m_childWindows;
  MDIChildBase* m_activeChildWindow;
};



// WARNING!  WARNING!  WARNING! :
// - We can't use Qt's signals'n'slots to communicate with the manager, as both of
//   these are templates!
// - We need to be able to speak to the manager, but because it is a template, we
//   need to be told what the manager's definite class is (e.g. MDIApplication<QObejct,QWidget>).
//
// TODO: rewrite the following!
/*
 * Convenience class
 *
 * Looks after the adding, removal and activation to/from the MDIApplication
 * Designed to be used with existing classes - such as those created with Qt Designer
 *
 * T should be a QWidget or a derivative of it.
 *
 * managerT should beMDIApplication or a derivative of it.
 *
 * Note: the DESTRUCTOR calls removeChildWindow(), NOT close()
 */
template <class T, class managerT>
  class MDIChild : public T, public MDIChildBase { // note: T is a QObject, so it must be first

 public:
    /*!
     * The parent should be left as null, as the main reason for this MDI 
     * framework is to have managed top-level children. However, it may be
     * set to a container (such as a big grey window, like Windows MDI apps do).
     *
     * Will automatically add ourself to the MDIApplication
     *
     * Asserts that the manager is non-null.
     */
    MDIChild( managerT* manager, QWidget* parent = 0 )
      : T(parent), m_manager(manager)
      {
	Q_ASSERT( m_manager != 0 );
	m_manager->addChildWindow( this );
      }

      /*! Calls the manager's removeChildWindow() method on us. */
      // Absolutely must be virtual, see MDIApplication::childWindowCloseRequest().
      virtual ~MDIChild()
	{
	  Q_ASSERT( m_manager != 0 );
	  m_manager->removeChildWindow( this );
	}

      /*! This assumes that this class is a derivative of QWidget. */
      virtual QWidget* widgetPtr() const
        {
          return (QWidget*)( this ); // note: a dynamic_cast or static_cast won't work here
        }

      /*!
       * Catches Close, WindowActivate, WindowDeactivate and WindowStateChange.
       *
       * On a Close event, this will call childWindowCloseRequest().
       *
       * Asserts that the event is non-null.
       */
      virtual bool event( QEvent* e )
      {
	Q_ASSERT( e != 0 );
	Q_ASSERT( m_manager != 0 );

	// FIXME: DOC:
	// Update eveything first, making sure that the window states are updated before be check them!
	bool ret = false;

	// TODO: optimise me!
	switch( e->type() )
	  {
	  case QEvent::Close: // TODO: doc that this is here!
	    /* WARNING!!! We are _NOT_ to de-allocate our self!
	     * The manager may need to ask the user something before closing the window
	     * (e.g. "Save changes before closing?) or perform some other processing
	     * before we are closed.
	     */
	    if ( m_manager->childWindowCloseRequest(this) )
	      return true; // if closed -> nothing left to do! (and if we made any further calls, we would be in trouble!)
	    else {
	      e->ignore(); // VERY IMPORTANT! Ignore the close event!
	      return true; // don't process the event any further
	    }

	  case QEvent::WindowActivate:
	    m_manager->setActiveChild( this );
	    break;

	  case QEvent::WindowDeactivate:
// TODO: re-doc me: - PUT IN A __BIG__ warning of never to do this!!!
// only setActiveChild(0) when we remove the last child window OR if the only window we have has been minimized!
//	    if ( m_manager->activeChildWindow() == this ) m_manager->setActiveChild( 0 );
//	    ret = true;
	    break;

	    // TODO: the following should not be needed, as I consider it a bug in Qt that windowActivationChange is not called on minimize or restore...
	    // REMOVE ME ONCE FIXED!
	  case QEvent::WindowStateChange:
	    //i.e. minimized, maximized, restored or made full-screen
	    // WARNING! When I used windowState() != Qt::WindowMinimized it returned false-positives (mac-snapshot-20050318)
	    if ( ! T::isMinimized() ) { // this is not the same as == maximized, as there are other states
	      m_manager->setActiveChild( this );
	    }
	    else if ( m_manager->activeChildWindow() == this ) { // redundant: && isMinimized() ) {
	      m_manager->setActiveChild( 0 );
	    }
	    ret = true;
	    break;

	  default:
	    break;
	  }

	// WARNING!!! Try the T's event (always give it a go, which may call close() or windowActivationChange()).
	//            If that fails, then and only then, default to the value of ret.
	return T::event(e) || ret;
      }


 private:
      managerT* m_manager;
  };


#endif // ! MDIAPPLICATION_H
