/*
 *  diagrameditor.cpp
 *  Digest
 * 
 *  Created by Aidan Lane on Thu Jun 20 2005.
 *  Copyright (c) 2005-2006 Optimisation and Constraint Solving Group,
 *  Monash University. All rights reserved.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#include "diagrameditor.h"

#include <QApplication>
#include <QDebug>  // TODO: remove me!
#include <QDesktopWidget>
#include <QFontMetricsF>
#include <QLineEdit>
#include <QMouseEvent>
#include <QPainter>
#include <QSettings>

#include <cmath> // for hypot()

#include "MvcDiagram/abstractdiagramelement.h"
#include "MvcDiagram/abstractdiagramelementattribute.h"
#include "MvcDiagram/diagram.h"
#include "MvcDiagram/diagramcontroller.h"
#include "MvcDiagram/diagramcontrollerevents.h"
#include "MvcDiagram/diagramgesture.h"
#include "MvcDiagram/diagrampolygon.h"
#include "MvcDiagram/diagrampolyline.h"
#include "MvcDiagram/diagramshape.h"
#include "MvcDiagram/diagramtext.h"
#include "MvcDiagram/diagramelementfactory.h"
#include "MvcDiagram/diagramelementattributefactory.h"
#include "MvcDiagram/diagramelementpolygon.h" // required for QPolygonF QVariant metatype declaration - remove me!
#include "MvcDiagram/diagramelementshape.h"   // required for its enum
#include "MvcDigestDb/digestdbmodel.h"
#include "MvcSettings/settingsmodel.h"
#include "GestureRecognition/abstractrecogniser.h"
#include "GestureRecognition/recogniserfactory.h"


// TODO: make these optional!
#define MOUSE_STROKE_PRESSURE  1.0 // intuitive, given button completly down,
//                                    also turns stroke data into "1" -> cheap!
//                                    as TODO: turn me into a property?
#define SINGLE_POINT_WIDTH     0.5


#define ANTIALIASING_ENABLED       true
#define TEXT_ANTIALIASING_ENABLED  true


/*!
 * Constructs a diagram editor with the given \em parent.
 */
DiagramEditor::DiagramEditor( JavaVM* jvm, QWidget* parent )
  : QWidget(parent),
    AbstractDiagramView(this),
    AbstractDigestDbView(this),
    AbstractSettingsView(this),
    m_jvm(jvm)
{
  init();
}


/*!
 * Constructs a diagram editor with the given \em parent and attaches itself to
 * the \em controller.
 */
DiagramEditor::DiagramEditor( AbstractController* diagramController,
			      AbstractController* digestDbController,
			      AbstractController* settingsController,
			      JavaVM* jvm,
			      QWidget* parent )
  : QWidget(parent),
    AbstractDiagramView(this),
    AbstractDigestDbView(this),
    AbstractSettingsView(this),
    m_jvm(jvm)
{
  init();

  Q_ASSERT( diagramController != 0 );
  Q_ASSERT( digestDbController != 0 );
  Q_ASSERT( settingsController != 0 );
  diagramController->attachView( static_cast<AbstractDiagramView*>(this) );
  digestDbController->attachView( static_cast<AbstractDigestDbView*>(this) );
  settingsController->attachView( static_cast<AbstractSettingsView*>(this) );
}


void DiagramEditor::init()
{
  m_currentTool = Digest::SelectionTool;
  m_mouseIsMovingSelection = false;
  m_stroking = false;
  m_recogniser = 0;
  m_recogniserId = -1; // i.e. currently invalid

  /* No background needs to be drawn for the window at all, since child widgets
   * cover the entire window. This reduces redraw time and possibly even flicker
   * when the window is resized.
   */
  // setAttribute( Qt::WA_NoBackground );
  //setAttribute( Qt::WA_NoSystemBackground ); // TODO: fixme: this may be dangerous


  // TODO: base the following on the current paper size
  // TODO: fix the following hack! - it's not the usful anyway!
  // Try to fill then screen vertically and then derive the width from that...
  Q_ASSERT( QApplication::desktop() != 0 );
  QRect availableGeometry = QApplication::desktop()->availableGeometry();
  int newHeight = (int)(availableGeometry.height() * 0.9); // 90% of it
  int newWidth = (int)(newHeight * 0.707070); // FIXME! rough approx. of A4 ratio - this is too limiting!
  // TODO: use pixel metrics to simplify the following!
  setMinimumSize( newWidth, newHeight );
  resize( newWidth, newHeight );

  setFocusPolicy( Qt::StrongFocus ); // we need to receive keyboard events
}


// TODO: FIXME:
QSize DiagramEditor::sizeHint() const {
  return minimumSize(); 
}


void DiagramEditor::setCurrentTool( Digest::Tool tool )
{
  m_currentTool = tool;

  // Update the mouse cursor to reflect the current tool
  QCursor cursor;
  switch ( m_currentTool )
    {
    case Digest::SelectionTool:
      cursor.setShape( Qt::ArrowCursor );
      break;
    case Digest::ZoomTool:
      // TODO: add zoom cursor support
      break;
    case Digest::ShapeTool:
    case Digest::PolygonTool:
    case Digest::PolylineTool:
      // This one looks better than Qt::CrossCursor :-)
      cursor = QCursor( QPixmap(":/images/Crosshair.png"), 7, 7 ); // not (8,8)
      break;
    case Digest::TextTool:
      cursor.setShape( Qt::IBeamCursor );
      break;
    }
  setCursor( cursor );
}


/*!
 * Copies any selected elements into the clipboard and the deletes them from the
 * Diagram.
 *
 * If there are no selected elements, nothing happens.
 */
void DiagramEditor::cut()
{
  copy();
  del();
}


/*!
 * Copies any selected diagram elements into the clipboard.
 *
 * If there are no selected elements, nothing happens.
 */
void DiagramEditor::copy()
{
  // TODO: impl me!
  qDebug("copy");
}


/*!
 * Deletes any selected diagram elements from the Diagram.
 *
 * If there are no selected elements, nothing happens.
 */
void DiagramEditor::del()
{
  postDiagramEvent( new CDiagramEvent
		    (CDiagramEvent::RemoveAllSelectedElements, this) );
}


/*!
 * Inserts any diagram elements that are in the clipboard into the Diagram.
 */
void DiagramEditor::paste()
{
  // TODO: impl me!
  qDebug("paste");
}


void DiagramEditor::selectAll()
{
  postDiagramEvent( new CChangeElementSelectionEvent
		    (AbstractElementSet::fromList(c_elementPictures.keys()),
		     AbstractElementSet(), this) );
}


bool DiagramEditor::tryBeginElementEditing( AbstractElement* e )
{
  Q_ASSERT( e != 0 );

  bool success = false;

  if ( e->type() == MvcDiagram::DE_Text )
    {
      if ( ! m_currentElementEditor.isNull() )
	m_currentElementEditor->deleteLater();

      // Give editor focus, but when it loses it or when enter / return is
      // pressed then update the element's text and then be destroy the editor.
      // Note: need ".toPointF().toPoint()", as the variant is actually a QPointF
      QLineEdit* editor = new QLineEdit( this );
      editor->move( e->attributeData(MvcDiagram::DEA_Origin).toPointF().toPoint() );
      editor->resize( e->attributeData(MvcDiagram::DEA_Size).toSizeF().toSize() );
      editor->setText( e->attributeData(MvcDiagram::DEA_Label).toString() );
      editor->setFont( QFont("Helvetica",14) ); // TODO: remove this hack!
      editor->setFrame( false );
      editor->show();
      editor->setFocus( Qt::PopupFocusReason );
      m_currentElementEditor = editor;
      m_currentElementInEdit = e;
      connect( editor, SIGNAL(editingFinished()),
	       SLOT(currentElementEditingFinished()) );
      success = true;
    }

  return success;
}


/*!
 * This slot is in charge of updating the element that is currently in edit and
 * any of its attributes and then destroying the current editor.
 *
 * 
 */
void DiagramEditor::currentElementEditingFinished()
{
  // Note: for this slot to be called, the following must be non-null:
  Q_ASSERT( diagramController() != 0 );
  Q_ASSERT( ! m_currentElementEditor.isNull() );
  Q_ASSERT( ! m_currentElementInEdit.isNull() );

  DiagramController* c = diagramController();

  if ( m_currentElementInEdit->type() == MvcDiagram::DE_Text )
    {
      QLineEdit* editor = qobject_cast<QLineEdit*>( m_currentElementEditor );
      Q_ASSERT( editor != 0 );
      QString label( editor->text() );
      // DEA_Label, DEA_Origin and DEA_Size come with DE_Text
      QFontMetricsF fm( QFont("Helvetica",14) ); // TODO: remove this hack!
      QSizeF size( fm.size(0, label) );
      c->postElementAttributeChangeData( m_currentElementInEdit,
					 MvcDiagram::DEA_Label,
					 label, this );
      // TODO: update the origin so that the text remains centred
      //  c->postElementAttributeChangeData( e, MvcDiagram::DEA_Origin,
      //				 event->pos() - QPointF(0.0, size.height()/2.0),
      //				 this );
      c->postElementAttributeChangeData( m_currentElementInEdit,
					 MvcDiagram::DEA_Size,
					 size, this );
  }

  m_currentElementEditor->deleteLater();
  m_currentElementInEdit = 0;
}


/*!
 * Calls AbstractDiagramView::dispatchEvent(),
 * AbstractDigestDbView::dispatchEvent(),
 * or AbstractSettingsView::dispatchEvent(),
 * depending on the event sender's module ID.
 */
void DiagramEditor::customEvent( QEvent* e )
{
  VEvent* ve = dynamic_cast<VEvent*>(e); // slow :-(
  if ( ve != 0 )
    {
      Q_ASSERT( ve->sender() != 0 );
      MvcModuleId_t m = ve->sender()->moduleId();
      if ( m == AbstractDiagramView::classModuleId() )
	AbstractDiagramView::dispatchEvent(ve);
      else if ( m == AbstractDigestDbView::classModuleId() )
	AbstractDigestDbView::dispatchEvent(ve);
      else if ( m == AbstractSettingsView::classModuleId() )
	AbstractSettingsView::dispatchEvent(ve);
    }
}


void DiagramEditor::diagramResetEvent( VEvent* )
{
  Q_ASSERT( diagram() != 0 );
  c_elementPictures.clear();
#if 0
  foreach ( AbstractElement* e, diagram()->elements() ) {
    Q_ASSERT( e != 0 );
    Q_ASSERT( !c_elementPictures.contains(e) ); // must NOT have a record yet
    c_elementPictures.insert( e, buildElementPath(e) );
  }
#endif
  update();
}


void DiagramEditor::elementAddedEvent( VElementEvent* event )
{
  Q_ASSERT( event != 0 );
  Q_ASSERT( event->element() != 0 );
  Q_ASSERT( !c_elementPictures.contains(event->element()) ); // must NOT have a record yet
  c_elementPictures.insert( event->element(), buildElementPicture(event->element()) );
  update(); // TODO: localise the update
  tryBeginElementEditing( event->element() );
}


void DiagramEditor::elementRemovedEvent( VElementEvent* event )
{
  Q_ASSERT( event != 0 );
  Q_ASSERT( event->element() != 0 );
  Q_ASSERT( c_elementPictures.contains(event->element()) ); // must have a record
  c_elementPictures.remove( event->element() );
  update();
}


void DiagramEditor::elementAttributeAddedEvent( VElementAttributeEvent* event ) {
  Q_ASSERT( event != 0 );
  elementAttributeModifiedWorker( event->element(), event->attribute() );
}


void DiagramEditor::elementAttributeRemovedEvent( VElementAttributeEvent* event ) {
  Q_ASSERT( event != 0 );
  elementAttributeModifiedWorker( event->element(), event->attribute() );
}


void DiagramEditor::elementAttributeDataChangedEvent( VElementAttributeEvent* event ) {
  Q_ASSERT( event != 0 );
  elementAttributeModifiedWorker( event->element(), event->attribute() );
}


void DiagramEditor::elementAttributeModifiedWorker( AbstractElement* e,
						    AbstractElementAttribute* a )
{ 
  Q_ASSERT( e != 0 );
  Q_ASSERT( a != 0 );
  Q_ASSERT( c_elementPictures.contains(e) );  // must have a record
  if ( a->type() != MvcDiagram::DEA_Origin )  // don't re-build for a move
    c_elementPictures.insert( e, buildElementPicture(e) );
  update(); // redraw the diagram's elements (at current origins) to reflect the changes
}


void DiagramEditor::elementSelectionChangedEvent( VElementSelectionChangedEvent* ) {
  update(); // redraw the diagram's elements to reflect the current selection
}


void DiagramEditor::elementSetOrderChangedEvent( VElementSetOrderChangedEvent* ) {
  update(); // redraw the diagram's elements to reflect the change(s)
}


/*!
 * Unconditionally calls refreshRecogniser().
 */
void DiagramEditor::settingsResetEvent( VEvent* )
{
  refreshRecogniser();
}


/*!
 * Calls refreshRecogniser() if recognition is enabled or disabled or if there
 * is a change in which trained recogniser is to be used.
 */
void DiagramEditor::settingsValueChangedEvent( VSettingsValueChangedEvent* event )
{
  Q_ASSERT( event != 0 );

  if ( event->key() == Digest::settingsDiagramRecogEnabledKey
       || event->key() == Digest::settingsDiagramRecogIdKey )
    refreshRecogniser();
}


/*!
 * Unconditionally calls refreshRecogniser().
 */
void DiagramEditor::digestDbResetEvent( VEvent* )
{
  refreshRecogniser();
}


/*!
 * Calls refreshRecogniser() if there is an active recogniser and the event
 * refers to its trained recognier record ID.
 */
void DiagramEditor::trainedRecogsEvent( VTrainedRecogsEvent* event )
{
  Q_ASSERT( event != 0 );

  if ( m_recogniser != 0
       && event->idSet().contains(m_recogniserId) )
    refreshRecogniser();
}


void DiagramEditor::keyPressEvent( QKeyEvent* event )
{
  Q_ASSERT( event != 0 );

  // Block input until we're ready
  if ( diagram() == 0 || diagramController() == 0 ) return;

  bool unknown = false; // until proven otherwise

  if ( event->modifiers() & Qt::ControlModifier )
    {
      switch ( event->key() )
	{
	case Qt::Key_A:
	  selectAll();
	  break;

	default:
	  unknown = true;
	}
    }
  else if ( event->modifiers() == Qt::NoModifier )
    {
      switch ( event->key() )
	{
	case Qt::Key_Delete:
#ifdef Q_WS_MAC
	  // Allow deletion via backspace on Macs, as some don't have a
	  // forward-delete button (e.g. PowerBooks).
	case Qt::Key_Backspace:
#endif
	  del();
	  break;

	default:
	  unknown = true;
	}
    }

  if ( unknown )
    event->ignore();
  else
    event->accept();
}


void DiagramEditor::mouseDoubleClickEvent( QMouseEvent* /*event*/ )
{
  // Block input until we're ready
  if ( diagram() == 0 || diagramController() == 0 ) return;

  const AbstractElementSet elementSelection
    = diagramController()->elementSelection();

  // If double clicked on a single selected element, then tryBeginElementEditing()
  if ( elementSelection.size() == 1 )
    {
      AbstractElement* e = elementSelection.toList().first();
      Q_ASSERT( e != 0 );
      tryBeginElementEditing( e );
    }
}


void DiagramEditor::mousePressEvent( QMouseEvent* event )
{
  Q_ASSERT( event != 0 );

  // Block input until we're ready
  if ( diagram() == 0 || diagramController() == 0 ) return;

  // TODO: cleanup!

  if ( event->button() == Qt::LeftButton )
    {
      switch ( m_currentTool )
	{
	case Digest::SelectionTool:
	  selectionPress( event->pos(), event->modifiers() );
	  break;

	case Digest::ZoomTool:
	  // TODO: add zoom support
	  break;

	case Digest::ShapeTool:
	  postDiagramEvent( new CElementAddEvent
			    (new DiagramShape(diagram(), DiagramElementShape::Circle,
					      event->pos() - QPointF(25.0, 25.0),// TODO: remove this hack!
					      QSizeF(50.0, 50.0)),// TODO: remove this hack!
			     this) );
	  break;

	case Digest::PolygonTool:
	  break;

	case Digest::PolylineTool:
	  break;

	case Digest::TextTool:
	  {
	    QString label( "Text" ); // TODO: remove this hack!
	    QFontMetricsF fm( QFont("Helvetica",14) ); // TODO: remove this hack!
	    QSizeF size( fm.size(0, label) );
	    QPointF origin( event->pos() - QPointF(0.0, size.height()/2.0) );
	    postDiagramEvent( new CElementAddEvent
			      (new DiagramText(diagram(), label, origin, size), this) );
	  }
	  break;
	}
    }
}


void DiagramEditor::mouseReleaseEvent( QMouseEvent* /*event*/ )
{
  // Block input until we're ready
  if ( diagram() == 0 || diagramController() == 0 ) return;

  switch ( m_currentTool )
     {
     case Digest::SelectionTool:
       selectionRelease(); // TODO: always call this - regardless of tool?
       break;

     default:
       break;
     }
}


void DiagramEditor::mouseMoveEvent( QMouseEvent* event )
{
  Q_ASSERT( event != 0 );

  // Block input until we're ready
  if ( diagram() == 0 || diagramController() == 0 ) return;

   switch ( m_currentTool )
     {
     case Digest::SelectionTool:
       selectionMove( event->pos() ); // TODO: always call this - regardless of tool?
       break;

     default:
       break;
     }
}



void DiagramEditor::tabletEvent( QTabletEvent* event )
{
  Q_ASSERT( event != 0 );

  // Block input until we're ready
  if ( diagram() == 0 || diagramController() == 0 ) return;

  QPoint widgetGlobalPos = mapToGlobal( QPoint(0,0) );
  float widgetGlobalPosX = (float)widgetGlobalPos.x(); // TODO: cache this!
  float widgetGlobalPosY = (float)widgetGlobalPos.y(); // TODO: cache this!
  float hiResLocalX = event->hiResGlobalX() - widgetGlobalPosX;
  float hiResLocalY = event->hiResGlobalY() - widgetGlobalPosY;
  
  /* Note: Disallow Eraser for recognition, instead of allowing only Pen,
   *       as UnknownPointer and Cursor are fairly legitimate too.
   *
   *       We are letting the Eraser act as a selection & movement tool.
   */
  // TODO: make this optional!
  // TODO: allow the use of the Eraser as a mouse (i.e. repects the current tool mode)
  if ( event->pointerType() == QTabletEvent::Eraser )
    {
      event->ignore(); // act as a mouse
#if 0 //  TODO: decide what to do here
      event->accept();
      switch ( event->type() ) {
      case QEvent::TabletPress:
	selectionPress( QPointF(hiResLocalX, hiResLocalY), event->modifiers() ); break;
      case QEvent::TabletRelease:
	selectionRelease(); break;
      case QEvent::TabletMove:
	selectionMove( QPointF(hiResLocalX, hiResLocalY) ); break;
      default:
	break;
      }
#endif
      return;
    }
  
  // TODO: add a "Disable Recognition" event that will call event->ignore();

  event->accept(); // don't pass this TABLET event on to the mouse event handlers!

  // TODO: make the use of high-res coordinates optional!!!

  switch ( event->type() )
    {
    case QEvent::TabletPress:
      // Our own TabletFocus. Needed for example if user uses a tablet to create
      // an element and then wants to press CTRL+A to select all - need focus!
      setFocus( Qt::MouseFocusReason );
      // Begin stroking
      if ( m_stroking ) endStroking(); // e.g. if MOUSE stroking in progress
      beginStroking( StrokePoint(hiResLocalX, hiResLocalY, event->pressure(),
				 (uint32_t)currentStrokeTime.elapsed()) );
      break;

    case QEvent::TabletRelease:
      if ( m_stroking ) {
	addPointToStroke( StrokePoint(hiResLocalX, hiResLocalY, event->pressure(),
				      (uint32_t)currentStrokeTime.elapsed()) );
	endStroking();
      }
      break;

    case QEvent::TabletMove:
      if ( m_stroking )
	addPointToStroke( StrokePoint(hiResLocalX, hiResLocalY, event->pressure(),
				      (uint32_t)currentStrokeTime.elapsed()) );
      break;

    default:
      break;
    }
}



void DiagramEditor::selectionPress( const QPointF& pos, Qt::KeyboardModifiers modifiers )
{
  // TODO: finish me:
  // TODO: CLEAN-UP AND OPTIMISE ME!
  Q_ASSERT( diagramController() != 0 );
  AbstractElement* element = findElementAt( pos );
  AbstractElementSet addSet;
  AbstractElementSet removeSet;
  AbstractElementSet currentSet = diagramController()->elementSelection();
  bool mouseHitSelection = (element != 0); // may change to false later on shift-press

  // Enable multi-selection only when holding the shift or Apple key down
  if ( modifiers & Qt::ShiftModifier
#ifdef Q_WS_MAC
       || modifiers & Qt::ControlModifier // Apple key
#endif
       )
    {
      // Toggle the element's selection status
      if ( element != 0 ) {
	if ( currentSet.contains(element) ) {
	  removeSet.insert( element );
	  // Element is no longer part of the selection
	  // -> not logical to allow movement (standard across graphics apps)
	  mouseHitSelection = false;
	}
	else
	  addSet.insert( element );
      }
    }
  else
    {
      // If the element is not part of the current selection
      // then make it the only one selected (includes the NULL element).
      if ( ! currentSet.contains(element) ) {
	removeSet = currentSet;
	if ( element != 0 )
	  addSet.insert( element );
      }
    }

  /* If user pressed on an element that is or is to be selected then
   * they may want to move the selection.
   *
   * Note: If the user shift-pressed on an element that was selected,
   *       then mouseHitSelection will be false.
   */
  if ( mouseHitSelection ) {
    m_mouseIsMovingSelection = true;
    m_selectionMoveStartPos = pos;

    m_elementOriginsBeforeMove.clear();
    foreach ( AbstractElement* e, currentSet-removeSet+addSet ) {
      AbstractElementAttribute* a = e->attribute( MvcDiagram::DEA_Origin );
      m_elementOriginsBeforeMove.insert( e,
					 a != 0
					 ? a->toVariant().toPointF()
					 : QPointF(0.0, 0.0) ); // as it may not have one
    }
  }

  postDiagramEvent( new CChangeElementSelectionEvent(addSet, removeSet, this) );
}


void DiagramEditor::selectionRelease()
{
  m_mouseIsMovingSelection = false;
  m_elementOriginsBeforeMove.clear();
}


void DiagramEditor::selectionMove( const QPointF& pos )
{
  if ( m_mouseIsMovingSelection )
    {
      /* Important:
       * Just translating the element by the difference between this move and the
       * previous would be too dangerous, especially if events are dropped.
       * This was happening when CIDER was being used.
       */
      QPointF diff = pos - m_selectionMoveStartPos; // cache it
      QHashIterator<AbstractElement*, QPointF> it( m_elementOriginsBeforeMove );
      while ( it.hasNext() ) {
	it.next();
	AbstractElementAttribute* a
	  = it.key()->attribute( MvcDiagram::DEA_Origin );
	if ( a != 0 ) // don't assert this, as we don't know what type the element is
	  postDiagramEvent( new CElementAttributeChangeDataEvent
			    (it.key(), a, it.value()+diff, this) );
      }
    }
}


void DiagramEditor::paintEvent( QPaintEvent* )
{
  // TODO: cache the entire pixmap, drawing often, updating it only when needed ?

  // TODO: possibly make the following adjustable
  static QPen pagePen( Qt::lightGray );
  static QBrush pageBrush( Qt::white );

  QPainter p( this );

  // Draw the page rect
  p.setPen( pagePen );
  p.setBrush( pageBrush );
  p.drawRect( 0, 0, width()-1, height()-1 );

  p.setRenderHint( QPainter::Antialiasing, ANTIALIASING_ENABLED );
  p.setRenderHint( QPainter::TextAntialiasing, TEXT_ANTIALIASING_ENABLED );

  p.setPen( Qt::black ); // TODO: remove me!
  p.setBrush( Qt::NoBrush ); // TODO: remove me!

  // Draw the diagram's elements.
  // Note: We can only to so if our controller has been set.
  if ( diagram() != 0 && diagramController() != 0 )
    {
      const AbstractElementSet elementSelection
	= diagramController()->elementSelection();
      QListIterator<AbstractElement*> it( diagram()->elements() ); // diagram order matters
      it.toBack(); // foremost element at front of list -> draw from back to front
      while ( it.hasPrevious() )
	{
	  AbstractElement* element = it.previous();
	  Q_ASSERT( element != 0 );

	  /* Note: We'll use the origin attribute if it's there, but given that
	   *       we don't know what the element is (and we don't really care,
	   *       as we want this to be extensible), we can't assume it has one.
	   */
	  QPointF origin;
	  AbstractElementAttribute* a = element->attribute( MvcDiagram::DEA_Origin );
	  if ( a != 0 )
	    origin = a->toVariant().toPointF();
	  
	  Q_ASSERT( c_elementPictures.contains(element) );
	  p.save();
	  if ( elementSelection.contains(element) )  
	    p.setPen( Qt::blue ); // TODO: do something a little more fancy here!
	  p.drawPicture( origin, c_elementPictures.value(element) );
	  p.restore();
	}
    }

  // If stroking, then draw the current (gesture) stroke that is being drawn
  if ( m_stroking )
    p.drawPath( m_currentStrokePath );
}


// TODO: re-write doc using the ORIGIN attribute...
/*!
 * \b Warning: The path return does \b not include the element's position.
 * This allows use to reuse a path even while it's being moved around.
 */
QPicture DiagramEditor::buildElementPicture( const AbstractElement* element ) const
{
  Q_ASSERT( element != 0 );

  QPicture picture;
  QPainter painter( &picture );

  // Init the variables using the element's attributes
  StrokeList ink;
  QString    label;
  QPolygonF  polygon;
  QSizeF     size;
  DiagramElementShape::Type shape = DiagramElementShape::Type(0);
  foreach ( const AbstractElementAttribute* a, element->attributes() )
    {
      Q_ASSERT( a != 0 );
      switch ( a->type() ) {
      case MvcDiagram::DEA_Ink:
	ink = a->toVariant().value<StrokeList>(); break;
      case MvcDiagram::DEA_Label:
	label = a->toVariant().toString(); break;
      case MvcDiagram::DEA_Polygon:
	polygon = a->toVariant().value<QPolygonF>(); break;     
      case MvcDiagram::DEA_Shape:
	shape = DiagramElementShape::Type(a->toVariant().toInt()); break;
      case MvcDiagram::DEA_Size:
	size = a->toVariant().toSizeF(); break;
      default:
	break;
      }
    }

  // TODO: impl me!
  // TODO: optimise me!
  switch ( element->type() )
    {
    case MvcDiagram::DE_Gesture:
      {
	StrokesPainterPath path;
	path.addStrokes( ink );
	painter.drawPath( path );
      }
      break;

    case MvcDiagram::DE_Polygon:    
      {
	StrokesPainterPath path;
	painter.setBrush( Qt::white ); // TODO: remove me!
	path.addPolygon( polygon ); // not closed unless start and end the same
	painter.drawPath( path );
      }
      break;

    case MvcDiagram::DE_Polyline:
      {
	StrokesPainterPath path;
	// TODO: remove this hack!
	if ( polygon.size() >= 3 ) {
	  path.moveTo( polygon.at(0) );
	  path.cubicTo( polygon.at(1), polygon.at(1), polygon.at(2) );
	}
	painter.drawPath( path );
      }
      break;

    case MvcDiagram::DE_Shape:
      {
	// TODO: finish me:
	QRectF r( QPointF(), size );
	painter.setBrush( QColor(255,255,255,128) ); // white at 50% alpha TODO: remove me!
	switch ( shape )
	  {
	  case DiagramElementShape::Circle:
	    painter.drawEllipse( r ); break;
	  case DiagramElementShape::Square:
	    painter.drawRect( r );
	  default:
	    break;
	  }
      }
      break;

    case MvcDiagram::DE_Text:
      painter.setFont( QFont("Helvetica",14) ); // TODO: add a font attribute and use it!
      painter.drawText( QPointF(0, size.height()), // yes, BOTTOM left - baseline
			label );
      break;

    default:
      break;
    }

  return picture;
}


// TODO: support element Z-axis ordering
// TODO: optimise and cleanup!
AbstractElement* DiagramEditor::findElementAt( const QPointF& pos ) const
{
  /* By looking at a smaller region first and then expanding it, we allow for
   * picking that is extremely accurate, while also being lenient if the user
   * did not click directly on an object.
   *
   * Example: Consider the case of two lines. IF WE WERE to look at a 3x3
   * region straight away where the foremost line only touches the pixel at
   * (3,3), while another line touches pixel (1,1), which is directly in the
   * centre, we would pick the former line. However, it would have been more
   * accurate to pick the latter line, as it is closer to the given pos.
   * Hence, this is why we check a region of 1x1 first, then 3x3.
   *
   * This algorithm is also faster if the user clicks directly on an element,
   * which is trivial to do with filled shapes.
   */

  uint unusedPixel = qRgba( 0, 0, 0, 0 );

  for ( int w=1; w <= 3; w+=2 ) // region must be odd for the center to be at POS
    {
      QImage img( w, w, QImage::Format_ARGB32_Premultiplied );

      // TODO: speed fix!
      // We MUST init the image - else it will be filled with random garbage
      for( int y=0; y < w; ++y ) 
	for ( int x=0; x < w; ++x )
	  img.setPixel( x,y, unusedPixel );

      QPainter p( &img );

      /* Important:
       * The render hints here MUST match paintEvent's, otherwise, for example the
       * user may click on a pixel that has been set (to something other than
       * [0,0,0,0]) due to anti-aliasing and hence may not register as a hit here.
       */
      p.setRenderHint( QPainter::Antialiasing, ANTIALIASING_ENABLED );
      p.setRenderHint( QPainter::TextAntialiasing, TEXT_ANTIALIASING_ENABLED );

      p.setClipping( true );
      p.setClipRect( 0, 0, w, w );
      p.translate( -pos + QPointF(w/2,w/2) );

      /* Search through the diagram's elements.
       *
       * Note: We must perform this search in the reverse order to what the
       *       diagram was drawn in, as the latter elements will be drawn op top
       *       of former elements and hence may obstruct them both visually and
       *       thus also the ability to "pick" them.
       *   ->  Since the foremost element is at the front of the list, we must
       *       work from front to back.
       */
      QListIterator<AbstractElement*> it( diagram()->elements() );
      while ( it.hasNext() )
	{
	  AbstractElement* element = it.next();
	  Q_ASSERT( element != 0 );

	  p.setPen(Qt::black); // TODO: remove me!
	  p.setBrush(Qt::NoBrush); // TODO: remove me!    
    
	  /* Note: We'll use the origin attribute if it's there, but given that
	   *       we don't know what the element is (and we don't really care,
	   *       as we want this to be extensible), we can't assume it has one.
	   */
	  QPointF origin;
	  AbstractElementAttribute* a = element->attribute( MvcDiagram::DEA_Origin );
	  if ( a != 0 )
	    origin = a->toVariant().toPointF();
	  
	  p.save();
	  p.drawPicture( origin, c_elementPictures.value(element) );
	  p.restore();

	  // TODO: speed fix!
	  for( int y=0; y < w; ++y )
	    for ( int x=0; x < w; ++x )
		if ( img.pixel(x,y) != unusedPixel )
		  return element;
	}
  
    }

  return 0;
}


/*!
 * Note:
 * Unlike endStroking(), this requires the x, y and pressure parameters because
 * this corresponds to a MOVE TO, vs. the addPointToStroke()'s LINE TO.
 * Also, you may want to call endStroking() without a point.
 *
 * Sets the modified flag to true.
 */
void DiagramEditor::beginStroking( const StrokePoint& pt )
{
  currentStrokeTime.restart(); // unlike start(), this's atomic

  /* VERY IMPORTANT:
   * pt.milliTime will have been calculated using the previous stroke start time,
   * but we've just updated the start time (that is, after the offset calc).
   * However, this is the start of a new stroke, so we know that milliTime is 0.
   */ 
  StrokePoint modPt = pt;
  modPt.milliTime = 0;

  StrokeCoordT x = modPt.x();
  StrokeCoordT y = modPt.y();

  m_currentStroke = Stroke(); // ensure that current stroke is empty
  m_currentStroke.append( modPt );

  m_currentStrokePath.moveTo( modPt );

  // Ensure that single-point strokes are visible
  m_currentStrokePath.addEllipse( x, y, SINGLE_POINT_WIDTH, SINGLE_POINT_WIDTH );

  // Update the effected region
  // Note: we have to account for the width of the pen
  ///TODO: re-enable me! float rad = pen().width() / 2.0 + 1.0; // + 1.0 to account for errors TODO: cache this!
  float rad = 0.5 + 1.0; // + 1.0 to account for errors TODO: remove me!
  QRectF r; // use adjust, as (modPt.x, modPt.y) is a point, not width and height
  r.adjust( x, y, x, y );
  update( r.adjusted(-rad, -rad, rad, rad).toRect() );

  m_stroking = true;
  // TODO: EMIT strokingStarted() with device ID
}


// TODO: fixme: still leaves marks on update!
void DiagramEditor::endStroking()
{
  m_stroking = false;

  // Note: controlPointRect() is as large as and is faster than boundingRect()
  // Note: we have to account for the width of the pen
  ///TODO: re-enable me! float rad = pen().width() / 2.0 + 1.0; // + 1.0 to account for errors TODO: cache this!
  float rad = 0.5 + 1.0; // + 1.0 to account for errors TODO: remove me!
  QRect updateRect
    = m_currentStrokePath.controlPointRect().adjusted(-rad,-rad,rad,rad).toRect();

  recogniseCurrentStroke();

  // Clear the current stroke
  m_currentStroke = Stroke();
  m_currentStrokePath = StrokesPainterPath();

  update( updateRect ); // get rid the stroke from the display

  // TODO: EMIT strokingFinished() with device ID
}


/*!
 * Asserts that isStroking() == true.
 */
void DiagramEditor::addPointToStroke( const StrokePoint& pt )
{
  Q_ASSERT( m_stroking == true );

  m_currentStroke.append( pt );
  m_currentStrokePath.lineTo( pt );

  // Update the effected region
  // TODO: keep track of the previous point instead of having to re-fetch it every time!
  // Note: we have to account for the width of the pen
  Q_ASSERT( m_currentStroke.size() >= 2 );
  const StrokePoint& prevPt
    = m_currentStroke.at(m_currentStroke.size()-2);
  ///TODO: re-enable me! float rad = pen().width() / 2.0 + 1.0; // + 1.0 to account for errors TODO: cache this!
  float rad = 0.5 + 1.0; // + 1.0 to account for errors TODO: remove me!
  QRectF r; // use adjust, as (pt.x, pt.y) is a point, not width and height
  r.adjust( prevPt.x(), prevPt.y(), pt.x(), pt.y() );
  update( r.normalized().adjusted(-rad, -rad, rad, rad).toRect() );
}


/*!
 * Destroys and if possible, recreates the gesture recogniser, according to the
 * settingsModel(), using the digestDbModel().
 *
 * This may safely be called if either of these models havn't been initialised,
 * as it will simply delete any existing recogniser and return.
 */
void DiagramEditor::refreshRecogniser()
{
  delete m_recogniser;
  m_recogniser = 0;
  m_recogniserId = -1; // i.e. currently invalid

  if ( settingsModel() != 0
       && digestDbModel() != 0 )
    {
      const QSettings* settings = settingsModel()->constSettings();
      
      if ( settings->value(Digest::settingsDiagramRecogEnabledKey, false).toBool() )
	{
	  bool intOk = false;
	  int id = settings->value(Digest::settingsDiagramRecogIdKey, -1).toInt(&intOk);
	  Q_ASSERT( intOk );
	  if ( id >= 0 ) {
	    DTrainedRecogRecord record = digestDbModel()->fetchTrainedRecog( id );
	    m_recogniser = RecogniserFactory::create( record.recogniserKey, m_jvm,
						      digestDbModel(), this );
	    if ( m_recogniser == 0 ) { // TODO: handle me properly!!!! - user may have got it wrong!
	      qDebug("ERROR: Invalid recogniser requested!");
	      m_recogniser = 0;
	      m_recogniserId = -1; // i.e. currently invalid
	      // TODO: update the settings?
	      return;
	    }
	    if ( ! m_recogniser->loadRecord(record) )
	      Q_ASSERT( !"recogniser loadRecord failed!" ); // TODO: handle this properly!
	    m_recogniserId = id;
	  }
	}
    }

  // TODO: make some graphical mark to indicate to the user of the recogniser's status
}


/*!
 * Attempts to recognise the current gesture's stroke, as drawn by the user
 * and then adds a new DiagramGesture to the diagram, using the ink data
 * and shape results.
 *
 * This may safely be called if there is no recogniser, it will just return
 * without doing anything.
 */
void DiagramEditor::recogniseCurrentStroke()
{
  // TODO: cleanup!

  if ( diagramController() == 0 ) return;
  DiagramController* c = diagramController();


  if ( m_recogniser != 0 )
    {
      DGestureRecord g;
      g.strokes += m_currentStroke;

      //
      // TODO: remove the following - leave it to CIDER!
      //

      ClassProbabilities results = m_recogniser->classify( g.strokes );

      int maxIndex = -1;   // i.e. init to invalid
      ClassProbabilityT maxValue = 0.0;

      ClassProbabilitiesIterator it( results );
      if ( it.hasNext() ) {   // init (do it now, don't constantly check later)
	it.next();
	maxIndex = it.key();
	maxValue = it.value();
      }
      while( it.hasNext() ) { // get the max over the entire results domain
	it.next();
	if ( it.value() > maxValue ) {
	  maxIndex = it.key();
	  maxValue = it.value();
	}
      }


      if ( maxIndex >= 0 )
	{
	  // TODO: cleanup!

	  Q_ASSERT( diagram() != 0 );

	  AbstractElement* element = 0;

#if 0
	  element = addNewElement( MvcDiagram::DE_Gesture );
	  Q_ASSERT( element != 0 );

	  StrokeList ink;
	  ink += m_currentStroke;

	  AbstractElementAttribute* attr // DEA_Ink comes with DE_Gesture
	    = element->attribute( MvcDiagram::DEA_Ink );
	  Q_ASSERT( attr != 0 );
	  postDiagramEvent( new CElementAttributeChangeDataEvent(element, attr,
								 QVariant::fromValue(ink), this) );
#else
	  GestureType type = UnknownGesture;

	  // TODO: remove this hack and make this mapping user settable!
	  if      ( maxIndex == 1 ) type = CircleGesture;
	  else if ( maxIndex == 2 ) type = SquareGesture;
	  else if ( maxIndex == 3 ) type = LineGesture;

	  if ( type == CircleGesture
	       || type == SquareGesture )
	    {
	      element = addNewElement( MvcDiagram::DE_Shape );
	      Q_ASSERT( element != 0 );

	      DiagramElementShape::Type shape = DiagramElementShape::Circle;
	      if ( type == SquareGesture ) shape = DiagramElementShape::Square;

	      // Note: We can't use the faster controlPointRect(), as we need
	      //       it to be accurate.
	      QRectF rect = m_currentStrokePath.boundingRect(); // TODO: keep aspect ratio at 1:1 (?)

	      // Note: DEA_Shape, DEA_Origin and DEA_Size come with DE_Shape
	      // Important: Set the size before the origin, as the circle's mid pos depends of the size
	      c->postElementAttributeChangeData( element, MvcDiagram::DEA_Shape,
						 shape, this );
	      c->postElementAttributeChangeData( element, MvcDiagram::DEA_Size,
						 rect.size(), this );
	      c->postElementAttributeChangeData( element, MvcDiagram::DEA_Origin,
						 rect.topLeft(), this );	      
	    }
	  else if ( type == LineGesture )
	    {
	      element = addNewElement( MvcDiagram::DE_Polyline );
	      Q_ASSERT( element != 0 );

	      QPolygonF polygon;
	      if ( ! m_currentStroke.isEmpty() )
		{
		  // TODO: optimise me!

		  // START point
		  polygon += QPointF(0.0, 0.0);

		  // MID point
		  /* TODO: DOC ME:
		   * Find a suitable mid-point.
		   * Finds the point that is closest to the half-way position.
		   * This is very intuitive when working with FSAs and works
		   * with all common FSA arrow shapes (straight, loop-back and bowed).
		   * (The first version found the most "extreme" point - that is
		   *  the point that ensured max travel time from start to end,
		   *  which was in effort to honour the overall shape.
		   *  However, this did not work well when the extreme point was
		   *  near the start or end point). - e.g. when you draw a fairly staright line...
		   */

		  if ( m_currentStroke.size() >= 3 )
		    {
		      QVector<float> runningSums( m_currentStroke.size() );
		      runningSums[0] = 0.0; // at start -> dist travelled == 0
		      for ( int i=1; i < m_currentStroke.size(); ++i ) { // can't be 1st
			const QPointF& pt = m_currentStroke.at(i);
			const QPointF& prevPt = m_currentStroke.at(i-1);
			runningSums[i] = ( runningSums.at(i-1)
					   + hypot(pt.x()-prevPt.x(),
						   pt.y()-prevPt.y()) );
		      }
		      float halfDist = runningSums.last() / 2.0; // target
		      int   bestIndex = 0;
		      float bestIndexError = runningSums.last()+1.0; // init to WORST+1.0 - must MINimise this
		      for ( int i=1; i < runningSums.size(); ++i ) { // can't be 1st
			float e = fabs( halfDist - runningSums.at(i) );
			if ( e < bestIndexError ) {
			  bestIndex = i;
			  bestIndexError = e;
			}
		      }
		      polygon += m_currentStroke.at(bestIndex) - m_currentStroke.first();
		    }

		  // END point
		  polygon += m_currentStroke.last() - m_currentStroke.first();
		}

	      c->postElementAttributeChangeData( element, MvcDiagram::DEA_Origin,
						 m_currentStroke.first(), this );
	      c->postElementAttributeChangeData( element, MvcDiagram::DEA_Polygon,
						 QVariant::fromValue(polygon), this );
	    }
#endif
	}
    }
}


/*!
 * Uses DiagramElementFactory to create the element and then posts a
 * CElementAddEvent to the Diagram's controller if it was successfully
 * created.
 *
 * Returns the pointer to the new element.
 *
 * Note: The \em type is an int, as to aid extensibility.
 */
// TODO: remove this method
AbstractElement* DiagramEditor::addNewElement( int type )
{
  AbstractElement* e
    = DiagramElementFactory::create
    ( (MvcDiagram::ElementType)type, diagram() );
  if ( e != 0 )
    postDiagramEvent( new CElementAddEvent(e, this) );
  return e;
}


/*!
 * Uses DiagramElementAttributeFactory to create the element and then posts a
 * CElementAttributeAddEvent to the Diagram's controller if it was successfully
 * created.
 *
 * Returns the pointer to the new element.
 *
 * Note: The \em type is an int, as to aid extensibility.
 */
AbstractElementAttribute*
DiagramEditor::addNewElementAttribute( int type,
				       AbstractElement* element )
{
  AbstractElementAttribute* a
    = DiagramElementAttributeFactory::create
    ( (MvcDiagram::ElementAttributeType)type, element );
  if ( a != 0 )
    postDiagramEvent( new CElementAttributeAddEvent(element, a, this) );
  return a;
}
