/*  -*- c++ -*-  (for Emacs)
 *
 *  digestprefsdialog.h
 *  Digest
 * 
 *  Created by Aidan Lane on Thu Aug 25 2005.
 *  Copyright (c) 2005-2006 Optimisation and Constraint Solving Group,
 *  Monash University. All rights reserved.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#include "digestprefsdialog.h"

#include <QApplication>
#include <QSettings>

#include "MvcDigestDb/digestdbmodel.h"
#include "MvcDigestDb/digestdbcontroller.h"
#include "MvcSettings/settingsmodel.h"
#include "MvcSettings/settingscontroller.h"

#include "digest.h"


/*!
 * \class DigestPrefsDialog.
 *
 * \brief The DigestPrefsDialog class provides a graphical interface for
 *        changing application-wide settings.
 *
 * It was designed for non-modal use, where all changes come in to effect
 * immediately.
 * Hence there is a "Close" push button (not shown on the Mac OS X version,
 * as to suit Apple HIG) and no "Ok" or "Cancel" push buttons.
 */


/*!
 * Constructs a preferences dialog with the given \em parent and \em flags
 * and attaches itself to the \em settingsController and \em digestDbController.
 *
 * The \em settingsController and \em digestDbController pointers may be null,
 * as they'll just be ignored.
 */
DigestPrefsDialog::DigestPrefsDialog( AbstractController* settingsController,
				      AbstractController* digestDbController,
				      QWidget* parent, Qt::WindowFlags flags )
   : QDialog(parent, flags),
     AbstractSettingsView(this),
     AbstractDigestDbView(this)
{
  m_ui.setupUi( this );

  // Hide the Close push button on the Mac - to match Apple's pref dialogs
#ifdef Q_WS_MAC
  Q_ASSERT( m_ui.closePushButton );
  m_ui.closePushButton->hide();
#endif

  if ( settingsController )
    settingsController->attachView( static_cast<AbstractSettingsView*>(this) );

  if ( digestDbController )
    digestDbController->attachView( static_cast<AbstractDigestDbView*>(this) );
}


/*!
 * Calls either AbstractSettingsView::dispatchEvent() or
 * AbstractDigestDbView::dispatchEvent(), depending on the event sender's
 * module ID.
 */
void DigestPrefsDialog::customEvent( QEvent* e )
{
  VEvent* ve = dynamic_cast<VEvent*>(e); // slow :-(
  if ( ve )
    {
      Q_ASSERT( ve->sender() );
      MvcModuleId_t m = ve->sender()->moduleId();
      if ( m == AbstractSettingsView::classModuleId() )
	AbstractSettingsView::dispatchEvent(ve);
      else if ( m == AbstractDigestDbView::classModuleId() )
	AbstractDigestDbView::dispatchEvent(ve);
    }
}


/*!
 * Resets the settings dialog and resynchronises it with the settings model.
 */
void DigestPrefsDialog::settingsResetEvent( VEvent* )
{
  Q_ASSERT( settingsModel() );
  Q_ASSERT( settingsModel()->constSettings() );
  const QSettings* settings = settingsModel()->constSettings();

  // Recognition enabled/disabled, Default=true
  Q_ASSERT( m_ui.recogGroupBox );
  m_ui.recogGroupBox->blockSignals( true );
  m_ui.recogGroupBox
    ->setChecked( settings->value(Digest::settingsDiagramRecogEnabledKey, true).toBool() );
  m_ui.recogGroupBox->blockSignals( false );

  // Recogniser ID
  refreshRecogComboBox(); // sync it to the settings and digestDb
}


/*!
 * Resets widgets that use data from the Digest Database.
 *
 * Calls refreshRecogComboBox().
 */
void DigestPrefsDialog::digestDbResetEvent( VEvent* )
{
  refreshRecogComboBox(); // sync it to the settings and digestDb
}


void DigestPrefsDialog::trainedRecogsEvent( VTrainedRecogsEvent* event )
{
  Q_ASSERT( event );
  Q_ASSERT( m_ui.recogComboBox );
  Q_ASSERT( digestDbModel() );
  DigestDbModel* m = digestDbModel();

  // TODO: re-sort the items alphabetically
  switch ( event->type() )
    {
    case VDigestDbEvent::TrainedRecogsAdded:
      foreach ( int id, event->idSet() )
	m_ui.recogComboBox->addItem( m->fetchTrainedRecogLabel(id), id );
      break;

    case VDigestDbEvent::TrainedRecogsRemoved:
      foreach ( int id, event->idSet() )
	m_ui.recogComboBox->removeItem( m_ui.recogComboBox->findData(id) );
      break;

    case VDigestDbEvent::TrainedRecogsUpdated:
      foreach ( int id, event->idSet() )
	m_ui.recogComboBox->setItemText( m_ui.recogComboBox->findData(id),
					 m->fetchTrainedRecogLabel(id) );
      break;

    default:
      break;
    }
}


void DigestPrefsDialog::on_recogGroupBox_toggled( bool on )
{
  Q_ASSERT( m_ui.recogGroupBox );
  QApplication
    ::postEvent( settingsController(),
		 new CSettingsChangeValueEvent(Digest::settingsDiagramRecogEnabledKey, on) );
}


void DigestPrefsDialog::on_recogComboBox_activated( int index )
{
  Q_ASSERT( m_ui.recogComboBox );
  QApplication
    ::postEvent( settingsController(),
		 new CSettingsChangeValueEvent(Digest::settingsDiagramRecogIdKey,
					       m_ui.recogComboBox->itemData(index)) );
}


void DigestPrefsDialog::on_closePushButton_clicked()
{
  close();
}


/*!
 * Completely clears and rebuilds the recogniser combo-box and sets its current
 * index to the one in the settings.
 *
 * This should only need to be called by the reset events, as
 * settingsValueChangedEvent() and trainedRecogEvent()
 * are in charge of keeping the combo-box in sync with the settings and database.
 *
 * If either settingsModel() or digestDbModel() are null, then the combobox will
 * simply be cleared.
 */
void DigestPrefsDialog::refreshRecogComboBox()
{
  int recogId = -1;
  int recogIndex = -1;
  bool intOk = false;

  Q_ASSERT( m_ui.recogComboBox );

  m_ui.recogComboBox->clear();

  // We need BOTH models to be valid before proceeding
  if ( settingsModel() == 0
       || digestDbModel() == 0 )
    return;

  // TODO: handle case where the index is invalid (i.e. recog id not found!)
  Q_ASSERT( settingsModel()->constSettings() );
  recogId = ( settingsModel()->constSettings()
	      ->value(Digest::settingsDiagramRecogIdKey).toInt(&intOk) );
  Q_ASSERT( intOk );

  // TODO: sort the items alphabetically
  QHashIterator<int, QString> it( digestDbModel()->fetchTrainedRecogsBasic() );
  while ( it.hasNext() ) {
    it.next();
    m_ui.recogComboBox->addItem( it.value(), it.key() );
    if ( it.key() == recogId ) recogIndex = m_ui.recogComboBox->count()-1;
  }

  m_ui.recogComboBox->blockSignals( true );
  m_ui.recogComboBox->setCurrentIndex( recogIndex );
  m_ui.recogComboBox->blockSignals( false );
}
