/*  -*- c++ -*-  (for Emacs)
 *
 *  categoryitemmodel.cpp
 *  Digest
 * 
 *  Created by Aidan Lane on Wed Jul 20 2005.
 *  Copyright (c) 2005-2006 Optimisation and Constraint Solving Group,
 *  Monash University. All rights reserved.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#include "categoryitemmodel.h"

#include <QDataStream>
#include <QList>
#include <QMimeData>
#include <QPixmap>
#include <QSet>
#include <QSqlTableModel>

#include "MvcDigestDb/digestdbmodel.h"
#include "MvcDigestDb/digestdbcontroller.h"


// TODO: move the following to a globals file!
#define GESTURE_MIME_FORMAT_STR  "text/x-gesture-id-set"

// TODO: move the following to digestdatabase.h
#define COLLECTION_TABLE_STR            "Collection"
#define COLLECTION_TABLE_ID_STR         "id"
#define COLLECTION_TABLE_ID_INDEX       0
#define COLLECTION_TABLE_LABEL_INDEX    1
// TODO: move the following to digestdatabase.h
// TODO: use the more flexible STRING indices!
#define GESTURE_TABLE_STR               "Gesture"
#define GESTURE_TABLE_ID_STR            "id"
#define GESTURE_TABLE_ID_INDEX          0
#define GESTURE_TABLE_LABEL_INDEX       1
#define GESTURE_TABLE_DATE_INDEX        2
#define GESTURE_TABLE_NOTES_INDEX       3
#define GESTURE_TABLE_NUMSTROKES_INDEX  4
#define GESTURE_TABLE_STROKES_INDEX     5

#define CLASSMAP_TABLE_STR            "ClassMap"
#define CLASSMAP_TABLE_GESTUREID_STR  "gestureId"
#define CLASSMAP_TABLE_CLASSID_STR    "classId"

#define COLLECTIONMAP_TABLE_STR              "CollectionMap"
#define COLLECTIONMAP_TABLE_GESTUREID_STR    "gestureId"
#define COLLECTIONMAP_TABLE_COLLECTIONID_STR "collectionId"


// Note: we store these as QVariants, QAbstractItemModel uses them for data()
static QString  s_libraryRootStr     = QObject::tr("Library");
static QString  s_classesRootStr     = QObject::tr("Classes");
static QString  s_collectionsRootStr = QObject::tr("Collections");
static QVariant s_libraryPix;
static QVariant s_collectionPix;
static QVariant s_classPix;


bool categoryItemPtrLessThan( const CategoryItemModel::CategoryItem* a,
			      const CategoryItemModel::CategoryItem* b );


struct CategoryItemModel::CategoryItem {

  CategoryItem( IndexType type = ClassIndex,
		int       dbId = -1,
		QString   label = QString(),
		QVariant  decoration = QVariant(),
		CategoryItem* parent = 0 )
    : type(type),
      dbId(dbId),
      label(label),
      decoration(decoration),
      checkState(Qt::Unchecked),
      parent(parent) {}
  ~CategoryItem() { qDeleteAll(children); }
  IndexType type;
  int       dbId; // specific to the table it lives in
  QString   label;
  QVariant  decoration; // ok to store one per item, thanks to implicit sharing
  Qt::CheckState checkState;
  CategoryItem*  parent;
  QList<CategoryItem*> children;
  QHash<CategoryItem*, QModelIndex> cachedIndexes;

  void sortChildren() {
    cachedIndexes.clear(); // invalidate cache - about to sort
    qStableSort( children.begin(), // stable sort will keep newer items later
		 children.end(),
		 categoryItemPtrLessThan );
  }
};


bool categoryItemPtrLessThan( const CategoryItemModel::CategoryItem* a,
			      const CategoryItemModel::CategoryItem* b ) {
  Q_ASSERT( a && b );
  return ( b->type != CategoryItemModel::LibraryIndex // the library is ALWAYS first
	   && a->label < b->label );
}



/*!
 * \class CategoryItemModel
 *
 * \brief The CategoryItemModel class provides a model that can be used to browse
 *        through collections and classes in a Digest Database (DigestDb).
 */


/*!
 * Constructs a category item model with the given \em parent.
 */
CategoryItemModel::CategoryItemModel( QObject* parent )
  : QAbstractItemModel(parent),
    AbstractDigestDbView(this)
{
  init();
}


/*!
 * Constructs a category item model with the given \em parent and attaches itself
 * to the \em controller.
 */
CategoryItemModel::CategoryItemModel( DigestDbController* controller, QObject* parent )
  : QAbstractItemModel(parent),
    AbstractDigestDbView(this)
{
  init();

  if ( controller )
    controller->attachView( this );
}


/*!
 * Destroys the category item model.
 */
CategoryItemModel::~CategoryItemModel()
{
  delete m_rootItem;  // ~CategoryItem() looks after de-allocating its children
}


void CategoryItemModel::init()
{
  m_ready = false;
  m_rootItem = 0;
  c_libraryItem = 0;

  m_modelFlags = ItemsAreSelectable | ItemsAreEditable;

  // Note: we can only use QPixmap once a QApplication has been instantiated
  if ( s_libraryPix.isNull() )
    s_libraryPix = QVariant::fromValue( QPixmap(":/images/LibraryIcon.png") );
  if ( s_classPix.isNull() )
    s_classPix = QVariant::fromValue( QPixmap(":/images/ClassIcon.png") );
  if ( s_collectionPix.isNull() )
    s_collectionPix = QVariant::fromValue( QPixmap(":/images/CollectionIcon.png") );

  // Note: resetModel() is called by resetEvent()
  // resetModel(); // don't call reset(), that only resets the Qt views
}


void CategoryItemModel::resetModel()
{
  /* WARNING:
   * It's not safe to call anything like setData() in here, as the views will not
   * yet have been updated, only the model (setData() calls dataChanged() which
   * will try to update the views).
   */

  m_ready = false;

  delete m_rootItem;  // the ~CategoryItem() looks after de-allocating its children

  m_rootItem = new CategoryItem;

  c_libraryItem = new CategoryItem( LibraryIndex, -1, s_libraryRootStr,
				    s_libraryPix, m_rootItem );
  m_rootItem->children.append( c_libraryItem );

  c_libraryIndex = index( c_libraryItem ); // ok to cache - never leaves row 0, column 0

  if ( digestDbModel() )
    {
      QHashIterator<int, QString> classIt( digestDbModel()->fetchClasses() );
      while ( classIt.hasNext() ) {
	classIt.next();
	CategoryItem* item = new CategoryItem( ClassIndex, classIt.key(),
					       classIt.value(), s_classPix,
					       c_libraryItem );
	c_libraryItem->children.append( item );
	m_classIdToItem.insert( classIt.key(), item );
      }
      c_libraryItem->sortChildren();
      
      QHashIterator<int, QString> collectIt( digestDbModel()->fetchCollections() );
      while ( collectIt.hasNext() ) {
	collectIt.next();
	CategoryItem* item = new CategoryItem( CollectionIndex, collectIt.key(),
					       collectIt.value(), s_collectionPix,
					       m_rootItem );
	m_rootItem->children.append( item );
	m_collectionIdToItem.insert( collectIt.key(), item );
      }
      m_rootItem->sortChildren();
    }

  m_ready = true;
}


/*!
 * Sets the category item model's flags.
 *
 * Defaults to: ItemsAreSelectable | ItemsAreEditable
 */
void CategoryItemModel::setModelFlags( ModelFlags flags )
{
  m_modelFlags = flags;
  reset();  // reset the views (there's no reason to reset the model though)
}


CategoryItemModel::IndexType
CategoryItemModel::indexType( const QModelIndex& index ) const
{
  if ( m_ready
       && index.isValid() ) {
    Q_ASSERT( index.internalPointer() );
    return static_cast<CategoryItem*>(index.internalPointer())->type;
  }
  return RootIndex;
}


/*!
 * This is specific to the SQL table that the index lives in.
 *
 * Use indexType() to determine which table the index lives in.
 */
int CategoryItemModel::indexDatabaseId( const QModelIndex& index ) const
{
  if ( m_ready
       && index.isValid() ) {
    Q_ASSERT( index.internalPointer() );
    return static_cast<CategoryItem*>(index.internalPointer())->dbId;
  }
  return -1;
}


Qt::ItemFlags CategoryItemModel::flags( const QModelIndex& index ) const
{
  if ( m_ready
       && index.isValid() )
    {
      Q_ASSERT( index.internalPointer() );
      CategoryItem* item = static_cast<CategoryItem*>( index.internalPointer() );

      return ( Qt::ItemIsEnabled
	       | (m_modelFlags & ItemsAreSelectable
		  ? Qt::ItemIsSelectable
		  : (Qt::ItemFlags)0)
	       | (m_modelFlags & ItemsAreEditable
		  && (item->type == ClassIndex || item->type == CollectionIndex)
		  ? Qt::ItemIsEditable
		  : (Qt::ItemFlags)0)
	       | (m_modelFlags & ItemsAreCheckable
		  ? Qt::ItemIsUserCheckable
		  : (Qt::ItemFlags)0)
	       | (m_modelFlags & ItemsAreDropEnabled
		  ? Qt::ItemIsDropEnabled
		  : (Qt::ItemFlags)0) );
    }

  return 0;
}


int CategoryItemModel::rowCount( const QModelIndex& parent ) const
{
  if ( !m_ready ) return 0;

  if ( parent.isValid() ) {
    Q_ASSERT( parent.internalPointer() );
    return static_cast<CategoryItem*>(parent.internalPointer())->children.count();
  }
  else { // !parent.isValid() -> top-level
    Q_ASSERT( m_rootItem );
    return m_rootItem->children.count();
  }
  return 0;
}


int CategoryItemModel::columnCount( const QModelIndex& /*parent*/ ) const {
  return m_ready ? 1 : 0;
}


QVariant CategoryItemModel::headerData( int /*section*/,
					Qt::Orientation /*orientation*/,
					int /*role*/ ) const
{
  return QVariant(); // TODO: fixme!
}


QModelIndex CategoryItemModel::parent( const QModelIndex& itemIndex ) const
{
  if ( m_ready
       && itemIndex.isValid() ) {
    Q_ASSERT( itemIndex.internalPointer() );
    return index( static_cast<CategoryItem*>(itemIndex.internalPointer())->parent );
  }
  return QModelIndex(); // top-level or failed
}


QModelIndex CategoryItemModel::index( int row, int column,
				      const QModelIndex& parent ) const
{
  /* VERY IMPORTANT:
   * Bounds checks ARE required, as something may call:
   *   index( -1, 0, ... )
   * such as during drag'n'drop when hovering the mouse over empty space
   * or even by the code from this class.
   * This has proved to save the app from exiting on QList bounds assertions.
   */
  if ( !m_ready
       || row < 0 || row > rowCount(parent)
       || column < 0 || column > columnCount() )
    return QModelIndex();

  if ( parent.isValid() ) {
    Q_ASSERT( parent.internalPointer() );
    return createIndex( row, column,
			static_cast<CategoryItem*>(parent.internalPointer())->children.at(row) );
  }
  else { // !parent.isValid() -> top-level
    Q_ASSERT( m_rootItem );
    return createIndex( row, column, m_rootItem->children.at(row) );
  }

  return QModelIndex(); // top-level or failed
}


/*!
 * We employ caching in this method, as it called very often by the views and
 * contains a very expensive line of code that includes createIndex() and the
 * very costly indexOf() (which performs a linear search).
 *
 * Asserts that the \em item is non-null.
 */
// TODO: perform more performance tests!
QModelIndex CategoryItemModel::index( CategoryItem* item ) const
{
  Q_ASSERT( item );

  CategoryItem* itemParent = item->parent;

  if ( item == m_rootItem ) {
    Q_ASSERT( itemParent == 0 ); // sanity check - root must have null parent
  }
  else {
    Q_ASSERT( itemParent != 0 ); // sanity check - non-root must have non-null parent

    if ( itemParent->cachedIndexes.contains(item) )
      return itemParent->cachedIndexes.value(item);
    else {
      // The following line is very expensive, due to createIndex() and indexOf()
      QModelIndex index = createIndex( itemParent->children.indexOf(item), 0, item );
      itemParent->cachedIndexes.insert( item, index );
      return index;
    }
  }

  return QModelIndex(); // root -> return QModelIndex()
}


QVariant CategoryItemModel::data( const QModelIndex& index, int role ) const
{
  if ( m_ready
       && index.isValid() )
    {
      Q_ASSERT( index.internalPointer() );
      CategoryItem* item = static_cast<CategoryItem*>( index.internalPointer() );

      if ( role == Qt::DisplayRole || role == Qt::EditRole )
	return item->label;
      else if ( role == Qt::DecorationRole )
	return item->decoration;
      else if ( role == Qt::CheckStateRole
		&& (m_modelFlags & ItemsAreCheckable) )
	return item->checkState;
    }
  return QVariant();
}


bool CategoryItemModel::setData( const QModelIndex& itemIndex,
				 const QVariant& value, int role )
{
  bool success = false;

  if ( m_ready
       && itemIndex.isValid() )
    {
      Q_ASSERT( itemIndex.internalPointer() );
      CategoryItem* item = static_cast<CategoryItem*>( itemIndex.internalPointer() );

      if ( role == Qt::EditRole )
	{
	  if ( item->type == ClassIndex )
	    postControllerEvent( new CClassUpdateEvent
				 (DClassRecord(item->dbId, value.toString()), this) );
	  else if ( item->type  == CollectionIndex )
	    postControllerEvent( new CCollectionUpdateEvent
				 (DCollectionRecord(item->dbId, value.toString()), this) );
	  // Note: we call dataChanged() upon receiving the update event (if it comes)
	  success = true;
	}
      else if ( role == Qt::CheckStateRole )
	{
	  Qt::CheckState checkState = (Qt::CheckState)value.toInt();

	  item->checkState = checkState;
	  dataChanged( itemIndex, itemIndex ); // needed, even though we return true

	  /* If this item has children, then they need to be toggled also.
	   *
	   * IMPORTANT: Qt::PartiallyChecked is not passed down to children,
	   *            only up to parents. This also helps to prevent inf recursion.
	   *
	   * IMPORTANT: Use setData(), as it will recursively update all children,
	   *            (i.e. their children, their childrens' children and so forth).
	   */
	  if ( checkState != Qt::PartiallyChecked // for parents - not passed on
	       && !item->children.isEmpty() )
	    {
	      foreach ( CategoryItem* c, item->children ) {
		if ( c->checkState != checkState ) // helps prevent inf recursion
		  setData( index(c), value, role );
	      }
	      dataChanged( index(item->children.first()),
			   index(item->children.last()) );
	    }

	  /* If this item has a parent, then it also needs to be updated.
	   * A parent should be (fully)checked IFF all of it's children are
	   * (fully)checked. A parent should be partially-checked IFF between
	   * 1 and |children|-1 children are (fully)checked or partially-checked.
	   * Otherwise, a parent must be unchecked.
	   */
	  CategoryItem* parent = item->parent;
	  if ( parent )
	    {	      
	      bool checkedOrPartialExists = false;
	      bool unCheckedExists = false;

	      foreach ( CategoryItem* pc, parent->children )
		{
		  Q_ASSERT( pc );

		  switch ( pc->checkState ) {
		  case Qt::Checked:
		  case Qt::PartiallyChecked:
		    checkedOrPartialExists = true;
		    break;
		  case Qt::Unchecked:
		    unCheckedExists = true;
		  }

		  // If 1+ checked and 1+ unchecked, no matter how many more we
		  // look at, the parent's check state must be Qt::PartiallyChecked.
		  if ( checkedOrPartialExists && unCheckedExists )
		    break;
		}

	      Qt::CheckState parentState = Qt::Unchecked;
	      if ( checkedOrPartialExists && unCheckedExists )
		parentState = Qt::PartiallyChecked;
	      else if ( checkedOrPartialExists )
		parentState = Qt::Checked;
	      if ( parent->checkState != parentState ) { // helps prevent inf recursion
		QModelIndex parentIndex = index( parent ); // safe to use over * calls?
		setData( parentIndex, parentState, role );
		dataChanged( parentIndex, parentIndex );
	      }
	    }


	  success = true;
	}
    }

  return success;
}


Qt::DropActions CategoryItemModel::supportedDropActions() const
{
  // TODO: support the move action?
  return Qt::CopyAction;
}



QStringList CategoryItemModel::mimeTypes() const
{
  QStringList types;
  types += GESTURE_MIME_FORMAT_STR;
  return types;
}


/*!
 * \b Warning: \em index has been renamed from \em parent, because the latter
 * lead to much confusion, as it really is the index of the item that the MIME
 * data was dropped onto.
 */
bool CategoryItemModel::dropMimeData( const QMimeData* data,
				      Qt::DropAction /*action*/,
				      int /*row*/, int /*column*/,
				      const QModelIndex& index )
{
  Q_ASSERT( data );

  if ( m_ready
       && data->hasFormat(GESTURE_MIME_FORMAT_STR) )
    {
      if ( index.isValid() )
	{
	  IndexType  itemType = LibraryIndex;
	  IdSet      itemSet;
	  IdSet      idSet;

	  Q_ASSERT( index.internalPointer() );
	  CategoryItem* item
	    = static_cast<CategoryItem*>( index.internalPointer() );
	  itemType = item->type;
	  itemSet += item->dbId;

	  // Gesture already in the Library -> no need to continue
	  if ( itemType == LibraryIndex )
	    return false;

	  QByteArray dataArray = data->data(GESTURE_MIME_FORMAT_STR);
	  QDataStream dataStream( &dataArray, QIODevice::ReadOnly );
	  dataStream >> idSet;

	  if ( itemType == ClassIndex )
	    postControllerEvent( new CGesturesChangeClassesEvent
				 (idSet, itemSet, IdSet(), this) );
	  else if ( itemType == CollectionIndex )
	    postControllerEvent( new CGesturesChangeCollectionsEvent
				 (idSet, itemSet, IdSet(), this) );

	  return true;
	}
    }

  return false;
}


/*!
 * Resets the model.
 *
 * Calls resetModel(), followed by reset() (which resets the views).
 */
void CategoryItemModel::resetEvent( VEvent* )
{
  resetModel();  // reset the model
  reset();       // reset the views
}


/*!
 * \b Note: "Classes" live in the Library, which I believe is logical.
 */
void CategoryItemModel::classesEvent( VClassesEvent* event )
{
  Q_ASSERT( event );
  Q_ASSERT( digestDbModel() );

  switch ( event->type() )
    {
    case VDigestDbEvent::ClassesAdded:
      {
	Q_ASSERT( c_libraryItem );
	int firstRow = c_libraryItem->children.count(); // get now before modify
	QList<CategoryItem*> items;
	foreach ( int id, event->idSet() )
	  items += new CategoryItem( ClassIndex, id,
				     digestDbModel()->fetchClassLabel(id),
				     s_classPix, c_libraryItem );
	beginInsertRows( c_libraryIndex, firstRow, firstRow + items.count() );
	c_libraryItem->children += items;
	foreach ( CategoryItem* item, items )
	  m_classIdToItem.insert( item->dbId, item );
	endInsertRows();
	c_libraryItem->sortChildren();
	reset(); // the safest way to update views after sort - esp. for selections
      }
      break;

    case VDigestDbEvent::ClassesRemoved:
      {
	/* Note: There is no need to sort the children on remove.
	 * Note: We call [begin,end]RemoveRows for each class, as they're most
	 *       likely not sequential in order.
	 */
	Q_ASSERT( c_libraryItem );
	foreach ( int id, event->idSet() ) {
	  Q_ASSERT( m_classIdToItem.contains(id) ); // must be true, else -> bug
	  int row = index(m_classIdToItem.value(id)).row();
	  beginRemoveRows( c_libraryIndex, row, row );
	  c_libraryItem->children.removeAt( row );
	  c_libraryItem->cachedIndexes.clear(); // invalidate cache
	  m_classIdToItem.remove( id );
	  endRemoveRows();
	}
      }
      break;

    case VDigestDbEvent::ClassesUpdated:
      {
	// Warning: Calling setData would cause infinite recursion
	foreach ( int id, event->idSet() ) {
	  Q_ASSERT( m_classIdToItem.contains(id) ); // must be true, else -> bug
	  CategoryItem* item = m_classIdToItem.value(id);
	  QModelIndex itemIndex = index( item );
	  item->label = digestDbModel()->fetchClassLabel(id);
	}
	Q_ASSERT( c_libraryItem );
	c_libraryItem->sortChildren();
	reset(); // the safest way to update views after sort - esp. for selections
      }
      break;

    default:
      break;
    }
}


void CategoryItemModel::collectionsEvent( VCollectionsEvent* event )
{
  Q_ASSERT( event );
  Q_ASSERT( digestDbModel() );

  switch ( event->type() )
    {
    case VDigestDbEvent::CollectionsAdded:
      {
	Q_ASSERT( m_rootItem );
	int firstRow = m_rootItem->children.count(); // get now before modify
	QList<CategoryItem*> items;
	foreach ( int id, event->idSet() )
	  items += new CategoryItem( CollectionIndex, id,
				     digestDbModel()->fetchCollectionLabel(id),
				     s_collectionPix, m_rootItem );
	beginInsertRows( QModelIndex(), firstRow, firstRow + items.count() );
	m_rootItem->children += items;
	foreach ( CategoryItem* item, items )
	  m_collectionIdToItem.insert( item->dbId, item );
	endInsertRows();
	m_rootItem->sortChildren();
	reset(); // the safest way to update views after sort - esp. for selections
      }
      break;

    case VDigestDbEvent::CollectionsRemoved:
      {
	/* Note: There is no need to sort the children on remove.
	 * Note: We call [begin,end]RemoveRows for each collection, as they're most
	 *       likely not sequential in order.
	 */
	Q_ASSERT( m_rootItem );
	foreach ( int id, event->idSet() ) {
	  Q_ASSERT( m_collectionIdToItem.contains(id) ); // must be true, else -> bug
	  int row = index(m_collectionIdToItem.value(id)).row();
	  beginRemoveRows( QModelIndex(), row, row );
	  m_rootItem->children.removeAt( row );
	  m_rootItem->cachedIndexes.clear(); // invalidate cache
	  m_collectionIdToItem.remove( id );
	  endRemoveRows();
	}
      }
      break;

    case VDigestDbEvent::CollectionsUpdated:
      {
	// Warning: Calling setData would cause infinite recursion
	foreach ( int id, event->idSet() ) {
	  Q_ASSERT( m_collectionIdToItem.contains(id) ); // must be true, else -> bug
	  CategoryItem* item = m_collectionIdToItem.value(id);
	  QModelIndex itemIndex = index( item );
	  item->label = digestDbModel()->fetchCollectionLabel(id);
	}
	Q_ASSERT( m_rootItem );
	m_rootItem->sortChildren();
	reset(); // the safest way to update views after sort - esp. for selections
      }
      break;

    default:
      break;
    }
}


QModelIndexList CategoryItemModel::checkedIndexes() const {
  return ( m_ready
	   ? checkedIndexes(m_rootItem)
	   : QModelIndexList() );
}


/*!
 * Light-weight, recursive method - used by checkedIndexes().
 *
 * Asserts m_ready and that \em parent is non-null.
 */
QModelIndexList CategoryItemModel::checkedIndexes( CategoryItem* parent ) const
{
  Q_ASSERT( m_ready );
  Q_ASSERT( parent );
  QModelIndexList indexes;
  foreach ( CategoryItem* child, parent->children ) {
    Q_ASSERT( child );
    if ( ! child->children.isEmpty() ) // only makes sense to check terminals
      indexes += checkedIndexes( child );
    else if ( child->checkState == Qt::Checked )
      indexes += index( child );
  }
  return indexes;
}


QString CategoryItemModel::gestureQueryForIndexes( const QModelIndexList& indexes ) const
{
  // TODO: cleanup

  QString query;

  bool chooseAll = false;

  QStringList classIdStrs;
  QStringList collectionIdStrs;
  foreach ( const QModelIndex& index, indexes ) {
    int type = indexType( index );
    if ( type == CategoryItemModel::LibraryIndex ) {
      chooseAll = true;
      break; // we finished - nothing more to choose
    }
    else if ( type == CategoryItemModel::ClassIndex )
      classIdStrs += QString::number( indexDatabaseId(index) );
    else if ( type == CategoryItemModel::CollectionIndex )
      collectionIdStrs += QString::number( indexDatabaseId(index) );
  }

  bool classesChosen = ! classIdStrs.isEmpty();
  bool collectionsChosen = ! collectionIdStrs.isEmpty();
  if ( chooseAll ) {
    query = "select id from gesture except select gestureid as id from classmap"
    " UNION ALL"
      " select A2.gestureid from (select id from class order by label) A1, "
      "(select gestureid, classid from classmap group by gestureid) A2 where A1.id = A2.classid";
  }
  else if ( classesChosen || collectionsChosen )
    {
      query = "select A2.gestureid from (";

      if ( classesChosen )
	/*
	query += ( "SELECT " CLASSMAP_TABLE_GESTUREID_STR
		   " FROM " CLASSMAP_TABLE_STR
		   " WHERE " CLASSMAP_TABLE_CLASSID_STR
		   " IN ("+ classIdStrs.join(",") + ") ORDER BY classId" );
	*/
	query += "select gestureid,classid from classmap A5 where A5.classid in ("\
	  + classIdStrs.join(",") + ") group by A5.gestureid";

      /*
	query = "select A1.gestureid as id, A2.label as label from"
	  " (select * from classmap where classid in ("
	  + classIdStrs.join(",") + ") group by gestureid)"
	  " A1, class A2 where A1.classid = A2.id"
	  " ORDER BY label";
      */

      if ( collectionsChosen ) {
	if ( classesChosen ) // both collections and classes chosen - need OR
	  query += " UNION ";
	/*
	query += ( "SELECT " COLLECTIONMAP_TABLE_GESTUREID_STR
		   " FROM " COLLECTIONMAP_TABLE_STR
		   " WHERE " COLLECTIONMAP_TABLE_COLLECTIONID_STR
		   " IN ("+ collectionIdStrs.join(",") + ")" );
	*/

	query += "select A3.gestureid as gestureId, A4.classid as classId from"
	  " (select gestureid from collectionmap where collectionid in"
	  " ("+ collectionIdStrs.join(",") + ")"
	  " group by gestureid) A3 left join classmap A4 "
	  " on A3.gestureid = A4.gestureid group by A3.gestureid";

      }

      query += ") A2 left join (select id from class order by label) A1 on A1.id = A2.classid";
    }

  return query;
}


/*!
 * Convenience method.
 *
 * Equivalent to gestureQueryForIndexes( checkedIndexes() );
 */
QString CategoryItemModel::gestureQueryForCheckedIndexes() const {
  return gestureQueryForIndexes( checkedIndexes() );
}
