/*
 *  gesturequerymodel.cpp
 *  Digest
 * 
 *  Created by Aidan Lane on Tue Jul 05 2005.
 *  Copyright (c) 2005-2006 Optimisation and Constraint Solving Group,
 *  Monash University. All rights reserved.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#include "gesturequerymodel.h"

#include <QDataStream>
#include <QDebug> // TODO: remove me!
#include <QMimeData>
#include <QPainter>
#include <QPen>
#include <QPixmap>
#include <QPixmapCache>
#include <QSet>
#include <QSqlRecord>

#include "Gesture/strokelist.h"
#include "MvcDigestDb/digestdbmodel.h"
#include "MvcDigestDb/digestdbcontroller.h"
#include "DigestGuiCore/strokespainterpath.h"


// TODO: find out the indices using a strings (e.g. "id") and then cache them!
#define GESTURE_TABLE_ID_INDEX       0
#define GESTURE_TABLE_STROKES_INDEX  5

// TODO: move the following to a globals file!
#define GESTURE_MIME_FORMAT_STR  "text/x-gesture-id-set"

// Note: The cache string is prefixed, as to reduce the probability of conflicts
#define GESTURE_ITEM_CACHE_STR(ID) ("G" + QString::number(ID))



/*!
 * \class GestureQueryModel
 *
 * \brief The GestureQueryModel class provides a thumbnail icon view of a set of
 *        gestures.
 *
 * This class extends QSqlQueryModel to provide gesture previews, in the form of
 * thumbnail icons. The thumbnail pixmaps are fetched by views by them calling
 * data() with the role set to Qt::DecorationRole.
 */


/*!
 * Constructs a category item model with the given \em parent.
 */
GestureQueryModel::GestureQueryModel( QObject* parent )
  : QSqlQueryModel(parent),
    AbstractDigestDbView(this)
{
  init();
}


/*!
 * Constructs a category item model with the given \em parent and attaches itself
 * to the \em controller.
 */
GestureQueryModel::GestureQueryModel( DigestDbController* controller, QObject* parent )
  : QSqlQueryModel(parent),
    AbstractDigestDbView(this)
{
  init();

  if ( controller )
    controller->attachView( this );
}



void GestureQueryModel::init()
{
  m_thumbnailSize = QSize( 52, 52 );
  m_strokeWidth = 3.0;
  m_tileMargin = 2;

  /* Note: QPixmapCache is better than QCache in this case, as it should pre-
   *       allocate memory, it natively allows for sharing across instances and
   *       because it also natively uses the size of the pixmap data (keeping
   *       memory usage under control).
   *
   * Note: At 16MB, using icons of at 52x52x32bit each we can store > 1500
   *       gestures. However, there may be other users of the cache.
   */
  if ( QPixmapCache::cacheLimit() < 16384 )
    QPixmapCache::setCacheLimit( 16384 ); // TODO: turn this into a property

  m_strokesPen = QPen( Qt::black, m_strokeWidth, Qt::SolidLine, Qt::RoundCap, Qt::RoundJoin );
  m_tilePen    = QPen( Qt::lightGray, 1, Qt::SolidLine, Qt::RoundCap, Qt::RoundJoin );
  m_tileBrush  = QBrush( QColor(246,246,246) ); /* light enough not to look selected,
						   dark enough to help real selection */
  setThumbnailSize( m_thumbnailSize ); // init thumbnail tile cache (+ test)
}


Qt::ItemFlags GestureQueryModel::flags( const QModelIndex& /*index*/ ) const {
  return ( Qt::ItemIsSelectable | Qt::ItemIsEnabled | Qt::ItemIsDragEnabled );
}


QVariant GestureQueryModel::data( const QModelIndex& index, int role ) const
{
  if ( ! index.isValid() )
    return QVariant();   

  if ( role == Qt::DisplayRole ) // TODO: make me optional
    return QVariant(); // don't display labels as they mess up the iconview
  else if ( role == Qt::DecorationRole ) {
    int id = rowGestureId( index.row() );
    return ( id >= 0
	     ? QVariant::fromValue(fetchThumbnail(id))
	     : QVariant() );
  }
  else
    return QSqlQueryModel::data( index, role );
}


QStringList GestureQueryModel::mimeTypes() const
{
  QStringList types;
  types += GESTURE_MIME_FORMAT_STR;
  return types;
}


QMimeData* GestureQueryModel::mimeData( const QModelIndexList& indexes ) const
{
  QMimeData* mimeData = 0;

  if ( !indexes.isEmpty() )
    {
      // Gather the IDs of the gestures that are identified by the indexes
      QSet<int> idSet; // QSet used to eliminate duplicates (vs. QList)
      idSet.reserve( indexes.size() ); // may help
      bool intOk = false;
      foreach ( const QModelIndex& index, indexes ) {
	idSet.insert( record(index.row())
		      .value(GESTURE_TABLE_ID_INDEX).toInt(&intOk) );
	Q_ASSERT( intOk );
      }

      QByteArray data;
      QDataStream dataStream( &data, QIODevice::WriteOnly );
      dataStream << idSet;

      mimeData = new QMimeData;
      mimeData->setData( GESTURE_MIME_FORMAT_STR, data );
    }

  return mimeData;
}


QModelIndex GestureQueryModel::findIndexOfGesture( int gestureId )
{
#if 0 // NOT usable anymore, as we now sort by the label of the first class of each gesture:
  // Use binary search to find the row of the gesture
  // The following assumes that the table is ordered/sorted by the ID column
  QModelIndex indexFound;
  bool intOk = false;
  int left = 0;
  int right = rowCount() - 1;
  int rowOfGesture = -1;  // init to invalid
  while ( left <= right )
    {
      int mid = (left+right) / 2;
      int midRowId = record(mid).value(0).toInt( &intOk );
      Q_ASSERT( intOk );
      if ( midRowId == gestureId ) {
	rowOfGesture = mid;
	break;
      }
      else if ( gestureId < midRowId ) right = mid - 1;
      else                             left  = mid + 1;
    }
#else
  // TODO: fixme, as this is too slow!
  QModelIndex indexFound;
  bool intOk = false;
  int rowOfGesture = -1;  // init to invalid
  for ( int r=0; r < rowCount(); ++r )
    {
      int id = record(r).value(0).toInt( &intOk );
      Q_ASSERT( intOk );
      if ( id == gestureId ) {
	rowOfGesture = r;
	break;
      }
    }
#endif

  if ( rowOfGesture >= 0 )
    indexFound = index( rowOfGesture, 0 );

  return indexFound;
}


/*!
 * Resets the model's views.
 */
void GestureQueryModel::resetEvent( VEvent* )
{
  /* Note: There is no need to invalidate the cache - old gestures will be
   *       overwritten as needed.
   *       Also, keeping them there allows for sharing between objects,
   *       especially as this method is called just after each creation.
   */
  reset(); // reset the views
}


/*!
 * Emits dataChanged() on VGesturesEvent::GesturesUpdated.
 */
void GestureQueryModel::gesturesEvent( VGesturesEvent* event )
{
  // Note: There is no need to invalidate the cache on a remove (only on update)
  //       - old gestures will be overwritten as needed.
  Q_ASSERT( event );
  if ( event->type() == (int)VGesturesEvent::GesturesUpdated ) {
    foreach ( int id, event->idSet() ) {
      QModelIndex index = findIndexOfGesture( id );
      QPixmapCache::remove( GESTURE_ITEM_CACHE_STR(id) ); // as it changed
      emit dataChanged( index, index );
    }
  }
}


/*!
 * Size of the thumbnail icon.
 *
 * Defaults to 52x52.
 */
const QSize& GestureQueryModel::thumbnailSize() const {
  return m_thumbnailSize;
}


/*!
 * Sets the thumbnailSize(), updates the thumbnail tile cache and clears the
 * entire thumbnail cache.
 *
 * Emists thumbnailSizeChanged() before returning.
 */
void GestureQueryModel::setThumbnailSize( const QSize& size )
{
  // Avoid clearing the cache unless we absolutely must
  // CRITICAL, even though it's harsh on other cache users - should be rare though
  if ( size != m_thumbnailSize )
    QPixmapCache::clear();

  m_thumbnailSize = size;
  c_thumbnailTile = generateThumbnailTile( size ); // needed at least at init

  reset(); // reset the views

  emit thumbnailSizeChanged( size );
}


/*!
 * This is an overloaded member function, provided for convenience.
 * It behaves essentially like the above function.
 */
void GestureQueryModel::setThumbnailSize( int w, int h ) {
  setThumbnailSize( QSize(w,h) );
}


/*!
 * Returns the gesture ID for the given \em row.
 *
 * Returns -1 on failure.
 */
int GestureQueryModel::rowGestureId( int row ) const
{
  int id = -1;
  bool intOk = false;
  QSqlRecord rec = record( row );
  if ( !rec.isEmpty() ) {
    QVariant val = rec.value( GESTURE_TABLE_ID_INDEX );
    if ( !val.isNull() ) {
      id = val.toInt(&intOk);
      Q_ASSERT( intOk );
    }
  }
  return id;
}


/*!
 * Attempts to get the thumbnail from a cache. If that fails, this calls
 * generateThumbnail(), updates the cache and then returns the result.
 *
 * Caching is absolutely required, as data() is called a lot, asking for the
 * thumbnail.
 */
QPixmap GestureQueryModel::fetchThumbnail( int gestureId ) const
{
  // WARNING! The following needs to be cheap, given that the views call it
  //          VERY frequently. Hence, we do lots of caching.

  // Note: We can't use the row as the key of the cache, as a gesture with a
  //        single ID may change rows (at any time, many times).
  QPixmap pm;
  if ( !QPixmapCache::find(GESTURE_ITEM_CACHE_STR(gestureId), pm) ) {
    pm = generateThumbnail( gestureId );
    if ( !pm.isNull() ) // don't add it if it was null (e.g. DB not ready yet)
      QPixmapCache::insert( GESTURE_ITEM_CACHE_STR(gestureId), pm );
  }
  return pm;
}



/*!
 * Generates a thumbnail pixmap for the given \em gestureId.
 *
 * Returns a null pixmap on failure.
 */
QPixmap GestureQueryModel::generateThumbnail( int gestureId ) const
{
  QPixmap pm = c_thumbnailTile;
  QPainter painter( &pm );
  StrokesPainterPath path;

  if ( digestDbModel() )
    path.addStrokes( digestDbModel()->fetchGesture(gestureId).strokes );
  else
    return QPixmap(); // error -> return null pixmap

  float halfStrokeWidth = m_strokeWidth / 2.0;
  int contentsWidth = m_thumbnailSize.width() - (m_tileMargin * 2);
  int contentsHeight = m_thumbnailSize.height() - (m_tileMargin * 2);
 
  /* From Qt4.0.0 doc re: controlPointRect()
   * "This function is significantly faster to compute than the exact
   * boundingRect();"
   *
   * Note: we need to account for the stroke width, so we don't cut it off.
   */
  const QRectF pathRect = path.controlPointRect().adjusted( -halfStrokeWidth,
							    -halfStrokeWidth,
							    halfStrokeWidth,
							    halfStrokeWidth );

  // TODO: introduce more math caching!
  float a = pathRect.width() / pathRect.height();
  float sW = contentsWidth / pathRect.width(); // scale width
  float sH = contentsHeight / pathRect.height(); // scale height
  float horizOffset = m_tileMargin; // used for centering
  float vertOffset = m_tileMargin; // ditto.
  if ( a >= 1.0 ) { // width > height
    sH /= a;
    vertOffset += (contentsHeight - (pathRect.height() * sH)) / 2.0;
  }
  else {
    sW *= a;
    horizOffset += (contentsWidth - (pathRect.width() * sW)) / 2.0;
  }

  // NOTE: The pen WILL be scaled
  painter.setRenderHint( QPainter::Antialiasing, true ); // TODO: make this optional
  painter.translate( horizOffset, vertOffset ); // MUST center it here, where we know the bounds
  painter.scale( sW, sH );
  painter.translate( -pathRect.left(), -pathRect.top() );
  painter.setPen( m_strokesPen );
  painter.drawPath( path );

  return pm;
}


QPixmap GestureQueryModel::generateThumbnailTile( const QSize& size ) const
{
  QPixmap pm( size );
  pm.fill( Qt::transparent ); // set background -> otherwise random mess!
  QPainter painter( &pm );
  painter.setRenderHint( QPainter::Antialiasing, true ); // TODO: make this optional
  painter.setBrush( m_tileBrush );
  painter.setPen( m_tilePen );
  painter.drawRoundRect( 0, 0, size.width()-1, size.height()-1, 10, 10 ); // TODO: make optional!
  return pm;
}
