/*
 *  strokeseditor.cpp
 *  Digest
 * 
 *  Created by Aidan Lane on Mon Jun 27 2005.
 *  Copyright (c) 2005-2006 Optimisation and Constraint Solving Group,
 *  Monash University. All rights reserved.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#include "strokeseditor.h"

#include <QCursor>
#include <QMouseEvent>
#include <QPixmap>
#include <QTabletEvent>
#include <QTime>


#define MOUSE_STROKE_PRESSURE  1.0 // intuitive, given button completly down,
//                                    also turns stroke data into "1" -> cheap!
//                                    TODO: turn me into a property?
#define SINGLE_POINT_WIDTH     0.5


StrokesEditor::StrokesEditor( QWidget* parent )
  : StrokesViewer(parent)
{
  init();
  setReadOnly( false ); // also updates the cursor
}


/*!
 * Convenience constructor.
 */
StrokesEditor::StrokesEditor( const StrokeList& strokes, QWidget* parent,
			      bool readOnly )
  : StrokesViewer(strokes, parent)
{
  init();
  setReadOnly( readOnly ); // also updates the cursor
}


void StrokesEditor::init()
{
  m_state        = StrokesEditor::EmptyState;
  m_autoClear    = true;
  m_modified     = false;
  m_readOnly     = false;
  m_strokeBeginDrawn     = false;
  m_recordHiRes  = false;
  m_multiStrokeTimeout = 1.0; // 1 second

  m_multiStrokeTimeoutTimer = new QTimer( this );
  m_multiStrokeTimeoutTimer->setSingleShot( true );
  connect( m_multiStrokeTimeoutTimer,
	   SIGNAL(timeout()), SLOT(onMultiStrokeTimedout()) );
  
  /* VERY IMPORTANT: We only need to call start() once, then we can call
   *                 restart() (which is atomic, unlike start()) for each stroke.
   *                 Note: It's safe to call this even if we don't use the it.
   */
  m_currentMultiStrokeTime.start();
}


StrokesEditor::State StrokesEditor::state() const {
  return m_state;
}


bool StrokesEditor::autoClear() const {
  return m_autoClear;
}

void StrokesEditor::setAutoClear( bool autoClear ) {
  m_autoClear = autoClear;
}


/*! \n Note: Calling setStrokes() resets the modified flag to false. */
bool StrokesEditor::isModified() const {
  return m_modified;
}

/*! \b Note: Calling setStrokes() resets the modified flag to false. */
void StrokesEditor::setModified( bool modified ) {
  m_modified = modified;
}


bool StrokesEditor::isReadOnly() const {
  return m_readOnly;
}

void StrokesEditor::setReadOnly( bool readOnly )
{
  m_readOnly = readOnly;

  if ( m_readOnly )
    unsetCursor();
  else  // This one looks better than Qt::CrossCursor :-)
    setCursor( QCursor(QPixmap(":/images/Crosshair.png"), 7, 7) ); // not (8,8)
}


bool StrokesEditor::isStrokeBeginDrawn() const {
  return m_strokeBeginDrawn;
}


bool StrokesEditor::recordHiRes() const {
  return m_recordHiRes;
}

void StrokesEditor::setRecordHiRes( bool enable ) {
  m_recordHiRes = enable;
  emit recordHiResToggled( m_recordHiRes );
}


/*!
 * Calling this will reset the modified flag to false.
 *
 * Also calls stopMultiStrokeTimoutTimer();
 */
void StrokesEditor::setStrokes( const StrokeList& strokes, bool updateWidget )
{
  stopMultiStrokeTimoutTimer();
  
  m_state = ( strokes.isEmpty()
	      ? StrokesEditor::EmptyState
	      : StrokesEditor::FinishedState );

  m_modified = false;

  // End stroking WITHOUT adding the current stroke to the strokes
  m_currentStroke = Stroke(); // clear current stroke
  m_strokeBeginDrawn = false;
  // TODO: EMIT strokingFinished() with device ID

  StrokesViewer::setStrokes( strokes, updateWidget );
}


/*!
  Returns the how much time can pass between multiple strokes before
  multiStrokeFinished() is emitted. The time is expressed in terms of seconds.
*/
double StrokesEditor::multiStrokeTimeout() const {
  return m_multiStrokeTimeout;
}

/*!
  Sets how much time can pass between multiple strokes before
  multiStrokeFinished() is emitted. The time is expressed in terms of seconds.
*/
void StrokesEditor::setMultiStrokeTimeout( double timeout ) {
  m_multiStrokeTimeout = timeout;
  emit multiStrokeTimeoutChanged( timeout );
}


void StrokesEditor::mousePressEvent( QMouseEvent* event )
{
  Q_ASSERT( event );
  if ( ! m_readOnly
       && event->button() == Qt::LeftButton ) {
    if ( m_strokeBeginDrawn ) endStroke(); // e.g. if TABLET stroking in progress
    beginNewStroke( StrokePoint(event->pos(),
			       MOUSE_STROKE_PRESSURE,
			       (uint32_t)m_currentMultiStrokeTime.elapsed()) );
  }
}


void StrokesEditor::mouseMoveEvent( QMouseEvent* event )
{
  Q_ASSERT( event );
  if ( ! m_readOnly
       && m_strokeBeginDrawn )
    addPointToStroke( StrokePoint(event->pos(),
				  MOUSE_STROKE_PRESSURE,
				  (uint32_t)m_currentMultiStrokeTime.elapsed()) );
}


void StrokesEditor::mouseReleaseEvent( QMouseEvent* event )
{
  Q_ASSERT( event );
  if ( ! m_readOnly
       && m_strokeBeginDrawn
       && event->button() == Qt::LeftButton ) {
    addPointToStroke( StrokePoint(event->pos(),
				  MOUSE_STROKE_PRESSURE,
				  (uint32_t)m_currentMultiStrokeTime.elapsed()) );
    endStroke();
  }
}


void StrokesEditor::tabletEvent( QTabletEvent* event )
{
  Q_ASSERT( event );
  
  if ( m_readOnly
       // Note: Disallow Eraser instead of allowing only Pen, as UnknownPointer
       //       and Cursor are fairly legitimate too, since the mouse is allowed.
       || event->pointerType() == QTabletEvent::Eraser ) return;

  event->accept(); // don't pass this TABLET event on to the mouse event handlers!

  QPointF localPos;
  uint32_t time = (uint32_t) m_currentMultiStrokeTime.elapsed();

  if ( m_recordHiRes ) {
    localPos = QPointF( event->hiResGlobalX(), event->hiResGlobalY() );
    localPos -= mapToGlobal( QPoint(0,0) );
  } else {
    localPos = event->pos();
  
  }

  switch ( event->type() )
    {
    case QEvent::TabletPress:
      if ( m_strokeBeginDrawn ) endStroke(); // e.g. if MOUSE stroking in progress
      beginNewStroke( StrokePoint(localPos, event->pressure(), time) );
      break;

    case QEvent::TabletMove:
      if ( m_strokeBeginDrawn )
	addPointToStroke( StrokePoint(localPos, event->pressure(), time) );
      break;

    case QEvent::TabletRelease:
      if ( m_strokeBeginDrawn ) {
	addPointToStroke( StrokePoint(localPos, event->pressure(), time) );
	endStroke();
      }
      break;

    default:
      break;
    }
}


/*!
 * \b Note:
 * Unlike endStroke(), this requires the x, y and pressure parameters because
 * this corresponds to a MOVE TO, vs. the addPointToStroke()'s LINE TO.
 * Also, you may want to call endStroke() without a point.
 *
 * If this is not the first stroke and the time between now and when the previous
 * stroke began is larger then the multi-stroke timeout, then the
 * multiStrokeFinished() signal is emitted first.
 *
 * Sets the modified flag to true.
 */
void StrokesEditor::beginNewStroke( StrokePoint pt )
{
  // Note: We don't reset the time between strokes, as that time information may
  //       be useful for recognition.
  if ( m_state != StrokesEditor::StartedState ) { // i.e. empty or finished
    if ( m_autoClear ) StrokesViewer::setStrokes( StrokeList() );
    m_currentMultiStrokeTime.restart(); // unlike start(), this's atomic
    pt.milliTime = 0;
    m_state = StrokesEditor::StartedState;
  }

  restartMultiStrokeTimoutTimer(); // point added -> restart timer

  m_modified = true;

  qreal x = pt.x();
  qreal y = pt.y();

  m_currentStroke = Stroke(); // ensure that current stroke is empty
  m_currentStroke.append( pt );

  pathRef().moveTo( pt );

  // Ensure that single-point strokes are visible
  pathRef().addEllipse( x, y, SINGLE_POINT_WIDTH, SINGLE_POINT_WIDTH );

  // Update the effected region
  // Note: we have to account for the width of the pen
  float rad = pen().width() / 2.0 + 1.0; // + 1.0 to account for error TODO: cache this!
  QRectF r; // use adjust, as (x, y) is a point, not width and height
  r.adjust( x, y, x, y );
  update( r.adjusted(-rad, -rad, rad, rad).toRect() );

  m_strokeBeginDrawn = true;
  
  emit strokingStarted(); // TODO: emit with device ID?
}


/*!
 * Asserts that isStroking() == true.
 */
void StrokesEditor::addPointToStroke( const StrokePoint& pt )
{
  Q_ASSERT( m_strokeBeginDrawn );

  restartMultiStrokeTimoutTimer(); // point added -> restart timer

  m_currentStroke.append( pt );
  pathRef().lineTo( pt );

  // Update the effected region
  // Note: we have to account for the width of the pen
  // TODO: keep track of the previous point instead of having to re-fetch it every time!
  Q_ASSERT( m_currentStroke.size() >= 2 );
  const StrokePoint& prevPt
    = m_currentStroke.at(m_currentStroke.size()-2);
  float rad = pen().width() / 2.0 + 1.0; // + 1.0 to account for error TODO: cache this!
  QRectF r; // use adjust, as (pt.x, pt.y) is a point, not width and height
  r.adjust( prevPt.x(), prevPt.y(), pt.x(), pt.y() );
  update( r.normalized().adjusted(-rad, -rad, rad, rad).toRect() );
}


void StrokesEditor::endStroke()
{
  // TODO: optimise me!

  /* Add the current stroke to the strokes
   * WARNING! Don't update the ref directly, as we want to call setStrokes,
   *          as that updates the graphic state. - TODO: remove this hack!
   */
  QList<Stroke> newStrokes = strokesRef();
  newStrokes += m_currentStroke;
  StrokesViewer::setStrokes( newStrokes ); /* see warning above!
					    * + don't call our version
					    *   - would clear our state!
					    */
  m_modified = true; // newSroke -> modified (needed, as setStrokes clears the mod flag)

  m_currentStroke = Stroke(); // clear current stroke
  m_strokeBeginDrawn = false;

  // NOTE: we wait for the multi-stroke timeout to fire before changing the state
  //       and emitting strokingFinished(), as there may be more strokes to come.
}


void StrokesEditor::onMultiStrokeTimedout()
{
  m_state = StrokesEditor::FinishedState;
  emit strokingFinished( strokesRef() );
}


void StrokesEditor::stopMultiStrokeTimoutTimer()
{
  Q_ASSERT( m_multiStrokeTimeoutTimer );
  m_multiStrokeTimeoutTimer->stop();
}


void StrokesEditor::restartMultiStrokeTimoutTimer()
{
  Q_ASSERT( m_multiStrokeTimeoutTimer );
  m_multiStrokeTimeoutTimer->stop();
  m_multiStrokeTimeoutTimer->setInterval( (int) (m_multiStrokeTimeout * 1000.0) );
  m_multiStrokeTimeoutTimer->start();
}
