/*
 *  strokesviewer.cpp
 *  Digest
 * 
 *  Created by Aidan Lane on Mon Jun 27 2005.
 *  Copyright (c) 2005-2006 Optimisation and Constraint Solving Group,
 *  Monash University. All rights reserved.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#include "strokesviewer.h"

#include <QPainter>


/*!
 * Default constructor.
 */
StrokesViewer::StrokesViewer( QWidget* parent )
  : QWidget(parent)
{
  init();
}


/*!
 * Convenience constructor.
 *
 * Calls setStrokes(). Hence, all strokes specific initialisation can be left to
 * setStrokes() or a re-implementation of it.
 */
StrokesViewer::StrokesViewer( const StrokeList& strokes, QWidget* parent )
  : QWidget(parent)
{
  init();
  setStrokes( strokes );
}


void StrokesViewer::init()
{
  animTimerId = 0;
  animQueuedPointWaiting = false;

   /* VERY IMPORTANT:
    * We only need to call start() once, then we can call restart()
    * (which is atomic, unlike start()) for each stroke.
    * Note: It's safe to call this even if we don't use the it.
    */
  animStrokeTime.start();

  setPen( QPen(Qt::black, 2.0, Qt::SolidLine, Qt::RoundCap, Qt::RoundJoin), false );
  setBackgroundBrush( QBrush(Qt::white), false );
  setGridPen( QPen(Qt::lightGray), false );
}


/*!
 * Changes the strokes that should be viewed and then calls for an update() if
 * requested.
 */
void StrokesViewer::setStrokes( const StrokeList& strokes, bool updateWidget )
{
  m_strokes = strokes;

  // Clear and re-build the path
  m_path = QPainterPath();
  m_path.addStrokes( strokes );

  if ( updateWidget ) update();
}


/*!
 * Changes the pen that should be used for painting and then calls for an
 * update() if requested.
 */
void StrokesViewer::setPen( const QPen& pen, bool updateWidget )
{
  m_pen = pen;
  if ( updateWidget ) update();
}


/*!
 * Changes the brush that should be used for the background and then calls for
 * an update() if requested.
 */
void StrokesViewer::setBackgroundBrush( const QBrush& brush, bool updateWidget )
{
  m_backgroundBrush = brush;
  if ( updateWidget ) update();
}


/*!
 * Changes the pen that should be used for the grid and then calls for an
 * update() if requested.
 */
void StrokesViewer::setGridPen( const QPen& pen, bool updateWidget )
{
  m_gridPen = pen;
  if ( updateWidget ) update();
}


/*!
 * Emits animationStarted().
 */
void StrokesViewer::animate()
{
  m_path = QPainterPath(); // clear the path
  animStrokeIndex = animStrokePointIndex = 0;
  animQueuedPointWaiting = false;
  animStrokeTime.restart();

  animTimerId = startTimer( 1000 / 30 ); // ~ 30FPS
  Q_ASSERT( animTimerId != 0 ); // timer must be valid

  emit animationStarted();
}


/*!
 * Emits animationStopped().
 */
void StrokesViewer::stopAnimation()
{
  killTimer( animTimerId );
  animTimerId = 0; // reset to 0, as it's used by isAnimating()

  // Re-build the path if the build process was terminated early
  // TODO: OPTIMISE ME: only build the bits that haven't been built
  // -> don't rebuild everything!
  setStrokes( m_strokes );

  emit animationStopped();
}


void StrokesViewer::paintEvent( QPaintEvent* /*event*/ )
{
  QPainter p( this );

  // Draw the background
  p.fillRect( 0, 0, width()-1, height()-1, m_backgroundBrush );

  // Draw the strokes
  p.setRenderHint( QPainter::Antialiasing, true ); // TODO: make this optional
  p.setPen( m_pen );
  p.drawPath( m_path );
}


/*!
 * Progressively builds the path given the current strokes's strokes.
 */
// TODO: OPTIMISE ME: there is no need to do the size checks if we've already got a point that we're waiting on to add
// TODO: FIXME: the current algo. does not allow the playback of faster than the timer interval! (e.g. more than 1 point per interval!!!) - BAD!
void StrokesViewer::timerEvent( QTimerEvent* event )
{
  Q_ASSERT( event != 0 );

  QWidget::timerEvent( event );

  forever
    {
      if ( ! animQueuedPointWaiting )
	{
	  // Terminate timer if all of the strokes have been exhausted
	  if ( animStrokeIndex >= m_strokes.size() ) {
	    stopAnimation();
	    return;
	  }
	  
	  // TODO: FIXME - AMBIGU with "s"
	  const Stroke& cs = m_strokes.at( animStrokeIndex ); // actually is fast
	  
	  // Move on to the next stroke if the current stroke's points have been exhausted
	  if ( animStrokePointIndex >= cs.size() )
	    {
	      ++animStrokeIndex;
	      animStrokePointIndex = 0;
	      
	      // Are we finished now?
	      if ( animStrokeIndex >= m_strokes.size() ) {
		stopAnimation();
		return;
	      }
	    }
	  
	  const Stroke& s = m_strokes.at( animStrokeIndex ); //actually is fast
	  animQueuedPoint = s.at( animStrokePointIndex ); //actually is fast
	  animQueuedPointWaiting = true;
	}

      // Note: This can't be else-if, as the last block may change it and we want
      //       to use the queued point if there is one there now.
      if ( animQueuedPointWaiting )
	{
	  // Add the point to the painter's path if it's time to do so
	  if ( animStrokePointIndex == 0 ) {
	    m_path.moveTo( animQueuedPoint );
	    // Ensure that single-point strokes are visible
	    m_path.addEllipse( animQueuedPoint.x(), animQueuedPoint.y(),
			       // TODO: cache end point width? even though the getter is inlined?
			       m_path.strokeEndPointWidth(), m_path.strokeEndPointWidth() );
	    animQueuedPointWaiting = false;
	  }
	  else if ( animQueuedPoint.milliTime
		    <= (uint32_t)animStrokeTime.elapsed() ) { // wait for turn
	    m_path.lineTo( animQueuedPoint );
	    animQueuedPointWaiting = false;
	  }
      
	  if ( ! animQueuedPointWaiting ) // i.e. POINT USED
	    {
	      ++animStrokePointIndex; // point used -> progress to next point
	      // TODO: only update the region between this point and the last one!
	      update();
	    }
	  else
	    // VERY IMPORTANT!!! - We'll wait until the next timer interval to
	    //                     use the queued point.
	    return;
	}
    }
}
