/*  -*- c++ -*-  (for Emacs)
 *
 *  experimentassistant.cpp
 *  Digest
 * 
 *  Created by Aidan Lane on Thu Nov 3 2005.
 *  Copyright (c) 2005-2006 Optimisation and Constraint Solving Group,
 *  Monash University. All rights reserved.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#include "experimentassistant.h"

#include <QDate>
#include <QDebug> // TODO: remove me!
#include <QDir> // TODO: remove me!
#include <QItemSelectionModel>
#include <QSqlQuery> // TODO: remove me!
#include <QStandardItemModel>
#include <QStringList>
#include <QTemporaryFile>

#include "MvcDigestDb/digestdbmodel.h"
#include "MvcDigestDb/digestdbcontroller.h"
#include "GestureRecognition/abstractrecogniser.h"
#include "GestureRecognition/recogniserfactory.h" // TODO: remove me?
#include "DigestGuiCore/categoryitemmodel.h"
#include "DigestGuiCore/gesturequerymodel.h"

// TODO: move the following to digestdatabase.h
#define CLASS_TABLE_STR          "Class"
#define CLASS_TABLE_ID_INDEX     0
#define CLASS_TABLE_LABEL_INDEX  1
// TODO: move the following to digestdatabase.h
// TODO: use the more flexible STRING indices!
#define GESTURE_TABLE_STR        "Gesture"
#define GESTURE_TABLE_ID_STR     "id"
#define GESTURE_TABLE_ID_INDEX   0


// Multiple instances of ExperimentAssistant may run at any given time, so we
// should share the following strings.
static bool s_strsInitialized = false;
static QStringList s_pageTitleStrs;
#if 0 // TODO: remove me (when used, user could see layout re-adjusting to it):
static QStringList s_pageDescStrs;
#endif


/*!
 * \class ExperimentAssistant
 *
 * \brief The ExperimentAssistant class provides a graphical interface for
 *        performing new experiments.
 */


ExperimentAssistant::ExperimentAssistant( DigestDbController* controller,
					  JavaVM* jvm,
					  QWidget* parent, Qt::WindowFlags flags )
  : GuiDbComponentDialog(controller, parent, flags),
    m_jvm(jvm),
    m_currentStep(ExperimentAssistant::Introduction),
    m_categoryModel(0),
    m_gestureModel(0)
{
  /*
   * WARNING: The use of the models is delayed until we receive VEvent::Reset.
   */
  if ( ! s_strsInitialized ) initStaticStrs();
  m_dummyModel = new QStandardItemModel( this );
  ui.setupUi( this );
  initWidgets();
  reflectCurrentStep();
}


void ExperimentAssistant::initStaticStrs()
{
  Q_ASSERT( ! s_strsInitialized ); // MUST only init once

  s_strsInitialized = true;

  s_pageTitleStrs
    << tr("Introduction")
    << tr("Select Trained Recognisers")
    << tr("Select Gestures")
    << tr("Final Details")
    << tr("Confirm Details")
    << tr("Performing Experiments")
    << tr("Finished");
}


void ExperimentAssistant::initWidgets()
{
  /*
   * WARNING: The use of the model is delayed until we receive VEvent::Reset.
   */

  /*
   * Step 1: Select Trained Recognisers - init performed in resetEvent()
   */

  /*
   * Step 2: Select Gestures
   */

  // Category view
  // Note: we don't pass in the controller, we're using addChildView()
  Q_ASSERT( m_categoryModel == 0 ); // MUST currently be null!
  m_categoryModel = new CategoryItemModel( this ); // see note above re: controller
  addChildView( m_categoryModel );
  // Note: We don't allow for selection of items, as we wouldn't do anything with
  //       the selection anyway and we don't want the user to think that we do.
  //       Also, we don't allow the labels to be edited, nor are items drop enabled.
  m_categoryModel->setModelFlags( CategoryItemModel::ItemsAreCheckable );
  Q_ASSERT( ui.categoryTreeView );
  ui.categoryTreeView->setHeaderText( tr("Source") );
  ui.categoryTreeView->setModel( m_categoryModel );

  connect( m_categoryModel,
	   SIGNAL(dataChanged(const QModelIndex&, const QModelIndex&)),
	   SLOT(onCategoryModelDataChanged()) );

  // Gesture view
  // Note: we don't pass in the controller, we're using addChildView()
  Q_ASSERT( m_gestureModel == 0 ); // MUST currently be null!
  m_gestureModel = new GestureQueryModel( this ); // see note above re: controller
  addChildView( m_gestureModel );
  Q_ASSERT( ui.gestureListView );
  // Note: These settings are so important, that we don't want to leave them in
  // the .ui file, as they may accidently be modified or reset without noticing.
  ui.gestureListView->setHeaderText( tr("Chosen Gestures") );
  ui.gestureListView->setSelectionMode( QAbstractItemView::NoSelection ); // for viewing purposes only
  ui.gestureListView->setViewMode( QListView::IconMode ); // affects other settings -> call this 1st
  ui.gestureListView->setIconSize( QSize(52,52) ); // TODO: use size from saved settings
  ui.gestureListView->setSpacing( 4 );
  ui.gestureListView->setLayoutMode( QListView::SinglePass ); // scrolls better than Batched
  ui.gestureListView->setResizeMode( QListView::Adjust ); // instant update
  ui.gestureListView->setEditTriggers( QAbstractItemView::NoEditTriggers ); // edit elsewhere
  ui.gestureListView->setUniformItemSizes( true ); // enable optimizations
  ui.gestureListView->setMovement( QListView::Static ); // for viewing purposes only
  ui.gestureListView->setDragEnabled( false ); // for viewing purposes only
  ui.gestureListView->setAcceptDrops( false );
}


void ExperimentAssistant::resetEvent( VEvent* )
{
  // Note: As categoryModel is a decedent of AbstractView, it has its own reset.
  //       (i.e. it will receive this event and perform any necessary tasks.)

  /*
   * Clean-up
   */
  Q_ASSERT( ui.trainedRecogListWidget );
  ui.trainedRecogListWidget->clear();

  Q_ASSERT( ui.gestureListView );
  ui.gestureListView->setModel( m_dummyModel ); // can't be null :-(
  ui.gestureListView->reset();


  /*
   * (Re)build
   */
  if ( digestDbModel() )
    {
      /*
       * Step 1: Select Trained Recognisers
       */
      // TODO: pre-prepare the QSqlQuery
      Q_ASSERT( ui.trainedRecogListWidget );
      QSqlQuery q( "SELECT id,label,ready FROM TrainedRecogniser", database() );
      while ( q.next() )
	{
	  QListWidgetItem* item = new QListWidgetItem( ui.trainedRecogListWidget );
	  item->setData( Qt::DisplayRole, q.value(1) ); // label
	  item->setData( Qt::UserRole, q.value(0) );    // ID - internal use
	  item->setFlags( Qt::ItemIsSelectable
			  | Qt::ItemIsUserCheckable
			  | (q.value(2).toBool() // only enabled if it's ready!
			     ? Qt::ItemIsEnabled
			     : (Qt::ItemFlag)0) );
	  item->setCheckState( Qt::Unchecked ); // need to set default for it to work
	}

      // "Check" the first item that is enabled
      // Note: Unlike the "Recogniser Test Pad", experiments are usually
      //       performed on a single recogniser.
      for ( int i=0; i < ui.trainedRecogListWidget->count(); ++i ) {
	QListWidgetItem* item = ui.trainedRecogListWidget->item(i);
	Q_ASSERT( item );
	if ( item->flags() & Qt::ItemIsEnabled ) {
	  item->setCheckState( Qt::Checked );
	  break;
	}
      }


      /*
       * Step 2: Select Gestures
       */
      // FIXME: This doesn't work - model is reset afterward?
      Q_ASSERT( m_categoryModel );
      m_categoryModel->setData( m_categoryModel->libraryIndex(), // good default:
				Qt::Checked, Qt::CheckStateRole );

      Q_ASSERT( ui.gestureListView );
      ui.gestureListView->setModel( m_gestureModel ); // *** note: set last, after properties
      onCategoryModelDataChanged(); // update the query now
    }
}


void ExperimentAssistant::on_goBackButton_clicked() {
  gotoStep( (Step)(m_currentStep-1) );
}


void ExperimentAssistant::on_continueButton_clicked() { // a.k.a. "Begin" and "Quit"
  gotoStep( (Step)(m_currentStep+1) );
}


void ExperimentAssistant::gotoStep( ExperimentAssistant::Step step )
{
  bool callBeginExperiments = false;

  m_currentStep = step;
  Q_ASSERT( m_currentStep >= ExperimentAssistant::Introduction ); // sanity check
  Q_ASSERT( m_currentStep <= ExperimentAssistant::Closed );       // ditto.
 
  // Perform tasks that are required for the current step
  switch ( m_currentStep )
    {
    case ExperimentAssistant::SelectGestures:
      // TODO: remove the need for this hack!
      //       - currently required to get the icons to appear in a grid from first view
      ui.gestureListView->setViewMode( QListView::ListMode );
      ui.gestureListView->setViewMode( QListView::IconMode );
      break;

    case ExperimentAssistant::FinalDetails:
      // TODO: Generate a default label for the experiment (e.g. recog type if 1 recog, or "multiple-" for 2+ recog)
      // TODO: ensure that the label is unique!
      Q_ASSERT( ui.labelEdit );
      if ( ! ui.labelEdit->isModified() )
	ui.labelEdit->setText( "Experiment" ); // use title
      break;

    case ExperimentAssistant::ConfirmDetails:
      gatherDetails();
      Q_ASSERT( ui.detailsHtmlWidget );
      ui.detailsHtmlWidget->setHtml( detailsAsHtml() );
      break;

    case ExperimentAssistant::PerformingExperiments:
      // Disable the continue button until we've not only trained the recogniser
      // but also have renamed it's file and its' READY for use. See Finished below:
      Q_ASSERT( ui.continueButton );
      ui.continueButton->setEnabled( false );
      callBeginExperiments = true;
      break;

    case ExperimentAssistant::Finished:
      Q_ASSERT( ui.continueButton );
      ui.continueButton->setEnabled( true );
      break;

    case ExperimentAssistant::Closed:
      close();
      return; // required - can't step any further!
    
    default:
      break;
    }

  reflectCurrentStep();

  /* This is a bit hacky, but it allows the user to see the Training page while
   * training is in progress, thus making the progress bar visible.
   */
  if ( callBeginExperiments ) beginExperiments();
}


void ExperimentAssistant::reflectCurrentStep()
{
  /* WARNING:
   * We don't set the description directly here, instead we have a different
   * label widget on each page of the stepWidgetStack.
   * Otherwise, the user would see the layout being re-adjusted to fit it (yuck!).
   */

  setUpdatesEnabled( false ); // start of many changes

  Q_ASSERT( ui.stepTitleLabel );
  Q_ASSERT( ui.stepWidgetStack );
  ui.stepTitleLabel->setText( s_pageTitleStrs.at(m_currentStep) );
  ui.stepWidgetStack->setCurrentIndex( m_currentStep );

  Q_ASSERT( ui.goBackButton );
  ui.goBackButton->setEnabled( m_currentStep > ExperimentAssistant::Introduction
			       && m_currentStep < ExperimentAssistant::PerformingExperiments ); // past point of no return

  Q_ASSERT( ui.continueButton );
  ui.continueButton->setText( (m_currentStep >= ExperimentAssistant::Finished)
				 ? tr("&Close")
				 : ((m_currentStep >= ExperimentAssistant::ConfirmDetails)
				    ? tr("&Begin")
				    : tr("&Continue")) );				 

  setUpdatesEnabled( true ); // end of many changes
}


void ExperimentAssistant::gatherDetails()
{
  bool intOk = false;

  // Step 1. Select Trained Recognisers
  Q_ASSERT( ui.trainedRecogListWidget );
  m_recogniserIds.clear(); // VERY IMPORTANT, as we may be called multiple times (back&forward)
  for ( int i=0; i < ui.trainedRecogListWidget->count(); ++i ) {
    QListWidgetItem * item = ui.trainedRecogListWidget->item(i);
    Q_ASSERT( item );
    if ( item->checkState() == Qt::Checked ) {
      m_recogniserIds += item->data(Qt::UserRole).toInt(&intOk);
      Q_ASSERT( intOk );
    }
  }

  // Step 2. Gestures
  Q_ASSERT( m_categoryModel );
  m_gestureIds.clear(); // VERY IMPORTANT, as we may be called multiple times
  m_gestureQuery = m_categoryModel->gestureQueryForCheckedIndexes(); // be safe -> update it!
  QSqlQuery q( m_gestureQuery );
  while ( q.next() ) {
    m_gestureIds += q.value(0).toInt(&intOk);
    Q_ASSERT( intOk );
  }

  // Step 3. Final Details
  Q_ASSERT( ui.labelEdit );
  Q_ASSERT( ui.notesEdit );
  m_labelStr = ui.labelEdit->text();
  m_notesStr = ui.notesEdit->text();
}


void ExperimentAssistant::beginExperiments()
{
  /* We post a CExperimentAddEvent for each recogniser here, as to safely reserve
   * an experiment ID for each of them before we start the actual tests.
   */

  QDate date = QDate::currentDate(); // use the same for all experiments
  foreach ( int id, m_recogniserIds )
    {
      // Note: the experiment's ID will be automatically generated
      DExperimentRecord rec;
      rec.label = m_labelStr;
      rec.external = false;
      rec.date = date;
      rec.notes = m_notesStr;
      rec.trainedRecogniserId = id;
       
      // Continues in ExperimentAssistant::experimentsEvent() ...
      postControllerEvent( new CExperimentAddEvent(rec, this) );
    }
}


void ExperimentAssistant::experimentsEvent( VExperimentsEvent* event )
{
  // TODO: cleanup!

  Q_ASSERT( event );
  Q_ASSERT( digestDbModel() );

  if ( event->originalSender() != this // don't catch what belongs to others
       || event->type() != (int)VDigestDbEvent::ExperimentsAdded )
    return;

  // Gather details about the event - ONE PER EXPERIMENT (thus also trained recogniser)

  // Add event -> always one ID
  // Can catch bad SQL insert, as it indirectly uses lastInsertedExperimentIdSet()
  Q_ASSERT( event->idSet().size() == 1 );

  int experimentId = * event->idSet().constBegin(); // hacky!
  DExperimentRecord rec = digestDbModel()->fetchExperiment( experimentId );
  m_recogToExper.insert( rec.trainedRecogniserId, experimentId );

  if ( m_recogToExper.size() == m_recogniserIds.size() ) // added them all yet?
    {
      // *** Finished adding / reserving experiment records, one for each recogniser.
      // PROCESS ***ALL*** experiments now - ***ALL*** recognisers

      QHash<int, AbstractRecogniser*> recogToInst;
      QHash<int, int> recogToCorrect;

      /*
       * By instantiating each of the recognisers first and then testing them on each
       * of the gestures, we can ensure that we load and parse the stroke data for
       * gesture only once.
       */
      foreach ( int id, m_recogniserIds )
	{
	  DTrainedRecogRecord rec = digestDbModel()->fetchTrainedRecog( id );

	  AbstractRecogniser* r
	    = RecogniserFactory::create( rec.recogniserKey, m_jvm,
					 digestDbModel(), this );
	  if ( !r ) { // TODO: handle me properly!!!! - user may have got it wrong!
	    qDebug("ERROR: Invalid recogniser requested!");
	    // TODO: update the settings?
	    continue;
	  }
	  if ( ! r->loadRecord(rec) ) {
	    qDebug( "recogniser loadRecord failed!" ); // TODO: handle this properly!
	    continue;
	  }
	  recogToInst.insert( id, r );
	}


      Q_ASSERT( ui.progressBar );
      ui.progressBar->setMinimum( 0 );
      ui.progressBar->setMaximum( 1000 ); /* safer # than possibly huge # of gestures,
					   * but also with enough resolution still
					   * (can matter with GUIs - especially on
					   *  different windowing systems)
					   */

      float iteratorToProgress = ( m_gestureIds.isEmpty()
				   ? ui.progressBar->maximum()
				   : ((float)ui.progressBar->maximum()
				      / (float)m_gestureIds.size()) );
  
      // TODO: move transaction code into DigestDbModel - dangerous here!
      QSqlDatabase db = database();
      db.transaction(); // BEGIN new transaction
      for ( int gi=0; gi < m_gestureIds.size(); ++gi )
	{
	  DGestureRecord grec = digestDbModel()->fetchGesture( m_gestureIds.at(gi) );

	  QHashIterator<int, AbstractRecogniser*> recogIt( recogToInst );
	  while ( recogIt.hasNext() )
	    {
	      recogIt.next();
	      Q_ASSERT( recogIt.value() ); // non-null instance

	      ClassProbabilities results = recogIt.value()->classify( grec.strokes );
	      ClassProbabilitiesIterator resultsIt( results );
	      int               testClassId = -1;
	      int               highest_classId = -1;
	      ClassProbabilityT highest_classProb = -1.0; // init to a probability value that anything should be >
	      QString           otherResultsStr;

	      // Find result with the probability
	      while ( resultsIt.hasNext() ) {
		resultsIt.next();
		if ( resultsIt.value() > highest_classProb ) {
		  highest_classId = resultsIt.key();
		  highest_classProb = resultsIt.value();
		}
	      }

	      // Build otherResultsStr - TODO: should this be kept sorted for the user's sake?
	      resultsIt.toFront();
	      while ( resultsIt.hasNext() ) {
		resultsIt.next();
		if ( resultsIt.key() != highest_classId ) // don't store highest - that's stored separately
		  otherResultsStr += QString("%1=%2,").arg(resultsIt.key()).arg(resultsIt.value());
	      }
	      if ( otherResultsStr.endsWith(",") ) otherResultsStr.chop(1);

	      // Try to find a match to highest_classId in grec.classes as the testClassId
	      if ( grec.classes.contains(highest_classId) ) {
		testClassId = highest_classId;
		recogToCorrect[recogIt.key()]++; // RECORD AS CORRECT
	      }
	      else if ( !grec.classes.isEmpty() ) { // MUST test this! - leave testClassId as -1 if it has none
		testClassId = * grec.classes.begin();
	      }

	      // TODO: optimise me!
	      // TODO: move this code into DigestDbModel (?)
	      QSqlQuery q;
	      q.prepare( "INSERT INTO ExperimentResult "
			 "VALUES (?,?,?,?,?,?)" );
	      q.addBindValue( m_recogToExper
			      .value(recogIt.key()) );  // experimentId
	      q.addBindValue( m_gestureIds.at(gi) );    // testGestureId
	      q.addBindValue( testClassId );            // testClassId
	      q.addBindValue( highest_classId );        // highestResultClassId
	      q.addBindValue( highest_classProb  );     // highestResultClassProb
	      q.addBindValue( otherResultsStr );        // otherResults
	      q.exec();
	      // TODO: perform checks!
	    }
      
	  /* In order to keep the UI responsive and properly updated during the tests,
	   * we have Qt process any pending events after processing blocks of 10 gestures.
	   *
	   * Note: It is important to processEvents() even if the integer version of the
	   *       progress hasn't increased due to not only the UI responsiveness, but
	   *       also the progress animation, which becomes important to the user when
	   *       they can't see bar increases.
	   */
	  ui.progressBar->setValue( (int)((float)gi*iteratorToProgress) );
	  if ( gi % 10 == 0 )
	    QCoreApplication::processEvents(); // don't call with time - it seems to use all of it!

	  // Commit the changes to disk in blocks of 100 gestures.
	  if ( gi % 100 == 0 ) {
	      db.commit();      // COMMIT current transaction
	      db.transaction(); // BEGIN new transaction
	    }
	}
      db.commit();      // COMMIT current transaction

      // Call at end to update progress bar, as total gestures may not be a multiple
      // of 10 and there could easily be error in the progress calculation.
      ui.progressBar->setValue( ui.progressBar->maximum() );
      QCoreApplication::processEvents(); // don't call with time - it seems to use all of it!

      // Update the experiments' cachedCorrectText and cachedIncorrectText
      QHashIterator<int, int> it( m_recogToExper );
      int totalGestures = m_gestureIds.size();
      while ( it.hasNext() )
	{
	  it.next();
	  DExperimentRecord rec = digestDbModel()->fetchExperiment( it.value() );
	  
	  int correct = recogToCorrect.value( it.key() );
	  rec.cachedCorrectText =
	    QString("%1\% (%2)")
	    .arg((float)correct/(float)totalGestures*100.0, 0, 'f', 2)
	    .arg(correct);
	  rec.cachedIncorrectText =
	    QString("%1\% (%2)")
	    .arg((float)(totalGestures-correct)/(float)totalGestures*100.0, 0, 'f', 2)
	    .arg(totalGestures-correct);
	  postControllerEvent( new CExperimentUpdateEvent(rec, this) );
	}

      gotoStep( Finished );
    }
}



void ExperimentAssistant::onCategoryModelDataChanged()
{
  Q_ASSERT( m_categoryModel );
  if ( m_gestureModel ) { // testing proved that we MUST check this
    m_gestureQuery = m_categoryModel->gestureQueryForCheckedIndexes();
    m_gestureModel->setQuery( m_gestureQuery );

    // Fetch ALL gestures - otherwise there are many issues to handle.
    while ( m_gestureModel->canFetchMore() )
      m_gestureModel->fetchMore();
  }
}


QString ExperimentAssistant::detailsAsHtml() const
{
  QString str
    = "<html>"
    "<body>"
    "<table cellspacing=\"6\">"
    "<tr><th align=\"right\"><font color=\"#888888\">Label</font></th><td>"
    + (m_labelStr.isEmpty() ? tr("(none)") : m_labelStr) + "</td></tr>"
    "<tr><th align=\"right\"><font color=\"#888888\">Notes</font></th><td>"
    + (m_notesStr.isEmpty() ? tr("(none)") : m_notesStr) + "</td></tr>"
    "<tr><th align=\"right\"><font color=\"#888888\">Recogniser Set</font>"
    "</th><td>";

  if ( m_recogniserIds.isEmpty() )
    str += tr("(none)");
  else {
    str += QString::number(m_recogniserIds.count()) + tr(" Recognisers - ");
    QStringList recogniserStrs;
    foreach ( int id, m_recogniserIds )
      recogniserStrs += QString::number(id); // TODO: look-up the recogniser's label!
    str += recogniserStrs.join( ", " );
  }

  str
    += "</td></tr>"
    "<tr><th align=\"right\"><font color=\"#888888\">Test Set</font></th><td>";

  if ( m_gestureIds.isEmpty() )
    str += tr("(none)");
  else {
    str += QString::number(m_gestureIds.count()) + tr(" Gestures - ");
    QStringList geatureStrs;
    QListIterator<int> it( m_gestureIds );
    while ( it.hasNext() )
      geatureStrs += QString::number(it.next());
    str += geatureStrs.join( ", " );
  }

  str
    += "</td></tr>"
    "</body>"
    "</html>";

  return str;
}
