/*  -*- c++ -*-  (for Emacs)
 *
 *  experimentbrowser.cpp
 *  Digest
 * 
 *  Created by Aidan Lane on Thu Nov 3 2005.
 *  Copyright (c) 2005-2006 Optimisation and Constraint Solving Group,
 *  Monash University. All rights reserved.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#include "experimentbrowser.h"

#include <QItemSelectionModel>
#include <QSqlRecord>
#include <QSqlQueryModel>
#include <QStandardItemModel>

#include "MvcDigestDb/digestdbmodel.h"
#include "MvcDigestDb/digestdbcontroller.h"
#include "MvcSettings/settingscontroller.h"
#include "MvcSettings/settingsmodel.h"

#include "digestsettings.h"


/*!
 * \class ExperimentBrowser
 *
 * \brief The ExperimentBrowser class provides a graphical interface for browsing
 *        through, adding and removing experiment records.
 */

ExperimentBrowser::ExperimentBrowser( DigestDbController* digestDbController,
				      SettingsController* settingsController,
				      JavaVM* jvm,
				      QWidget* parent, Qt::WindowFlags flags )
  : GuiDbComponentDialog(digestDbController, parent, flags),
    m_settingsController(settingsController),
    m_jvm(jvm),
    m_experimentsModel(0),
    m_selectionModel(0),
    m_resultsModel(0),
    m_dummyModel(0)
{
  /*
   * WARNING: Init of the query models is delayed until we receive VEvent::Reset.
   */
  m_dummyModel = new QStandardItemModel( this );
  m_ui.setupUi( this );

  // Set UI default state:
  Q_ASSERT( m_ui.removeButton );
  Q_ASSERT( m_ui.resultsTabWidget );
  m_ui.removeButton->setEnabled( false );
  m_ui.resultsTabWidget->setEnabled( false );

  if ( m_settingsController ) {
    m_settingsController->bind( m_ui.filterComboBox, "currentIndex",
				SIGNAL(currentIndexChanged(int)),
				DigestSettings::exprBrowserResultsFilterKey );
    m_settingsController->bind( m_ui.testGestureCheckBox, "checked",
				SIGNAL(toggled(bool)),
				DigestSettings::exprBrowserShowTestGestureKey );
    m_settingsController->bind( m_ui.testClassCheckBox, "checked",
				SIGNAL(toggled(bool)),
				DigestSettings::exprBrowserShowTestClassKey );
    m_settingsController->bind( m_ui.resultClassCheckBox, "checked",
				SIGNAL(toggled(bool)),
				DigestSettings::exprBrowserShowResultClassKey );
    m_settingsController->bind( m_ui.resultProbCheckBox, "checked",
				SIGNAL(toggled(bool)),
				DigestSettings::exprBrowserShowResultProbKey );
    m_settingsController->bind( m_ui.exprIdCheckBox, "checked",
				SIGNAL(toggled(bool)),
				DigestSettings::exprBrowserShowExprIdKey );
    m_settingsController->bind( m_ui.exprLabelCheckBox, "checked",
				SIGNAL(toggled(bool)),
				DigestSettings::exprBrowserShowExprLabelKey );
    m_settingsController->bind( m_ui.exprDateCheckBox, "checked",
				SIGNAL(toggled(bool)),
				DigestSettings::exprBrowserShowExprDateKey );
    m_settingsController->bind( m_ui.exprOthersCheckBox, "checked",
				SIGNAL(toggled(bool)),
				DigestSettings::exprBrowserShowExprOthersKey );
    m_settingsController->bind( m_ui.exprRecogCheckBox, "checked",
				SIGNAL(toggled(bool)),
				DigestSettings::exprBrowserShowExprRecogKey );
    m_settingsController->bind( m_ui.genIsCorrectCheckBox, "checked",
				SIGNAL(toggled(bool)),
				DigestSettings::exprBrowserShowGenIsCorrectKey );
    if ( m_settingsController->settingsModel() ) {
      Q_ASSERT( m_ui.topSplitter );
      Q_ASSERT( m_ui.detailsSplitter );
      SettingsModel* m = m_settingsController->settingsModel();
      m_ui.topSplitter->restoreState
	( m->value(DigestSettings::exprBrowserTopSplitterStateKey).toByteArray() );
      m_ui.detailsSplitter->restoreState
	( m->value(DigestSettings::exprBrowserDetailsSplitterStateKey).toByteArray() );
    }
  }

  /* Important:
   * These connections must be made AFTER the they are bound to settings,
   * otherwise, the bind() calls may trigger them, causing updateResults() to be
   * called before resetEvent(), and thus also before the models are setup.
   */
  connect( m_ui.filterComboBox, SIGNAL(currentIndexChanged(int)), SLOT(updateResults()) );
  connect( m_ui.testGestureCheckBox,  SIGNAL(toggled(bool)), SLOT(updateResults()) );
  connect( m_ui.testClassCheckBox,    SIGNAL(toggled(bool)), SLOT(updateResults()) );
  connect( m_ui.resultClassCheckBox,  SIGNAL(toggled(bool)), SLOT(updateResults()) );
  connect( m_ui.resultProbCheckBox,   SIGNAL(toggled(bool)), SLOT(updateResults()) );
  connect( m_ui.exprIdCheckBox,       SIGNAL(toggled(bool)), SLOT(updateResults()) );
  connect( m_ui.exprLabelCheckBox,    SIGNAL(toggled(bool)), SLOT(updateResults()) );
  connect( m_ui.exprDateCheckBox,     SIGNAL(toggled(bool)), SLOT(updateResults()) );
  connect( m_ui.exprOthersCheckBox,   SIGNAL(toggled(bool)), SLOT(updateResults()) );
  connect( m_ui.exprRecogCheckBox,    SIGNAL(toggled(bool)), SLOT(updateResults()) );
  connect( m_ui.genIsCorrectCheckBox, SIGNAL(toggled(bool)), SLOT(updateResults()) );
}


ExperimentBrowser::~ExperimentBrowser()
{
  if ( m_settingsController ) {
    Q_ASSERT( m_ui.topSplitter );
    Q_ASSERT( m_ui.detailsSplitter );
    QApplication::postEvent( m_settingsController,
			     new CSettingsChangeValueEvent
			     (DigestSettings::exprBrowserTopSplitterStateKey,
			      m_ui.topSplitter->saveState(), this) );
    QApplication::postEvent( m_settingsController,
			     new CSettingsChangeValueEvent
			     (DigestSettings::exprBrowserDetailsSplitterStateKey,
			      m_ui.detailsSplitter->saveState(), this) );
  }
}


/*!
 * Re-syncs the GUI with the database if digestDbModel() is non-null, as it may
 * have changed for some unknown reason without sending update events to us.
 *
 * Users tend to expect a refresh if a window is hidden and then shown again.
 *
 * It then then passes event onto GuiDbComponentDialog.
 */
void ExperimentBrowser::showEvent( QShowEvent* event )
{
  if ( digestDbModel() ) refreshExperimentsModel();
  GuiDbComponentDialog::showEvent( event );
}


void ExperimentBrowser::resetEvent( VEvent* )
{
  Q_ASSERT( m_ui.treeView );
  Q_ASSERT( m_ui.resultsTableView );

  /*
   * Clean-up
   */
  //m_ui.treeView->setSelectionModel( 0 );
  m_ui.treeView->setModel( m_dummyModel ); // can't be null :-(
  m_ui.treeView->reset();
  delete m_selectionModel;
  delete m_experimentsModel;
  m_selectionModel = 0;
  m_experimentsModel = 0;

  m_ui.resultsTableView->setModel( m_dummyModel ); // can't be null :-(
  m_ui.resultsTableView->reset();
  delete m_resultsModel;
  m_resultsModel = 0;


  /*
   * (Re)build
   */
  if ( digestDbModel() )
    {
      m_experimentsModel = new QSqlQueryModel( this );
      m_selectionModel = new QItemSelectionModel( m_experimentsModel );
      connect( m_selectionModel,
	       SIGNAL(selectionChanged(const QItemSelection&, const QItemSelection&)),
	       SLOT(onSelectionChanged()) );
      m_ui.treeView->setModel( m_experimentsModel );
      m_ui.treeView->setSelectionModel( m_selectionModel );

      m_resultsModel = new QSqlQueryModel( this );
      m_ui.resultsTableView->setModel( m_resultsModel );

      /* The following method calls onSelectionChanged(), which itself calls:
       *  - updateDetailsText()
       *  - updateResults()
       */
      refreshExperimentsModel();
    }
}



void ExperimentBrowser::experimentsEvent( VExperimentsEvent* event )
{
  Q_ASSERT( event );

  /* Ensure that we're in sync with the database.
   * Ignore ExperimentsAboutToBeRemoved, as refreshExperimentsModel() only needs to be
   * called when the database has actually changed.
   * Hence, we wait for ExperimentsRemoved.
   */
  if ( (VDigestDbEvent::Type)event->type()
       != VDigestDbEvent::ExperimentsAboutToBeRemoved )
    refreshExperimentsModel();
}


void ExperimentBrowser::refreshExperimentsModel()
{
  if ( digestDbModel() ) // should check this - it may not be ready yet
    {
      // Important: Other methods rely on 0=ID and 1=label
      Q_ASSERT( m_experimentsModel );
      m_experimentsModel->setQuery( "SELECT id, label, date, "
				    "cachedCorrectText, "
				    "cachedIncorrectText "
				    "FROM Experiment" );
      
      m_experimentsModel->setHeaderData( 0, Qt::Horizontal, tr("ID") );
      m_experimentsModel->setHeaderData( 1, Qt::Horizontal, tr("Label") );
      m_experimentsModel->setHeaderData( 2, Qt::Horizontal, tr("Date") );
      m_experimentsModel->setHeaderData( 3, Qt::Horizontal, tr("Correct") );
      m_experimentsModel->setHeaderData( 4, Qt::Horizontal, tr("Incorrect") );
      
      // Note: We get strange behaviour if we don't perform the following
      Q_ASSERT( m_ui.treeView );
      m_ui.treeView->setSelectionModel( m_selectionModel );
      
      onSelectionChanged(); // update selection dependants (e.g. info & results)
    }
}


void ExperimentBrowser::on_newExperimentButton_clicked()
{ emit request( "new ExperimentAssistant" ); }


void ExperimentBrowser::on_removeButton_clicked()
{
  // TODO: ask the user if they are REALLY sure - because it may take some
  // time to recreate an experiment (with results)!

  // Delete the selected experiments
  Q_ASSERT( m_selectionModel );
  Q_ASSERT( m_experimentsModel );
  IdSet idSet;
  foreach ( const QModelIndex& index, m_selectionModel->selectedIndexes() ) {
    bool intOk = false;
    if ( index.column() ) continue; // only process 1st column of each row
    idSet += m_experimentsModel->record(index.row()).value(0).toInt(&intOk);
    Q_ASSERT( intOk );
  }
  if ( !idSet.isEmpty() )
    postControllerEvent( new CExperimentsRemoveEvent(idSet, this) );
}


void ExperimentBrowser::on_showGesturesButton_clicked()
{
  // Note: We can't just iterate through the m_resultsModel, as the gestureId
  //       field may not be shown / have been fetched.
  // TODO: use optimised code if the gestureId has been fetched for m_resultsModel ?

  // TODO: factorize code with updateResults()
  // TODO: cleanup!
  // TODO: set connection to "local"

  Q_ASSERT( digestDbModel() );
  Q_ASSERT( m_selectionModel );
  Q_ASSERT( m_experimentsModel );
  Q_ASSERT( m_ui.filterComboBox );

  QModelIndexList indexes = m_selectionModel->selectedIndexes();

  if ( ! indexes.isEmpty() )
    {
      QString qStr;
      QString message = "show GestureBrowser searchtype=\"sqlwhere\" : search=\"ID in (";
      QString idsStr;
      QString filter;
      bool showAll = false;

      foreach ( const QModelIndex& idx, indexes )
	if ( idx.column() == 0 ) // only consider each row once - when column == 0
	  idsStr += m_experimentsModel->record(idx.row()).value(0).toString() + ",";
      idsStr.chop( 1 ); // chop last comma - safe given we know there's >= 1 index

      // TODO: handle case where there are multiple results for a test
      //       and multiple classes for a gesture.
      switch ( m_ui.filterComboBox->currentIndex() ) {
      case 0: // All
	showAll = true;
	break;
      case 1: // Correct
	filter = "AND A1.testClassId=A1.highestResultClassId";
	break;
      case 2: // Incorrect
	filter = "AND NOT A1.testClassId=A1.highestResultClassId";
	break;
      }

      qStr = "SELECT A1.testGestureId FROM ExperimentResult A1 ";

      if (showAll)
	qStr += QString("WHERE experimentId IN (%1) ")
	  .arg(idsStr);
      else
	qStr += // Get the max highestResultClassProb for each
	  // experimentId vs. testGestureId
	  QString( ", "
		   "(SELECT experimentId, testGestureId, "
		   " max(highestResultClassProb) AS maxResultClassProb "
		   " FROM ExperimentResult "
		   " WHERE experimentId IN (%1) " // don't use filter here - we want the real maximums
		   " GROUP BY testGestureId) A3 "
		   "WHERE A1.experimentId = A3.experimentId "
		   "AND A1.testGestureId = A3.testGestureId "
		   "AND A1.highestResultClassProb = A3.maxResultClassProb ")
	  .arg(idsStr);

      qStr += filter;

      // TODO: error checks
      QSqlQuery q( qStr );
      while ( q.next() )
	message += q.value(0).toString() + ",";
      if ( message.endsWith(",") ) message.chop(1);
      message += ")\"";

      emit request( message );
    }
}


void ExperimentBrowser::onSelectionChanged()
{
  Q_ASSERT( m_selectionModel );
  Q_ASSERT( m_ui.removeButton );
  Q_ASSERT( m_ui.resultsTabWidget );
  bool s = ! m_selectionModel->selectedIndexes().isEmpty();
  m_ui.removeButton->setEnabled( s );
  m_ui.resultsTabWidget->setEnabled( s );
  updateDetailsText();
  updateResults();
}


void ExperimentBrowser::updateDetailsText()
{
  Q_ASSERT( digestDbModel() );
  Q_ASSERT( m_selectionModel );
  Q_ASSERT( m_experimentsModel );
  Q_ASSERT( m_ui.detailsHtmlWidget );

  QString detailsText;
  bool intOk = false;

  // TODO: if multiple items are selected, then print "multiple selection" info
  if ( ! m_selectionModel->selectedIndexes().isEmpty() ) {
    int id = m_experimentsModel
      ->record(m_selectionModel->selectedIndexes().first().row()).value(0).toInt(&intOk);
    Q_ASSERT( intOk );
    DExperimentRecord record = digestDbModel()->fetchExperiment( id );
    detailsText = experimentRecordAsHtml( record );
  }
  else {
    detailsText =
      "<html><body><table width=\"100%\">"
      "<tr><td align=\"center\"><font color=\"#AAAAAA\" size=\"+1\"><b>"
      "<br><br>Nothing Selected"
      "</b></font></td></tr>"
      "</table></body></html>";
  }

  m_ui.detailsHtmlWidget->setHtml( detailsText );
}


void ExperimentBrowser::updateResults()
{
  // TODO: result classes as columns
  // TODO: add recogniser fields - label, date, etc...
  // TODO: handle case where there are multiple classes for a gesture.
  // TODO: optimise this!

  Q_ASSERT( digestDbModel() );
  Q_ASSERT( m_selectionModel );
  Q_ASSERT( m_experimentsModel );
  Q_ASSERT( m_resultsModel );
  Q_ASSERT( m_ui.filterComboBox );
 
  QString qStr;
  QModelIndexList indexes = m_selectionModel->selectedIndexes();

  if ( ! indexes.isEmpty() )
    {
      QString r_fields;  // experiment result fields
      QString e_fields;  // experiment-wide info fields
      QString idsStr;
      QString filter;

      // If experimentId is to be shown, then it will (intentionally) always be first

      if ( m_ui.exprIdCheckBox->isChecked() )      e_fields += "id,";
      if ( m_ui.exprLabelCheckBox->isChecked() )   e_fields += "label AS experimentLabel,";
      if ( m_ui.exprDateCheckBox->isChecked() )    e_fields += "date AS experimentDate,";
      if ( m_ui.exprOthersCheckBox->isChecked() )  e_fields += ("external AS "
								"experimentExternal,"
								"notes AS "
								"experimentNotes,");
      if ( m_ui.exprRecogCheckBox->isChecked() )   e_fields += "trainedRecogniserId,";
      if ( e_fields.endsWith(",") ) e_fields.chop(1);

      if ( m_ui.testGestureCheckBox->isChecked() )  r_fields += "testGestureId,";
      if ( m_ui.testClassCheckBox->isChecked() )    r_fields += "testClassId,";
      if ( m_ui.resultClassCheckBox->isChecked() )  r_fields += "highestResultClassId,";
      if ( m_ui.resultProbCheckBox->isChecked() )   r_fields += "highestResultClassProb,";
      if ( m_ui.genIsCorrectCheckBox->isChecked() ) r_fields += ("testClassId="
								 "highestResultClassId AS "
								 "isCorrect,");
      if ( r_fields.endsWith(",") ) r_fields.chop(1);

      foreach ( const QModelIndex& idx, indexes )
	if ( idx.column() == 0 ) // only consider each row once - when column == 0
	  idsStr += m_experimentsModel->record(idx.row()).value(0).toString() + ",";
      idsStr.chop( 1 ); // chop last comma - safe given we know there's >= 1 index

      switch ( m_ui.filterComboBox->currentIndex() ) {
      case 0: // All
	break;
      case 1: // Correct
	filter = "AND testClassId=highestResultClassId";
	break;
      case 2: // Incorrect
	filter = "AND NOT testClassId=highestResultClassId";
	break;
      }

      if ( e_fields.isEmpty() )
	qStr = QString( "SELECT %1 FROM ExperimentResult "
			"WHERE experimentId IN (%2) "
			"%3" )
	  .arg(r_fields)
	  .arg(idsStr)
	  .arg(filter);
      else
	qStr = QString( "SELECT %1%2 FROM ExperimentResult, Experiment "
			"WHERE experimentId IN (%3) "
			"AND experimentId = id "
			"%4" )
	  .arg(e_fields)
	  .arg(r_fields.isEmpty() ? "" : ","+r_fields)
	  .arg(idsStr)
	  .arg(filter);
  }

  m_resultsModel->setQuery( qStr );
}


QString ExperimentBrowser::experimentRecordAsHtml( const DExperimentRecord& record ) const
{
  return
    "<html>"
    "<body>"
    "<table cellspacing=\"6\">"
    "<tr><th align=\"right\"><font color=\"#888888\">ID</font></th><td>"
    + QString::number(record.id) + "</td></tr>"
    "<tr><th align=\"right\"><font color=\"#888888\">Label</font></th><td>"
    + (record.label.isEmpty() ? tr("(none)") : record.label) + "</td></tr>"
    "<tr><th align=\"right\"><font color=\"#888888\">External</font></th><td>"
    + (record.external ? tr("Yes") : tr("No")) + "</td></tr>"
    "<tr><th align=\"right\"><font color=\"#888888\">Date</font></th><td>"
    + (record.date.isValid() ? record.date.toString(Qt::LocalDate) : tr("(not set)")) +
    "</td></tr>"
    "<tr><th align=\"right\"><font color=\"#888888\">Notes</font></th><td>"
    + (record.notes.isEmpty() ? tr("(none)") : record.notes) + "</td></tr>"
    "<tr><th align=\"right\"><font color=\"#888888\">Trained Recogniser</font></th><td>"
    + QString::number(record.trainedRecogniserId) + "</td></tr>"
    "<tr><th align=\"right\"><font color=\"#888888\">Correct</font></th><td>"
    + (record.cachedCorrectText.isEmpty()
       ? tr("(unknown)")
       : record.cachedCorrectText) + "</td></tr>"
    "<tr><th align=\"right\"><font color=\"#888888\">Incorrect</font></th><td>"
    + (record.cachedIncorrectText.isEmpty()
       ? tr("(unknown)")
       : record.cachedIncorrectText) + "</td></tr>"
    "</table>"
    "</body>"
    "</html>";
}
