/*  -*- c++ -*-  (for Emacs)
 *
 *  strokelist.cpp
 *  Digest
 * 
 *  Created by Aidan Lane on Tue Jul 05 2005.
 *  Copyright (c) 2005 Optimisation and Constraint Solving Group,
 *  Monash University. All rights reserved.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#include "strokelist.h"

#include <QStringList>

// Separators
#define VALUE_SEP   ","
#define VEC_SEP     ";"
#define STROKE_SEP  "$"


static bool metaTypeUnregistered = true;


StrokeList::StrokeList()
{
  init();
}

StrokeList::StrokeList( const StrokeList& other )
  : QList<Stroke>(other)
{
  init();
}

StrokeList::StrokeList( const QList<Stroke>& other )
  : QList<Stroke>(other)
{
  init();
}


void StrokeList::init()
{
  if ( metaTypeUnregistered ) {
    metaTypeUnregistered = true;
    qRegisterMetaType<StrokeList>("StrokeList");
  }
}


/*!
 * Returns the strokes as a QString.
 *
 * This method is required to store the strokes in a record of the Gesture table
 * (in a QSqlDatabase). Hence, the QString returned is be compact while also
 * easy enough for fromString() to parse (or by some external parser).
 *
 * The format of the string also allows for extensibility, as each field is
 * explicitly stated as a "tag" (like XML element attributes).
 * However, as compactness is important for storage, the fields values are not
 * grouped by StrokePoint. Instead they are stored as a vector for each field
 * type for each stroke. This format makes it easy to fetch data from specific
 * fields without having to parse all of the data (e.g. parse only "x" and "y",
 * not "pressure").
 *
 * Also of note: field values, field vectors and strokes are separated using
 * single characters, not brackets (they being ",", ";" and "$", respectively).
 * The decision to not use bracketing wasn't driven by compactness (although
 * that is a nice side-effect), but instead by the ease of parsing, given the
 * availability of a "split-on" function.
 *
 * Lastly, a textual string has been used instead of binary data, as this code
 * is designed to be cross platform, where the database could reside on a machine
 * of any architecture, communicating with a client of also any architecture.
 * It also makes the data (mildly) human-readable.
 *
 * Example of 3 strokes, with 12 points in total:
 * \code
 *   "x=0,1,2,1,2;y=3,1,2,4,5;pressure=.1,.5,1,.4,.01
 *    |x=2,4,1;y=3,0,2;pressure=.2,.9,.1
 *    |x=7,3,5,2;y=8,7,6,4;pressure=.4,.9,.9,.3"
 * \endcode
 *
 */
QString StrokeList::toString() const
{
  QString str;

  /* Note: We add add separators unconditionally in the loops, so that we don't
   *       have to continually evaluate a condition that will return the same
   *       result until the very last element. Thus, following the loops we
   *       perform an unconditional removal of the last separator, as to keep
   *       the final string as concise as possible.
   */
  foreach ( const Stroke& stroke, *this )
    {
      // Ensure that there is 1+ point -> required for the unconditional removal
      // of the last separator.
      if ( stroke.isEmpty() ) continue;
	
      QString xStr = "x=";
      QString yStr = "y=";
      QString pStr = "pressure=";
      QString tStr = "milliTime="; // note: it's important to state the time unit
      foreach ( const StrokePoint& pt, stroke ) {
	xStr += QString::number(pt.x()) + VALUE_SEP;
	yStr += QString::number(pt.y()) + VALUE_SEP;
	pStr += QString::number(pt.pressure) + VALUE_SEP;
	tStr += QString::number(pt.milliTime) + VALUE_SEP;
      }
      xStr.chop( 1 ); // remove the last separator
      yStr.chop( 1 ); // ditto.
      pStr.chop( 1 ); // ditto.
      tStr.chop( 1 ); // ditto.
      str += xStr + VEC_SEP + yStr + VEC_SEP + pStr + VEC_SEP + tStr;
	
      str += STROKE_SEP;
    }
  if ( ! str.isEmpty() )
    str.chop( 1 ); // remove the last separator
  
  return str;
}


/*!
 * Returns a StrokeList object with the data contained in the given string.
 * The order of the elements in the StrokeList is the same as in string.
 *
 * This method is required to load strokes from a record of the Gesture table
 * (in a QSqlDatabase). This method parses data that has been generated using
 * toString().
 */
StrokeList StrokeList::fromString( const QString& str )
{
  StrokeList strokeList;

  if ( str.isEmpty() ) return StrokeList();

  QStringList strokeListStr = str.split( STROKE_SEP );
  foreach ( const QString& strokeStr, strokeListStr )
    {
      if ( strokeStr.isEmpty() ) continue;

      Stroke stroke;
      foreach ( const QString& vecStr,
		strokeStr.split(VEC_SEP, QString::SkipEmptyParts) )
	{
	  if ( vecStr.isEmpty() ) continue;

	  QStringList fieldAndValues = vecStr.split( "=" );
	  //TODO: as this is imput data, alert user if fieldAndValues.size != 2
	  Q_ASSERT( fieldAndValues.size() == 2 ); // TODO: REMOVE DANGEROUS ASSERT
	  const QString& field = fieldAndValues.at(0);
	  QStringList values = fieldAndValues.at(1).split( VALUE_SEP,
							   QString::SkipEmptyParts );
	  bool convOk = false;

	  // TODO: optimise and clean this up
	  // TODO: factorize code!
	  // TODO DOC: a bit of checking, but don't need there's no "double-handling" of data
	  if ( field == "x" )
	    for ( int i=0; i < values.size(); ++i ) {
	      if ( i >= stroke.size() ) stroke += StrokePoint();
	      stroke[i].rx() = values.at(i).toFloat(&convOk);
	      //TODO: as this is input data, alert user if toFloat not OK - ask if -> 0 ok
	      Q_ASSERT( convOk ); // TODO: REMOVE DANGEROUS ASSERT
	    }
	  else if ( field == "y" )
	    for ( int i=0; i < values.size(); ++i ) {
	      if ( i >= stroke.size() ) stroke += StrokePoint();
	      stroke[i].ry() = values.at(i).toFloat(&convOk);
	      //TODO: as this is input data, alert user if toFloat not OK - ask if -> 0 ok
	      Q_ASSERT( convOk ); // TODO: REMOVE DANGEROUS ASSERT
	    }
	  else if ( field == "pressure" )
	    for ( int i=0; i < values.size(); ++i ) {
	      if ( i >= stroke.size() ) stroke += StrokePoint();
	      stroke[i].pressure = values.at(i).toFloat(&convOk);
	      //TODO: as this is input data, alert user if toFloat not OK - ask if -> 0 ok
	      Q_ASSERT( convOk ); // TODO: REMOVE DANGEROUS ASSERT
	    }
	  else if ( field == "milliTime" )
	    for ( int i=0; i < values.size(); ++i ) {
	      if ( i >= stroke.size() ) stroke += StrokePoint();
	      stroke[i].milliTime = values.at(i).toUInt(&convOk);
	      //TODO: as this is input data, alert user if toFloat not OK - ask if -> 0 ok
	      Q_ASSERT( convOk ); // TODO: REMOVE DANGEROUS ASSERT
	    } 
	}

      strokeList += stroke;
    }

  return strokeList;
}
