/*  -*- c++ -*-  (for Emacs)
 *
 *  strokelist.h
 *  Digest
 * 
 *  Created by Aidan Lane on Thu Jul 7 2005.
 *  Copyright (c) 2005 Optimisation and Constraint Solving Group,
 *  Monash University. All rights reserved.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#ifndef STROKELIST_H
#define STROKELIST_H


#include <QList>
#include <QMetaType>
#include <QPointF>
#include <QString>


typedef QPointF   StrokePointBase;
typedef float     StrokeCoordT;
typedef float     StrokePressureT;
typedef uint32_t  StrokeTimeT;


/* WARNING! Using floats for x and y may be overkill, but it provides a lot of
 *          flexibility. However the strokes editor should provide an option to
 *          record rounded or truncated values, which when coded as a string
 *          (using QString::number()), will use the same # of chars as an integer.
 */
class StrokePoint : public StrokePointBase {
public:
  StrokePoint( StrokeCoordT x = 0.0,
	       StrokeCoordT y = 0.0,
	       StrokePressureT pressure = 0.0,
	       StrokeTimeT milliTime = 0 )
    : StrokePointBase(x,y),
      pressure(pressure),
      milliTime(milliTime) {}
  StrokePoint( const StrokePointBase& pt,
	       StrokePressureT pressure = 0.0,
	       StrokeTimeT milliTime = 0 )
    : StrokePointBase(pt),
      pressure(pressure),
      milliTime(milliTime) {}
  StrokePressureT pressure;  /*! 0.0-1.0, which allows for higher
			       or lower precise tablets. */
  StrokeTimeT     milliTime; /*! OFFSET from the time when the STROKE began. */
#if 0
  float rotation;
  float xTilt;
  float yTilt;
#endif
};



class Stroke : public QList<StrokePoint> {

public:
  inline void translate( StrokeCoordT dx, StrokeCoordT dy );
  inline void translate( const StrokePointBase& p );
  inline Stroke translated( StrokeCoordT dx, StrokeCoordT dy ) const;
  inline Stroke translated( const StrokePointBase& p ) const;
};

inline void Stroke::translate( StrokeCoordT dx, StrokeCoordT dy )
{ translate( StrokePointBase(dx, dy) ); }

inline void Stroke::translate( const StrokePointBase& p ) {
  QMutableListIterator<StrokePoint> it( *this );
  while ( it.hasNext() ) it.next() += p;
}

inline Stroke Stroke::translated( StrokeCoordT dx, StrokeCoordT dy ) const
{ return translated( StrokePointBase(dx, dy) ); }

inline Stroke Stroke::translated( const StrokePointBase& p ) const {
  Stroke t;
  QListIterator<StrokePoint> it( *this );
  while ( it.hasNext() ) t.append( it.next() + p );
  return t;
}



class StrokeList : public QList<Stroke> {

public:
  StrokeList();
  StrokeList( const StrokeList& other );
  StrokeList( const QList<Stroke>& other );

  inline void translate( StrokeCoordT dx, StrokeCoordT dy );
  inline void translate( const StrokePointBase& p );
  inline StrokeList translated( StrokeCoordT dx, StrokeCoordT dy ) const;
  inline StrokeList translated( const StrokePointBase& p ) const;

  QString toString() const;

  static StrokeList fromString( const QString& str );

private:
  void init();
};
Q_DECLARE_METATYPE( StrokeList );


inline void StrokeList::translate( StrokeCoordT dx, StrokeCoordT dy )
{ translate( StrokePointBase(dx, dy) ); }

inline void StrokeList::translate( const StrokePointBase& p ) {
  QMutableListIterator<Stroke> it( *this );
  while ( it.hasNext() ) it.next().translate(p);
}

inline StrokeList StrokeList::translated( StrokeCoordT dx, StrokeCoordT dy ) const
{ return translated( StrokePointBase(dx, dy) ); }

inline StrokeList StrokeList::translated( const StrokePointBase& p ) const {
  StrokeList t;
  QListIterator<Stroke> it( *this );
  while ( it.hasNext() ) t.append( it.next().translated(p) );
  return t;
}


#endif  // ! STROKELIST_H
