/*  -*- c++ -*-  (for Emacs)
 *
 *  gesturebrowser.cpp
 *  Digest
 * 
 *  Created by Aidan Lane on Tue Jun 21 2005.
 *  Copyright (c) 2005-2006 Optimisation and Constraint Solving Group,
 *  Monash University. All rights reserved.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#include "gesturebrowser.h"

#include <QAction>
#include <QHBoxLayout>
#include <QHeaderView>
#include <QInputDialog>
#include <QItemSelectionModel>
#include <QMenu>
#include <QMessageBox>
#include <QRegExp>
#include <QSqlRecord>
#include <QStandardItemModel>

#ifdef Q_WS_MAC
#include <QMacStyle>
#endif

#include "DigestGuiCore/categoryitemmodel.h"
#include "DigestGuiCore/gesturequerymodel.h"
#include "MvcDigestDb/digestdbmodel.h"
#include "MvcDigestDb/digestdbcontroller.h"
#include "MvcSettings/settingscontroller.h"
#include "MvcSettings/settingsmodel.h"
#include "WidgetPack/searchwidget.h"

#include "digestsettings.h"
#include "splitcollectiondialog.h"


// TODO: check which symbol is best for which platform (other than Mac OS X)
#ifdef Q_WS_MAC // _IS_ Mac
#define ELLIPSIS_STR  QChar(0x2026)  // the single "..." character - used on Mac OS X
#else
#define ELLIPSIS_STR  "..."
#endif

// TODO: move the following to digestdatabase.h
#define COLLECTION_TABLE_STR            "Collection"
#define COLLECTION_TABLE_ID_STR         "id"
#define COLLECTION_TABLE_ID_INDEX       0
#define COLLECTION_TABLE_LABEL_INDEX    1
// TODO: move the following to digestdatabase.h
// TODO: use the more flexible STRING indices!
#define GESTURE_TABLE_STR               "Gesture"
#define GESTURE_TABLE_ID_STR            "id"
#define GESTURE_TABLE_ID_INDEX          0
#define GESTURE_TABLE_LABEL_INDEX       1
#define GESTURE_TABLE_DATE_INDEX        2
#define GESTURE_TABLE_NOTES_INDEX       3
#define GESTURE_TABLE_NUMSTROKES_INDEX  4
#define GESTURE_TABLE_STROKES_INDEX     5

#define CLASSMAP_TABLE_STR            "ClassMap"
#define CLASSMAP_TABLE_GESTUREID_STR  "gestureId"
#define CLASSMAP_TABLE_CLASSID_STR    "classId"

#define COLLECTIONMAP_TABLE_STR              "CollectionMap"
#define COLLECTIONMAP_TABLE_GESTUREID_STR    "gestureId"
#define COLLECTIONMAP_TABLE_COLLECTIONID_STR "collectionId"

#define STACKED_FIELD_WIDGET_VIEW_INDEX  0
#define STACKED_FIELD_WIDGET_EDIT_INDEX  1


// TODO: remove me:
#ifdef Q_WS_MAC
#include <QMacStyle>
#include <QSplitter>
/*!
 * Draws splitters as thin 1pix QPalette::dark() lines.
 */
class MacStyle2 : public QMacStyle {
public:
  virtual void drawControl( ControlElement element, const QStyleOption * option, QPainter * painter, const QWidget * widget = 0 ) const
  {
    Q_ASSERT( painter && widget );
    bool drawStandard = true;

    if ( element == QStyle::CE_Splitter ) {
      const QSplitter* s = qobject_cast<const QSplitter*>( widget );
      if ( s )
	{
	  painter->setPen( QPen(s->palette().dark().color()) );
	  if ( s->orientation() == Qt::Horizontal )
	    painter->drawLine( 0, 0, 0, s->height() );
	  else
	    painter->drawLine( 0, 0, s->width(), 0 );
	  drawStandard = false;
	}
    }

    if ( drawStandard )
      QMacStyle::drawControl( element, option, painter, widget );
  }
  virtual int pixelMetric( PixelMetric metric, const QStyleOption * option = 0, const QWidget * widget = 0 ) const
  {
    Q_ASSERT( widget );
    if ( metric == QStyle::PM_SplitterWidth )
        return 1;
    else
    return QMacStyle::pixelMetric( metric, option, widget );
  }
};
#endif


// Utility method for sorting QAction pointers alphabetically based on their text.
bool actionLessThan( const QAction* a, const QAction* b ) {
  Q_ASSERT( a );
  Q_ASSERT( b );
  return a->text() < b->text();
}



/*!
 * \class GestureBrowser
 *
 * \brief The GestureBrowser class provides a graphical interface for browsing,
 *        adding, removing, modifying and in general organising gestures.
 */


// TODO: re-factorize this method!
GestureBrowser::GestureBrowser( DigestDbController* digestDbController,
				SettingsController* settingsController,
				QWidget* parent, Qt::WindowFlags flags )
  : GuiDbComponentMainWindow(digestDbController, parent, flags),
    m_settingsController(settingsController),
    m_categoryModel(0),
    m_categorySelectionModel(0),
    m_gestureModel(0),
    m_gestureSelectionModel(0),
    m_currentGestureMiscModified(false),
    m_gestureSelected(false),
    m_gestureEditing(false),
    m_addGestureAction(0),
    m_searchWidget(0),
    m_searchMenu(0),
    m_addRemoveClassesMenu(0),
    m_addRemoveCollectionsMenu(0),
    m_newClassAction(0),
    m_noClassesPseudoAction(0),
    m_newCollectionAction(),
    m_noCollectionsPseudoAction(0),
    m_classesActionGroup(0),
    m_collectionsActionGroup(0)
{
  /*
   * WARNING: The use of the query models is delayed until we receive VEvent::Reset.
   */

  m_dummyModel = new QStandardItemModel( this );

  m_ui.setupUi( this );

  initToolBar();

  // TODO: remove the need for this!
  // Set background to white without affecting other palette values (e.g. making text boxes grey)
  Q_ASSERT( m_ui.gestureSubFrame );
  QPalette pal( m_ui.gestureSubFrame->palette() );
  pal.setColor( QPalette::Window, Qt::white );
  pal.setColor( QPalette::Base, Qt::white );
  m_ui.gestureSubFrame->setPalette( pal );
  // TODO: remove the need for this!
  Q_ASSERT( m_ui.strokesEditorFrame );
  pal.setColor( QPalette::Foreground, Qt::gray );
  m_ui.strokesEditorFrame->setPalette( pal );

#ifdef Q_WS_MAC
  Q_ASSERT( m_ui.gestureIconZoomSlider );
  QMacStyle::setWidgetSizePolicy( m_ui.gestureIconZoomSlider, QMacStyle::SizeSmall );
#endif

  // TODO: remove me:
#ifdef Q_WS_MAC
  Q_ASSERT( m_ui.topSplitter );
  m_ui.topSplitter->setStyle( new MacStyle2() );
  for ( int i=0; i < m_ui.topSplitter->count(); ++i ) {
    Q_ASSERT( m_ui.topSplitter->handle(i) );
    m_ui.topSplitter->handle(i)->setStyle( new MacStyle2() );
  }

  Q_ASSERT( m_ui.categoryInfoSplitter );
  m_ui.categoryInfoSplitter->setStyle( new MacStyle2() );
  for ( int i=0; i < m_ui.categoryInfoSplitter->count(); ++i ) {
    Q_ASSERT( m_ui.categoryInfoSplitter->handle(i) );
    m_ui.categoryInfoSplitter->handle(i)->setStyle( new MacStyle2() );
  }
#endif

  QPalette greyLinePal = m_ui.greyLine1->palette();
  greyLinePal.setColor( QPalette::Foreground, greyLinePal.dark().color() );
  Q_ASSERT( m_ui.greyLine1 );
  Q_ASSERT( m_ui.greyLine2 );
  Q_ASSERT( m_ui.greyLine3 );
  m_ui.greyLine1->setPalette( greyLinePal );
  m_ui.greyLine2->setPalette( greyLinePal );
  m_ui.greyLine3->setPalette( greyLinePal );


  Q_ASSERT( m_ui.infoTextBrowser );
  m_ui.infoTextBrowser->setHeaderText( tr("Information") );


  /* Set the background icons for various UI buttons (icon() is set in Ui file).
   *
   * Note: By using a separate icon() and backgroundIcon(), we have more control
   *       over the look of the button. For example, with the +/- twin buttons,
   *       when disabled, the alpha of the icon() should be reduced, but
   *       backgroundIcon() should stay the same. Also, this representation
   *       allows for the re-use of icon() and backgroundIcon() pixmaps.
   *
   * Important: We set QIcon::Disabled, as to override Qt's ghosted default.
   */
  QIcon left30Icon( ":/images/TwinLeft30Normal.png" );
  left30Icon.addFile( ":/images/TwinLeft30Pressed.png", QSize(), QIcon::Active );
  left30Icon.addFile( ":/images/TwinLeft30Normal.png", QSize(), QIcon::Disabled );
  QIcon leftIcon( ":/images/TwinLeftNormal.png" );
  leftIcon.addFile( ":/images/TwinLeftPressed.png", QSize(), QIcon::Active );
  leftIcon.addFile( ":/images/TwinLeftNormal.png", QSize(), QIcon::Disabled );
  QIcon rightIcon( ":/images/TwinRightNormal.png" );
  rightIcon.addFile( ":/images/TwinRightPressed.png", QSize(), QIcon::Active );
  rightIcon.addFile( ":/images/TwinRightNormal.png", QSize(), QIcon::Disabled );
  QIcon flat30Icon( ":/images/FlatButton30Normal.png" );
  flat30Icon.addFile( ":/images/FlatButton30Pressed.png", QSize(), QIcon::Active );
  flat30Icon.addFile( ":/images/FlatButton30Normal.png", QSize(), QIcon::Disabled );
  QIcon flat50Icon( ":/images/FlatButton50Normal.png" );
  flat50Icon.addFile( ":/images/FlatButton50Pressed.png", QSize(), QIcon::Active );
  flat50Icon.addFile( ":/images/FlatButton50Normal.png", QSize(), QIcon::Disabled );
  QIcon flat60Icon( ":/images/FlatButton60Normal.png" );
  flat60Icon.addFile( ":/images/FlatButton60Pressed.png", QSize(), QIcon::Active );
  flat60Icon.addFile( ":/images/FlatButton60Normal.png", QSize(), QIcon::Disabled );
  Q_ASSERT( m_ui.addCategoryButton );
  Q_ASSERT( m_ui.removeCategoryButton );
  Q_ASSERT( m_ui.categoryActionButton );
  Q_ASSERT( m_ui.addGestureButton );
  Q_ASSERT( m_ui.removeGestureButton );
  Q_ASSERT( m_ui.addRemoveClassesButton );
  Q_ASSERT( m_ui.addRemoveCollectionsButton );
  Q_ASSERT( m_ui.playAndStopButton );
  Q_ASSERT( m_ui.clearButton );
  Q_ASSERT( m_ui.duplicateButton );
  Q_ASSERT( m_ui.editAndDoneButton );
  m_ui.addCategoryButton->setBackgroundIcon( left30Icon );
  m_ui.removeCategoryButton->setBackgroundIcon( rightIcon );
  m_ui.categoryActionButton->setBackgroundIcon( flat30Icon );
  m_ui.addGestureButton->setBackgroundIcon( leftIcon );
  m_ui.removeGestureButton->setBackgroundIcon( rightIcon );
  m_ui.addRemoveClassesButton->setBackgroundIcon( flat50Icon );
  m_ui.addRemoveCollectionsButton->setBackgroundIcon( flat50Icon );
  m_ui.playAndStopButton->setBackgroundIcon( flat60Icon );
  m_ui.clearButton->setBackgroundIcon( flat60Icon );
  m_ui.duplicateButton->setBackgroundIcon( flat60Icon );
  m_ui.editAndDoneButton->setBackgroundIcon( flat60Icon );

  connect( m_ui.removeCategoryButton,
	   SIGNAL(clicked()), SLOT(removeSelectedCategories()) );
  connect( m_ui.removeGestureButton,
	   SIGNAL(clicked()), SLOT(removeSelectedGestures()) );

  QFont menuFont = font();
  menuFont.setPointSize( 11 );

  Q_ASSERT( m_ui.addCategoryButton );
  QMenu* addCategoryButtonMenu = new QMenu( m_ui.addCategoryButton );
  addCategoryButtonMenu->setFont( menuFont );
  addCategoryButtonMenu->addAction( tr("New Class") + ELLIPSIS_STR,
				    this, SLOT(addNewClass()) );
  addCategoryButtonMenu->addAction( tr("New Collection") + ELLIPSIS_STR,
				    this, SLOT(addNewCollection()) );
  m_ui.addCategoryButton->setMenu( addCategoryButtonMenu );

  Q_ASSERT( m_ui.addRemoveClassesButton );
  m_addRemoveClassesMenu = new QMenu( m_ui.addRemoveClassesButton );
  m_addRemoveClassesMenu->setFont( menuFont );
  m_newClassAction = new QAction( tr("New Class") + ELLIPSIS_STR, this );
  m_noClassesPseudoAction = new QAction( tr("No Classes"), this );
  m_noClassesPseudoAction->setEnabled( false );
  connect( m_newClassAction, SIGNAL(triggered()), SLOT(addNewClass()) );
  m_addRemoveClassesMenu->addAction( m_newClassAction );
  m_addRemoveClassesMenu->addAction( m_noClassesPseudoAction );
  m_ui.addRemoveClassesButton->setMenu( m_addRemoveClassesMenu );

  Q_ASSERT( m_ui.categoryActionButton );
  QMenu* categoryActionButtonMenu = new QMenu( m_ui.categoryActionButton );
  categoryActionButtonMenu->setFont( menuFont );
  m_ui.categoryActionButton->setMenu( categoryActionButtonMenu );
  m_splitCollectionAction = new QAction( tr("Split Collection") + ELLIPSIS_STR,
					 categoryActionButtonMenu );
  categoryActionButtonMenu->addAction( m_splitCollectionAction );
  connect(m_splitCollectionAction, SIGNAL(triggered()), SLOT(splitCollection()));

  Q_ASSERT( m_ui.addRemoveCollectionsButton );
  m_addRemoveCollectionsMenu = new QMenu( m_ui.addRemoveCollectionsButton );
  m_addRemoveCollectionsMenu->setFont( menuFont );
  m_newCollectionAction = new QAction( tr("New Collection") + ELLIPSIS_STR, this );
  m_noCollectionsPseudoAction = new QAction( tr("No Collections"), this );
  m_noCollectionsPseudoAction->setEnabled( false );
  connect( m_newCollectionAction, SIGNAL(triggered()), SLOT(addNewCollection()) );
  m_addRemoveCollectionsMenu->addAction( m_noCollectionsPseudoAction );
  m_ui.addRemoveCollectionsButton->setMenu( m_addRemoveCollectionsMenu );

  m_editPen = QPen( Qt::black, 2.0, Qt::SolidLine, Qt::RoundCap, Qt::RoundJoin );
  m_viewPen = QPen( Qt::darkGray, 2.0, Qt::SolidLine, Qt::RoundCap, Qt::RoundJoin );
  m_editBackgroundBrush = QBrush( Qt::white );
  m_viewBackgroundBrush = QBrush( QColor(240,240,240) );

  // Set the gesture editor to read only mode
  Q_ASSERT( m_ui.strokesEditor );
  m_ui.strokesEditor->setReadOnly( true );
  m_ui.strokesEditor->setPen( m_viewPen );                         // make mode obvious
  //TODO: remove me: m_ui.strokesEditor->setBackgroundBrush( m_viewBackgroundBrush ); // make mode obvious
  connect( m_ui.strokesEditor,
	   SIGNAL(animationStarted()), SLOT(animationStateChanged()) );
  connect( m_ui.strokesEditor,
	   SIGNAL(animationStopped()), SLOT(animationStateChanged()) );


  // Category model
  // Note: we don't pass in the controller, we're using addChildView()
  m_categoryModel = new CategoryItemModel( this ); // see note above re: controller
  addChildView( m_categoryModel );
  m_categoryModel->setModelFlags( CategoryItemModel::ItemsAreSelectable
				  | CategoryItemModel::ItemsAreEditable
				  | CategoryItemModel::ItemsAreDropEnabled );
  m_categorySelectionModel = new QItemSelectionModel( m_categoryModel );
  connect( m_categorySelectionModel,
	   SIGNAL(selectionChanged(const QItemSelection&, const QItemSelection&)),
	   SLOT(onCategorySelectionChanged()) );

  // Category view
  // Warning: Don't enable the edit trigger QAbstractItemView::EditKeyPressed,
  //          as it tended to trigger accidentally VERY easily.
  Q_ASSERT( m_ui.categoryTreeView );
  Q_ASSERT( m_ui.categoryTreeView->header() );
  m_ui.categoryTreeView->header()->hide();
  //m_ui.categoryTreeView->setHeaderText( tr("Category") );
  m_ui.categoryTreeView->setSelectionMode( QAbstractItemView::ExtendedSelection ); 
  m_ui.categoryTreeView->setAcceptDrops( true );
  // Can select multiple categories -> showing gesture of all those categories:
  m_ui.categoryTreeView->setEditTriggers( QAbstractItemView::DoubleClicked );
  m_ui.categoryTreeView->setHorizontalScrollBarPolicy( Qt::ScrollBarAlwaysOff ); // TODO: put in Designer
  m_ui.categoryTreeView->installEventFilter( this ); // catch delete key events


  // Gesture model
  // Note: we don't pass in the controller, we're using addChildView()
  m_gestureModel = new GestureQueryModel( this ); // see note above re: controller
  addChildView( m_gestureModel );
  m_gestureModel->setQuery( "SELECT " GESTURE_TABLE_ID_STR // ID is all that we need :-)
			    " FROM " GESTURE_TABLE_STR );
  m_gestureSelectionModel = new QItemSelectionModel( m_gestureModel );
  connect( m_gestureModel,
	   SIGNAL(thumbnailSizeChanged(const QSize&)),
	   SLOT(onGestureModelThumbnailSizeChanged(const QSize&)) );
  connect( m_gestureSelectionModel,
	   SIGNAL(selectionChanged(const QItemSelection&, const QItemSelection&)),
	   SLOT(onGestureSelectionChanged()) );


  // Gesture view
  // Note: These settings are so important, that we don't want to leave them in
  // the .ui file, as they may accidently be modified or reset without noticing.
  Q_ASSERT( m_ui.gestureModelView );
  m_ui.gestureModelView->setModel( m_gestureModel );
  m_ui.gestureModelView->setModelColumn( GESTURE_TABLE_ID_INDEX );
  m_ui.gestureModelView->setSelectionModel( m_gestureSelectionModel );
  m_ui.gestureModelView->setSelectionMode( QAbstractItemView::ExtendedSelection );
  m_ui.gestureModelView->setViewMode( QListView::IconMode ); // affects other settings -> call this 1st
  m_ui.gestureModelView->setIconSize( m_gestureModel->thumbnailSize() );
  m_ui.gestureModelView->setSpacing( 4 );
  m_ui.gestureModelView->setLayoutMode( QListView::SinglePass ); // scrolls better than Batched
  m_ui.gestureModelView->setResizeMode( QListView::Adjust ); // instant update
  m_ui.gestureModelView->setEditTriggers( QAbstractItemView::NoEditTriggers ); // edit elsewhere
  m_ui.gestureModelView->setUniformItemSizes( true ); // enable optimizations
  m_ui.gestureModelView->setMovement( QListView::Snap ); // can't use static - need drag'n'drop
  m_ui.gestureModelView->setDragEnabled( true ); // allow dragging to the category view
  m_ui.gestureModelView->setAcceptDrops( false );
  m_ui.gestureModelView->installEventFilter( this ); // catch delete key events

  if ( m_settingsController ) {
    m_settingsController->bind( m_ui.strokesEditor, "recordHiRes",
				SIGNAL(recordHiResToggled(bool)),
				DigestSettings::inputRecordHiResKey );
    m_settingsController->bind( m_ui.strokesEditor, "multiStrokeTimeout",
				SIGNAL(multiStrokeTimeoutChanged(double)),
				DigestSettings::inputMultiStrokeTimeoutKey );
    m_settingsController->bind( m_gestureModel, "thumbnailSize",
				SIGNAL(thumbnailSizeChanged(const QSize&)),
				DigestSettings::gestureBrowserThumbnailSizeKey );

    if ( m_settingsController->settingsModel() ) {
      Q_ASSERT( m_ui.topSplitter );
      SettingsModel* m = m_settingsController->settingsModel();
      m_ui.topSplitter->restoreState
	( m->value(DigestSettings::gestureBrowserTopSplitterStateKey).toByteArray() );
    }
  }
}


/*!
 * Calls submitEverything() and finalises settings.
 */
GestureBrowser::~GestureBrowser()
{
  submitEverything();

  if ( m_settingsController ) {
    Q_ASSERT( m_ui.topSplitter );
    QApplication::postEvent( m_settingsController,
			     new CSettingsChangeValueEvent
			     (DigestSettings::gestureBrowserTopSplitterStateKey,
			      m_ui.topSplitter->saveState(), this) );
  }
}


/*!
 * Re-implemented to allow the use of the Delete key (and also the backspace
 * on the Mac) to be used to delete the selected categories or gestures,
 * depending on which widget (categoryTreeView or gestureModelView) is in focus.
 */
bool GestureBrowser::eventFilter( QObject* watched, QEvent* event )
{
  Q_ASSERT( watched );
  Q_ASSERT( event );

  if ( event->type() == QEvent::KeyPress )
    {
      QKeyEvent* keyEvent = static_cast<QKeyEvent*>( event );
      if ( keyEvent->key() == Qt::Key_Delete // forward delete
#ifdef Q_WS_MAC // some macs don't have forward-delete, allow use of back-space:
	   || keyEvent->key() == Qt::Key_Backspace
#endif
	   )
	{
	  if ( watched == m_ui.categoryTreeView ) {
	    removeSelectedCategories();
	    return true;
	  }
	  else if ( watched == m_ui.gestureModelView ) {
	    removeSelectedGestures();
	    return true;
	  }
	}
    }

  // Pass the event on to the parent class
  return GuiDbComponentMainWindow::eventFilter( watched, event );
}


/*!
 * Re-syncs the GUI with the database if digestDbModel() is non-null, as it may
 * have changed for some unknown reason without sending update events to us.
 *
 * Users tend to expect a refresh if a window is hidden and then shown again.
 *
 * It then then passes event onto GuiDbComponentMainWindow.
 */
void GestureBrowser::showEvent( QShowEvent* event )
{
  // Update GUI to reflect the models
  if ( digestDbModel() ) {
    onCategorySelectionChanged();
    setEditCurrentGesture( false ); // ensure that we're not in edit mode
    onGestureSelectionChanged();       // called last, in case of "No Selection"
  }

  GuiDbComponentMainWindow::showEvent( event );
}


/*!
 * Calls submitEverything() and then passes event onto GuiDbComponentMainWindow.
 */
void GestureBrowser::hideEvent( QHideEvent* event )
{
  submitEverything();
  GuiDbComponentMainWindow::hideEvent( event );
}



// TODO: FIXME: submitCurrentGesture isn't being called when it should when this method is called by the destructor
void GestureBrowser::submitEverything()
{
  submitCurrentGesture();
}



void GestureBrowser::initToolBar()
{
  Q_ASSERT( m_ui.toolBar );

  QWidget* spacer1 = new QWidget( m_ui.toolBar );
  spacer1->setSizePolicy( QSizePolicy::Expanding, QSizePolicy::Fixed );
  m_ui.toolBar->addWidget( spacer1 );

  // Create "Search" widget
  m_searchWidget = new SearchWidget( m_ui.toolBar );
  m_searchWidget->setMinimumWidth( 200 );
  m_searchWidget->setMaximumWidth( 300 );
  m_ui.toolBar->addWidget( m_searchWidget );
  connect( m_searchWidget,
	   SIGNAL(beginSearch(const QString&)), SLOT(beginSearch(const QString&)) );
  connect( m_searchWidget,
	   SIGNAL(endSearch()), SLOT(endSearch()) );

  // Create the menu for the "Search" widget
  m_searchMenu = new QMenu( m_ui.toolBar );
  QActionGroup* m_searchTypeActionGroup = new QActionGroup( this );
  m_searchTypeActionGroup->setExclusive( true );
  QAction* fakeAction1  = new QAction( tr("Search"),            m_searchTypeActionGroup );
  m_searchAllAction     = new QAction( "   " + tr("All"),       m_searchTypeActionGroup );
  m_searchIdsAction     = new QAction( "   " + tr("IDs"),       m_searchTypeActionGroup );
  m_searchLabelsAction  = new QAction( "   " + tr("Labels"),    m_searchTypeActionGroup );
  m_searchDatesAction   = new QAction( "   " + tr("Dates"),     m_searchTypeActionGroup );
  m_searchNotesAction   = new QAction( "   " + tr("Notes"),     m_searchTypeActionGroup );
  m_searchClassesAction = new QAction( "   " + tr("Classes"),   m_searchTypeActionGroup );
  m_searchCollectionsAction = new QAction( "   " + tr("Collections"),   m_searchTypeActionGroup );
  QAction* fakeAction2  = new QAction(                          m_searchTypeActionGroup );
  m_sqlWhereAction      = new QAction( "   " + tr("SQL Where"), m_searchTypeActionGroup );
  fakeAction1->setEnabled( false );
  m_searchAllAction->setCheckable( true );
  m_searchIdsAction->setCheckable( true );
  m_searchLabelsAction->setCheckable( true );
  m_searchDatesAction->setCheckable( true );
  m_searchNotesAction->setCheckable( true );
  m_searchClassesAction->setCheckable( true );
  m_searchCollectionsAction->setCheckable( true );
  fakeAction2->setSeparator( true );
  m_sqlWhereAction->setCheckable( true );
  m_searchMenu->addActions( m_searchTypeActionGroup->actions() );
  QFont f = m_searchMenu->font();
  f.setPointSize( 11 );
  m_searchMenu->setFont( f );
  m_searchWidget->setMenu( m_searchMenu );
  m_searchAllAction->setChecked( true ); // "All" is the default - TODO: make this QSettings dictated!

  // Add an end spacer to the toolbar
  QWidget* spacer2 = new QWidget( m_ui.toolBar );
  spacer2->setMinimumWidth( 6 );
  spacer2->setMaximumWidth( 6 );
  spacer2->setSizePolicy( QSizePolicy::Fixed, QSizePolicy::Fixed );
  m_ui.toolBar->addWidget( spacer2 );
}


void GestureBrowser::resetEvent( VEvent* )
{
  /*
   * Clean-up
   */
  Q_ASSERT( m_ui.gestureModelView );
  m_ui.gestureModelView->setModel( m_dummyModel ); // can't be null :-(
  m_ui.gestureModelView->reset();

  Q_ASSERT( m_ui.categoryTreeView );
  m_ui.categoryTreeView->setModel( m_dummyModel ); // can't be null :-(
  m_ui.categoryTreeView->reset();


  /*
   * (Re)build
   */
  if ( digestDbModel() )
    {
      Q_ASSERT( m_ui.categoryTreeView );
      m_ui.categoryTreeView->setModel( m_categoryModel );
      m_ui.categoryTreeView->setSelectionModel( m_categorySelectionModel );

      Q_ASSERT( m_ui.gestureModelView );
      m_gestureModel->setQuery( "SELECT " GESTURE_TABLE_ID_STR // ID is all that we need :-)
				" FROM " GESTURE_TABLE_STR );
      m_ui.gestureModelView->setModel( m_gestureModel );
      m_ui.gestureModelView->setModelColumn( GESTURE_TABLE_ID_INDEX );
      m_ui.gestureModelView->setSelectionModel( m_gestureSelectionModel );

      // Re-sync the GUI
      onCategorySelectionChanged();
      setEditCurrentGesture( false ); // ensure that we're not in edit mode
      onGestureSelectionChanged();       // called last, in case of "No Selection"
    }
}


void GestureBrowser::classesEvent( VClassesEvent* event )
{
  Q_ASSERT( event );

  VDigestDbEvent::Type eventType = (VDigestDbEvent::Type)event->type();

  if ( ! m_gestureEditing )
    {
      // Update the information (e.g. gestureFieldsHtmlWidget) for any selected
      // gesture(s) if a class was removed or updated.
      if ( eventType == VDigestDbEvent::ClassesRemoved
	   || eventType == VDigestDbEvent::ClassesUpdated )
	onGestureSelectionChanged();
    }
  else
    {
      bool updateClassesWidget = false;

      switch ( eventType )
	{
	case VDigestDbEvent::ClassesAdded:
	  // User chose to add a class while editing a gesture
	  // -> they most likely want add it to the gesture then, so do it for them!
	  // HOWEVER: Don't respond to adds by others, such as the ExperimentBrowser.
	  if ( event->originalSender() == this ) {
	    m_currentGesture.classes += event->idSet(); // auto avoids duplicates
	    m_currentGestureMiscModified = true;
	    updateClassesWidget = true;
	  }
	  break;

	case VDigestDbEvent::ClassesRemoved:
	  // TODO: refresh gestures model if there is a match between the removed & selected classes
	  // Note: we need to perform this regardless of who sent it.
	  m_currentGesture.classes -= event->idSet(); // if it has it -> remove it!
	  updateClassesWidget = true;
	  break;

	case VDigestDbEvent::ClassesUpdated:
	  updateClassesWidget = true;
	  break;

	default:
	  break;
	}

      // Update the GUI
      if ( updateClassesWidget )
	{
	  Q_ASSERT( digestDbModel() );
	  Q_ASSERT( m_ui.gestureClassesWidget );
	  m_ui.gestureClassesWidget
	    ->setText( digestDbModel()->classesToString(m_currentGesture.classes) );
	}

      // Update the menu, since a class has been either added, removed or updated.
      // IMPORTANT: also updates check marks after m_currentGesture.classes.insert()
      refreshAddRemoveClassesMenu();
    }

  onCategorySelectionChanged(); // classes affect source list which affects gesture list
}


void GestureBrowser::collectionsEvent( VCollectionsEvent* event )
{
  Q_ASSERT( event );

  VDigestDbEvent::Type eventType = (VDigestDbEvent::Type)event->type();

  if ( ! m_gestureEditing )
    {
      // Update the information (e.g. gestureFieldsHtmlWidget) for any selected
      // gesture(s) if a collection was removed or updated.
      if ( eventType == VDigestDbEvent::CollectionsRemoved
	   || eventType == VDigestDbEvent::CollectionsUpdated )
	onGestureSelectionChanged();
    }
  else
    {
      bool updateCollectionsWidget = false;

      switch ( eventType )
	{
	case VDigestDbEvent::CollectionsAdded:
	  // User chose to add a collection while editing a gesture
	  // -> they most likely want add it to the gesture then, so do it for them!
	  // HOWEVER: Don't respond to adds by others, such as the CollectionBrowser.
	  if ( event->originalSender() == this ) {
	    m_currentGesture.collections += event->idSet(); // auto avoids duplicates
	    m_currentGestureMiscModified = true;
	    updateCollectionsWidget = true;
	  }
	  break;

	case VDigestDbEvent::CollectionsRemoved:
	  // TODO: refresh gestures model if there is a match between the removed & selected collections
	  // Note: we need to perform this regardless of who sent it.
	  m_currentGesture.collections -= event->idSet(); // if it has it -> remove it!
	  updateCollectionsWidget = true;
	  break;

	case VDigestDbEvent::CollectionsUpdated:
	  updateCollectionsWidget = true;
	  break;
	
	default:
	  break;
	}

      // Update the GUI
      if ( updateCollectionsWidget )
	{
	  Q_ASSERT( digestDbModel() );
	  Q_ASSERT( m_ui.gestureCollectionsWidget );
	  m_ui.gestureCollectionsWidget
	    ->setText( digestDbModel()->collectionsToString(m_currentGesture.collections) );
	}

      // Update the menu, since a collection has been either added, removed or updated.
      // IMPORTANT: also updates check marks after m_currentGesture.collections.insert()
      refreshAddRemoveCollectionsMenu();
    }

  onCategorySelectionChanged(); // classes affect source list which affects gesture list
}


void GestureBrowser::gesturesEvent( VGesturesEvent* event )
{
  // TODO: cleanup
  
  Q_ASSERT( event );
  Q_ASSERT( m_gestureSelectionModel );
  Q_ASSERT( m_ui.gestureModelView );

  VDigestDbEvent::Type eventType = (VDigestDbEvent::Type)event->type();

  Q_ASSERT( m_gestureModel );

  if ( eventType == VDigestDbEvent::GesturesUpdated )
    {
      // Change the selection to only those which were updated
      // Note: There is no need to re-execute the query in this case
      //       - which is what onCategorySelectionChanged() does.
      m_ui.gestureModelView->clearSelection();
      QModelIndex firstIndex;
      foreach ( int id, event->idSet() ) {
	QModelIndex index = m_gestureModel->findIndexOfGesture( id );
	if ( index.isValid() ) { // check, as the gesture may not be in the current query
	  if ( !firstIndex.isValid() ) firstIndex = index;
	  m_gestureSelectionModel->select( index, QItemSelectionModel::Select );
	}
      }
      if ( firstIndex.isValid() ) // scroll to first updated:
	m_ui.gestureModelView->scrollTo( firstIndex );
    }
  else // GesturesAdded, GesturesChangedClasses and GesturesChangedCollections:
    onCategorySelectionChanged();  // CATEGORY selection changed

  onGestureSelectionChanged();     // GESTURE selection changed
  

  if ( eventType == VDigestDbEvent::GesturesAdded )
    {
      // Now that onCategorySelectionChanged() has been called,
      // we can try findIndexOfGesture on the new gestures.

      // Change the selection to only those which were added
      m_ui.gestureModelView->clearSelection();
      QModelIndex firstIndex;
      foreach ( int id, event->idSet() ) {
	QModelIndex index = m_gestureModel->findIndexOfGesture( id );
	if ( index.isValid() ) {
	  if ( !firstIndex.isValid() ) firstIndex = index;
	  m_gestureSelectionModel->select( index, QItemSelectionModel::Select );
	}
      }
      if ( firstIndex.isValid() ) // scroll to first updated:
	m_ui.gestureModelView->scrollTo( firstIndex );

      // If only one gesture was added, then try to open it for editing
      if ( event->idSet().size() == 1 ) {
	int id = * event->idSet().begin(); // hacky
	if ( setCurrentGesture(id) )
	  setEditCurrentGesture( true );
      }
    }
}


void GestureBrowser::execMessage( const QString& message )
{
  // FIXME: this is hacky!
  // TODO: cleanup!
  // TODO: have fields auto extracted elsewhere

  Q_ASSERT( m_searchWidget );

  bool searchAffected = false;
  SearchWidget::SearchTriggers old_t = m_searchWidget->searchTriggers();
  m_searchWidget->setSearchTriggers( SearchWidget::NoTriggers );

  QStringList tokens = message.split( ":", QString::SkipEmptyParts );
  foreach ( const QString& token, tokens )
    {
      QStringList pairs = token.split( "=", QString::SkipEmptyParts );
      if ( pairs.size() != 2 ) continue;
      QString key   = pairs.value(0).trimmed().toLower(); 
      QString value = pairs.value(1).trimmed().toLower();
      if ( value.left(1) == "\"" ) value.remove(0, 1);
      if ( value.right(1) == "\"" ) value.chop(1);

      if ( key == "searchtype" )
	{
	  Q_ASSERT( m_sqlWhereAction );
	  if ( value == "sqlwhere" )
	    m_sqlWhereAction->trigger();
	  searchAffected = true;
	}
      else if ( key == "search" )
	{
	  m_searchWidget->setText( value, false );
	  searchAffected = true;
	}
    }

  // Only begin search once all search related properties have been set
  // - not after each change
  m_searchWidget->setSearchTriggers( old_t );
  if ( searchAffected )
    m_searchWidget->search();
}


void GestureBrowser::on_actionTestPad_triggered() {
  // don't use RecogniserTestPad::key() - reduce coupling and dependancies
  emit request( "show RecogniserTestPad" );
}

void GestureBrowser::on_actionRecognisers_triggered() {
  // don't use RecogniserBrowser::key() - reduce coupling and dependancies
  emit request( "show RecogniserBrowser" );
}

void GestureBrowser::on_actionExperiments_triggered() {
  // don't use ExperimentBrowser::key() - reduce coupling and dependancies
  emit request( "show ExperimentBrowser" );
}

void GestureBrowser::on_actionDatabase_triggered() {
  // don't use SqlBrowser::key() - reduce coupling and dependancies
  emit request( "show SqlBrowser" );
}


void GestureBrowser::removeSelectedCategories()
{
  // TODO: ask user if they want to actually delete the gestures or just the association
  Q_ASSERT( m_categoryModel );
  Q_ASSERT( m_categorySelectionModel );
  IdSet classes;
  IdSet collections;
  foreach ( const QModelIndex& index, m_categorySelectionModel->selectedIndexes() ) {
    // Note: You can't delete the "Library"
    CategoryItemModel::IndexType type = m_categoryModel->indexType(index);
    int id = m_categoryModel->indexDatabaseId( index );
    if ( type == CategoryItemModel::ClassIndex ) classes += id;
    else if ( type == CategoryItemModel::CollectionIndex ) collections += id;
  }
  if ( !classes.isEmpty() )
    postControllerEvent( new CClassesRemoveEvent(classes) );
  if ( !collections.isEmpty() )
    postControllerEvent( new CCollectionsRemoveEvent(collections) );
}


/*!
 * Adds a new gesture to the database.
 *
 * The gesture will be added to any classes or collections that are currently
 * selected in the category tree view.
 */
void GestureBrowser::on_addGestureButton_clicked()
{
  DGestureRecord g;
  g.label = "Gesture"; // TODO: generate unique name!
  g.date = QDate::currentDate();

  /* VERY IMPORTANT:
   * We need to clear the gesture selection NOW, before the new gesture is added.
   * Otherwise, when we're notified that the gesture has been added and then
   * proceed to select it, the current selection would be cleared, which closes
   * any editing, which would post an update. The problem now is that the updated
   * gesture would be selected, not the new one.
   * Hence, we clear the selection first, which causes all updates to occur before
   * the new gesture is added, thus, new gestures will be selected and opened for
   * editing.
   */
  Q_ASSERT( m_ui.gestureModelView );
  m_ui.gestureModelView->clearSelection();
  Q_ASSERT( !m_gestureEditing );
  Q_ASSERT( !m_gestureSelected );

  // Get all of the selected sources
  Q_ASSERT( m_categorySelectionModel );
  Q_ASSERT( g.classes.isEmpty() );     // assumed true by the following
  Q_ASSERT( g.collections.isEmpty() ); // ditto.
  foreach ( const QModelIndex& index, m_categorySelectionModel->selectedIndexes() ) {
    CategoryItemModel::IndexType type = m_categoryModel->indexType(index);
    if ( type == CategoryItemModel::ClassIndex )
      g.classes += m_categoryModel->indexDatabaseId(index);
    else if ( type == CategoryItemModel::CollectionIndex )
      g.collections += m_categoryModel->indexDatabaseId(index);
  }

  postControllerEvent( new CGestureAddEvent(g, this) );
}


void GestureBrowser::removeSelectedGestures()
{
  bool   librarySelected = false;
  IdSet  classes;
  IdSet  collections;
  IdSet  gestures;

  // TODO: add a "Do not ask me again" checkbox
  QMessageBox msg( QString(),
		   QString(),
		   QMessageBox::Question,
		   QMessageBox::Cancel | QMessageBox::Escape,
		   QMessageBox::Ok | QMessageBox::Default,
		   QMessageBox::NoButton,
		   this );
  msg.setButtonText( QMessageBox::Cancel, tr("&Cancel") );
  msg.setButtonText( QMessageBox::Ok, tr("&Remove") );

  

  // Get all of the selected categories
  Q_ASSERT( m_categoryModel );
  Q_ASSERT( m_categorySelectionModel );
  foreach ( const QModelIndex& index,
	    m_categorySelectionModel->selectedIndexes() ) {
    CategoryItemModel::IndexType type = m_categoryModel->indexType(index);
    if ( type == CategoryItemModel::LibraryIndex )
      librarySelected = true;
    else if ( type == CategoryItemModel::ClassIndex )
      classes += m_categoryModel->indexDatabaseId(index);
    else if ( type == CategoryItemModel::CollectionIndex )
      collections += m_categoryModel->indexDatabaseId(index);
  }

  // Get all of the selected gestures
  Q_ASSERT( m_gestureModel );
  Q_ASSERT( m_gestureSelectionModel );
  foreach ( const QModelIndex& index,
	    m_gestureSelectionModel->selectedIndexes() ) {
    bool intOk = false;
    if ( index.isValid() ) {
      gestures += m_gestureModel->record(index.row()).value(0).toInt(&intOk);
      Q_ASSERT( intOk );
    }
  }

  if ( !librarySelected
       && (!classes.isEmpty()
	   || !collections.isEmpty()) )
    {
      // 1st check - only classes or collections selected (not library at all)
      //           - offer to just remove the gestures from these categories

      msg.setText( tr("Are you sure you want to remove the selected "
		      "gestures from\n"
		      "the chosen categories?") );
      msg.setMinimumSize( 490, 120 );
      msg.setMaximumSize( 490, 120 );
      if ( msg.exec() == QMessageBox::Cancel ) return;

      if ( !gestures.isEmpty() ) {
	postControllerEvent( new CGesturesChangeClassesEvent
			     (gestures, IdSet(), classes, this) );
	postControllerEvent( new CGesturesChangeCollectionsEvent
			     (gestures, IdSet(), collections, this) );
      }
    }
  else if ( librarySelected )
    {
      // 2nd check - library + one or more classes or collections selected
      //           - offer to remove the gestures from the library

      msg.setText( tr("<h4>Are you sure you want to remove the selected "
		      "gestures from the library?</h4>\n\n"
		      "<span style=\"font-size:11pt;\">"
		      "The gestures will be removed from the database and any "
		      "classes or collections that refer to them will be updated."
		      "</span>") );
      msg.setMinimumSize( 490, 160 );
      msg.setMaximumSize( 490, 160 );
      if ( msg.exec() == QMessageBox::Cancel ) return;

      if ( !gestures.isEmpty() )
	postControllerEvent( new CGesturesRemoveEvent(gestures, this) );
    }
}


void GestureBrowser::on_gestureIconZoomMin_clicked() {
  Q_ASSERT( m_ui.gestureIconZoomSlider );
  m_ui.gestureIconZoomSlider->setValue( m_ui.gestureIconZoomSlider->minimum() );
}

void GestureBrowser::on_gestureIconZoomMax_clicked() {
  Q_ASSERT( m_ui.gestureIconZoomSlider );
  m_ui.gestureIconZoomSlider->setValue( m_ui.gestureIconZoomSlider->maximum() );
}


/*!
 * The slider's value range is 0 to 56.
 *
 * To get the actual icon size, we first add 8 and then multiply by 2.
 * Hence, we get the icon sizes: 16x16, 18x18, ... 32x32, 34x34, ... 128x128.
 *
 * Note: This uses Qt's auto-connection feature.
 */
void GestureBrowser::on_gestureIconZoomSlider_valueChanged( int value )
{
  int w = (value + 8) * 2;
  Q_ASSERT( m_gestureModel );;
  m_gestureModel->setThumbnailSize( w, w );
}


void GestureBrowser::onGestureModelThumbnailSizeChanged( const QSize& size )
{
  Q_ASSERT( m_ui.gestureIconZoomSlider );
  Q_ASSERT( m_ui.gestureModelView );
  m_ui.gestureIconZoomSlider->blockSignals( true );
  m_ui.gestureIconZoomSlider->setValue( (size.width() / 2) - 8 );
  m_ui.gestureIconZoomSlider->blockSignals( false );
  m_ui.gestureModelView->setIconSize( size );
}


/*!
 * Note: This uses Qt's auto-connection feature.
 */
void GestureBrowser::on_playAndStopButton_clicked()
{
  // NOTE: animationStateChanged() looks after updating our UI
  Q_ASSERT( m_gestureSelected ); // sanity check
  Q_ASSERT( m_ui.strokesEditor );
  if ( m_ui.strokesEditor->isAnimating() )
     m_ui.strokesEditor->stopAnimation();
  else
     m_ui.strokesEditor->animate();
}


/*!
 * Note: This uses Qt's auto-connection feature.
 */
void GestureBrowser::on_clearButton_clicked()
{
  Q_ASSERT( m_ui.strokesEditor );
  m_ui.strokesEditor->setStrokes( StrokeList() );
}


/*!
 * \b Note: All fields of the current gesture will be duplicated, except for the
 *          date, which will be set to the current date.
 */
void GestureBrowser::on_duplicateButton_clicked()
{
  Q_ASSERT( m_gestureSelected ); // sanity check
  updateCurrentGestureFromUi();

  DGestureRecord g = m_currentGesture;
  //g.label = "Gesture"; // TODO: generate unique name!
  g.date = QDate::currentDate();

  /* VERY IMPORTANT:
   * We need to clear the gesture selection NOW, before the new gesture is added.
   * Otherwise, when we're notified that the gesture has been added and then
   * proceed to select it, the current selection would be cleared, which closes
   * any editing, which would post an update. The problem now is that the updated
   * gesture would be selected, not the new one.
   * Hence, we clear the selection first, which causes all updates to occur before
   * the new gesture is added, thus, new gestures will be selected and opened for
   * editing.
   *
   * Note: This is safe to call now, as the current gesture data has been copied.
   */
  m_ui.gestureModelView->clearSelection();
  Q_ASSERT( !m_gestureEditing );
  Q_ASSERT( !m_gestureSelected );

  postControllerEvent( new CGestureAddEvent(g, this) );
}


/*!
 * Note: This uses Qt's auto-connection feature.
 */
void GestureBrowser::on_editAndDoneButton_clicked()
{
  setEditCurrentGesture( !m_gestureEditing );
}


void GestureBrowser::animationStateChanged()
{
  Q_ASSERT( m_ui.strokesEditor );
  bool animating = m_ui.strokesEditor->isAnimating();

  Q_ASSERT( m_ui.playAndStopButton );
  m_ui.playAndStopButton->setText( animating ? tr("&Stop") : tr("&Play") );

  // Make the button accessible by pressing Enter (or Return) or Space
  m_ui.playAndStopButton->setFocus();

  // Prevent any editing during animation!
  Q_ASSERT( m_ui.duplicateButton );
  Q_ASSERT( m_ui.editAndDoneButton );
  m_ui.duplicateButton->setEnabled( !animating );
  m_ui.editAndDoneButton->setEnabled( !animating );
}



void GestureBrowser::onCategorySelectionChanged()
{
  Q_ASSERT( m_categoryModel );
  Q_ASSERT( m_categorySelectionModel );

  // TODO: move this hacking code elsewehere -> really should only be on a selection change
  if ( m_gestureEditing )
    setEditCurrentGesture( false ); // stop editing first

  QModelIndexList selectedIndexes = m_categorySelectionModel->selectedIndexes();

  // Determine if only the "Library" is selected
  bool onlyLibrarySelected
    = ( selectedIndexes.size() == 1
	&& (m_categoryModel->indexType(selectedIndexes.first())
	    == CategoryItemModel::LibraryIndex) );

  m_gestureQueryStr = m_categoryModel->gestureQueryForIndexes( selectedIndexes );

  Q_ASSERT( m_gestureModel );
  m_gestureModel->setQuery( m_gestureQueryStr );

  // Fetch ALL gestures - otherwise there are many issues to handle.
  while ( m_gestureModel->canFetchMore() )
    m_gestureModel->fetchMore();

  updateNumGesturesLabel();

  updateInfoTextBrowser();

  // Update removeCategoryButton - enabled IFF one or more collections are
  //                               AND !onlyLibrarySelected selected.
  Q_ASSERT( m_ui.removeCategoryButton );
  m_ui.removeCategoryButton->setEnabled( !onlyLibrarySelected
					 && !selectedIndexes.isEmpty() );

  // Update splitCollectionAction - enabled IFF a SINGLE COLLECTION is selected
  Q_ASSERT( m_splitCollectionAction );
  m_splitCollectionAction->setEnabled(selectedIndexes.size() == 1
				      && (m_categoryModel->indexType(selectedIndexes.first())
					  == CategoryItemModel::CollectionIndex) );

  // Update addGestureButton - enabled IFF one or more collections are selected
  Q_ASSERT( m_ui.addGestureButton );
  m_ui.addGestureButton->setEnabled( !selectedIndexes.isEmpty() );
}


// TODO: change this to something like on_gestureModel_selectionChanged
void GestureBrowser::onGestureSelectionChanged()
{
  Q_ASSERT( m_gestureModel );
  Q_ASSERT( m_gestureSelectionModel );

  QModelIndexList selectedIndexes = m_gestureSelectionModel->selectedIndexes();
  int id = -1; // init to invalid

  if ( !selectedIndexes.isEmpty()
       && selectedIndexes.first().isValid() ) { // proved to catch failures
    bool intOk = false;
    id = ( m_gestureModel->record(selectedIndexes.first().row())
	   .value(GESTURE_TABLE_ID_INDEX).toInt(&intOk) );
    Q_ASSERT(intOk);
  }

  setCurrentGesture( id );
}


bool GestureBrowser::setCurrentGesture( int id )
{
  // TODO: move this hacky code elsewehere -> really should only be on a selection change
  if ( m_gestureEditing )
    setEditCurrentGesture( false );

  Q_ASSERT( digestDbModel() );
  
  m_currentGesture = digestDbModel()->fetchGesture( id, &m_gestureSelected );

  // Show/hide widgets that depend on there being a gesture selected
  Q_ASSERT( m_ui.strokesEditorFrame );
  Q_ASSERT( m_ui.playAndStopButton );
  Q_ASSERT( m_ui.dateHtmlWidget );
  m_ui.strokesEditorFrame->setVisible( m_gestureSelected );
  m_ui.playAndStopButton->setVisible( m_gestureSelected );
  m_ui.dateHtmlWidget->setVisible( m_gestureSelected );

  // Enable/Disable widgets that depend on there being a gesture selected
  Q_ASSERT( m_ui.removeGestureButton );
  Q_ASSERT( m_ui.duplicateButton );
  Q_ASSERT( m_ui.editAndDoneButton );
  m_ui.removeGestureButton->setEnabled( m_gestureSelected );
  m_ui.duplicateButton->setEnabled( m_gestureSelected );
  m_ui.editAndDoneButton->setEnabled( m_gestureSelected );

  Q_ASSERT( m_ui.stackedFieldWidget );
  m_ui.stackedFieldWidget->setCurrentIndex( m_gestureEditing
					    ? STACKED_FIELD_WIDGET_EDIT_INDEX
					    : STACKED_FIELD_WIDGET_VIEW_INDEX );

  if ( m_gestureSelected )
    {
      // Update the gesture fields - reflect selected gesture
      Q_ASSERT( m_ui.gestureFieldsHtmlWidget );
      Q_ASSERT( m_ui.strokesEditor );
      Q_ASSERT( m_ui.dateHtmlWidget );
      m_ui.gestureFieldsHtmlWidget->setHtml( gestureFieldsAsHtml(m_currentGesture) );
      m_ui.strokesEditor->setStrokes( m_currentGesture.strokes );
      m_ui.dateHtmlWidget->setHtml( gestureDateAsHtml(m_currentGesture.date) );
    }
  else
    {
      m_currentGesture = DGestureRecord(); // clear data structure, invalidating it (i.e. ID == -1)
      Q_ASSERT( m_ui.gestureFieldsHtmlWidget );
      m_ui.gestureFieldsHtmlWidget->setHtml( "<html><body><table width=\"100%\">"
					     "<tr><td align=\"center\">"
					     "<font size=\"+1\" color=\"#AAAAAA\">"
					     "<br><br><br>"
					     "<strong>No Gesture Selected</strong>"
					     "</font>"
					     "</td></tr>"
					     "</table></body></html>" );
    }

  return m_gestureSelected;
}


bool GestureBrowser::setEditCurrentGesture( bool edit )
{
  Q_ASSERT( digestDbModel() );
  Q_ASSERT( m_ui.strokesEditor );
  
  bool wasEditing = m_gestureEditing;

  // If we WERE In editing mode, submit all modifications
  if ( wasEditing ) submitCurrentGesture();

  m_gestureEditing = edit;

  // If NOW in editing mode, update the widgets that are soley used for editing
  if ( m_gestureEditing )
    {      
      Q_ASSERT( m_ui.gestureIdLabel );
      Q_ASSERT( m_ui.gestureLabelEdit );
      Q_ASSERT( m_ui.gestureNotesEdit );
      Q_ASSERT( m_ui.gestureClassesWidget );
      Q_ASSERT( m_ui.gestureCollectionsWidget );
      m_ui.gestureIdLabel->setText( QString::number(m_currentGesture.id) );
      m_ui.gestureLabelEdit->setText( m_currentGesture.label );
      m_ui.gestureNotesEdit->setPlainText( m_currentGesture.notes );
      if ( m_ui.gestureNotesEdit->document() )
	m_ui.gestureNotesEdit->document()->setModified( false );
      m_ui.gestureClassesWidget
	->setText( digestDbModel()->classesToString(m_currentGesture.classes) );
      m_ui.gestureCollectionsWidget
	->setText( digestDbModel()->collectionsToString(m_currentGesture.collections) );

      // TODO: remove the need to call these here -> only on class add remove call them!
      refreshAddRemoveClassesMenu();
      refreshAddRemoveCollectionsMenu();
    }
  else
    {
      // Update the gesture fields - reflect selected gesture
      Q_ASSERT( m_ui.gestureFieldsHtmlWidget );
      Q_ASSERT( m_ui.dateHtmlWidget );
      m_ui.gestureFieldsHtmlWidget->setHtml( gestureFieldsAsHtml(m_currentGesture) );
      m_ui.dateHtmlWidget->setHtml( gestureDateAsHtml(m_currentGesture.date) );
    }

  Q_ASSERT( m_ui.editAndDoneButton );
  m_ui.editAndDoneButton->setText( m_gestureEditing ? tr("&Done") : tr("&Edit") );

  // Make the button accessible by pressing Enter (or Return) or Space
  m_ui.editAndDoneButton->setFocus();

  m_ui.strokesEditor->setReadOnly( ! m_gestureEditing );
  m_ui.strokesEditor->setPen( m_gestureEditing ? m_editPen : m_viewPen ); // make mode obvious

  Q_ASSERT( m_ui.stackedFieldWidget );
  m_ui.stackedFieldWidget->setCurrentIndex( m_gestureEditing
					     ? STACKED_FIELD_WIDGET_EDIT_INDEX
					     : STACKED_FIELD_WIDGET_VIEW_INDEX );

  // The clear button should only be VISIBLE while editing
  Q_ASSERT( m_ui.clearButton );
  m_ui.clearButton->setVisible( m_gestureEditing );

  // The play/stop button should only be ENABLED when NOT editing - reduces issues
  Q_ASSERT( m_ui.playAndStopButton );
  m_ui.playAndStopButton->setEnabled( !m_gestureEditing );

  Q_ASSERT( m_ui.dateHtmlWidget );
  Q_ASSERT( m_gestureSelectionModel );
  m_ui.dateHtmlWidget->setVisible( ! m_gestureEditing
				   && ! m_gestureSelectionModel->selection().isEmpty() );
    

  // Re-evaluate the query for the gesture model, as we've just modified a gesture.
  // The collections it belongs to may have changed, so this is necessary.
  // TODO: just call m_ui.gestureModelView->update(); is the classes havent changed
  if ( wasEditing ) {
    Q_ASSERT( m_gestureModel );
    m_gestureModel->setQuery( m_gestureQueryStr );
  }


  return m_gestureEditing;
}


void GestureBrowser::updateNumGesturesLabel()
{
  Q_ASSERT( m_ui.numGesturesLabel );
  int n = m_gestureModel->rowCount(); // top-level row count -> # items in list
  m_ui.numGesturesLabel->setText( (n==0 ? tr("No") : QString::number(n))
				+ tr(" gesture")
				+ (n==1 ? "" : "s") );
}


/*!
 * Update the info box with the labels of the classes that the gestures in the
 * gesture browser belong to. These gestures may have been chosen to be shown
 * via either the category list or by a "search" query.
 */
void GestureBrowser::updateInfoTextBrowser()
{
  Q_ASSERT( m_gestureModel );
  Q_ASSERT( m_ui.infoTextBrowser );
  QStringList classLabels;
  QString gestureIdsStr;
  int gestureCount = m_gestureModel->rowCount();
  gestureIdsStr.reserve( gestureCount * 5 ); // typical 4 digit ID + ","
  for ( int i=0; i < gestureCount; ++i )
    gestureIdsStr += QString::number(m_gestureModel->rowGestureId(i)) + ',';
  if ( gestureIdsStr.endsWith(',') ) gestureIdsStr.chop(1);
  gestureIdsStr.squeeze();
  QSqlQuery q( "select A2.label from "
	       "(select classid from classmap where gestureid in ("
	       + gestureIdsStr +
	       ") group by classid) A1 left join class A2 on A1.classid = A2.id",
	       database() );
  while ( q.next() ) classLabels += q.value(0).toString();
  classLabels.sort();
  m_ui.infoTextBrowser->setPlainText
    ( gestureCount == 0
      ? tr("No Gestures Shown")
      : (QString("The %1 to %2 class%3:\n\n%4")
	 .arg(gestureCount == 1 ? "gesture belongs" : "gestures belong")
	 .arg(classLabels.count())
	 .arg(classLabels.count() == 1 ? "" : "es")
	 .arg(classLabels.join("\n")))
      );
}



bool GestureBrowser::updateCurrentGestureFromUi()
{
  Q_ASSERT( m_ui.gestureLabelEdit );
  Q_ASSERT( m_ui.gestureNotesEdit );
  Q_ASSERT( m_ui.strokesEditor );

  // Return if nothing has been modified
  if ( !m_currentGestureMiscModified
       && !m_ui.strokesEditor->isModified()
       && !m_ui.gestureLabelEdit->isModified()
       && (!m_ui.gestureNotesEdit->document()
	   || !m_ui.gestureNotesEdit->document()->isModified()) ) // eval doc iff exists
    return false;


  // First, update the gesture object with the edit widgets
  // NOTE: classes and collections are updated interactively and directly
  // NOTE: don't change the date, as that's only set when it's created
  m_currentGesture.label = m_ui.gestureLabelEdit->text();
  m_currentGesture.notes = m_ui.gestureNotesEdit->toPlainText();
  if ( m_ui.strokesEditor->isModified() )  // only copy if we absolutely have to
    m_currentGesture.strokes = m_ui.strokesEditor->strokes();

  m_currentGestureMiscModified = false;

  return true;
}



/*!
 * Submits the current gesture to the database iff it is valid and has been
 * modified.
 */
void GestureBrowser::submitCurrentGesture()
{
  if ( updateCurrentGestureFromUi() )
    postControllerEvent( new CGestureUpdateEvent(m_currentGesture, this) );
}



/*!
 * 
 */
void GestureBrowser::beginSearch( const QString& text )
{
  // TODO: support more types of queries!
  // TODO: support multiple forms of dates (e.g. store format: 20050801, 01/08/2005, etc)
  QStringList searchFields;

  Q_ASSERT( m_searchAllAction );
  Q_ASSERT( m_searchIdsAction );
  Q_ASSERT( m_searchLabelsAction );
  Q_ASSERT( m_searchDatesAction );
  Q_ASSERT( m_searchNotesAction );
  Q_ASSERT( m_searchClassesAction );
  Q_ASSERT( m_sqlWhereAction );
  if ( m_searchAllAction->isChecked() )
    searchFields << "id" << "label" << "date" << "notes";
  else if ( m_searchIdsAction->isChecked() )
    searchFields << "id";
  else if ( m_searchLabelsAction->isChecked() )
    searchFields << "label";
  else if ( m_searchDatesAction->isChecked() )
    searchFields << "date";
  else if ( m_searchNotesAction->isChecked() )
    searchFields << "notes";
  else if ( m_searchClassesAction->isChecked() )
    {
      // TODO: write me!
    }
  else if ( m_sqlWhereAction->isChecked() ) // SQL WHERE
    {
      m_gestureQueryStr =
	"SELECT " GESTURE_TABLE_ID_STR
	" FROM " GESTURE_TABLE_STR
	" WHERE " + text;
    }

  // searchFields not empty -> search in ID, Label, Date or Note, or combo of them
  if ( ! searchFields.isEmpty() )
    {
      // TODO: see if we can optimise this!
      QStringList items = text.split( QRegExp("\\W+"), QString::SkipEmptyParts );
      m_gestureQueryStr =
	"SELECT " GESTURE_TABLE_ID_STR
	" FROM " GESTURE_TABLE_STR
	" WHERE ";
      QStringList conditions;
      foreach ( const QString& field, searchFields )
	conditions += ( field + " LIKE '%"
			+ items.join("%' OR "+ field + " LIKE '%") + "%'" ); // TODO: replace OR with AND & fix prob
      m_gestureQueryStr += conditions.join( " OR " );
    }

  Q_ASSERT( m_gestureModel );
  m_gestureModel->setQuery( m_gestureQueryStr );

  // Fetch ALL gestures - otherwise there are many issues to handle.
  while ( m_gestureModel->canFetchMore() )
    m_gestureModel->fetchMore();

  updateNumGesturesLabel();
  updateInfoTextBrowser();
}


void GestureBrowser::endSearch()
{
  // Re-evaluate the query based on the selected collections
  onCategorySelectionChanged();
}


void GestureBrowser::addNewClass()
{
  // TODO: remove close, max and min buttons from the Mac OS X version of dialog
  bool ok;
  DClassRecord record;
  record.label
    = QInputDialog::getText( this, tr("New Class"), tr("New class label:"),
			     QLineEdit::Normal, QString(), &ok
#ifdef Q_WS_MAC
                             , Qt::Sheet
#endif
                             );

  if ( !ok || record.label.isEmpty() )
    return;

  Q_ASSERT( digestDbModel() );

  if ( digestDbModel()->classLabelExists(record.label) ) {
    QMessageBox::information( 0, tr("Class label already exists"),
			      tr("A class already exists with the label ")
			      + "\"" + record.label + "\". "
			      + tr("A duplicate will not be added."),
			      QMessageBox::Ok );
    return;
  }

  postControllerEvent( new CClassAddEvent(record, this) );
}



void GestureBrowser::addNewCollection()
{
  // TODO: remove close, max and min buttons from the Mac OS X version of dialog
  bool ok;
  DCollectionRecord record;
  record.label
    = QInputDialog::getText( this, tr("New Collection"), tr("New collection label:"),
			     QLineEdit::Normal, QString(), &ok
#ifdef Q_WS_MAC
                             , Qt::Sheet
#endif
                             );
  
  if ( !ok || record.label.isEmpty() )
    return;

  Q_ASSERT( digestDbModel() );

  if ( digestDbModel()->collectionLabelExists(record.label) ) {
    QMessageBox::information( 0, tr("Collection label already exists"),
			      tr("A collection already exists with the label ")
			      + "\"" + record.label + "\". "
			      + tr("A duplicate will not be added."),
			      QMessageBox::Ok );
    return;
  }

  postControllerEvent( new CCollectionAddEvent(record, this) );
}


// TODO: allow anything to be split - event the library or a single class? - what about a selection?
/*!
 * Opens the "Split Collection" dialog, which allows the user to split the
 * currenty selected collection automatically into training and test sets.
 *
 * \b Note: This won't do anything unless exactly one collection is selected
 *          Thus the "Split Collection..." action, which calls this method,
 *          should only be enabled when this condition is satisfied.
 *          This condition is not asserted, it's always checked, as this is a
 *          high-level method and may be called by other code in the future.
 *
 * \b Note: This asserts that the category model and it's selection model are
 *          both non-null.
 */
void GestureBrowser::splitCollection()
{
  Q_ASSERT( m_categoryModel );
  Q_ASSERT( m_categorySelectionModel );
  QModelIndexList selectedIndexes = m_categorySelectionModel->selectedIndexes();
  if ( selectedIndexes.size() == 1
       && (m_categoryModel->indexType(selectedIndexes.first())
	   == CategoryItemModel::CollectionIndex) );
  {
    // TODO: show alt message if the collection contains no gestures
    int id = m_categoryModel->indexDatabaseId( selectedIndexes.first() );
    SplitCollectionDialog* dlg
      = new SplitCollectionDialog(id, digestDbController(), this
#ifdef Q_WS_MAC
				  ,Qt::Sheet
#endif
				  );
    dlg->setAttribute(Qt::WA_DeleteOnClose);
    dlg->setWindowModality(Qt::ApplicationModal);// TODO: Qt::WindowModal); when it can handle DB changes while open
    dlg->show();
  }
}


void GestureBrowser::onClassActionTriggered( QAction* action )
{
  if ( action == 0 ) return;

  bool intOk = false;
  int id = action->data().toInt(&intOk);
  Q_ASSERT( intOk );

  if ( action->isChecked() )
    m_currentGesture.classes.insert( id ); // auto avoids duplicates
  else
    m_currentGesture.classes.remove( id );

  m_currentGestureMiscModified = true;

  // Update the GUI
  Q_ASSERT( digestDbModel() );
  Q_ASSERT( m_ui.gestureClassesWidget );
  m_ui.gestureClassesWidget
	->setText( digestDbModel()->classesToString(m_currentGesture.classes) );
}


void GestureBrowser::onCollectionActionTriggered( QAction* action )
{
  if ( action == 0 ) return;

  bool intOk = false;
  int id = action->data().toInt(&intOk);
  Q_ASSERT( intOk );

  if ( action->isChecked() )
    m_currentGesture.collections.insert( id ); // auto avoids duplicates
  else
    m_currentGesture.collections.remove( id );

  m_currentGestureMiscModified = true;

  // Update the GUI
  Q_ASSERT( digestDbModel() );
  Q_ASSERT( m_ui.gestureCollectionsWidget );
  m_ui.gestureCollectionsWidget
    ->setText( digestDbModel()->collectionsToString(m_currentGesture.collections) );
}


void GestureBrowser::refreshAddRemoveClassesMenu()
{
  // TODO: optimise this - build it as we add/remove classes?
  // TODO: sort the classes!
  delete m_classesActionGroup; // owner of existing class actions -> delete them too :-)
  m_classesActionGroup = new QActionGroup( this );
  m_classesActionGroup->setExclusive( false );
  connect( m_classesActionGroup,
	   SIGNAL(triggered(QAction*)),
	   SLOT(onClassActionTriggered(QAction*)) );

  Q_ASSERT( digestDbModel() );
  QHashIterator<int, QString> it( digestDbModel()->fetchClasses() );
  while ( it.hasNext() ) {
    it.next();
    QAction* a = new QAction( it.value(), m_classesActionGroup );
    a->setCheckable( true );
    a->setChecked( m_currentGesture.classes.contains(it.key()) );
    a->setData( it.key() );
  }

  Q_ASSERT( m_addRemoveClassesMenu );
  m_addRemoveClassesMenu->clear();
  m_addRemoveClassesMenu->addAction( m_newClassAction );
  m_addRemoveClassesMenu->addSeparator();

  if ( m_classesActionGroup->actions().isEmpty() )
    m_addRemoveClassesMenu->addAction( m_noClassesPseudoAction );
  else {
    QList<QAction*> actions = m_classesActionGroup->actions();
    qSort( actions.begin(), actions.end(), actionLessThan );
    m_addRemoveClassesMenu->addActions( actions );
  }
}


void GestureBrowser::refreshAddRemoveCollectionsMenu()
{
  // TODO: optimise this - build it as we add/remove collections?
  // TODO: sort the collections!
  delete m_collectionsActionGroup; // owner of existing collection actions -> delete them too :-)
  m_collectionsActionGroup = new QActionGroup( this );
  m_collectionsActionGroup->setExclusive( false );
  connect( m_collectionsActionGroup,
	   SIGNAL(triggered(QAction*)),
	   SLOT(onCollectionActionTriggered(QAction*)) );

  Q_ASSERT( digestDbModel() );
  QHashIterator<int, QString> it( digestDbModel()->fetchCollections() );
  while ( it.hasNext() ) {
    it.next();
    QAction* a = new QAction( it.value(), m_collectionsActionGroup );
    a->setCheckable( true );
    a->setChecked( m_currentGesture.collections.contains(it.key()) );
    a->setData( it.key() );
  }

  Q_ASSERT( m_addRemoveCollectionsMenu );
  m_addRemoveCollectionsMenu->clear();
  m_addRemoveCollectionsMenu->addAction( m_newCollectionAction );
  m_addRemoveCollectionsMenu->addSeparator();

  if ( m_collectionsActionGroup->actions().isEmpty() )
    m_addRemoveCollectionsMenu->addAction( m_noCollectionsPseudoAction );
  else {
    QList<QAction*> actions = m_collectionsActionGroup->actions();
    qSort( actions.begin(), actions.end(), actionLessThan );
    m_addRemoveCollectionsMenu->addActions( actions );
  }
}


QString GestureBrowser::gestureFieldsAsHtml( const DGestureRecord& g ) const
{
  /* HTML table layout:
   *
   *          ID **
   *       Label *****************
   *       Notes *****************
   *     Classes *****************
   * Collections *****************
   *     Strokes **
   */

  Q_ASSERT( digestDbModel() );

  // NOTE: The DATE is shown elsewhere, lower down the dialog in dateHtmlWidget
  return ( "<html>"
	   "<body>"
	   "<table cellspacing=\"6\">"
	   "<tr><th align=\"right\"><font color=\"#888888\">ID</font></th><td>"
	   + QString::number(g.id) + "</td></tr>"
	   "<tr><th align=\"right\"><font color=\"#888888\">Label</font></th><td>"
	   + (g.label.isEmpty() ? tr("(none)") : g.label) + "</td></tr>"
	   "<tr><th align=\"right\"><font color=\"#888888\">Notes</font></th><td>"
	   + (g.notes.isEmpty() ? tr("(none)") : g.notes) + "</td></tr>"
	   "<tr><th align=\"right\"><font color=\"#888888\">Classes</font></th><td>"
	   + ( g.classes.isEmpty()
	       ? tr("(none)")
	       : digestDbModel()->classesToString(g.classes) ) + "</td></tr>"
	   "<tr><th align=\"right\"><font color=\"#888888\">Collections</font></th><td>"
	   + ( g.collections.isEmpty()
	       ? tr("(none)")
	       : digestDbModel()->collectionsToString(g.collections) ) + "</td></tr>"
	   "<tr><th align=\"right\"><font color=\"#888888\">Strokes</font></th><td>"
	   + (g.strokes.isEmpty() ? QString("0") : QString::number(g.strokes.size())) +
	   "</td></tr>"
	   "</table>"
	   "</body>"
	   "</html>" );
}


QString GestureBrowser::gestureDateAsHtml( const QDate& date ) const
{
  return ( "<html>"
	   "<body>"
	   "<p align=\"right\"><font color=\"#AAAAAA\"><strong>"
	   + tr("Created") +
	   ": "
	   + date.toString(Qt::LocalDate) +
	   "</strong></font></p>"
	   "</body>"
	   "</html>" );
}
